import React, { useState, useEffect } from 'react';
import { Users, Filter, Clock, CheckCircle, XCircle } from 'lucide-react';

interface Lead {
  id: number;
  property_id: number;
  // Basic Details
  buyer_name: string;
  buyer_phone: string;
  buyer_email?: string;
  
  // Lead Classification
  lead_type: string;
  requirement: string;
  
  // Location & Budget
  city_location: string;
  budget?: string;
  
  // System fields
  source: string;
  status: string;
  priority: string;
  created_at: string;
  notes?: string;
}

const LeadsDashboard: React.FC = () => {
  const [leads, setLeads] = useState<Lead[]>([]);
  const [loading, setLoading] = useState(true);
  const [statusFilter, setStatusFilter] = useState<string>('');
  const [sourceFilter, setSourceFilter] = useState<string>('');
  const [expandedCard, setExpandedCard] = useState<number | null>(null);

  const fetchLeads = async () => {
    try {
      const params = new URLSearchParams();
      if (statusFilter) params.append('status', statusFilter);
      if (sourceFilter) params.append('source', sourceFilter);
      
      const response = await fetch(`/api/leads/?${params}`);
      if (response.ok) {
        const data = await response.json();
        setLeads(data);
      }
    } catch (error) {
      console.error('Failed to fetch leads:', error);
    } finally {
      setLoading(false);
    }
  };

  const updateLeadStatus = async (leadId: number, status: string) => {
    try {
      const response = await fetch(`/api/leads/${leadId}`, {
        method: 'PUT',
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify({ status })
      });
      
      if (response.ok) {
        fetchLeads(); // Refresh the list
      }
    } catch (error) {
      console.error('Failed to update lead:', error);
    }
  };

  useEffect(() => {
    fetchLeads();
  }, [statusFilter, sourceFilter]);

  const getStatusColor = (status: string) => {
    switch (status) {
      case 'New': return 'bg-blue-100 text-blue-800';
      case 'Contacted': return 'bg-yellow-100 text-yellow-800';
      case 'Qualified': return 'bg-green-100 text-green-800';
      case 'Lost': return 'bg-red-100 text-red-800';
      case 'Converted': return 'bg-purple-100 text-purple-800';
      default: return 'bg-gray-100 text-gray-800';
    }
  };

  const getPriorityColor = (priority: string) => {
    switch (priority) {
      case 'High': return 'text-red-600';
      case 'Medium': return 'text-yellow-600';
      case 'Low': return 'text-green-600';
      default: return 'text-gray-600';
    }
  };

  if (loading) {
    return <div className="flex justify-center p-8">Loading leads...</div>;
  }

  return (
    <div className="p-6 bg-white min-h-screen">
      <div className="flex items-center justify-between mb-6">
        <h1 className="text-2xl font-bold flex items-center gap-2">
          <Users className="w-6 h-6" />
          Leads Dashboard
        </h1>
        
        <div className="flex gap-4">
          <select
            value={statusFilter}
            onChange={(e) => setStatusFilter(e.target.value)}
            className="px-3 py-2 border border-gray-300 rounded-md"
          >
            <option value="">All Status</option>
            <option value="New">New</option>
            <option value="Contacted">Contacted</option>
            <option value="Qualified">Qualified</option>
            <option value="Lost">Lost</option>
            <option value="Converted">Converted</option>
          </select>
          
          <select
            value={sourceFilter}
            onChange={(e) => setSourceFilter(e.target.value)}
            className="px-3 py-2 border border-gray-300 rounded-md"
          >
            <option value="">All Sources</option>
            <option value="Website">Website</option>
            <option value="WhatsApp">WhatsApp</option>
            <option value="Campaign">Campaign</option>
            <option value="Walk-in">Walk-in</option>
            <option value="Referral">Referral</option>
            <option value="Agent Entry">Agent Entry</option>
            <option value="Social Media">Social Media</option>
            <option value="Third-Party Portal">Third-Party Portal</option>
            <option value="Direct">Direct</option>
          </select>
        </div>
      </div>

      <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-4">
        {leads.map((lead) => (
          <div key={lead.id} className="bg-white rounded-lg shadow-sm border border-gray-200 p-2 hover:shadow-md transition-all duration-300 cursor-pointer max-h-48 overflow-y-auto"
          onClick={() => setExpandedCard(expandedCard === lead.id ? null : lead.id)}>
            {/* Header */}
            <div className="flex justify-between items-start mb-2">
              <div className="flex-1">
                <h3 className="font-medium text-gray-900 text-xs">{lead.buyer_name}</h3>
                <p className="text-xs text-gray-500">{lead.buyer_phone}</p>
              </div>
              <div className="flex gap-1" onClick={(e) => e.stopPropagation()}>
                {lead.status === 'New' && (
                  <button
                    onClick={() => updateLeadStatus(lead.id, 'Contacted')}
                    className="text-blue-600 hover:text-blue-900 p-1"
                    title="Mark as Contacted"
                  >
                    <Clock className="w-3 h-3" />
                  </button>
                )}
                {lead.status === 'Contacted' && (
                  <button
                    onClick={() => updateLeadStatus(lead.id, 'Qualified')}
                    className="text-green-600 hover:text-green-900 p-1"
                    title="Mark as Qualified"
                  >
                    <CheckCircle className="w-3 h-3" />
                  </button>
                )}
                <button
                  onClick={() => updateLeadStatus(lead.id, 'Lost')}
                  className="text-red-600 hover:text-red-900 p-1"
                  title="Mark as Lost"
                >
                  <XCircle className="w-3 h-3" />
                </button>
              </div>
            </div>
            
            {/* Tags */}
            <div className="flex flex-wrap gap-1 mb-1">
              <span className={`px-1 py-0.5 text-xs rounded-full ${
                lead.lead_type === 'Buyer' ? 'bg-blue-100 text-blue-800' : 'bg-green-100 text-green-800'
              }`}>
                {lead.lead_type}
              </span>
              <span className={`px-1 py-0.5 text-xs rounded-full ${getStatusColor(lead.status)}`}>
                {lead.status}
              </span>
            </div>
            
            {/* Details */}
            <div className="space-y-0.5 text-xs text-gray-600">
              <div className="flex justify-between">
                <span>Location:</span>
                <span className="font-medium">{lead.city_location}</span>
              </div>
              {lead.budget && (
                <div className="flex justify-between">
                  <span>Budget:</span>
                  <span className="font-medium">{lead.budget}</span>
                </div>
              )}
              <div className="flex justify-between">
                <span>Source:</span>
                <span className="font-medium">{lead.source}</span>
              </div>
              <div className="flex justify-between">
                <span>Priority:</span>
                <span className={`font-medium ${getPriorityColor(lead.priority)}`}>
                  {lead.priority}
                </span>
              </div>
              <div className="flex justify-between">
                <span>Created:</span>
                <span className="font-medium">{new Date(lead.created_at).toLocaleDateString()}</span>
              </div>
            </div>
            
            {/* Expanded Property Details */}
            {expandedCard === lead.id && (
              <div className="mt-2 pt-2 border-t border-gray-200 max-h-32 overflow-y-auto">
                <div className="flex justify-between items-center mb-2">
                  <h4 className="font-medium text-gray-900 text-xs">Property Details (ID: {lead.property_id})</h4>
                  <button
                    onClick={(e) => {
                      e.stopPropagation();
                      window.open(`http://localhost:5173/property/${lead.property_id}`, '_blank');
                    }}
                    className="px-2 py-1 bg-blue-600 text-white text-xs rounded hover:bg-blue-700"
                  >
                    I am Interested
                  </button>
                </div>
                <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-4">
                  <div className="space-y-2">
                    <h5 className="font-medium text-sm text-gray-700">Contact Information</h5>
                    <div className="text-xs text-gray-600 space-y-1">
                      <div>Name: {lead.buyer_name}</div>
                      <div>Phone: {lead.buyer_phone}</div>
                      {lead.buyer_email && <div>Email: {lead.buyer_email}</div>}
                    </div>
                  </div>
                  <div className="space-y-2">
                    <h5 className="font-medium text-sm text-gray-700">Lead Classification</h5>
                    <div className="text-xs text-gray-600 space-y-1">
                      <div>Type: {lead.lead_type}</div>
                      <div>Requirement: {lead.requirement}</div>
                      <div>Priority: <span className={getPriorityColor(lead.priority)}>{lead.priority}</span></div>
                    </div>
                  </div>
                  <div className="space-y-2">
                    <h5 className="font-medium text-sm text-gray-700">Additional Info</h5>
                    <div className="text-xs text-gray-600 space-y-1">
                      <div>Source: {lead.source}</div>
                      <div>Status: <span className={`px-1 py-0.5 rounded text-xs ${getStatusColor(lead.status)}`}>{lead.status}</span></div>
                      {lead.notes && <div>Notes: {lead.notes}</div>}
                    </div>
                  </div>
                </div>
              </div>
            )}
          </div>
        ))}
      </div>
      
      {leads.length === 0 && (
        <div className="text-center py-8 text-gray-500">
          No leads found matching your filters.
        </div>
      )}
    </div>
  );
};

export default LeadsDashboard;