import React, { useState } from 'react';
import { LocationConnectivity } from './LocationConnectivity';
import { checkLocationConnectivity, testLocationConnectivity } from '../utils/locationServices';

export const LocationConnectivityTest: React.FC = () => {
  const [latitude, setLatitude] = useState<string>('12.9716');
  const [longitude, setLongitude] = useState<string>('77.5946');
  const [useTestData, setUseTestData] = useState(true);

  const location = {
    latitude: parseFloat(latitude) || 0,
    longitude: parseFloat(longitude) || 0
  };

  const handleTestAPI = async () => {
    console.log('🧪 Testing Location Connectivity API...');
    
    try {
      // Test with real API
      console.log('📡 Testing real API endpoints...');
      const realResult = await checkLocationConnectivity(location);
      console.log('✅ Real API Result:', realResult);
      
      // Test with mock data
      console.log('🎭 Testing with mock data...');
      const mockResult = await testLocationConnectivity(location);
      console.log('✅ Mock Data Result:', mockResult);
      
    } catch (error) {
      console.error('❌ Test failed:', error);
    }
  };

  return (
    <div className="max-w-4xl mx-auto p-6 space-y-6">
      <div className="bg-white rounded-lg border p-6">
        <h2 className="text-2xl font-bold mb-4">Location Connectivity Test</h2>
        
        {/* Input Controls */}
        <div className="grid grid-cols-1 md:grid-cols-3 gap-4 mb-6">
          <div>
            <label className="block text-sm font-medium text-gray-700 mb-2">
              Latitude
            </label>
            <input
              type="number"
              step="any"
              value={latitude}
              onChange={(e) => setLatitude(e.target.value)}
              className="w-full px-3 py-2 border border-gray-300 rounded-md focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
              placeholder="12.9716"
            />
          </div>
          
          <div>
            <label className="block text-sm font-medium text-gray-700 mb-2">
              Longitude
            </label>
            <input
              type="number"
              step="any"
              value={longitude}
              onChange={(e) => setLongitude(e.target.value)}
              className="w-full px-3 py-2 border border-gray-300 rounded-md focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
              placeholder="77.5946"
            />
          </div>
          
          <div className="flex items-end">
            <label className="flex items-center space-x-2">
              <input
                type="checkbox"
                checked={useTestData}
                onChange={(e) => setUseTestData(e.target.checked)}
                className="w-4 h-4 text-blue-600 border-gray-300 rounded focus:ring-blue-500"
              />
              <span className="text-sm text-gray-700">Use Test Data</span>
            </label>
          </div>
        </div>

        {/* Test Button */}
        <button
          onClick={handleTestAPI}
          className="mb-6 px-4 py-2 bg-blue-600 text-white rounded-md hover:bg-blue-700 transition-colors"
        >
          Test API in Console
        </button>

        {/* Location Display */}
        <div className="mb-4 p-3 bg-gray-50 rounded-md">
          <p className="text-sm text-gray-600">
            Testing location: <span className="font-mono">{location.latitude.toFixed(4)}, {location.longitude.toFixed(4)}</span>
          </p>
          <p className="text-xs text-gray-500 mt-1">
            {useTestData ? 'Using mock data for demonstration' : 'Using real API endpoints'}
          </p>
        </div>
      </div>

      {/* Connectivity Component */}
      <LocationConnectivity 
        location={location}
        useTestData={useTestData}
        className="shadow-lg"
      />

      {/* API Endpoints Info */}
      <div className="bg-blue-50 border border-blue-200 rounded-lg p-4">
        <h3 className="font-semibold text-blue-900 mb-2">Expected API Endpoints:</h3>
        <div className="text-sm text-blue-800 space-y-1">
          <p><code>GET /api/infrastructure/nearby?lat={location.latitude}&lng={location.longitude}&radius=5</code></p>
          <p><code>GET /api/zoning/check?lat={location.latitude}&lng={location.longitude}</code></p>
          <p><code>GET /api/metro-lines/nearby?lat={location.latitude}&lng={location.longitude}&radius=3</code></p>
        </div>
      </div>
    </div>
  );
};