import React, { useState, useEffect } from 'react';
import { X, MapPin, Navigation, Search } from 'lucide-react';

interface LocationPickerProps {
  isOpen: boolean;
  onClose: () => void;
  onLocationSelect: (location: { lat: number; lng: number; address: string }) => void;
  initialCoordinates?: { lat: number; lng: number };
  initialAddress?: string;
}

export const LocationPicker: React.FC<LocationPickerProps> = ({
  isOpen,
  onClose,
  onLocationSelect,
  initialCoordinates,
  initialAddress
}) => {
  const [coordinates, setCoordinates] = useState(initialCoordinates || { lat: 12.9716, lng: 77.5946 });
  const [address, setAddress] = useState(initialAddress || '');
  const [isGettingLocation, setIsGettingLocation] = useState(false);
  const [manualCoords, setManualCoords] = useState('');

  useEffect(() => {
    if (initialCoordinates) {
      setCoordinates(initialCoordinates);
      setManualCoords(`${initialCoordinates.lat}, ${initialCoordinates.lng}`);
    }
  }, [initialCoordinates]);

  const getCurrentLocation = () => {
    if (!navigator.geolocation) {
      alert('Geolocation is not supported by this browser.');
      return;
    }

    setIsGettingLocation(true);
    
    navigator.geolocation.getCurrentPosition(
      (position) => {
        const { latitude, longitude } = position.coords;
        const newCoords = { lat: latitude, lng: longitude };
        setCoordinates(newCoords);
        setManualCoords(`${latitude}, ${longitude}`);
        setAddress(`Current Location: ${latitude.toFixed(6)}, ${longitude.toFixed(6)}`);
        setIsGettingLocation(false);
      },
      (error) => {
        setIsGettingLocation(false);
        alert('Could not get your current location. Please try again or enter coordinates manually.');
      },
      {
        enableHighAccuracy: true,
        timeout: 10000,
        maximumAge: 60000
      }
    );
  };

  const openGoogleMaps = () => {
    const mapsUrl = `https://www.google.com/maps/@${coordinates.lat},${coordinates.lng},15z`;
    window.open(mapsUrl, '_blank', 'width=1000,height=700');
  };

  const handleManualCoordinates = () => {
    const coords = manualCoords.split(',').map(coord => parseFloat(coord.trim()));
    if (coords.length === 2 && !isNaN(coords[0]) && !isNaN(coords[1])) {
      const newCoords = { lat: coords[0], lng: coords[1] };
      setCoordinates(newCoords);
      setAddress(`Manual Location: ${coords[0].toFixed(6)}, ${coords[1].toFixed(6)}`);
    } else {
      alert('Invalid coordinates format. Please use: latitude, longitude');
    }
  };

  const handleConfirm = () => {
    onLocationSelect({
      lat: coordinates.lat,
      lng: coordinates.lng,
      address: address || `${coordinates.lat.toFixed(6)}, ${coordinates.lng.toFixed(6)}`
    });
    onClose();
  };

  if (!isOpen) return null;

  return (
    <div className="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50 p-4">
      <div className="bg-white rounded-lg shadow-xl max-w-md w-full max-h-[90vh] overflow-y-auto">
        {/* Header */}
        <div className="flex items-center justify-between p-4 border-b border-gray-200">
          <h3 className="text-lg font-semibold text-gray-900">Select Location</h3>
          <button
            onClick={onClose}
            className="text-gray-400 hover:text-gray-600 transition-colors"
          >
            <X className="w-5 h-5" />
          </button>
        </div>

        {/* Content */}
        <div className="p-4 space-y-4">
          {/* Current Location Button */}
          <button
            onClick={getCurrentLocation}
            disabled={isGettingLocation}
            className="w-full flex items-center justify-center space-x-2 px-4 py-3 bg-blue-50 border border-blue-200 rounded-lg hover:bg-blue-100 transition-colors disabled:opacity-50"
          >
            {isGettingLocation ? (
              <>
                <div className="animate-spin rounded-full h-5 w-5 border-b-2 border-blue-600"></div>
                <span>Getting your location...</span>
              </>
            ) : (
              <>
                <Navigation className="w-5 h-5 text-blue-600" />
                <span className="text-blue-700 font-medium">Use My Current Location</span>
              </>
            )}
          </button>

          {/* Google Maps Button */}
          <button
            onClick={openGoogleMaps}
            className="w-full flex items-center justify-center space-x-2 px-4 py-3 bg-green-50 border border-green-200 rounded-lg hover:bg-green-100 transition-colors"
          >
            <MapPin className="w-5 h-5 text-green-600" />
            <span className="text-green-700 font-medium">Open Google Maps</span>
          </button>

          <div className="text-xs text-gray-500 bg-gray-50 p-3 rounded-lg">
            <strong>Google Maps Instructions:</strong>
            <ol className="mt-1 space-y-1 list-decimal list-inside">
              <li>Click on your desired location</li>
              <li>Right-click and select "What's here?"</li>
              <li>Copy the coordinates from the popup</li>
              <li>Paste them in the field below</li>
            </ol>
          </div>

          {/* Manual Coordinates Input */}
          <div>
            <label className="block text-sm font-medium text-gray-700 mb-2">
              Enter Coordinates Manually
            </label>
            <div className="flex space-x-2">
              <input
                type="text"
                value={manualCoords}
                onChange={(e) => setManualCoords(e.target.value)}
                placeholder="12.9716, 77.5946"
                className="flex-1 px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent text-sm"
              />
              <button
                onClick={handleManualCoordinates}
                className="px-3 py-2 bg-gray-100 text-gray-700 rounded-lg hover:bg-gray-200 transition-colors text-sm"
              >
                Set
              </button>
            </div>
          </div>

          {/* Current Selection Display */}
          <div className="bg-gray-50 border border-gray-200 rounded-lg p-3">
            <h4 className="text-sm font-medium text-gray-900 mb-2">Selected Location:</h4>
            <div className="space-y-1 text-sm text-gray-600">
              <p><strong>Latitude:</strong> {coordinates.lat.toFixed(6)}</p>
              <p><strong>Longitude:</strong> {coordinates.lng.toFixed(6)}</p>
              {address && <p><strong>Address:</strong> {address}</p>}
            </div>
          </div>

          {/* Address Input */}
          <div>
            <label className="block text-sm font-medium text-gray-700 mb-2">
              Address Description (Optional)
            </label>
            <input
              type="text"
              value={address}
              onChange={(e) => setAddress(e.target.value)}
              placeholder="Enter a description for this location"
              className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent text-sm"
            />
          </div>
        </div>

        {/* Footer */}
        <div className="flex space-x-3 p-4 border-t border-gray-200">
          <button
            onClick={onClose}
            className="flex-1 px-4 py-2 border border-gray-300 text-gray-700 rounded-lg hover:bg-gray-50 transition-colors"
          >
            Cancel
          </button>
          <button
            onClick={handleConfirm}
            className="flex-1 px-4 py-2 bg-[#2B256D] text-white rounded-lg hover:bg-[#1f1a5a] transition-colors"
          >
            Confirm Location
          </button>
        </div>
      </div>
    </div>
  );
};

