import { useNavigate } from "react-router-dom";
import { Card, CardContent } from "./ui/card";
import { Button } from "./ui/button";
import { User, Building2, Users, X } from "lucide-react";

interface LoginTypeModalProps {
  isOpen: boolean;
  onClose: () => void;
  context?: 'login' | 'post-property';
}

export function LoginTypeModal({ isOpen, onClose, context = 'login' }: LoginTypeModalProps) {
  const navigate = useNavigate();

  if (!isOpen) return null;

  const handleSelection = (type: string) => {
    onClose();
    if (context === 'post-property') {
      // For post property context, redirect to appropriate login with return URL
      const returnUrl = encodeURIComponent('/list-property');
      if (type === "buyer") {
        navigate(`/login?returnUrl=${returnUrl}`);
      } else if (type === "seller") {
        navigate(`/seller?returnUrl=${returnUrl}`);
      } else if (type === "agent") {
        navigate(`/agent?returnUrl=${returnUrl}`);
      }
    } else {
      // Regular login context
      if (type === "buyer") {
        navigate("/login");
      } else if (type === "seller") {
        navigate("/seller");
      } else if (type === "agent") {
        navigate("/agent");
      }
    }
  };

  return (
    <div className="fixed inset-0 bg-black/50 flex items-center justify-center z-[9999]" onClick={onClose}>
      <Card className="w-full max-w-md mx-4" onClick={(e) => e.stopPropagation()}>
        <CardContent className="p-6">
          <div className="flex justify-between items-center mb-6">
            <h2 className="text-2xl font-bold text-gray-900">
              {context === 'post-property' ? 'Login to Post Property' : 'Login As'}
            </h2>
            <button onClick={onClose} className="text-gray-400 hover:text-gray-600">
              <X className="w-5 h-5" />
            </button>
          </div>
          
          <div className="space-y-3">
            <button
              onClick={() => handleSelection("buyer")}
              className="w-full h-16 bg-white text-gray-900 border-2 border-gray-200 flex items-center justify-start gap-4 px-6 transition-all duration-300 rounded-lg"
              onMouseEnter={(e) => {
                e.currentTarget.style.backgroundColor = '#dbeafe';
                e.currentTarget.style.borderColor = '#3b82f6';
              }}
              onMouseLeave={(e) => {
                e.currentTarget.style.backgroundColor = '#ffffff';
                e.currentTarget.style.borderColor = '#e5e7eb';
              }}
            >
              <div className="w-10 h-10 bg-blue-100 rounded-full flex items-center justify-center">
                <User className="w-5 h-5 text-[#0056D2]" />
              </div>
              <div className="text-left">
                <div className="font-semibold text-black">Buyer</div>
                <div className="text-xs text-black">
                  {context === 'post-property' ? 'Login as buyer to post property' : 'Looking to buy property'}
                </div>
              </div>
            </button>

            <button
              onClick={() => handleSelection("seller")}
              className="w-full h-16 bg-white text-gray-900 border-2 border-gray-200 flex items-center justify-start gap-4 px-6 transition-all duration-300 rounded-lg"
              onMouseEnter={(e) => {
                e.currentTarget.style.backgroundColor = '#dbeafe';
                e.currentTarget.style.borderColor = '#3b82f6';
              }}
              onMouseLeave={(e) => {
                e.currentTarget.style.backgroundColor = '#ffffff';
                e.currentTarget.style.borderColor = '#e5e7eb';
              }}
            >
              <div className="w-10 h-10 bg-green-100 rounded-full flex items-center justify-center">
                <Building2 className="w-5 h-5 text-green-600" />
              </div>
              <div className="text-left">
                <div className="font-semibold">Seller</div>
                <div className="text-xs text-gray-500">
                  {context === 'post-property' ? 'Login as seller to post property' : 'Want to sell property'}
                </div>
              </div>
            </button>

            <button
              onClick={() => handleSelection("agent")}
              className="w-full h-16 bg-white text-gray-900 border-2 border-gray-200 flex items-center justify-start gap-4 px-6 transition-all duration-300 rounded-lg"
              onMouseEnter={(e) => {
                e.currentTarget.style.backgroundColor = '#dbeafe';
                e.currentTarget.style.borderColor = '#3b82f6';
              }}
              onMouseLeave={(e) => {
                e.currentTarget.style.backgroundColor = '#ffffff';
                e.currentTarget.style.borderColor = '#e5e7eb';
              }}
            >
              <div className="w-10 h-10 bg-purple-100 rounded-full flex items-center justify-center">
                <Users className="w-5 h-5 text-purple-600" />
              </div>
              <div className="text-left">
                <div className="font-semibold">Agent</div>
                <div className="text-xs text-gray-500">
                  {context === 'post-property' ? 'Login as agent to post property' : 'Real estate professional'}
                </div>
              </div>
            </button>
          </div>
        </CardContent>
      </Card>
    </div>
  );
}
