import React, { useRef, useEffect, useState } from 'react';
import Map, { Marker, NavigationControl, GeolocateControl } from 'react-map-gl/maplibre';
import 'maplibre-gl/dist/maplibre-gl.css';

interface MapLibreMapProps {
  latitude?: number;
  longitude?: number;
  zoom?: number;
  onCoordinateChange?: (lat: number, lng: number) => void;
  showMarker?: boolean;
  interactive?: boolean;
  className?: string;
  style?: React.CSSProperties;
}

const MapLibreMap: React.FC<MapLibreMapProps> = ({
  latitude = 12.9716,
  longitude = 77.5946,
  zoom = 10,
  onCoordinateChange,
  showMarker = true,
  interactive = true,
  className = "w-full h-96",
  style
}) => {
  const mapRef = useRef<any>(null);
  const [viewState, setViewState] = useState({
    longitude,
    latitude,
    zoom
  });

  const [markerPosition, setMarkerPosition] = useState({
    latitude,
    longitude
  });

  useEffect(() => {
    console.log('MapLibre props:', { latitude, longitude, zoom });
    setViewState({ longitude, latitude, zoom: Math.min(zoom, 18) });
    setMarkerPosition({ latitude, longitude });
  }, [latitude, longitude, zoom]);

  const handleMapClick = (event: { lngLat: { lng: number; lat: number } }) => {
    if (!interactive || !onCoordinateChange) return;
    
    const { lng, lat } = event.lngLat;
    setMarkerPosition({ latitude: lat, longitude: lng });
    onCoordinateChange(lat, lng);
  };

  return (
    <div className={className} style={{ border: '2px solid #4a90e2', borderRadius: '8px', overflow: 'hidden', height: '600px', width: '100%', ...style }}>
      <Map
        ref={mapRef}
        {...viewState}
        onMove={evt => setViewState(evt.viewState)}
        onClick={handleMapClick}
        mapStyle={{
          version: 8,
          sources: {
            'satellite': {
              type: 'raster',
              tiles: [
                'https://server.arcgisonline.com/ArcGIS/rest/services/World_Imagery/MapServer/tile/{z}/{y}/{x}'
              ],
              tileSize: 256,
              minzoom: 0,
              maxzoom: 19,
              attribution: '© Esri, Maxar, Earthstar Geographics'
            },
            'osm-fallback': {
              type: 'raster',
              tiles: [
                'https://tile.openstreetmap.org/{z}/{x}/{y}.png'
              ],
              tileSize: 256,
              minzoom: 0,
              maxzoom: 19,
              attribution: '© OpenStreetMap contributors'
            }
          },
          layers: [
            {
              id: 'satellite',
              type: 'raster',
              source: 'satellite'
            }
          ]
        }}
        attributionControl={true}
        style={{ width: '100%', height: '600px' }}
        onLoad={() => console.log('Map loaded successfully')}
        onError={(e) => console.error('Map error:', e)}
      >
        {showMarker && (
          <Marker
            longitude={markerPosition.longitude}
            latitude={markerPosition.latitude}
            anchor="bottom"
          >
            <div className="w-8 h-8 bg-red-500 rounded-full border-4 border-white shadow-lg flex items-center justify-center">
              <div className="w-3 h-3 bg-white rounded-full"></div>
            </div>
          </Marker>
        )}
        
        <NavigationControl position="top-right" />
        <GeolocateControl position="top-right" />
      </Map>
    </div>
  );
};

export default MapLibreMap;