import { useState, useEffect } from "react";
import { PropertyHeatmap } from './heatmap/PropertyHeatmap';
import { Property } from './heatmap/types';

export function MarketInsights() {
  const [properties, setProperties] = useState<Property[]>([]);
  const [loading, setLoading] = useState(true);

  useEffect(() => {
    fetchProperties();
  }, []);

  const fetchProperties = async () => {
    try {
      const response = await fetch('/api/properties/?limit=100');
      const data = await response.json();
      
      const mappedProperties: Property[] = data.properties
        .filter((p: any) => p.latitude && p.longitude)
        .map((p: any) => ({
          id: p.id,
          latitude: p.latitude,
          longitude: p.longitude,
          sale_price: p.sale_price || 0,
          monthly_rent: p.monthly_rent || 0,
          property_type: p.property_type,
          listing_type: p.listing_type,
          city: p.city
        }));

      setProperties(mappedProperties);
    } catch (error) {
      console.error('Error fetching properties:', error);
    } finally {
      setLoading(false);
    }
  };

  return (
    <section className="py-16 bg-white">
      <div className="w-full px-4 sm:px-6 lg:px-8">
        <div className="text-center mb-12">
          <h2 className="text-3xl md:text-4xl font-bold text-gray-900 mb-3">Heatmap</h2>
          <p className="text-gray-600 max-w-3xl mx-auto">
          
          </p>
        </div>

        <div className="w-full rounded-xl overflow-hidden border border-gray-300">
          {loading ? (
            <div className="h-[500px] bg-gray-200 animate-pulse flex items-center justify-center">
              <span className="text-gray-500">Loading properties...</span>
            </div>
          ) : (
            <div style={{ height: 500, width: "100%" }}>
              <PropertyHeatmap properties={properties} />
            </div>
          )}
        </div>
      </div>
    </section>
  );
}