// src/components/NLSearchFeedback.tsx
/**
 * Natural Language Search Feedback Component
 * Shows extracted filters as chips and provides real-time parsing feedback.
 */

import React from 'react';
import { X, MapPin, Home, BedDouble, IndianRupee, Sparkles, Building, Key } from 'lucide-react';
import { ParsedQuery, ExtractedEntity } from '../api/properties';

interface NLSearchFeedbackProps {
    parsedQuery: ParsedQuery | null;
    onRemoveFilter?: (entityType: string) => void;
    isLoading?: boolean;
    className?: string;
}

// Entity display configuration
const entityConfig: Record<string, { icon: React.ReactNode; label: string; color: string }> = {
    bhk: {
        icon: <BedDouble className="w-3 h-3" />,
        label: 'BHK',
        color: 'bg-blue-100 text-blue-800 border-blue-200'
    },
    location: {
        icon: <MapPin className="w-3 h-3" />,
        label: 'Location',
        color: 'bg-green-100 text-green-800 border-green-200'
    },
    property_type: {
        icon: <Building className="w-3 h-3" />,
        label: 'Type',
        color: 'bg-purple-100 text-purple-800 border-purple-200'
    },
    listing_type: {
        icon: <Key className="w-3 h-3" />,
        label: 'Intent',
        color: 'bg-orange-100 text-orange-800 border-orange-200'
    },
    price: {
        icon: <IndianRupee className="w-3 h-3" />,
        label: 'Budget',
        color: 'bg-yellow-100 text-yellow-800 border-yellow-200'
    },
    furnishing: {
        icon: <Home className="w-3 h-3" />,
        label: 'Furnishing',
        color: 'bg-pink-100 text-pink-800 border-pink-200'
    },
    amenities: {
        icon: <Sparkles className="w-3 h-3" />,
        label: 'Amenities',
        color: 'bg-indigo-100 text-indigo-800 border-indigo-200'
    }
};

function formatEntityValue(entity: ExtractedEntity): string {
    const { entity_type, value } = entity;

    switch (entity_type) {
        case 'bhk':
            return `${value} BHK`;
        case 'location':
            return value.name || value;
        case 'property_type':
            return value.charAt(0).toUpperCase() + value.slice(1).replace('-', ' ');
        case 'listing_type':
            if (value === 'sell') return 'Buy';
            return value.charAt(0).toUpperCase() + value.slice(1).replace('-', ' ');
        case 'price':
            if (value >= 10000000) return `₹${(value / 10000000).toFixed(1)} Cr`;
            if (value >= 100000) return `₹${(value / 100000).toFixed(0)} L`;
            if (value >= 1000) return `₹${(value / 1000).toFixed(0)}K`;
            return `₹${value}`;
        case 'furnishing':
            return value.replace('-', ' ').split(' ').map((w: string) =>
                w.charAt(0).toUpperCase() + w.slice(1)
            ).join(' ');
        case 'amenities':
            if (Array.isArray(value)) {
                return value.map(v => v.replace('_', ' ')).join(', ');
            }
            return value;
        default:
            return String(value);
    }
}

export function NLSearchFeedback({
    parsedQuery,
    onRemoveFilter,
    isLoading = false,
    className = ''
}: NLSearchFeedbackProps) {
    if (isLoading) {
        return (
            <div className={`flex items-center gap-2 text-sm text-gray-500 ${className}`}>
                <div className="animate-pulse flex gap-2">
                    <div className="h-6 w-16 bg-gray-200 rounded-full"></div>
                    <div className="h-6 w-20 bg-gray-200 rounded-full"></div>
                    <div className="h-6 w-14 bg-gray-200 rounded-full"></div>
                </div>
            </div>
        );
    }

    if (!parsedQuery || parsedQuery.extracted_entities.length === 0) {
        return null;
    }

    return (
        <div className={`flex flex-col gap-2 ${className}`}>
            {/* Filter chips */}
            <div className="flex flex-wrap items-center gap-2">
                <span className="text-xs text-gray-500 font-medium">Searching for:</span>
                {parsedQuery.extracted_entities.map((entity, index) => {
                    const config = entityConfig[entity.entity_type] || {
                        icon: null,
                        label: entity.entity_type,
                        color: 'bg-gray-100 text-gray-800 border-gray-200'
                    };

                    return (
                        <div
                            key={`${entity.entity_type}-${index}`}
                            className={`
                inline-flex items-center gap-1.5 px-2.5 py-1 rounded-full text-xs font-medium
                border ${config.color} transition-all hover:shadow-sm
              `}
                        >
                            {config.icon}
                            <span>{formatEntityValue(entity)}</span>
                            {onRemoveFilter && (
                                <button
                                    onClick={() => onRemoveFilter(entity.entity_type)}
                                    className="ml-1 hover:bg-black/10 rounded-full p-0.5"
                                    aria-label={`Remove ${config.label} filter`}
                                >
                                    <X className="w-3 h-3" />
                                </button>
                            )}
                        </div>
                    );
                })}

                {/* Confidence indicator */}
                {parsedQuery.confidence < 0.7 && (
                    <span className="text-xs text-amber-600 bg-amber-50 px-2 py-0.5 rounded">
                        Low confidence
                    </span>
                )}
            </div>

            {/* Summary text */}
            {parsedQuery.filter_summary && (
                <p className="text-xs text-gray-600">
                    {parsedQuery.filter_summary}
                </p>
            )}
        </div>
    );
}

export default NLSearchFeedback;
