import React, { useState, useEffect } from 'react';
import { Bell, X, CheckCircle, AlertCircle, Info, Wifi, WifiOff } from 'lucide-react';
import { useEventBus } from '../context/EventBusContext';
import { formatDistanceToNow } from 'date-fns';
import { getUserNotifications, markUserNotificationAsRead, markAllUserNotificationsAsRead } from '../api/notifications';

interface NotificationPanelProps {
  className?: string;
}

export const NotificationPanel: React.FC<NotificationPanelProps> = ({ className = '' }) => {
  const { notifications: realtimeNotifications, isConnected, connectionStatus, clearNotifications, refreshNotifications } = useEventBus();
  const [isOpen, setIsOpen] = useState(false);
  const [dbNotifications, setDbNotifications] = useState<any[]>([]);
  const [unreadCount, setUnreadCount] = useState(0);
  const [isLoading, setIsLoading] = useState(false);

  // Get user ID from localStorage
  const getUserId = () => {
    try {
      // Try seller ID first (for seller dashboard)
      const sellerId = localStorage.getItem('sellerId');
      if (sellerId) return sellerId;
      
      // Try user object
      const user = JSON.parse(localStorage.getItem('user') || '{}');
      return user.user_id || user.id || null;
    } catch {
      return null;
    }
  };

  // Fetch notifications from database
  const fetchNotifications = async () => {
    const userId = getUserId();
    console.log('🔔 NotificationPanel - Fetching notifications for user:', userId);
    console.log('🔔 NotificationPanel - User from localStorage:', localStorage.getItem('user'));
    console.log('🔔 NotificationPanel - SellerId from localStorage:', localStorage.getItem('sellerId'));
    if (!userId) {
      console.log('No user ID found');
      return;
    }

    setIsLoading(true);
    try {
      const response = await getUserNotifications(userId);
      console.log('📬 NotificationPanel - API Response:', response);
      console.log('📬 NotificationPanel - Notifications count:', response.notifications?.length || 0);
      setDbNotifications(response.notifications || []);
      setUnreadCount(response.unread_count || 0);
    } catch (error) {
      console.error('Failed to fetch notifications:', error);
    } finally {
      setIsLoading(false);
    }
  };

  // Fetch notifications on mount
  useEffect(() => {
    fetchNotifications();
    
    // Auto-refresh every 10 seconds
    const interval = setInterval(fetchNotifications, 10000);
    return () => clearInterval(interval);
  }, []);

  // Fetch when panel opens
  useEffect(() => {
    if (isOpen) {
      fetchNotifications();
    }
  }, [isOpen]);

  // Combine database notifications only (no realtime)
  const allNotifications = dbNotifications;
  console.log('All notifications:', allNotifications);

  // Mark notification as read
  const handleMarkAsRead = async (notificationId: string) => {
    try {
      await markUserNotificationAsRead(notificationId);
      await fetchNotifications();
    } catch (error) {
      console.error('Failed to mark notification as read:', error);
    }
  };

  // Mark all as read
  const handleMarkAllAsRead = async () => {
    const userId = getUserId();
    if (!userId) return;

    try {
      await markAllUserNotificationsAsRead(userId);
      await fetchNotifications();
    } catch (error) {
      console.error('Failed to mark all as read:', error);
    }
  };

  // Clear all notifications
  const handleClearAll = () => {
    clearNotifications();
    fetchNotifications();
  };

  const getEventIcon = (eventType?: string) => {
    switch (eventType) {
      case 'property_approved':
      case 'visit_approved':
      case 'document_verified':
      case 'auction_won':
        return <CheckCircle className="w-4 h-4 text-green-500" />;
      case 'property_rejected':
      case 'visit_rejected':
        return <AlertCircle className="w-4 h-4 text-red-500" />;
      case 'auction_bid':
      case 'inquiry_received':
        return <Info className="w-4 h-4 text-blue-500" />;
      default:
        return <Bell className="w-4 h-4 text-gray-500" />;
    }
  };

  const getConnectionStatusColor = () => {
    switch (connectionStatus) {
      case 'connected':
        return 'text-green-500';
      case 'connecting':
        return 'text-yellow-500';
      case 'error':
        return 'text-red-500';
      default:
        return 'text-gray-500';
    }
  };

  const formatEventTitle = (notification: any) => {
    const eventType = notification.event_type;
    const data = notification.data || {};
    
    switch (eventType) {
      case 'property_approved':
        return `Property "${data.property_title}" Approved`;
      case 'property_rejected':
        return `Property "${data.property_title}" Rejected`;
      case 'visit_approved':
        return `Visit for "${data.property_title}" Approved`;
      case 'visit_rejected':
        return `Visit for "${data.property_title}" Rejected`;
      case 'inquiry_received':
        return `New Inquiry for "${data.property_title}"`;
      case 'auction_bid':
        return `New Bid: ₹${data.amount?.toLocaleString()}`;
      case 'auction_won':
        return `Auction Won: "${data.property_title}"`;
      case 'document_verified':
        return 'Document Verified';
      default:
        return data.title || 'New Notification';
    }
  };

  return (
    <div className={`relative ${className}`}>
      {/* Notification Bell Button */}
      <button
        onClick={() => setIsOpen(!isOpen)}
        className="relative p-2 text-gray-600 hover:text-gray-900 hover:bg-gray-100 rounded-full transition-colors"
      >
        <Bell className="w-6 h-6" />
        
        {/* Unread Count Badge */}
        {(unreadCount > 0 || allNotifications.length > 0) && (
          <span className="absolute -top-1 -right-1 bg-red-500 text-white text-xs rounded-full w-5 h-5 flex items-center justify-center">
            {unreadCount > 99 ? '99+' : (unreadCount || allNotifications.length)}
          </span>
        )}
        
        {/* Connection Status - Always show disconnected */}
        <div className={`absolute -bottom-1 -right-1 text-gray-400`}>
          <WifiOff className="w-3 h-3" />
        </div>
      </button>

      {/* Notification Panel */}
      {isOpen && (
        <div className="absolute right-0 top-full mt-2 w-80 bg-white rounded-lg shadow-lg border border-gray-200 z-50 max-h-96 overflow-hidden">
          {/* Header */}
          <div className="flex items-center justify-between p-4 border-b border-gray-200">
            <div className="flex items-center space-x-2">
              <h3 className="font-semibold text-gray-900">Notifications</h3>
              <span className="text-xs px-2 py-1 rounded-full bg-gray-100 text-gray-800">
                polling
              </span>
            </div>
            <div className="flex items-center space-x-2">
              {allNotifications.length > 0 && (
                <>
                  <button
                    onClick={handleMarkAllAsRead}
                    className="text-xs text-blue-600 hover:text-blue-800"
                  >
                    Mark All Read
                  </button>
                  <button
                    onClick={handleClearAll}
                    className="text-xs text-red-600 hover:text-red-800"
                  >
                    Clear All
                  </button>
                </>
              )}
              <button
                onClick={() => setIsOpen(false)}
                className="text-gray-400 hover:text-gray-600"
              >
                <X className="w-4 h-4" />
              </button>
            </div>
          </div>

          {/* Notifications List */}
          <div className="max-h-80 overflow-y-auto">
            {isLoading ? (
              <div className="p-4 text-center text-gray-500">
                <div className="animate-spin rounded-full h-8 w-8 border-b-2 border-blue-600 mx-auto mb-2"></div>
                <p className="text-xs">Loading notifications...</p>
              </div>
            ) : allNotifications.length === 0 ? (
              <div className="p-4 text-center text-gray-500">
                <Bell className="w-8 h-8 mx-auto mb-2 text-gray-300" />
                <p>No notifications yet</p>
                <p className="text-xs">You'll see real-time updates here</p>
              </div>
            ) : (
              <div className="divide-y divide-gray-100">
                {allNotifications.map((notification, index) => {
                  const isDbNotification = notification.id && !notification.timestamp;
                  const isUnread = isDbNotification && !notification.read;
                  
                  return (
                    <div 
                      key={notification.id || index} 
                      className={`p-4 hover:bg-gray-50 transition-colors cursor-pointer ${
                        isUnread ? 'bg-blue-50' : ''
                      }`}
                      onClick={() => isDbNotification && isUnread && handleMarkAsRead(notification.id)}
                    >
                      <div className="flex items-start space-x-3">
                        <div className="flex-shrink-0 mt-1">
                          {getEventIcon(notification.event_type || notification.type)}
                        </div>
                        <div className="flex-1 min-w-0">
                          <div className="flex items-center justify-between">
                            <p className={`text-sm font-medium text-gray-900 truncate ${
                              isUnread ? 'font-bold' : ''
                            }`}>
                              {notification.title || formatEventTitle(notification)}
                            </p>
                            {isUnread && (
                              <span className="ml-2 w-2 h-2 bg-blue-600 rounded-full"></span>
                            )}
                          </div>
                          <p className="text-sm text-gray-600 mt-1">
                            {notification.message || notification.data?.message}
                          </p>
                          <p className="text-xs text-gray-400 mt-1">
                            {notification.time ? 
                              formatDistanceToNow(new Date(notification.time), { addSuffix: true }) :
                              notification.timestamp ? 
                                formatDistanceToNow(new Date(notification.timestamp), { addSuffix: true }) :
                                'Just now'
                            }
                          </p>
                        </div>
                      </div>
                    </div>
                  );
                })}
              </div>
            )}
          </div>

          {/* Footer */}
          <div className="p-3 border-t border-gray-200 bg-gray-50">
            <div className="flex items-center justify-between text-xs text-gray-500">
              <span>Auto-refresh every 10s</span>
              <div className="flex items-center space-x-1">
                <div className="w-2 h-2 rounded-full bg-gray-400"></div>
                <span>Polling</span>
              </div>
            </div>
          </div>
        </div>
      )}
    </div>
  );
};