import React, { useState } from 'react';

interface PasswordProtectedUploadProps {
  onUpload: (file: File, password?: string) => void;
  isUploading?: boolean;
}

const PasswordProtectedUpload: React.FC<PasswordProtectedUploadProps> = ({
  onUpload,
  isUploading = false
}) => {
  const [selectedFile, setSelectedFile] = useState<File | null>(null);
  const [isEncrypted, setIsEncrypted] = useState(false);
  const [password, setPassword] = useState('');
  const [confirmPassword, setConfirmPassword] = useState('');
  const [passwordError, setPasswordError] = useState('');

  const handleFileSelect = (event: React.ChangeEvent<HTMLInputElement>) => {
    const file = event.target.files?.[0];
    if (file) {
      setSelectedFile(file);
    }
  };

  const handlePasswordChange = (value: string) => {
    setPassword(value);
    if (confirmPassword && value !== confirmPassword) {
      setPasswordError('Passwords do not match');
    } else {
      setPasswordError('');
    }
  };

  const handleConfirmPasswordChange = (value: string) => {
    setConfirmPassword(value);
    if (password && value !== password) {
      setPasswordError('Passwords do not match');
    } else {
      setPasswordError('');
    }
  };

  const handleUpload = () => {
    if (!selectedFile) return;

    if (isEncrypted) {
      if (!password) {
        setPasswordError('Password is required for encrypted upload');
        return;
      }
      if (password !== confirmPassword) {
        setPasswordError('Passwords do not match');
        return;
      }
      if (password.length < 6) {
        setPasswordError('Password must be at least 6 characters');
        return;
      }
    }

    onUpload(selectedFile, isEncrypted ? password : undefined);
  };

  return (
    <div className="border border-gray-300 rounded-lg p-4 bg-white">
      <div className="space-y-4">
        {/* File Selection */}
        <div>
          <label className="block text-sm font-medium text-gray-700 mb-2">
            Choose Document
          </label>
          <input
            type="file"
            accept=".pdf,.jpg,.jpeg,.png"
            onChange={handleFileSelect}
            className="block w-full text-sm text-gray-500 file:mr-4 file:py-2 file:px-4 file:rounded-md file:border-0 file:text-sm file:font-semibold file:bg-blue-50 file:text-blue-700 hover:file:bg-blue-100"
          />
          {selectedFile && (
            <p className="mt-1 text-sm text-gray-600">
              Selected: {selectedFile.name}
            </p>
          )}
        </div>

        {/* Encryption Toggle */}
        <div className="flex items-center space-x-3">
          <input
            type="checkbox"
            id="encryption"
            checked={isEncrypted}
            onChange={(e) => setIsEncrypted(e.target.checked)}
            className="h-4 w-4 text-blue-600 focus:ring-blue-500 border-gray-300 rounded"
          />
          <label htmlFor="encryption" className="flex items-center text-sm font-medium text-gray-700">
            🔒 Password Protect This Document
          </label>
        </div>

        {/* Password Fields */}
        {isEncrypted && (
          <div className="space-y-3 p-3 bg-blue-50 rounded-md border border-blue-200">
            <div>
              <label className="block text-sm font-medium text-gray-700 mb-1">
                🔑 Set Password
              </label>
              <input
                type="password"
                value={password}
                onChange={(e) => handlePasswordChange(e.target.value)}
                placeholder="Enter password (min 6 characters)"
                className="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500"
              />
            </div>
            <div>
              <label className="block text-sm font-medium text-gray-700 mb-1">
                Confirm Password
              </label>
              <input
                type="password"
                value={confirmPassword}
                onChange={(e) => handleConfirmPasswordChange(e.target.value)}
                placeholder="Confirm password"
                className="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500"
              />
            </div>
            {passwordError && (
              <p className="text-sm text-red-600">{passwordError}</p>
            )}
          </div>
        )}

        {/* Upload Button */}
        <button
          onClick={handleUpload}
          disabled={!selectedFile || isUploading || (isEncrypted && passwordError)}
          className={`w-full py-2 px-4 rounded-md font-medium ${
            !selectedFile || isUploading || (isEncrypted && passwordError)
              ? 'bg-gray-300 text-gray-500 cursor-not-allowed'
              : isEncrypted
              ? 'bg-blue-600 text-white hover:bg-blue-700'
              : 'bg-green-600 text-white hover:bg-green-700'
          }`}
        >
          {isUploading ? (
            <span className="flex items-center justify-center">
              <svg className="animate-spin -ml-1 mr-3 h-5 w-5 text-white" xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 24 24">
                <circle className="opacity-25" cx="12" cy="12" r="10" stroke="currentColor" strokeWidth="4"></circle>
                <path className="opacity-75" fill="currentColor" d="M4 12a8 8 0 018-8V0C5.373 0 0 5.373 0 12h4zm2 5.291A7.962 7.962 0 014 12H0c0 3.042 1.135 5.824 3 7.938l3-2.647z"></path>
              </svg>
              Uploading...
            </span>
          ) : isEncrypted ? (
            '🔒 Upload Securely'
          ) : (
            'Upload Document'
          )}
        </button>

        {/* Status */}
        {selectedFile && isEncrypted && !passwordError && password && confirmPassword && (
          <div className="text-center text-sm text-green-600 bg-green-50 p-2 rounded">
            ✅ Ready for secure upload
          </div>
        )}
      </div>
    </div>
  );
};

export default PasswordProtectedUpload;