import React, { useRef, useEffect, useState } from "react";

interface PlacesAutocompleteInputProps {
  value: string;
  onChange: (value: string) => void;
  onPlaceSelect?: (coordinates: { lat: number; lng: number }, address: string, placeDetails?: any) => void;
  placeholder?: string;
  className?: string;
  types?: string[];
  disabled?: boolean;
}

export function PlacesAutocompleteInput({
  value,
  onChange,
  onPlaceSelect,
  placeholder = "Search for a location...",
  className = "",
  types = ["establishment", "geocode"],
  disabled = false
}: PlacesAutocompleteInputProps) {
  const inputRef = useRef<HTMLInputElement>(null);
  const autocompleteRef = useRef<google.maps.places.Autocomplete | null>(null);
  const [isGoogleMapsReady, setIsGoogleMapsReady] = useState(false);

  useEffect(() => {
    const checkGoogleMaps = () => {
      if (window.google?.maps?.places) {
        setIsGoogleMapsReady(true);
      } else {
        setTimeout(checkGoogleMaps, 100);
      }
    };
    checkGoogleMaps();
  }, []);

  useEffect(() => {
    if (!isGoogleMapsReady || !inputRef.current || disabled) return;

    try {
      // Initialize autocomplete
      autocompleteRef.current = new window.google.maps.places.Autocomplete(
        inputRef.current,
        {
          types,
          fields: ["place_id", "geometry", "name", "formatted_address", "address_components"]
        }
      );

      // Add place changed listener
      const listener = autocompleteRef.current.addListener("place_changed", () => {
        const place = autocompleteRef.current?.getPlace();
        
        if (place?.geometry?.location && onPlaceSelect) {
          const lat = place.geometry.location.lat();
          const lng = place.geometry.location.lng();
          const address = place.formatted_address || place.name || "";
          
          onPlaceSelect({ lat, lng }, address, place);
        }
      });

      return () => {
        if (listener) {
          window.google.maps.event.removeListener(listener);
        }
      };
    } catch (error) {
      console.error('Places Autocomplete error:', error);
    }
  }, [isGoogleMapsReady, onPlaceSelect, types, disabled]);

  return (
    <input
      ref={inputRef}
      type="text"
      value={value}
      onChange={(e) => onChange(e.target.value)}
      placeholder={placeholder}
      className={className}
      autoComplete="off"
      disabled={disabled}
    />
  );
}