import React, { useState } from 'react';
import { Trash2, Shield, AlertTriangle, Download } from 'lucide-react';

const API_BASE = import.meta.env.VITE_API_BASE || 'http://localhost:8090/api';

interface PrivacyControlsProps {
  onDataDeleted?: () => void;
  onDataAnonymized?: () => void;
}

export const PrivacyControls: React.FC<PrivacyControlsProps> = ({
  onDataDeleted,
  onDataAnonymized
}) => {
  const [loading, setLoading] = useState(false);
  const [showDeleteConfirm, setShowDeleteConfirm] = useState(false);
  const [showAnonymizeConfirm, setShowAnonymizeConfirm] = useState(false);

  const getAuthHeaders = () => {
    const token = localStorage.getItem('access_token');
    return token ? { 'Authorization': `Bearer ${token}` } : {};
  };

  const handleDeleteData = async () => {
    setLoading(true);
    try {
      const response = await fetch(`${API_BASE}/behavior/my-data`, {
        method: 'DELETE',
        headers: getAuthHeaders()
      });
      
      if (response.ok) {
        alert('Your behavior data has been permanently deleted.');
        onDataDeleted?.();
      } else {
        throw new Error('Failed to delete data');
      }
    } catch (error) {
      alert('Error deleting data. Please try again.');
    } finally {
      setLoading(false);
      setShowDeleteConfirm(false);
    }
  };

  const handleAnonymizeData = async () => {
    setLoading(true);
    try {
      const response = await fetch(`${API_BASE}/behavior/anonymize-data`, {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
          ...getAuthHeaders()
        }
      });
      
      if (response.ok) {
        alert('Your behavior data has been anonymized.');
        onDataAnonymized?.();
      } else {
        throw new Error('Failed to anonymize data');
      }
    } catch (error) {
      alert('Error anonymizing data. Please try again.');
    } finally {
      setLoading(false);
      setShowAnonymizeConfirm(false);
    }
  };

  const handleExportData = async () => {
    setLoading(true);
    try {
      const [insights, events] = await Promise.all([
        fetch(`${API_BASE}/behavior/my-insights`, { headers: getAuthHeaders() }),
        fetch(`${API_BASE}/behavior/my-events?limit=1000`, { headers: getAuthHeaders() })
      ]);
      
      const data = {
        insights: insights.ok ? await insights.json() : null,
        events: events.ok ? await events.json() : [],
        exported_at: new Date().toISOString()
      };
      
      const blob = new Blob([JSON.stringify(data, null, 2)], { type: 'application/json' });
      const url = URL.createObjectURL(blob);
      const a = document.createElement('a');
      a.href = url;
      a.download = `behavior-data-${new Date().toISOString().split('T')[0]}.json`;
      document.body.appendChild(a);
      a.click();
      document.body.removeChild(a);
      URL.revokeObjectURL(url);
    } catch (error) {
      alert('Error exporting data. Please try again.');
    } finally {
      setLoading(false);
    }
  };

  return (
    <div className="bg-white rounded-lg shadow-lg border border-gray-200 p-6 relative z-10">
      <div className="flex items-center gap-2 mb-4">
        <Shield className="w-5 h-5 text-blue-600" />
        <h3 className="font-semibold text-lg text-gray-900">Privacy Controls</h3>
      </div>
      
      <p className="text-sm text-gray-600 mb-6">
        Manage your behavior data and privacy settings. These actions cannot be undone.
      </p>
      
      <div className="space-y-4">
        <div className="flex items-center justify-between p-4 bg-blue-50 rounded-lg">
          <div className="flex items-center gap-3">
            <Download className="w-5 h-5 text-blue-600" />
            <div>
              <h4 className="font-medium text-blue-900">Export My Data</h4>
              <p className="text-sm text-blue-700">Download all your behavior data</p>
            </div>
          </div>
          <button
            onClick={handleExportData}
            disabled={loading}
            style={{ backgroundColor: '#2563eb', color: '#ffffff' }}
            className="flex items-center space-x-1 px-4 py-2 rounded-lg hover:bg-blue-700 transition-colors shadow-sm font-medium text-sm disabled:bg-blue-400"
          >
            {loading ? 'Exporting...' : 'Export'}
          </button>
        </div>
        
        <div className="flex items-center justify-between p-4 bg-yellow-50 rounded-lg">
          <div className="flex items-center gap-3">
            <Shield className="w-5 h-5 text-yellow-600" />
            <div>
              <h4 className="font-medium text-yellow-900">Anonymize Data</h4>
              <p className="text-sm text-yellow-700">Remove personal identifiers from your data</p>
            </div>
          </div>
          <button
            onClick={() => setShowAnonymizeConfirm(true)}
            disabled={loading}
            style={{ backgroundColor: '#2563eb', color: '#ffffff' }}
            className="flex items-center space-x-1 px-4 py-2 rounded-lg hover:bg-blue-700 transition-colors shadow-sm font-medium text-sm disabled:bg-blue-400"
          >
            Anonymize
          </button>
        </div>
        
        <div className="flex items-center justify-between p-4 bg-red-50 rounded-lg">
          <div className="flex items-center gap-3">
            <Trash2 className="w-5 h-5 text-red-600" />
            <div>
              <h4 className="font-medium text-red-900">Delete All Data</h4>
              <p className="text-sm text-red-700">Permanently remove all your behavior data</p>
            </div>
          </div>
          <button
            onClick={() => setShowDeleteConfirm(true)}
            disabled={loading}
            className="px-4 py-2 bg-red-600 text-white rounded-lg hover:bg-red-700 disabled:opacity-50"
          >
            Delete
          </button>
        </div>
      </div>
      
      {showDeleteConfirm && (
        <div className="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50">
          <div className="bg-white rounded-lg p-6 max-w-md mx-4">
            <div className="flex items-center gap-3 mb-4">
              <AlertTriangle className="w-6 h-6 text-red-600" />
              <h3 className="font-semibold text-lg">Confirm Data Deletion</h3>
            </div>
            <p className="text-gray-600 mb-6">
              This will permanently delete all your behavior data. This action cannot be undone.
            </p>
            <div className="flex gap-3">
              <button
                onClick={() => setShowDeleteConfirm(false)}
                className="flex-1 px-4 py-2 border border-gray-300 rounded-lg hover:bg-gray-50"
              >
                Cancel
              </button>
              <button
                onClick={handleDeleteData}
                disabled={loading}
                className="flex-1 px-4 py-2 bg-red-600 text-white rounded-lg hover:bg-red-700 disabled:opacity-50"
              >
                {loading ? 'Deleting...' : 'Delete'}
              </button>
            </div>
          </div>
        </div>
      )}
      
      {showAnonymizeConfirm && (
        <div className="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50">
          <div className="bg-white rounded-lg p-6 max-w-md mx-4">
            <div className="flex items-center gap-3 mb-4">
              <Shield className="w-6 h-6 text-yellow-600" />
              <h3 className="font-semibold text-lg">Confirm Data Anonymization</h3>
            </div>
            <p className="text-gray-600 mb-6">
              This will remove all personal identifiers from your behavior data while keeping the data for analytics.
            </p>
            <div className="flex gap-3">
              <button
                onClick={() => setShowAnonymizeConfirm(false)}
                className="flex-1 px-4 py-2 border border-gray-300 rounded-lg hover:bg-gray-50"
              >
                Cancel
              </button>
              <button
                onClick={handleAnonymizeData}
                disabled={loading}
                className="flex-1 px-4 py-2 bg-yellow-600 text-white rounded-lg hover:bg-yellow-700 disabled:opacity-50"
              >
                {loading ? 'Anonymizing...' : 'Anonymize'}
              </button>
            </div>
          </div>
        </div>
      )}
    </div>
  );
};