import React, { useState, useEffect } from 'react';
import { 
  FaShieldAlt, FaDownload, FaTrashAlt, FaUser, FaSpinner,
  FaInfoCircle, FaExclamationTriangle, FaCheck, FaEye, FaDatabase,
  FaCalendarAlt, FaChevronDown, FaChevronUp
} from 'react-icons/fa';
import DataRetentionAPI from '../api/dataRetention';
import './PrivacySettings.css';

const PrivacySettings: React.FC = () => {
  const [requesting, setRequesting] = useState(false);
  const [showErasureModal, setShowErasureModal] = useState(false);
  const [erasureReason, setErasureReason] = useState('');
  const [message, setMessage] = useState('');
  const [showDataSummary, setShowDataSummary] = useState(false);
  const [dataSummary, setDataSummary] = useState<any>(null);
  const [loadingData, setLoadingData] = useState(false);

  const handleViewMyData = async () => {
    try {
      setLoadingData(true);
      const summary = await DataRetentionAPI.getUserDataSummary();
      setDataSummary(summary);
      setShowDataSummary(true);
    } catch (err) {
      setMessage('❌ Failed to load data summary: ' + (err instanceof Error ? err.message : 'Unknown error'));
    } finally {
      setLoadingData(false);
    }
  };

  const formatDataSize = (sizeInMB: number) => {
    if (sizeInMB < 1) {
      return `${(sizeInMB * 1024).toFixed(0)} KB`;
    }
    return `${sizeInMB.toFixed(2)} MB`;
  };

  const handleQuickExport = async (exportType: 'personal_data' | 'complete') => {
    try {
      setRequesting(true);
      setMessage('');
      
      console.log('🚀 Requesting export:', exportType);
      const exportResult = await DataRetentionAPI.requestDataExport(exportType);
      console.log('📦 Export result:', exportResult);
      
      if (exportResult.status === 'completed') {
        console.log('✅ Export completed, starting download...');
        // Auto-download the file
        const blob = await DataRetentionAPI.downloadDataExport(exportResult.id);
        const url = window.URL.createObjectURL(blob);
        const a = document.createElement('a');
        a.href = url;
        a.download = `data_export_${exportType}_${exportResult.id}.zip`;
        document.body.appendChild(a);
        a.click();
        window.URL.revokeObjectURL(url);
        document.body.removeChild(a);
        
        setMessage('✅ Export completed! Download started automatically.');
      } else {
        console.log('⏳ Export status:', exportResult.status);
        setMessage('✅ Export request submitted! You will be notified when it\'s ready.');
      }
    } catch (err) {
      console.error('❌ Export error:', err);
      setMessage('❌ Failed to request export: ' + (err instanceof Error ? err.message : 'Unknown error'));
    } finally {
      setRequesting(false);
    }
  };

  const handleErasureRequest = async () => {
    if (!erasureReason.trim()) {
      alert('Please provide a reason for the erasure request.');
      return;
    }

    try {
      setRequesting(true);
      setMessage('');
      await DataRetentionAPI.createErasureRequest({
        request_type: 'complete',
        reason: erasureReason
      });
      setMessage('✅ Erasure request submitted! It will be reviewed by our team.');
      setShowErasureModal(false);
      setErasureReason('');
    } catch (err) {
      setMessage('❌ Failed to submit erasure request: ' + (err instanceof Error ? err.message : 'Unknown error'));
    } finally {
      setRequesting(false);
    }
  };

  return (
    <div className="privacy-settings">
      <div className="privacy-header">
        <h2><FaShieldAlt /> Privacy & Data Settings</h2>
        <p>Manage your personal data and privacy preferences</p>
      </div>

      {message && (
        <div className={`message-box ${message.includes('✅') ? 'success' : 'error'}`}>
          {message}
        </div>
      )}

      <div className="privacy-actions-grid">
        <div className="privacy-action-card">
          <div className="action-icon view">
            <FaEye />
          </div>
          <h3>View My Data</h3>
          <p>See what personal data we store about you and how it's organized.</p>
          <div className="action-buttons">
            <button
              className="btn btn-info"
              onClick={handleViewMyData}
              disabled={loadingData}
            >
              {loadingData ? <FaSpinner className="spinning" /> : <FaEye />}
              View My Data
            </button>
          </div>
        </div>

        <div className="privacy-action-card">
          <div className="action-icon export">
            <FaDownload />
          </div>
          <h3>Export My Data</h3>
          <p>Download a copy of your personal data for backup or transfer to another service.</p>
          <div className="action-buttons">
            <button
              className="btn btn-primary"
              onClick={() => handleQuickExport('personal_data')}
              disabled={requesting}
            >
              {requesting ? <FaSpinner className="spinning" /> : <FaDownload />}
              Export Personal Data
            </button>
            <button
              className="btn btn-secondary"
              onClick={() => handleQuickExport('complete')}
              disabled={requesting}
            >
              Export All Data
            </button>
          </div>
        </div>

        <div className="privacy-action-card">
          <div className="action-icon delete">
            <FaTrashAlt />
          </div>
          <h3>Delete My Account</h3>
          <p>Permanently delete your account and all associated data. This action cannot be undone.</p>
          <div className="action-buttons">
            <button
              className="btn btn-danger"
              onClick={() => setShowErasureModal(true)}
            >
              <FaTrashAlt />
              Request Account Deletion
            </button>
          </div>
        </div>
      </div>

      {showDataSummary && dataSummary && (
        <div className="data-summary-section">
          <div className="section-header">
            <h3><FaDatabase /> My Data Summary</h3>
            <button 
              className="btn btn-sm btn-secondary"
              onClick={() => setShowDataSummary(false)}
            >
              <FaChevronUp /> Hide
            </button>
          </div>

          <div className="user-info-card">
            <div className="user-info">
              <FaUser className="user-icon" />
              <div>
                <h4>{dataSummary.username || 'User'}</h4>
                {dataSummary.email && <p>{dataSummary.email}</p>}
                {dataSummary.phone_number && <p>{dataSummary.phone_number}</p>}
              </div>
            </div>
            
            <div className="user-stats">
              <div className="stat">
                <span className="label">Account Created:</span>
                <span className="value">
                  <FaCalendarAlt />
                  {new Date(dataSummary.account_created).toLocaleDateString()}
                </span>
              </div>
              
              {dataSummary.last_activity && (
                <div className="stat">
                  <span className="label">Last Activity:</span>
                  <span className="value">
                    <FaCalendarAlt />
                    {new Date(dataSummary.last_activity).toLocaleDateString()}
                  </span>
                </div>
              )}
              
              <div className="stat">
                <span className="label">Total Data Size:</span>
                <span className="value">
                  <FaDatabase />
                  {formatDataSize(dataSummary.total_data_size_mb)}
                </span>
              </div>
            </div>
          </div>

          <div className="data-categories">
            <h4>Data Categories</h4>
            <div className="categories-grid">
              {dataSummary.data_categories.map((category: any) => (
                <div key={category.category} className="category-card">
                  <div className="category-header">
                    <FaDatabase />
                    <h5>{category.category.replace('_', ' ').toUpperCase()}</h5>
                  </div>
                  
                  <div className="category-content">
                    <p>{category.description}</p>
                    
                    <div className="category-stats">
                      <div className="stat">
                        <span className="label">Records:</span>
                        <span className="value">{category.data_count.toLocaleString()}</span>
                      </div>
                      
                      <div className="stat">
                        <span className="label">Last Updated:</span>
                        <span className="value">
                          {new Date(category.last_updated).toLocaleDateString()}
                        </span>
                      </div>
                      
                      <div className="stat">
                        <span className="label">Retention:</span>
                        <span className="value">{category.retention_days} days</span>
                      </div>
                    </div>
                  </div>
                </div>
              ))}
            </div>
          </div>
        </div>
      )}

      <div className="privacy-info-section">
        <h3><FaInfoCircle /> Your Privacy Rights</h3>
        <div className="rights-list">
          <div className="right-item">
            <FaCheck className="check-icon" />
            <div>
              <h4>Right to Access</h4>
              <p>You can request information about what personal data we hold about you.</p>
            </div>
          </div>
          
          <div className="right-item">
            <FaCheck className="check-icon" />
            <div>
              <h4>Right to Portability</h4>
              <p>You can request a copy of your data in a machine-readable format.</p>
            </div>
          </div>
          
          <div className="right-item">
            <FaCheck className="check-icon" />
            <div>
              <h4>Right to Erasure</h4>
              <p>You can request deletion of your personal data under certain conditions.</p>
            </div>
          </div>
          
          <div className="right-item">
            <FaCheck className="check-icon" />
            <div>
              <h4>Right to Rectification</h4>
              <p>You can request correction of inaccurate or incomplete personal data.</p>
            </div>
          </div>
        </div>
      </div>

      {showErasureModal && (
        <div className="modal-overlay" onClick={() => setShowErasureModal(false)}>
          <div className="modal-content" onClick={(e) => e.stopPropagation()}>
            <div className="modal-header">
              <h3>Request Account Deletion</h3>
              <button 
                className="modal-close"
                onClick={() => setShowErasureModal(false)}
              >
                ×
              </button>
            </div>
            
            <div className="modal-body">
              <div className="warning-notice">
                <FaExclamationTriangle />
                <div>
                  <h4>This action is permanent</h4>
                  <p>
                    Deleting your account will permanently remove all your data including:
                    saved properties, search history, visit schedules, and profile information.
                    This action cannot be undone.
                  </p>
                </div>
              </div>

              <div className="form-group">
                <label htmlFor="erasureReason">Please tell us why you want to delete your account *</label>
                <textarea
                  id="erasureReason"
                  value={erasureReason}
                  onChange={(e) => setErasureReason(e.target.value)}
                  placeholder="Your feedback helps us improve our service..."
                  rows={4}
                  required
                />
              </div>

              <div className="modal-actions">
                <button
                  className="btn btn-danger"
                  onClick={handleErasureRequest}
                  disabled={requesting || !erasureReason.trim()}
                >
                  {requesting ? <FaSpinner className="spinning" /> : <FaTrashAlt />}
                  Delete My Account
                </button>
                <button
                  className="btn btn-secondary"
                  onClick={() => setShowErasureModal(false)}
                >
                  Cancel
                </button>
              </div>
            </div>
          </div>
        </div>
      )}
    </div>
  );
};

export default PrivacySettings;
