import React, { useState, useEffect } from 'react';
import { Shield, CheckCircle, Clock, AlertTriangle } from 'lucide-react';
import { toast } from 'sonner';

interface KYCData {
  verification_status: string;
  full_name: string;
  date_of_birth: string;
  aadhaar_number: string;
  pan_number: string;
  aadhaar_verified: boolean;
  pan_verified: boolean;
  created_at: string;
}

interface KYCVerificationSectionProps {
  onNavigateToKYC: () => void;
}

export const KYCVerificationSection: React.FC<KYCVerificationSectionProps> = ({ onNavigateToKYC }) => {
  const [kycData, setKycData] = useState<KYCData | null>(null);
  const [loading, setLoading] = useState(true);

  useEffect(() => {
    fetchKYCStatus();
    
    const handleKYCUpdate = () => {
      fetchKYCStatus();
    };
    
    window.addEventListener('kycUpdated', handleKYCUpdate);
    return () => window.removeEventListener('kycUpdated', handleKYCUpdate);
  }, []);

  const fetchKYCStatus = async () => {
    setLoading(true);
    try {
      const user = JSON.parse(localStorage.getItem('user') || '{}');
      const userId = user.user_id;
      
      const response = await fetch(`http://localhost:8090/api/user-account/kyc/profile?user_id=${userId}`);
      const data = await response.json();
      
      if (data.success && data.kyc_data) {
        setKycData(data.kyc_data);
      } else {
        setKycData(null);
      }
    } catch (error) {
      console.error('Failed to fetch KYC status:', error);
      setKycData(null);
    } finally {
      setLoading(false);
    }
  };

  const getKYCStatusIcon = (status?: string) => {
    switch (status) {
      case 'verified':
        return <CheckCircle className="h-5 w-5 text-green-500" />;
      case 'pending':
        return <Clock className="h-5 w-5 text-blue-500" />;
      case 'rejected':
        return <AlertTriangle className="h-5 w-5 text-red-500" />;
      default:
        return <Shield className="h-5 w-5 text-gray-400" />;
    }
  };

  const getKYCStatusBadge = (status?: string) => {
    switch (status) {
      case 'verified':
        return <span className="inline-flex items-center px-2 py-1 rounded-full text-xs font-medium bg-green-100 text-green-800">Verified</span>;
      case 'pending':
        return <span className="inline-flex items-center px-2 py-1 rounded-full text-xs font-medium bg-blue-100 text-blue-800">Pending</span>;
      case 'rejected':
        return <span className="inline-flex items-center px-2 py-1 rounded-full text-xs font-medium bg-red-100 text-red-800">Rejected</span>;
      default:
        return <span className="inline-flex items-center px-2 py-1 rounded-full text-xs font-medium bg-gray-100 text-gray-800">Not Started</span>;
    }
  };

  const getProgressPercentage = () => {
    if (!kycData) return 0;
    if (kycData.verification_status === 'verified') return 100;
    if (kycData.aadhaar_verified && kycData.pan_verified) return 80;
    if (kycData.aadhaar_verified || kycData.pan_verified) return 50;
    return 20;
  };

  if (loading) {
    return (
      <div className="bg-white rounded-2xl shadow-lg border border-gray-200/50 p-6 hover:shadow-xl transition-shadow duration-300">
        <div className="flex items-center justify-center py-8">
          <div className="animate-spin rounded-full h-6 w-6 border-b-2 border-blue-600"></div>
          <span className="ml-2 text-gray-600">Loading KYC status...</span>
        </div>
      </div>
    );
  }

  return (
    <div className="bg-white rounded-2xl shadow-lg border border-gray-200/50 p-6 hover:shadow-xl transition-shadow duration-300">
      <div className="flex items-center justify-between mb-6 pb-4 border-b border-gray-100">
        <div className="flex items-center gap-3">
          <div className="p-2 bg-blue-100 rounded-lg">
            <Shield className="h-5 w-5 text-blue-600" />
          </div>
          <h3 className="text-xl font-bold text-gray-900">KYC Verification</h3>
        </div>
        <button
          onClick={onNavigateToKYC}
          className="text-blue-600 hover:text-blue-800 text-sm font-medium bg-blue-50 px-3 py-2 rounded-lg hover:bg-blue-100 transition-colors"
        >
          {kycData ? 'Manage KYC' : 'Start KYC'} →
        </button>
      </div>
      
      {kycData ? (
        <div className="space-y-4">
          <div className="flex items-center justify-between">
            <div className="flex items-center gap-3">
              {getKYCStatusIcon(kycData.verification_status)}
              <div>
                <p className="font-medium text-gray-900">Verification Status</p>
                <p className="text-sm text-gray-600">
                  {kycData.verification_status === 'verified' 
                    ? 'Your identity has been successfully verified' 
                    : 'Identity verification in progress'
                  }
                </p>
              </div>
            </div>
            {getKYCStatusBadge(kycData.verification_status)}
          </div>
          
          <div className="bg-gray-50 rounded-lg p-4">
            <div className="flex items-center justify-between mb-2">
              <span className="text-sm font-medium text-gray-700">Progress</span>
              <span className="text-sm text-gray-600">{getProgressPercentage()}%</span>
            </div>
            <div className="w-full bg-gray-200 rounded-full h-2">
              <div 
                className={`h-2 rounded-full transition-all duration-300 ${
                  kycData.verification_status === 'verified' ? 'bg-green-600' : 'bg-blue-600'
                }`}
                style={{ width: `${getProgressPercentage()}%` }}
              ></div>
            </div>
          </div>
          
          {kycData.verification_status === 'verified' && (
            <div className="bg-green-50 border border-green-200 rounded-lg p-4">
              <div className="flex items-center gap-2">
                <CheckCircle className="h-5 w-5 text-green-600" />
                <p className="text-green-800 font-medium">KYC Verification Complete</p>
              </div>
              <p className="text-green-700 text-sm mt-1">
                Your Aadhaar and PAN have been successfully verified.
              </p>
            </div>
          )}

          {kycData.verification_status !== 'verified' && (
            <div className="grid grid-cols-2 gap-4 text-sm">
              <div className="flex items-center gap-2">
                {kycData.aadhaar_verified ? (
                  <CheckCircle className="h-4 w-4 text-green-500" />
                ) : (
                  <Clock className="h-4 w-4 text-gray-400" />
                )}
                <span className={kycData.aadhaar_verified ? 'text-green-700' : 'text-gray-600'}>
                  Aadhaar {kycData.aadhaar_verified ? 'Verified' : 'Pending'}
                </span>
              </div>
              <div className="flex items-center gap-2">
                {kycData.pan_verified ? (
                  <CheckCircle className="h-4 w-4 text-green-500" />
                ) : (
                  <Clock className="h-4 w-4 text-gray-400" />
                )}
                <span className={kycData.pan_verified ? 'text-green-700' : 'text-gray-600'}>
                  PAN {kycData.pan_verified ? 'Verified' : 'Pending'}
                </span>
              </div>
            </div>
          )}
        </div>
      ) : (
        <div className="text-center py-8">
          <Shield className="h-12 w-12 text-gray-300 mx-auto mb-4" />
          <h4 className="text-lg font-medium text-gray-900 mb-2">Start KYC Verification</h4>
          <p className="text-gray-600 mb-4">Verify your identity to access all platform features and build trust with other users.</p>
          <button
            onClick={onNavigateToKYC}
            className="bg-gradient-to-r from-blue-600 to-blue-700 text-black px-6 py-3 rounded-lg hover:from-blue-700 hover:to-blue-800 transition-all duration-200 shadow-md hover:shadow-lg font-medium"
          >
            Start Verification
          </button>
        </div>
      )}
    </div>
  );
};