import React, { useState, useEffect, useRef } from 'react';
import { ArrowLeft, Edit3, Save, X, User, Mail, Phone, MapPin, Calendar, Camera, Trash2, LogOut } from 'lucide-react';
import { useNavigate } from 'react-router-dom';
import ProfileCompletionModal from './ProfileCompletionModal';
import { getProfile, updateProfile } from '../../api/profile';
import { logout } from '../../api/auth';
import { KYCVerificationSection } from './KYCVerificationSection';

interface UserProfile {
  user_id: string;
  phone_number: string;
  first_name: string;
  last_name: string;
  email: string;
  date_of_birth: string;
  gender: 'male' | 'female' | 'other';
  country: string;
  city: string;
  address: string;
  postal_code: string;
  profile_picture_url: string;
  bio: string;
  preferences: Record<string, any>;
  profile_completion_status: 'complete' | 'incomplete';
  profile_completion_percentage: number;
  is_verified: boolean;
  status: string;
  created_at: string;
  updated_at: string;
  last_login: string;
}

const DEFAULT_PROFILE_IMAGE = 'https://thumbs.dreamstime.com/b/default-avatar-profile-icon-vector-social-media-user-photo-concept-285140929.jpg';

const Profile: React.FC = () => {
  const navigate = useNavigate();
  const [profile, setProfile] = useState<UserProfile | null>(null);
  const [loading, setLoading] = useState(true);
  const [saving, setSaving] = useState(false);
  const [error, setError] = useState('');
  const [success, setSuccess] = useState('');
  const [isEditing, setIsEditing] = useState(false);
  const [showProfileModal, setShowProfileModal] = useState(false);
  const [editData, setEditData] = useState<Partial<UserProfile>>({});
  const [imageRemoved, setImageRemoved] = useState(false);
  const fileInputRef = useRef<HTMLInputElement | null>(null);
  const fetchingRef = useRef(false);

  useEffect(() => {
    fetchProfile();
  }, []);

  const fetchProfile = async () => {
    const accessToken = localStorage.getItem('access_token');
    const sellerToken = localStorage.getItem('sellerToken');
    const agentToken = localStorage.getItem('agentToken');
    
    if (!accessToken && !sellerToken && !agentToken) {
      setError('No access token');
      setLoading(false);
      return;
    }

    // Prevent multiple concurrent API calls using ref
    if (fetchingRef.current) {
      setLoading(false);
      return;
    }
    
    fetchingRef.current = true;

    try {
      let data;
      
      // Try seller-specific API first if seller is logged in, fallback to regular API
      if (sellerToken) {
        try {
          const { getSellerProfile } = await import('../../api/sellerProfile');
          data = await getSellerProfile();
          // Map seller profile fields to standard profile format
          data = {
            ...data,
            first_name: data.full_name?.split(' ')[0] || 'User',
            last_name: data.full_name?.split(' ').slice(1).join(' ') || 'Profile',
            profile_phone: data.phone_number,
            profile_completion_percentage: 100,
            profile_completion_status: 'complete'
          };
        } catch (sellerError) {
          console.warn('Seller API failed, creating profile from localStorage:', sellerError);
          // Create profile from seller localStorage data
          const sellerEmail = localStorage.getItem('sellerEmail') || localStorage.getItem('login_email') || '';
          data = {
            user_id: localStorage.getItem('user_id') || '',
            email: sellerEmail,
            first_name: 'Seller',
            last_name: 'User',
            phone_number: '',
            date_of_birth: '',
            gender: 'male',
            country: '',
            city: '',
            address: '',
            postal_code: '',
            profile_picture_url: '',
            bio: '',
            preferences: {},
            profile_completion_status: 'incomplete',
            profile_completion_percentage: 50,
            is_verified: false,
            status: 'active',
            created_at: new Date().toISOString(),
            updated_at: new Date().toISOString(),
            last_login: new Date().toISOString()
          };
        }
      } else {
        data = await getProfile();
      }
      
      console.log('Profile data received:', data);
      
      // For sellers, get email from localStorage if not in profile
      let email = data.email;
      if (!email && sellerToken) {
        email = localStorage.getItem('sellerEmail') || localStorage.getItem('login_email') || '';
      }
      
      // Ensure all fields are properly mapped
      const profileData = {
        ...data,
        phone_number: data.phone_number || (data as any).profile_phone || '',
        email: email || localStorage.getItem('login_email') || '',
        first_name: data.first_name || 'User',
        last_name: data.last_name || 'Profile',
        date_of_birth: data.date_of_birth || '',
        gender: data.gender || 'male',
        country: data.country || '',
        city: data.city || '',
        address: data.address || '',
        postal_code: data.postal_code || '',
        profile_picture_url: data.profile_picture_url || '',
        bio: data.bio || '',
        preferences: data.preferences || {},
        profile_completion_status: data.profile_completion_status || 'incomplete',
        profile_completion_percentage: data.profile_completion_percentage || 0,
        is_verified: data.is_verified || false,
        status: data.status || 'active',
        created_at: data.created_at || new Date().toISOString(),
        updated_at: data.updated_at || new Date().toISOString(),
        last_login: data.last_login || new Date().toISOString()
      };
      
      setProfile(profileData);
      setEditData(profileData);
      setImageRemoved(false);
      localStorage.setItem('user', JSON.stringify(profileData));
    } catch (err: any) {
      if (err.message?.includes('401')) {
        setError('Session expired. Please login again.');
        localStorage.removeItem('access_token');
        localStorage.removeItem('refresh_token');
        localStorage.removeItem('sellerToken');
        localStorage.removeItem('agentToken');
      } else {
        setError(err.message || 'Failed to load profile');
      }
    } finally {
      setLoading(false);
      fetchingRef.current = false;
    }
  };

  const handleEdit = () => {
    setIsEditing(true);
    setError('');
    setSuccess('');
  };

  const handleCancel = () => {
    setIsEditing(false);
    setEditData(profile || {});
    setError('');
    setSuccess('');
    setImageRemoved(false);
  };

  const handleProfileImageChange = async (file?: File | null) => {
    if (!file) return;
    
    // Show preview immediately
    const reader = new FileReader();
    reader.onload = () => {
      setEditData(prev => ({ ...prev, profile_picture_url: reader.result as string }));
      setImageRemoved(false);
    };
    reader.readAsDataURL(file);
    
    // Upload to backend
    try {
      setSaving(true);
      const { uploadProfileImage } = await import('../../api/profile');
      const result = await uploadProfileImage(file);
      setProfile(result.profile);
      setEditData(result.profile);
      setSuccess('Profile image uploaded successfully!');
      setTimeout(() => setSuccess(''), 3000);
    } catch (err: any) {
      setError('Failed to upload image. Please try again.');
      setTimeout(() => setError(''), 3000);
    } finally {
      setSaving(false);
    }
  };

  const onPickImage = () => fileInputRef.current?.click();

  const removeProfileImage = async () => {
    try {
      setSaving(true);
      const { removeProfileImage: removeImage } = await import('../../api/profile');
      const result = await removeImage();
      setProfile(result.profile);
      setEditData(result.profile);
      setImageRemoved(true);
      setSuccess('Profile image removed successfully!');
      setTimeout(() => setSuccess(''), 3000);
    } catch (err: any) {
      setError('Failed to remove image. Please try again.');
      setTimeout(() => setError(''), 3000);
    } finally {
      setSaving(false);
    }
  };

  const handleLogout = async () => {
    try {
      await logout();
    } catch (err) {
      console.error('Logout API failed:', err);
    } finally {
      localStorage.removeItem('access_token');
      localStorage.removeItem('refresh_token');
      navigate('/login');
    }
  };

  const handleSave = async () => {
    setSaving(true);
    setError('');
    setSuccess('');

    const sellerToken = localStorage.getItem('sellerToken');
    
    if (sellerToken) {
      // For sellers, just update local state and show success
      try {
        const updatedProfile = {
          ...profile,
          first_name: editData.first_name?.trim() || profile?.first_name || "User",
          last_name: editData.last_name?.trim() || profile?.last_name || "Profile",
          date_of_birth: editData.date_of_birth || profile?.date_of_birth || '',
          gender: editData.gender || profile?.gender || 'male',
          country: editData.country?.trim() || profile?.country || '',
          city: editData.city?.trim() || profile?.city || '',
          address: editData.address?.trim() || profile?.address || '',
          postal_code: editData.postal_code?.trim() || profile?.postal_code || '',
          bio: editData.bio?.trim() || profile?.bio || ''
        };
        
        setProfile(updatedProfile);
        setEditData(updatedProfile);
        localStorage.setItem('user', JSON.stringify(updatedProfile));
        
        setIsEditing(false);
        setImageRemoved(false);
        setSuccess('Profile updated successfully!');
        setTimeout(() => setSuccess(''), 3000);
      } catch (err) {
        setError('Failed to update profile. Please try again.');
      } finally {
        setSaving(false);
      }
      return;
    }

    const accessToken = localStorage.getItem('access_token');
    const agentToken = localStorage.getItem('agentToken');
    
    if (!accessToken && !agentToken) {
      setError('Please login first');
      setSaving(false);
      return;
    }

    try {
      const profileData = {
        first_name: editData.first_name?.trim() || "User",
        last_name: editData.last_name?.trim() || "Profile",
        email: profile?.email || editData.email?.trim(),
        phone_number: profile?.phone_number || (profile as any)?.profile_phone,
        date_of_birth: editData.date_of_birth ? (editData.date_of_birth.includes('T') ? editData.date_of_birth : editData.date_of_birth + 'T00:00:00.000Z') : "2000-01-01T00:00:00.000Z",
        gender: editData.gender || "male",
        country: editData.country?.trim() || "India",
        city: editData.city?.trim() || "City",
        address: editData.address?.trim() || "Address",
        postal_code: editData.postal_code?.trim() || "000000",
        profile_picture_url: imageRemoved ? "" : (editData.profile_picture_url?.trim() || ""),
        bio: editData.bio?.trim() || "Bio not updated",
        preferences: editData.preferences && Object.keys(editData.preferences).length > 0 ? editData.preferences : { "default": "preferences" }
      };
      
      const updatedProfile = await updateProfile(profileData);
      
      if (!updatedProfile.phone_number && (updatedProfile as any).profile_phone) {
        updatedProfile.phone_number = (updatedProfile as any).profile_phone;
      }
      
      setProfile(updatedProfile);
      setEditData(updatedProfile);
      localStorage.setItem('user', JSON.stringify(updatedProfile));
      window.dispatchEvent(new Event('profileUpdated'));
      
      setIsEditing(false);
      setImageRemoved(false);
      setSuccess('Profile updated successfully!');
      setTimeout(() => setSuccess(''), 3000);
    } catch (err) {
      setError('Failed to update profile. Please try again.');
    } finally {
      setSaving(false);
    }
  };

  const handleInputChange = (e: React.ChangeEvent<HTMLInputElement | HTMLSelectElement | HTMLTextAreaElement>) => {
    const { name, value } = e.target;
    setEditData(prev => ({ ...prev, [name]: value }));
  };

  const formatDate = (dateString?: string) => {
    if (!dateString) return 'Not provided';
    return new Date(dateString).toLocaleDateString('en-US', {
      year: 'numeric',
      month: 'long',
      day: 'numeric'
    });
  };

  const formatPhoneNumber = (phoneNumber?: string, profilePhone?: string) => {
    // Use profile_phone if phone_number is empty
    const actualPhone = phoneNumber || profilePhone;
    if (!actualPhone) return 'No phone number provided';
    // Filter out invalid phone numbers that start with 'email_' or contain non-numeric characters
    if (actualPhone.startsWith('email_') || actualPhone.includes('_') || !/^[+]?[0-9\s-()]+$/.test(actualPhone)) {
      return 'No phone number provided';
    }
    return actualPhone;
  };

  const getCompletionColor = (percentage: number) => {
    if (percentage >= 80) return 'bg-green-500';
    if (percentage >= 50) return 'bg-yellow-500';
    return 'bg-red-500';
  };

  if (loading) {
    return (
      <div className="min-h-screen bg-gray-50 flex items-center justify-center">
        <div className="text-center">
          <div className="animate-spin rounded-full h-12 w-12 border-b-2 border-blue-600 mx-auto mb-4"></div>
          <p className="text-gray-600">Loading profile...</p>
        </div>
      </div>
    );
  }

  if (error && !profile) {
    return (
      <div className="min-h-screen bg-gray-50 flex items-center justify-center">
        <div className="text-center max-w-md mx-auto p-6">
          <div className="text-red-600 text-lg mb-4">{error}</div>
          <button 
            onClick={() => fetchProfile()}
            className="bg-blue-600 text-white px-6 py-2 rounded-lg hover:bg-blue-700 transition-colors"
          >
            Try Again
          </button>
        </div>
      </div>
    );
  }

  if (!profile) {
    return (
      <div className="min-h-screen bg-gray-50 flex items-center justify-center">
        <div className="text-gray-600">No profile data found</div>
      </div>
    );
  }

  return (
    <div className="min-h-screen bg-gradient-to-br from-gray-50 via-blue-50 to-gray-100 relative">
      {/* Saving Overlay */}
      {saving && (
        <div className="fixed inset-0 backdrop-blur-md bg-white/10 flex items-center justify-center z-50">
          <div className="bg-white/95 backdrop-blur-xl rounded-2xl p-8 shadow-2xl border border-white/20 flex flex-col items-center space-y-4 max-w-sm mx-4">
            <div className="relative">
              <div className="w-16 h-16 border-4 border-blue-200 rounded-full animate-spin">
                <div className="absolute top-0 left-0 w-16 h-16 border-4 border-blue-600 rounded-full border-t-transparent animate-spin"></div>
              </div>
              <div className="absolute inset-0 flex items-center justify-center">
                <div className="w-2 h-2 bg-blue-600 rounded-full animate-bounce"></div>
              </div>
            </div>
         <div className="fixed inset-0 backdrop-blur-sm bg-opacity-50 flex items-center justify-center z-50">
  <div className="bg-white rounded-xl shadow-lg p-6 w-full max-w-md mx-4">
    <div className="text-center">
      <h3 className="text-lg font-semibold text-gray-900 mb-1">Saving Profile</h3>
      <p className="text-sm text-gray-600">Please wait while we update your information...</p>
    </div>
  </div>
</div>

            <div className="flex space-x-1">
              <div className="w-2 h-2 bg-blue-600 rounded-full animate-bounce" style={{animationDelay: '0ms'}}></div>
              <div className="w-2 h-2 bg-blue-600 rounded-full animate-bounce" style={{animationDelay: '150ms'}}></div>
              <div className="w-2 h-2 bg-blue-600 rounded-full animate-bounce" style={{animationDelay: '300ms'}}></div>
            </div>
          </div>
        </div>
      )}
      {/* Header */}
      <div className="bg-white border-b border-gray-200 sticky top-0 z-40">
        <div className="max-w-6xl mx-auto px-4 sm:px-6 lg:px-8">
          <div className="flex items-center justify-between h-16">
            <div className="flex items-center space-x-4">
              <button
                onClick={() => navigate('/')}
                className="p-2 hover:bg-gray-100 rounded-full transition-colors"
              >
                <ArrowLeft className="w-5 h-5 text-gray-600" />
              </button>
              <div>
                <h1 className="text-xl font-semibold text-gray-900">My Profile</h1>
                <p className="text-sm text-gray-500">Manage your personal information</p>
              </div>
            </div>
            
            <div className="flex items-center gap-3">
              {!isEditing ? (
                <>
                  <button
                    onClick={handleEdit}
                    className="flex items-center gap-2 px-4 py-2 text-blue-600 hover:bg-blue-50 rounded-lg transition-colors border border-blue-200"
                  >
                    <Edit3 className="w-4 h-4" />
                    <span>Edit Profile</span>
                  </button>
                  <button
                    onClick={handleLogout}
                    className="flex items-center gap-2 px-4 py-2 text-red-600 hover:bg-red-50 rounded-lg transition-colors border border-red-200"
                  >
                    <LogOut className="w-4 h-4" />
                    <span>Logout</span>
                  </button>
                </>
              ) : (
                <>
                  <button
                    onClick={handleCancel}
                    className="flex items-center gap-2 px-4 py-2 text-gray-600 hover:bg-gray-100 rounded-lg transition-colors border border-gray-300"
                  >
                    <X className="w-4 h-4" />
                    <span>Cancel</span>
                  </button>
                  <button
                    onClick={handleSave}
                    disabled={saving}
                    className="flex items-center gap-2 px-4 py-2 text-green-600 hover:bg-green-50 rounded-lg transition-colors border border-green-200 disabled:opacity-50"
                  >
                    {saving ? (
                      <div className="w-4 h-4 border-2 border-green-600 border-t-transparent rounded-full animate-spin"></div>
                    ) : (
                      <Save className="w-4 h-4" />
                    )}
                    <span>{saving ? 'Saving...' : 'Save Changes'}</span>
                  </button>
                </>
              )}
            </div>
          </div>
        </div>
      </div>

      <div className="max-w-6xl mx-auto px-4 sm:px-6 lg:px-8 py-8">
        {/* Success/Error Messages */}
        {error && (
          <div className="bg-red-50 border border-red-200 text-red-700 px-4 py-3 rounded-lg mb-6">
            {error}
          </div>
        )}

        {success && (
          <div className="bg-green-50 border border-green-200 text-green-700 px-4 py-3 rounded-lg mb-6">
            {success}
          </div>
        )}

        {/* Profile Header Card */}
        <div className="bg-white rounded-lg shadow-sm border border-gray-200 overflow-hidden mb-6">
          <div className="bg-gradient-to-r from-blue-600 to-blue-700 h-32 relative">
            <div className="absolute inset-0 bg-black/10"></div>
          </div>
          <div className="px-6 pb-6">
            <div className="flex flex-col sm:flex-row items-center sm:items-end justify-between -mt-12 mb-4 gap-4">
              <div className="relative group">
                <div className="w-20 h-20 bg-white rounded-full p-1 shadow-lg cursor-pointer" onClick={onPickImage}>
                  <img 
                    src={imageRemoved ? DEFAULT_PROFILE_IMAGE : (profile.profile_picture_url ? `${import.meta.env.VITE_API_BASE || 'http://localhost:8090/api'}${profile.profile_picture_url}` : DEFAULT_PROFILE_IMAGE)} 
                    alt="Profile" 
                    className="w-full h-full rounded-full object-cover hover:opacity-80 transition-opacity"
                    onError={(e) => {
                      const target = e.target as HTMLImageElement;
                      target.src = DEFAULT_PROFILE_IMAGE;
                    }}
                  />
                  <div className="absolute inset-0 flex items-center justify-center bg-black bg-opacity-50 rounded-full opacity-0 group-hover:opacity-100 transition-opacity">
                    <Camera className="w-4 h-4 text-white" />
                  </div>
                </div>
                <input ref={fileInputRef} type="file" accept="image/*" className="hidden" onChange={(e) => handleProfileImageChange(e.target.files?.[0] ?? null)} />
              </div>
              
              <div className="text-center sm:text-right">
                {profile.profile_completion_percentage < 100 && (
                  <div className="flex items-center gap-2 justify-center sm:justify-end mb-2">
                    <div className="w-24 bg-gray-200 rounded-full h-2">
                      <div className={`h-2 rounded-full ${getCompletionColor(profile.profile_completion_percentage || 0)}`} style={{ width: `${profile.profile_completion_percentage || 0}%` }}></div>
                    </div>
                    <span className="text-sm font-medium text-gray-700">{profile.profile_completion_percentage || 0}%</span>
                  </div>
                )}
              </div>
            </div>

            <div className="text-center sm:text-left">
              <h2 className="text-2xl font-bold text-gray-900 mb-2">
                {profile.first_name} {profile.last_name}
              </h2>
              <div className="flex flex-wrap items-center justify-center sm:justify-start gap-3 text-sm text-gray-600">
                <div className="flex items-center gap-1">
                  <Mail className="w-4 h-4 text-blue-600" />
                  <span>{profile.email || 'No email provided'}</span>
                </div>
                <div className="flex items-center gap-1">
                  <Phone className="w-4 h-4 text-green-600" />
                  <span>{formatPhoneNumber(profile.phone_number, (profile as any).profile_phone)}</span>
                </div>
                {profile.city && (
                  <div className="flex items-center gap-1">
                    <MapPin className="w-4 h-4 text-red-600" />
                    <span>{profile.city}, {profile.country}</span>
                  </div>
                )}
              </div>
            </div>
          </div>
        </div>

        {/* Profile Details */}
        <div className="grid grid-cols-1 lg:grid-cols-2 gap-6">
          {/* Personal Information */}
          <div className="bg-white rounded-lg shadow-sm border border-gray-200 p-6">
            <div className="flex items-center gap-3 mb-4 pb-3 border-b border-gray-100">
              <div className="p-2 bg-blue-50 rounded-lg">
                <User className="w-5 h-5 text-blue-600" />
              </div>
              <h3 className="text-lg font-semibold text-gray-900">Personal Information</h3>
            </div>
            <div className="space-y-6">
              <div className="grid grid-cols-2 gap-4">
                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-2">First Name</label>
                  {isEditing ? (
                    <input
                      type="text"
                      name="first_name"
                      value={editData.first_name || ''}
                      onChange={handleInputChange}
                      placeholder="Enter your first name"
                      className="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500 transition-all duration-200 bg-gray-50 focus:bg-white"
                    />
                  ) : (
                    <p className="text-gray-900 py-3 font-medium">{profile.first_name || <span className="text-gray-400 italic font-normal">Not provided</span>}</p>
                  )}
                </div>
                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-2">Last Name</label>
                  {isEditing ? (
                    <input
                      type="text"
                      name="last_name"
                      value={editData.last_name || ''}
                      onChange={handleInputChange}
                      placeholder="Enter your last name"
                      className="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500 transition-all duration-200 bg-gray-50 focus:bg-white"
                    />
                  ) : (
                    <p className="text-gray-900 py-3 font-medium">{profile.last_name || <span className="text-gray-400 italic font-normal">Not provided</span>}</p>
                  )}
                </div>
              </div>

              <div>
                <label className="block text-sm font-medium text-gray-700 mb-2">Email Address</label>
                <div className="relative">
                  <input
                    type="email"
                    name="email"
                    value={profile.email || ''}
                    readOnly
                    className="w-full px-4 py-3 border border-gray-200 rounded-lg bg-gradient-to-r from-gray-50 to-gray-100 text-gray-700 cursor-not-allowed font-medium"
                  />
                  <span className="absolute right-3 top-3 text-xs text-gray-500 bg-white px-2 py-1 rounded-md border border-gray-200">🔒 Locked</span>
                </div>
              </div>

              <div>
                <label className="block text-sm font-medium text-gray-700 mb-2">Phone Number</label>
                <div className="relative">
                  <input
                    type="tel"
                    name="phone_number"
                    value={formatPhoneNumber(profile.phone_number, (profile as any).profile_phone) === 'No phone number provided' ? '' : (profile.phone_number || (profile as any).profile_phone || '')}
                    readOnly
                    placeholder="No phone number provided"
                    className="w-full px-4 py-3 border border-gray-200 rounded-lg bg-gradient-to-r from-gray-50 to-gray-100 text-gray-700 cursor-not-allowed font-medium"
                  />
                  <span className="absolute right-3 top-3 text-xs text-gray-500 bg-white px-2 py-1 rounded-md border border-gray-200">🔒 Locked</span>
                </div>
              </div>

              <div className="grid grid-cols-2 gap-4">
                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-2">Date of Birth</label>
                  {isEditing ? (
                    <input
                      type="date"
                      name="date_of_birth"
                      value={editData.date_of_birth?.split('T')[0] || ''}
                      onChange={handleInputChange}
                      className="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500 transition-all duration-200 bg-gray-50 focus:bg-white"
                    />
                  ) : (
                    <p className="text-gray-900 py-3 font-medium">{profile.date_of_birth ? formatDate(profile.date_of_birth) : <span className="text-gray-400 italic font-normal">Not provided</span>}</p>
                  )}
                </div>
                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-2">Gender</label>
                  {isEditing ? (
                    <select
                      name="gender"
                      value={editData.gender || ''}
                      onChange={handleInputChange}
                      className="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500 transition-all duration-200 bg-gray-50 focus:bg-white"
                    >
                      <option value="">Select Gender</option>
                      <option value="male">Male</option>
                      <option value="female">Female</option>
                      <option value="other">Other</option>
                    </select>
                  ) : (
                    <p className="text-gray-900 py-3 capitalize font-medium">{profile.gender || <span className="text-gray-400 italic font-normal">Not provided</span>}</p>
                  )} 
                </div>
              </div>
            </div>
          </div>

          {/* Location Information */}
          <div className="bg-white rounded-lg shadow-sm border border-gray-200 p-6">
            <div className="flex items-center gap-3 mb-4 pb-3 border-b border-gray-100">
              <div className="p-2 bg-green-50 rounded-lg">
                <MapPin className="w-5 h-5 text-green-600" />
              </div>
              <h3 className="text-lg font-semibold text-gray-900">Location Information</h3>
            </div>
            <div className="space-y-6">
              <div className="grid grid-cols-2 gap-4">
                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-2">Country</label>
                  {isEditing ? (
                    <input
                      type="text"
                      name="country"
                      value={editData.country || ''}
                      onChange={handleInputChange}
                      placeholder="Enter your country"
                      className="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500 transition-all duration-200 bg-gray-50 focus:bg-white"
                    />
                  ) : (
                    <p className="text-gray-900 py-3 font-medium">{profile.country || <span className="text-gray-400 italic font-normal">Not provided</span>}</p>
                  )}
                </div>
                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-2">City</label>
                  {isEditing ? (
                    <input
                      type="text"
                      name="city"
                      value={editData.city || ''}
                      onChange={handleInputChange}
                      placeholder="Enter your city"
                      className="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500 transition-all duration-200 bg-gray-50 focus:bg-white"
                    />
                  ) : (
                    <p className="text-gray-900 py-3 font-medium">{profile.city || <span className="text-gray-400 italic font-normal">Not provided</span>}</p>
                  )}
                </div>
              </div>

              <div>
                <label className="block text-sm font-medium text-gray-700 mb-2">Address</label>
                {isEditing ? (
                  <input
                    type="text"
                    name="address"
                    value={editData.address || ''}
                    onChange={handleInputChange}
                    placeholder="Enter your address"
                    className="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500 transition-all duration-200 bg-gray-50 focus:bg-white"
                  />
                ) : (
                  <p className="text-gray-900 py-3 font-medium">{profile.address || <span className="text-gray-400 italic font-normal">Not provided</span>}</p>
                )}
              </div>

              <div>
                <label className="block text-sm font-medium text-gray-700 mb-2">Postal Code</label>
                {isEditing ? (
                  <input
                    type="text"
                    name="postal_code"
                    value={editData.postal_code || ''}
                    onChange={handleInputChange}
                    placeholder="Enter postal code"
                    className="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500 transition-all duration-200 bg-gray-50 focus:bg-white"
                  />
                ) : (
                  <p className="text-gray-900 py-3 font-medium">{profile.postal_code || <span className="text-gray-400 italic font-normal">Not provided</span>}</p>
                )}
              </div>
            </div>
          </div>

          {/* Bio Section */}
          <div className="bg-white rounded-lg shadow-sm border border-gray-200 p-6 lg:col-span-2">
            <div className="flex items-center gap-3 mb-4 pb-3 border-b border-gray-100">
              <div className="p-2 bg-purple-50 rounded-lg">
                <User className="w-5 h-5 text-purple-600" />
              </div>
              <h3 className="text-lg font-semibold text-gray-900">About Me</h3>
            </div>
            {isEditing ? (
              <textarea
                name="bio"
                value={editData.bio || ''}
                onChange={handleInputChange}
                rows={4}
                className="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500 transition-all duration-200 resize-none bg-gray-50 focus:bg-white"
                placeholder="Tell us about yourself..."
              />
            ) : (
              <p className="text-gray-700 leading-relaxed">{profile.bio || <span className="text-gray-400 italic">Add your bio to tell others about yourself</span>}</p>
            )}
          </div>

          {/* KYC Verification Section */}
          <div className="lg:col-span-2 mb-6">
            <KYCVerificationSection onNavigateToKYC={() => navigate('/kyc')} />
          </div>

          {/* Account Information */}
          <div className="bg-white rounded-lg shadow-sm border border-gray-200 p-6 lg:col-span-2">
            <div className="flex items-center gap-3 mb-4 pb-3 border-b border-gray-100">
              <div className="p-2 bg-orange-50 rounded-lg">
                <Calendar className="w-5 h-5 text-orange-600" />
              </div>
              <h3 className="text-lg font-semibold text-gray-900">Account Information</h3>
            </div>
            <div className="grid grid-cols-1 md:grid-cols-3 gap-6">
              <div className="bg-gradient-to-br from-green-50 to-green-100/50 p-4 rounded-xl border border-green-200">
                <label className="block text-sm font-semibold text-gray-600 mb-2">Account Status</label>
                <span className={`inline-flex items-center px-4 py-2 rounded-full text-sm font-bold shadow-sm ${
                  profile.status === 'active' ? 'bg-green-500 text-white' : 'bg-red-500 text-white'
                }`}>
                  ● {profile.status.toUpperCase()}
                </span>
              </div>
              <div className="bg-gradient-to-br from-blue-50 to-blue-100/50 p-4 rounded-xl border border-blue-200">
                <label className="block text-sm font-semibold text-gray-600 mb-2">Member Since</label>
                <p className="text-gray-900 font-semibold">{formatDate(profile.created_at)}</p>
              </div>
              <div className="bg-gradient-to-br from-purple-50 to-purple-100/50 p-4 rounded-xl border border-purple-200">
                <label className="block text-sm font-semibold text-gray-600 mb-2">Last Login</label>
                <p className="text-gray-900 font-semibold">{formatDate(profile.last_login)}</p>
              </div>
            </div>
          </div>
        </div>
      </div>

      <ProfileCompletionModal 
        isOpen={showProfileModal} 
        onClose={() => setShowProfileModal(false)}
        onComplete={() => {
          setShowProfileModal(false);
          fetchProfile(); // Refresh profile data
        }}
      />
    </div>
  );
};

export default Profile;