import React, { useState, useEffect, useRef } from 'react';
import { completeProfile, getProfile, updateUserProfile } from '../../api/profile';
import { Upload, Check, User, MapPin, Settings, Calendar } from 'lucide-react';
import DatePicker from 'react-datepicker';
import 'react-datepicker/dist/react-datepicker.css';
import './DatePicker.css';

interface ProfileCompletionModalProps {
  isOpen: boolean;
  onClose: () => void;
  onComplete: () => void;
}

interface ProfileData {
  first_name: string;
  last_name: string;
  email: string;
  phone_number: string;
  date_of_birth: string;
  gender: 'male' | 'female' | 'other' | '';
  country: string;
  city: string;
  address: string;
  postal_code: string;
  profile_picture_url: string;
  bio: string;
  preferences: Record<string, any>;
}

const ProfileCompletionModal: React.FC<ProfileCompletionModalProps> = ({ isOpen, onClose, onComplete }) => {
  const [currentStep, setCurrentStep] = useState(1);
  const [loading, setLoading] = useState(false);
  const [error, setError] = useState('');
  const [success, setSuccess] = useState('');
  const [profileImage, setProfileImage] = useState<File | null>(null);
  const [imagePreview, setImagePreview] = useState<string>('');
  const [fieldErrors, setFieldErrors] = useState<Record<string, string>>({});
  const [showValidationMessage, setShowValidationMessage] = useState(false);
  const [isFirstTimeUser, setIsFirstTimeUser] = useState(true);
  const [isProfileComplete, setIsProfileComplete] = useState(false);
  const fileInputRef = useRef<HTMLInputElement>(null);
  
  // OTP verification states
  const [isEmailVerified, setIsEmailVerified] = useState(false);
  const [isPhoneVerified, setIsPhoneVerified] = useState(false);
  const [emailOtp, setEmailOtp] = useState('');
  const [phoneOtp, setPhoneOtp] = useState('');
  const [otpSentEmail, setOtpSentEmail] = useState(false);
  const [otpSentPhone, setOtpSentPhone] = useState(false);
  const [emailReadOnly, setEmailReadOnly] = useState(false);
  const [phoneReadOnly, setPhoneReadOnly] = useState(false);
  const [loginMethod, setLoginMethod] = useState<'email' | 'phone' | null>(null);
  
  const [profileData, setProfileData] = useState<ProfileData>({
    first_name: '',
    last_name: '',
    email: (() => {
      const storedUser = JSON.parse(localStorage.getItem('user') || '{}');
      return storedUser.email || localStorage.getItem('login_email') || '';
    })(),
    phone_number: '+91',
    date_of_birth: '',
    gender: '',
    country: 'India',
    city: '',
    address: '',
    postal_code: '',
    profile_picture_url: '',
    bio: '',
    preferences: {}
  });
  
  const [preferencesText, setPreferencesText] = useState('');

  useEffect(() => {
    if (isOpen) {
      // Check for agent/seller tokens and close modal immediately
      const agentToken = localStorage.getItem('agentToken');
      const sellerToken = localStorage.getItem('sellerToken');
      
      if (agentToken || sellerToken) {
        onClose();
        return;
      }
      
      // Set email immediately
      const storedUser = JSON.parse(localStorage.getItem('user') || '{}');
      const emailFromStorage = storedUser.email || localStorage.getItem('login_email') || '';
      if (emailFromStorage) {
        setProfileData(prev => ({ ...prev, email: emailFromStorage }));
      }
      
      const accessToken = localStorage.getItem('access_token');
      const isAdminLoggedIn = localStorage.getItem('adminLoggedIn') === 'true';

      // Skip authentication check for admin users
      if (!accessToken && !isAdminLoggedIn) {
        setError('Authentication required. Please log in again.');
        return;
      }

      // For admin users, skip profile completion
      if (isAdminLoggedIn) {
        onClose();
        return;
      }
      
      determineLoginMethod();
      
      // Skip API call for new users - start at step 1
      if (storedUser.is_new_user) {
        setIsFirstTimeUser(true);
        setCurrentStep(1);
      } else {
        // Load profile for existing users
        loadExistingProfile();
      }
    }
  }, [isOpen]);
  
  const determineLoginMethod = () => {
    const loginType = localStorage.getItem('login_type');
    const storedUser = JSON.parse(localStorage.getItem('user') || '{}');
    const loginEmail = localStorage.getItem('login_email');
    
    // If user logged in with email/password or has verified email, mark as verified
    if (loginType === 'email' || storedUser.email_verified || loginEmail) {
      setLoginMethod('email');
      setEmailReadOnly(true);
      setIsEmailVerified(true);
      setIsPhoneVerified(false);
      setPhoneReadOnly(false);
    } else if (loginType === 'phone') {
      setLoginMethod('phone');
      setPhoneReadOnly(true);
      setIsPhoneVerified(true);
      setIsEmailVerified(false);
      setEmailReadOnly(false);
    } else {
      // If no login method detected, allow both
      setEmailReadOnly(false);
      setPhoneReadOnly(false);
    }
  };



  const loadExistingProfile = async () => {
    if (sessionStorage.getItem('profile_modal_load_in_progress')) return;
    sessionStorage.setItem('profile_modal_load_in_progress', 'true');
    
    try {
      const existingProfile = await getProfile();
      
      const hasCompletedMandatory = existingProfile.first_name && existingProfile.last_name && 
                                   existingProfile.email && existingProfile.phone_number && 
                                   existingProfile.date_of_birth && existingProfile.gender;
      
      setIsFirstTimeUser(!hasCompletedMandatory);
      
      if (!hasCompletedMandatory) {
        setCurrentStep(1);
      } else if (!existingProfile.country || !existingProfile.city || !existingProfile.address) {
        setCurrentStep(2);
      } else {
        setCurrentStep(3);
      }
      
      const emailToUse = existingProfile.email || localStorage.getItem('login_email') || '';
      const cleanPhoneNumber = existingProfile.phone_number?.startsWith('+') ? existingProfile.phone_number : '+91';
      
      setProfileData({
        first_name: existingProfile.first_name || '',
        last_name: existingProfile.last_name || '',
        email: emailToUse,
        phone_number: cleanPhoneNumber,
        date_of_birth: existingProfile.date_of_birth?.split('T')[0] || '',
        gender: (existingProfile.gender as 'male' | 'female' | 'other') || '',
        country: existingProfile.country || 'India',
        city: existingProfile.city || '',
        address: existingProfile.address || '',
        postal_code: existingProfile.postal_code || '',
        profile_picture_url: existingProfile.profile_picture_url || '',
        bio: existingProfile.bio || '',
        preferences: existingProfile.preferences || {}
      });
      
      if (existingProfile.profile_picture_url) {
        setImagePreview(existingProfile.profile_picture_url);
      }
      
      if (existingProfile.preferences && Object.keys(existingProfile.preferences).length > 0) {
        if (existingProfile.preferences.default === "preferences" && Object.keys(existingProfile.preferences).length === 1) {
          setPreferencesText('');
        } else if (existingProfile.preferences.notes && Object.keys(existingProfile.preferences).length === 1) {
          setPreferencesText(existingProfile.preferences.notes);
        } else {
          setPreferencesText(JSON.stringify(existingProfile.preferences, null, 2));
        }
      }
    } catch (err: any) {
      console.error('Failed to load existing profile:', err);
      setIsFirstTimeUser(true);
      setCurrentStep(1);
    } finally {
      sessionStorage.removeItem('profile_modal_load_in_progress');
    }
  };

  const handleInputChange = (e: React.ChangeEvent<HTMLInputElement | HTMLSelectElement | HTMLTextAreaElement>) => {
    const { name, value } = e.target;
    
    // Clear field error when user starts typing
    if (fieldErrors[name]) {
      setFieldErrors(prev => ({ ...prev, [name]: '' }));
    }
    
    if (name === 'phone_number') {
      // Ensure +91 prefix is always present
      if (!value.startsWith('+91')) {
        setProfileData(prev => ({ ...prev, [name]: '+91' + value.replace(/^\+91/, '') }));
      } else {
        setProfileData(prev => ({ ...prev, [name]: value }));
      }
    } else {
      setProfileData(prev => ({ ...prev, [name]: value }));
    }
  };

  const handleDateChange = (date: Date | null) => {
    // Clear field error when date is selected
    if (fieldErrors.date_of_birth) {
      setFieldErrors(prev => ({ ...prev, date_of_birth: '' }));
    }
    
    const dateString = date ? date.toISOString().split('T')[0] : '';
    setProfileData(prev => ({ ...prev, date_of_birth: dateString }));
  };

  const handleImageUpload = (e: React.ChangeEvent<HTMLInputElement>) => {
    const file = e.target.files?.[0];
    if (file) {
      // Validate file type
      if (!file.type.startsWith('image/')) {
        setError('Please select a valid image file (JPEG, PNG, etc.)');
        return;
      }
      
      // Validate file size (2MB limit)
      if (file.size > 2 * 1024 * 1024) {
        setError('Image size must be less than 2MB');
        return;
      }
      
      setProfileImage(file);
      const reader = new FileReader();
      reader.onload = (e) => {
        setImagePreview(e.target?.result as string);
      };
      reader.readAsDataURL(file);
      setError('');
    }
  };

  const handleRemoveImage = () => {
    setProfileImage(null);
    setImagePreview('');
    setProfileData(prev => ({ ...prev, profile_picture_url: '' }));
    if (fileInputRef.current) {
      fileInputRef.current.value = '';
    }
  };

  const validateStep1 = () => {
    const { first_name, last_name, email, phone_number, date_of_birth, gender } = profileData;
    const errors: Record<string, string> = {};
    
    if (!first_name.trim()) errors.first_name = 'Please fill out this field to proceed.';
    if (!last_name.trim()) errors.last_name = 'Please fill out this field to proceed.';
    if (!email.trim()) errors.email = 'Please fill out this field to proceed.';
    // Phone number is now optional - no validation error if empty
    if (!date_of_birth) errors.date_of_birth = 'Please fill out this field to proceed.';
    if (!gender) errors.gender = 'Please fill out this field to proceed.';
    
    // Skip validation errors if fields are verified
    if (isEmailVerified && errors.email === 'Please fill out this field to proceed.') {
      delete errors.email;
    }
    
    // Email is always verified during login/registration - no additional verification needed
    
    setFieldErrors(errors);
    return Object.keys(errors).length === 0;
  };

  const canCloseModal = () => {
    return !isFirstTimeUser;
  };

  const handleModalClose = () => {
    if (canCloseModal()) {
      onClose();
    } else {
      setShowValidationMessage(true);
      setTimeout(() => setShowValidationMessage(false), 3000);
    }
  };

  const handleOverlayClick = (e: React.MouseEvent) => {
    if (e.target === e.currentTarget) {
      handleModalClose();
    }
  };

  const handleNextStep = () => {
    if (currentStep === 1) {
      if (validateStep1()) {
        setCurrentStep(2);
        setError('');
      } else {
        // Focus first empty field
        const firstErrorField = Object.keys(fieldErrors)[0];
        if (firstErrorField) {
          const element = document.querySelector(`[name="${firstErrorField}"]`) as HTMLElement;
          element?.focus();
        }
      }
    } else if (currentStep === 2) {
      setCurrentStep(3);
    } else {
      handleFinalSubmit();
    }
  };

  // OTP Functions
  const sendEmailOTP = async () => {
    try {
      setLoading(true);
      const accessToken = localStorage.getItem('access_token');
      const response = await fetch(`${import.meta.env.VITE_API_BASE || 'http://localhost:8090'}/auth/send-email-otp`, {
        method: 'POST',
        headers: { 
          'Content-Type': 'application/json',
          'Authorization': `Bearer ${accessToken}`
        },
        body: JSON.stringify({ email: profileData.email })
      });
      
      if (response.ok) {
        setOtpSentEmail(true);
        setSuccess('OTP sent to your email');
      } else {
        setError('Failed to send OTP');
      }
    } catch (err) {
      setError('Failed to send OTP');
    } finally {
      setLoading(false);
    }
  };

  const verifyEmailOTP = async () => {
    try {
      setLoading(true);
      const accessToken = localStorage.getItem('access_token');
      const response = await fetch(`${import.meta.env.VITE_API_BASE || 'http://localhost:8090'}/auth/verify-email-otp`, {
        method: 'POST',
        headers: { 
          'Content-Type': 'application/json',
          'Authorization': `Bearer ${accessToken}`
        },
        body: JSON.stringify({ email: profileData.email, otp_code: emailOtp })
      });
      
      if (response.ok) {
        setIsEmailVerified(true);
        setEmailReadOnly(true);
        setSuccess('Email verified successfully');
        setEmailOtp('');
        setOtpSentEmail(false);
      } else {
        setError('Invalid OTP');
      }
    } catch (err) {
      setError('Failed to verify OTP');
    } finally {
      setLoading(false);
    }
  };

  const sendPhoneOTP = async () => {
    try {
      setLoading(true);
      const accessToken = localStorage.getItem('access_token');
      const response = await fetch(`${import.meta.env.VITE_API_BASE || 'http://localhost:8090'}/auth/send-otp`, {
        method: 'POST',
        headers: { 
          'Content-Type': 'application/json',
          'Authorization': `Bearer ${accessToken}`
        },
        body: JSON.stringify({ phone_number: profileData.phone_number })
      });
      
      const responseData = await response.json();
      console.log('Phone OTP Response:', responseData);
      
      if (response.ok) {
        if (responseData.otp) {
          console.log('🔐 Phone OTP Code:', responseData.otp);
        }
        setOtpSentPhone(true);
        setSuccess('OTP sent to your phone');
      } else {
        setError('Failed to send OTP');
      }
    } catch (err) {
      setError('Failed to send OTP');
    } finally {
      setLoading(false);
    }
  };

  const verifyPhoneOTP = async () => {
    try {
      setLoading(true);
      const accessToken = localStorage.getItem('access_token');
      const response = await fetch(`${import.meta.env.VITE_API_BASE || 'http://localhost:8090'}/auth/verify-otp`, {
        method: 'POST',
        headers: { 
          'Content-Type': 'application/json',
          'Authorization': `Bearer ${accessToken}`
        },
        body: JSON.stringify({ phone_number: profileData.phone_number, otp_code: phoneOtp })
      });
      
      if (response.ok) {
        setIsPhoneVerified(true);
        setPhoneReadOnly(true);
        setSuccess('Phone verified successfully');
        setPhoneOtp('');
        setOtpSentPhone(false);
      } else {
        setError('Invalid OTP');
      }
    } catch (err) {
      setError('Failed to verify OTP');
    } finally {
      setLoading(false);
    }
  };

  const isValidEmail = (email: string) => {
    return /^[^\s@]+@[^\s@]+\.[^\s@]+$/.test(email);
  };

  const isValidPhone = (phone: string) => {
    return /^\+91[0-9]{10}$/.test(phone);
  };

  const handleStep1Submit = (e: React.FormEvent) => {
    e.preventDefault();
    handleNextStep();
  };
  
  const handleStep2Submit = (e: React.FormEvent) => {
    e.preventDefault();
    handleNextStep();
  };

  const handleFinalSubmit = async (e?: React.FormEvent) => {
    if (e) e.preventDefault();

    // Check if user is admin - skip profile completion for admins
    const isAdminLoggedIn = localStorage.getItem('adminLoggedIn') === 'true';
    if (isAdminLoggedIn) {
      setSuccess('Admin profile setup completed!');
      setTimeout(() => {
        onComplete();
      }, 1500);
      return;
    }

    setLoading(true);
    setError('');
    setSuccess('');

    let parsedPreferences = {};
    if (preferencesText.trim()) {
      try {
        parsedPreferences = JSON.parse(preferencesText);
      } catch {
        parsedPreferences = { notes: preferencesText.trim() };
      }
    }

    const payload = {
      profile_data: {
        first_name: profileData.first_name.trim(),
        last_name: profileData.last_name.trim(),
        email: localStorage.getItem('login_email') || profileData.email.trim(),
        phone_number: profileData.phone_number.trim() !== '+91' ? profileData.phone_number.trim() : undefined,
        date_of_birth: profileData.date_of_birth ? new Date(profileData.date_of_birth).toISOString() : undefined,
        gender: profileData.gender || undefined,
        country: profileData.country.trim() || undefined,
        city: profileData.city.trim() || undefined,
        address: profileData.address.trim() || undefined,
        postal_code: profileData.postal_code.trim() || undefined,
        profile_picture_url: profileData.profile_picture_url.trim() || undefined,
        bio: profileData.bio.trim() || undefined,
        preferences: Object.keys(parsedPreferences).length > 0 ? parsedPreferences : undefined
      }
    };

    try {
      console.log('Submitting profile data:', payload);
      const data = await completeProfile(payload);
      console.log('Profile completion response:', data);
      setSuccess('Profile completed successfully!');
      // Mark profile as completed - this will prevent modal from showing again
      localStorage.setItem('profile_completed', 'true');
      // Update localStorage with fresh profile data
      localStorage.setItem('user', JSON.stringify(data));
      // Dispatch custom event to refresh header
      window.dispatchEvent(new Event('profileUpdated'));
      setTimeout(() => {
        onComplete();
      }, 1500);
    } catch (err: any) {
      console.error('Profile completion error:', err);

      // Try to get more detailed error info
      if (err.response) {
        try {
          const errorData = await err.response.json();
          console.error('Backend error response:', errorData);
          
          // Handle validation errors specifically
          if (errorData.detail && Array.isArray(errorData.detail)) {
            const validationErrors = errorData.detail.map(e => `${e.loc?.join('.')}: ${e.msg}`).join(', ');
            setError(`Validation error: ${validationErrors}`);
          } else {
            setError(errorData.message || errorData.detail?.message || 'Failed to complete profile');
          }
        } catch {
          setError(`Server error: ${err.response.status} ${err.response.statusText}`);
        }
      } else if (err.message === 'Failed to fetch') {
        setError('Unable to connect to server. Please check your internet connection.');
      } else {
        setError(err.message || 'Failed to complete profile. Please try again.');
      }
    } finally {
      setLoading(false);
    }
  };

  // Don't show modal if not authenticated or already shown this session or profile is 100% complete
  const accessToken = localStorage.getItem('access_token');
  const modalShownThisSession = localStorage.getItem('profile_modal_shown_this_login') === 'true';
  const profileCompleted = localStorage.getItem('profile_completed') === 'true';
  if (!isOpen || !accessToken || modalShownThisSession || profileCompleted) return null;

  const steps = [
    { number: 1, title: 'Personal Info', icon: User, active: currentStep >= 1, completed: currentStep > 1 },
    { number: 2, title: 'Location', icon: MapPin, active: currentStep >= 2, completed: currentStep > 2 },
    { number: 3, title: 'Preferences', icon: Settings, active: currentStep >= 3, completed: currentStep > 3 }
  ];

  return (
    <div 
      className="fixed inset-0 bg-black/50 backdrop-blur-sm flex items-center justify-center p-4 z-50"
      onClick={handleOverlayClick}
    >
      <div 
        className="bg-white rounded-2xl w-full max-w-3xl flex flex-col border border-gray-200 overflow-hidden"
        style={{
          boxShadow: '0 25px 50px -12px rgba(0, 0, 0, 0.25), 0 0 0 1px rgba(0, 0, 0, 0.05), 0 10px 15px -3px rgba(0, 0, 0, 0.1)',
          maxHeight: '90vh'
        }}

        onClick={(e) => e.stopPropagation()}
      >
        {/* Header */}
        <div className="px-6 py-4 border-b border-gray-100">
          <div className="flex items-center justify-between mb-4">
            <div className="flex-1">
              <h2 className="text-xl font-bold text-gray-900">
                {isFirstTimeUser ? 'Complete Your Profile' : 'Update Your Profile'}
              </h2>
              <p className="text-gray-600 text-sm mt-1">
                {isFirstTimeUser 
                  ? 'Help us personalize your experience' 
                  : 'Review and update your information as needed'
                }
              </p>
              {!isFirstTimeUser && (
                <div className="mt-2 text-sm text-blue-600 bg-blue-50 px-3 py-2 rounded-lg border border-blue-200">
                  Your previous information is pre-filled. Update any fields you'd like to change.
                </div>
              )}
              {showValidationMessage && (
                <div className="mt-2 text-sm text-red-600 bg-red-50 px-3 py-2 rounded-lg border border-red-200">
                  {isFirstTimeUser 
                    ? 'Please complete your profile to continue.' 
                    : 'Please complete the required fields before continuing.'
                  }
                </div>
              )}
            </div>
            
            {/* Close button only for returning users */}
            {!isFirstTimeUser && (
              <button
                onClick={() => {
                  localStorage.setItem('profile_modal_shown_this_login', 'true');
                  onClose();
                }}
                className="p-2 hover:bg-gray-100 rounded-full transition-colors ml-4"
                title="Close"
              >
                <svg className="w-5 h-5 text-gray-500" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                  <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M6 18L18 6M6 6l12 12" />
                </svg>
              </button>
            )}
          </div>

          {/* Progress Steps */}
          <div className="flex items-center justify-between">
            {steps.map((step, index) => (
              <React.Fragment key={step.number}>
                <div className="flex items-center">
                  <div className={`w-8 h-8 rounded-full flex items-center justify-center transition-all ${
                    step.completed 
                      ? 'bg-green-500 text-white' 
                      : step.active 
                        ? 'bg-blue-500 text-white' 
                        : 'bg-gray-200 text-gray-500'
                  }`}>
                    {step.completed ? (
                      <Check className="w-4 h-4" />
                    ) : (
                      <step.icon className="w-4 h-4" />
                    )}
                  </div>
                  <div className="ml-2">
                    <p className={`text-xs font-medium ${step.active ? 'text-gray-900' : 'text-gray-500'}`}>
                      {step.title}
                    </p>
                  </div>
                </div>
                {index < steps.length - 1 && (
                  <div className={`flex-1 h-0.5 mx-2 ${step.completed ? 'bg-green-500' : 'bg-gray-200'}`} />
                )}
              </React.Fragment>
            ))}
          </div>
        </div>

        {/* Content */}
        <div 
          className="flex-1" 
          style={{ 
            minHeight: 0,
            overflowY: 'auto',
            scrollbarWidth: 'thin',
            scrollbarColor: '#cbd5e1 #f1f5f9'
          }}
        >
          {error && (
            <div className="mx-8 mt-6 bg-red-50 border border-red-200 text-red-700 px-4 py-3 rounded-lg text-sm">
              {error}
            </div>
          )}

          {success && (
            <div className="mx-8 mt-6 bg-green-50 border border-green-200 text-green-700 px-4 py-3 rounded-lg text-sm flex items-center">
              <Check className="w-4 h-4 mr-2" />
              {success}
            </div>
          )}

          {/* Step 1: Personal Information */}
          {currentStep === 1 && (
            <form onSubmit={handleStep1Submit} className="p-6 space-y-4">
              <div className="grid grid-cols-2 gap-4">
                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-1">
                    First Name <span style={{color: '#FF0000'}}>*</span>
                  </label>
                  <input
                    type="text"
                    name="first_name"
                    value={profileData.first_name}
                    onChange={handleInputChange}
                    className={`w-full px-3 py-2 border rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent transition-all ${
                      fieldErrors.first_name ? 'border-red-500' : 'border-gray-300'
                    }`}
                  />
                  {fieldErrors.first_name && (
                    <p className="text-red-500 text-xs mt-1">{fieldErrors.first_name}</p>
                  )}
                </div>
                
                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-1">
                    Last Name <span style={{color: '#FF0000'}}>*</span>
                  </label>
                  <input
                    type="text"
                    name="last_name"
                    value={profileData.last_name}
                    onChange={handleInputChange}
                    className={`w-full px-3 py-2 border rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent transition-all ${
                      fieldErrors.last_name ? 'border-red-500' : 'border-gray-300'
                    }`}
                  />
                  {fieldErrors.last_name && (
                    <p className="text-red-500 text-xs mt-1">{fieldErrors.last_name}</p>
                  )}
                </div>
              </div>

              <div>
                <label className="block text-sm font-medium text-gray-700 mb-1">
                  Email Address <span style={{color: '#FF0000'}}>*</span>
                </label>
                <div className="space-y-2">
                  <div className="flex gap-2">
                    <input
                      type="email"
                      name="email"
                      value={profileData.email || JSON.parse(localStorage.getItem('user') || '{}').email || localStorage.getItem('login_email') || ''}
                      onChange={handleInputChange}
                      readOnly={true}
                      className={`flex-1 px-3 py-2 border rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent transition-all ${
                        fieldErrors.email ? 'border-red-500' : 'border-gray-300'
                      } bg-gray-100`}
                    />
                    <span className="flex items-center px-3 py-2 text-green-600 text-sm">
                      ✅ Verified
                    </span>
                  </div>
                  <p className="text-xs text-gray-500">Email verified during login</p>
                </div>
                {fieldErrors.email && (
                  <p className="text-red-500 text-xs mt-1">{fieldErrors.email}</p>
                )}
              </div>

              <div>
                <label className="block text-sm font-medium text-gray-700 mb-1">
                  Phone Number <span className="text-gray-500 text-xs">(Optional)</span>
                </label>
                <div className="space-y-2">
                  <div className="flex gap-2">
                    <input
                      type="tel"
                      name="phone_number"
                      value={profileData.phone_number}
                      onChange={handleInputChange}
                      readOnly={phoneReadOnly}
                      className={`flex-1 px-3 py-2 border rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent transition-all ${
                        fieldErrors.phone_number ? 'border-red-500' : 'border-gray-300'
                      } ${phoneReadOnly ? 'bg-gray-100' : ''}`}
                      placeholder="+91 98765 43210"
                    />
                    {!phoneReadOnly && !isPhoneVerified && loginMethod !== 'phone' && (
                      <button
                        type="button"
                        onClick={sendPhoneOTP}
                        disabled={!isValidPhone(profileData.phone_number) || loading}
                        className="px-3 py-2 bg-blue-500 text-white rounded-lg text-sm disabled:bg-gray-300"
                      >
                        Send OTP
                      </button>
                    )}
                    {(isPhoneVerified || loginMethod === 'phone') && (
                      <span className="flex items-center px-3 py-2 text-green-600 text-sm">
                        ✅ Verified
                      </span>
                    )}
                  </div>
                  {otpSentPhone && !isPhoneVerified && loginMethod !== 'phone' && (
                    <div className="flex gap-2">
                      <input
                        type="text"
                        placeholder="Enter OTP"
                        value={phoneOtp}
                        onChange={(e) => setPhoneOtp(e.target.value)}
                        className="flex-1 px-3 py-2 border border-gray-300 rounded-lg"
                        maxLength={6}
                      />
                      <button
                        type="button"
                        onClick={verifyPhoneOTP}
                        disabled={phoneOtp.length !== 6 || loading}
                        className="px-3 py-2 bg-green-500 text-white rounded-lg text-sm disabled:bg-gray-300"
                      >
                        Verify
                      </button>
                    </div>
                  )}
                  {otpSentPhone && (
                    <p className="text-xs text-gray-500">OTP valid for 5 mins</p>
                  )}
                </div>
                {fieldErrors.phone_number && (
                  <p className="text-red-500 text-xs mt-1">{fieldErrors.phone_number}</p>
                )}
              </div>

              <div className="grid grid-cols-2 gap-4">
                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-1">
                    Date of Birth <span style={{color: '#FF0000'}}>*</span>
                  </label>
                  <div className="relative">
                    <DatePicker
                      selected={profileData.date_of_birth ? new Date(profileData.date_of_birth) : null}
                      onChange={handleDateChange}
                      maxDate={new Date()}
                      showYearDropdown
                      showMonthDropdown
                      dropdownMode="select"
                      dateFormat="dd/MM/yyyy"
                      placeholderText="dd/MM/yyyy"
                      className={`w-full px-3 py-2 border rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent transition-all ${
                        fieldErrors.date_of_birth ? 'border-red-500' : 'border-gray-300'
                      }`}
                      wrapperClassName="w-full"
                    />
                  
                  </div>
                  {fieldErrors.date_of_birth && (
                    <p className="text-red-500 text-xs mt-1">{fieldErrors.date_of_birth}</p>
                  )}
                </div>
                
                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-1">
                    Gender <span style={{color: '#FF0000'}}>*</span>
                  </label>
                  <select
                    name="gender"
                    value={profileData.gender}
                    onChange={handleInputChange}
                    className={`w-full px-3 py-2 border rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent transition-all ${
                      fieldErrors.gender ? 'border-red-500' : 'border-gray-300'
                    }`}
                  >
                    <option value="">Select Gender</option>
                    <option value="male">Male</option>
                    <option value="female">Female</option>
                    <option value="other">Other</option>
                  </select>
                  {fieldErrors.gender && (
                    <p className="text-red-500 text-xs mt-1">{fieldErrors.gender}</p>
                  )}
                </div>
              </div>
            </form>
          )}

          {/* Step 2: Location Details */}
          {currentStep === 2 && (
            <form onSubmit={handleStep2Submit} className="p-6 space-y-4">
              <div className="grid grid-cols-2 gap-4">
                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-1">Country</label>
                  <input
                    type="text"
                    name="country"
                    value={profileData.country}
                    onChange={handleInputChange}
                    className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent transition-all"
                  />
                </div>
                
                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-1">City</label>
                  <input
                    type="text"
                    name="city"
                    value={profileData.city}
                    onChange={handleInputChange}
                    className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent transition-all"
                  />
                </div>
              </div>

              <div>
                <label className="block text-sm font-medium text-gray-700 mb-1">Address</label>
                <input
                  type="text"
                  name="address"
                  value={profileData.address}
                  onChange={handleInputChange}
                  className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent transition-all"
                />
              </div>

              <div>
                <label className="block text-sm font-medium text-gray-700 mb-1">Postal Code</label>
                <input
                  type="text"
                  name="postal_code"
                  value={profileData.postal_code}
                  onChange={handleInputChange}
                  className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent transition-all"
                />
              </div>
            </form>
          )}

          {/* Step 3: Profile Picture & Preferences */}
          {currentStep === 3 && (
            <form onSubmit={handleFinalSubmit} className="p-6 space-y-4">
              {/* Profile Picture Upload */}
              <div>
                <label className="block text-sm font-medium text-gray-700 mb-1">Profile Picture</label>
                <div className="flex items-center space-x-4">
                  <div className="w-16 h-16 rounded-full bg-gray-100 flex items-center justify-center overflow-hidden">
                    {imagePreview ? (
                      <img src={imagePreview} alt="Profile" className="w-full h-full object-cover" />
                    ) : (
                      <User className="w-6 h-6 text-gray-400" />
                    )}
                  </div>
                  <div className="flex gap-2">
                    <button
                      type="button"
                      onClick={() => fileInputRef.current?.click()}
                      className="inline-flex items-center px-3 py-2 border border-gray-300 rounded-lg text-sm font-medium text-gray-700 bg-white hover:bg-gray-50 transition-colors"
                    >
                      <Upload className="w-4 h-4 mr-2" />
                      {imagePreview ? 'Change' : 'Browse'}
                    </button>
                    {imagePreview && (
                      <button
                        type="button"
                        onClick={handleRemoveImage}
                        className="inline-flex items-center px-3 py-2 border border-red-300 rounded-lg text-sm font-medium text-red-700 bg-white hover:bg-red-50 transition-colors"
                      >
                        Remove
                      </button>
                    )}
                    <input
                      ref={fileInputRef}
                      type="file"
                      accept="image/*"
                      onChange={handleImageUpload}
                      className="hidden"
                    />
                  </div>
                </div>
                <p className="text-xs text-gray-500 mt-1">JPEG, PNG up to 2MB</p>
              </div>

              <div>
                <label className="block text-sm font-medium text-gray-700 mb-1">Bio</label>
                <textarea
                  name="bio"
                  value={profileData.bio}
                  onChange={handleInputChange}
                  rows={2}
                  className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent resize-none transition-all"
                  placeholder="Tell us about yourself..."
                />
              </div>

              <div>
                <label className="block text-sm font-medium text-gray-700 mb-1">Preferences</label>
                <textarea
                  value={preferencesText}
                  onChange={(e) => setPreferencesText(e.target.value)}
                  rows={3}
                  className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent resize-none transition-all"
                  placeholder="Tell us your property preferences (e.g., property type, budget range, preferred locations, amenities, etc.)"
                />
                <p className="text-xs text-gray-500 mt-1">
                  You can write in plain text 
                </p>
              </div>
            </form>
          )}
        </div>

        {/* Footer */}
        <div className="px-6 py-4 border-t border-gray-100 bg-gray-50">
          <div className={`flex items-center  ${
            currentStep === 2 && !isFirstTimeUser ? 'justify-center' : 'justify-between'
          }`}>
            {/* Hide back button only for returning users in step 2 */}
            {!(currentStep === 2 && !isFirstTimeUser) && (
              <button
                type="button"
                onClick={() => currentStep > 1 ? setCurrentStep(currentStep - 1) : handleModalClose()}
                className={`px-4 py-2 font-medium rounded-lg transition-all${
                  currentStep > 1 
                    ? 'bg-gray-500 hover:bg-gray-600 text-white bg-blue-500'
                    : isFirstTimeUser
                      ? 'bg-gray-300 text-white cursor-not-allowed bg-blue-500'
                      : 'bg-gray-500 hover:bg-gray-600 text-white '
                }`}
                disabled={currentStep === 1 && isFirstTimeUser}
              >
                {currentStep > 1 ? 'Back' : 'Cancel'}
              </button>
            )}
            
            <button
              type="button"
              onClick={handleNextStep}
              disabled={loading}
              className={`px-6 py-2 font-medium rounded-lg transition-all flex items-center ${
                loading
                  ? 'bg-gray-300 text-gray-500 cursor-not-allowed'
                  : 'bg-blue-600 hover:bg-blue-700 text-white hover:scale-105 hover:shadow-md bg-blue-500'
              }`}
            >
              {loading && <div className="w-4 h-4 border-2 border-white border-t-transparent rounded-full animate-spin mr-2" />}
              {currentStep === 3 ? (loading ? 'Completing...' : 'Complete Profile') : 'Next'}
            </button>
          </div>
        </div>
      </div>
    </div>
  );
};

export default ProfileCompletionModal;