import React, { useEffect, useState } from "react";
import { useNavigate, useSearchParams } from "react-router-dom";
import { MapPin, Navigation, Mic, MicOff, Camera } from "lucide-react";
import { api } from "../utils/api";
import { searchHistoryAPI } from "../api/searchHistory";
import { PropertyCard } from "./PropertyCard";
import { GoogleMap } from "./GoogleMap";
import { SearchSuggestions } from "./SearchSuggestions";
import { LocationPermissionModal } from "./LocationPermissionModal";
import { isAuthenticated } from "../utils/auth";
import { toast } from "sonner";



interface Property {
  id: number;
  title: string;
  location: string;
  beds: string | number;
  baths: string | number;
  area: string;
  price: string;
  imageUrl?: string;
  urgent: boolean;
  property_type: string;
  city: string;
  locality: string;
  sale_price: number;
  latitude?: number;
  longitude?: number;
  listing_type?: string;
  priceNegotiable?: boolean;
  loanAvailable?: boolean;
}

interface Filters {
  search: string;
  bedrooms: string[];
  bathrooms: string[];
  property_type: string[];
  min_price: string;
  max_price: string;
  city: string;
  sort: string;
  furnishing: string[];
  monthly_rent: string;
  loan_available: string;
  amenities: string[];
  listing_type: string[];
  locality?: string[];
  // Tenant-specific filters
  rent_range_min: string;
  rent_range_max: string;
  deposit_amount: string;
  maintenance_cost: string;
  tenant_preference: string;
  lease_duration: string;
  // Buyer-specific filters
  possession_status: string;
  possession_date: string;
  rera_approved: string;
  // Common filters
  verified_only: boolean;
}

// Generic location normalization function
const normalizeLocation = (location: string): string => {
  const locationLower = location.toLowerCase().trim();

  // Common city name variations
  const cityVariations: { [key: string]: string } = {
    bengaluru: "Bangalore",
    bangalore: "Bangalore",
    mumbai: "Mumbai",
    bombay: "Mumbai",
    delhi: "Delhi",
    "new delhi": "Delhi",
    chennai: "Chennai",
    madras: "Chennai",
    hyderabad: "Hyderabad",
    secunderabad: "Hyderabad",
    pune: "Pune",
    poona: "Pune",
    kolkata: "Kolkata",
    calcutta: "Kolkata",
  };

  // Check for exact matches first
  if (cityVariations[locationLower]) {
    return cityVariations[locationLower];
  }

  // Check for partial matches
  for (const [variation, normalized] of Object.entries(cityVariations)) {
    if (locationLower.includes(variation)) {
      return normalized;
    }
  }

  // Return original with proper capitalization
  return location.charAt(0).toUpperCase() + location.slice(1).toLowerCase();
};

// Generate fallback search terms for any location
const generateFallbackSearches = (searchTerm: string): string[] => {
  const fallbacks = new Set<string>();
  const words = searchTerm.trim().split(/\s+/);

  // Add original term
  fallbacks.add(searchTerm);

  // Add normalized version
  fallbacks.add(normalizeLocation(searchTerm));

  // Add case variations
  fallbacks.add(searchTerm.toLowerCase());
  fallbacks.add(searchTerm.toUpperCase());
  fallbacks.add(
    searchTerm.charAt(0).toUpperCase() + searchTerm.slice(1).toLowerCase()
  );

  // Add individual words
  words.forEach((word) => {
    if (word.length > 2) {
      fallbacks.add(word);
      fallbacks.add(word.toLowerCase());
      fallbacks.add(word.charAt(0).toUpperCase() + word.slice(1).toLowerCase());
      fallbacks.add(normalizeLocation(word));
    }
  });

  // Add partial matches for compound words
  if (searchTerm.length > 5) {
    for (let i = 3; i <= searchTerm.length - 2; i++) {
      fallbacks.add(searchTerm.substring(0, i));
      fallbacks.add(searchTerm.substring(i));
    }
  }

  return Array.from(fallbacks).filter((term) => term.length > 1);
};

// Parse price input with crore/lakh support
const parsePrice = (input: string): number | null => {
  if (!input || input.trim() === "") return null;

  const value = input.toLowerCase().trim();
  const numberMatch = value.match(/([0-9.]+)/);

  if (!numberMatch) return null;

  const number = parseFloat(numberMatch[1]);
  if (isNaN(number)) return null;

  // Check for crore indicators
  if (value.includes("cr") || value.includes("crore")) {
    return Math.floor(number * 10000000); // Convert crores to rupees
  }

  // Check for lakh indicators
  if (value.includes("l") || value.includes("lakh") || value.includes("lac")) {
    return Math.floor(number * 100000); // Convert lakhs to rupees
  }

  // If no unit specified, treat as rupees if reasonable, otherwise as lakhs
  if (number >= 1000000) {
    return Math.floor(number); // Already in rupees
  } else if (number >= 10) {
    return Math.floor(number * 100000); // Assume lakhs
  } else {
    return Math.floor(number * 10000000); // Assume crores
  }
};

export function Properties() {
  const navigate = useNavigate();
  const [searchParams] = useSearchParams();
  const [properties, setProperties] = useState<Property[]>([]);
  const [filteredProperties, setFilteredProperties] = useState<Property[]>([]);
  const [loading, setLoading] = useState(true);
  const [clickedHearts, setClickedHearts] = useState<Set<number>>(new Set());
  const [recommendations, setRecommendations] = useState<Property[]>([]);
  const [currentLocation, setCurrentLocation] = useState<{
    lat: number;
    lng: number;
    name: string;
  } | null>(null);
  const [mapCenter, setMapCenter] = useState<{
    lat: number;
    lng: number;
  } | null>(null);

  // Cities and their localities data
  const citiesData = {
    Mumbai: [
      "Andheri",
      "Bandra",
      "Juhu",
      "Powai",
      "Malad",
      "Borivali",
      "Thane",
      "Navi Mumbai",
      "Worli",
      "Lower Parel",
      "Goregaon",
      "Kandivali",
      "Santacruz",
      "Vile Parle",
      "Khar",
      "Linking Road",
      "Colaba",
      "Fort",
      "Churchgate",
      "Marine Drive",
    ],
    Delhi: [
      "Connaught Place",
      "Karol Bagh",
      "Lajpat Nagar",
      "South Extension",
      "Vasant Kunj",
      "Dwarka",
      "Rohini",
      "Pitampura",
      "Janakpuri",
      "Laxmi Nagar",
      "Preet Vihar",
      "Mayur Vihar",
      "Saket",
      "Hauz Khas",
      "Green Park",
      "Defence Colony",
      "Greater Kailash",
      "Nehru Place",
      "Okhla",
      "Noida",
    ],
    Bangalore: [
      "Koramangala",
      "Indiranagar",
      "Whitefield",
      "Electronic City",
      "HSR Layout",
      "BTM Layout",
      "Jayanagar",
      "Malleshwaram",
      "Rajajinagar",
      "Basavanagudi",
      "JP Nagar",
      "Banashankari",
      "Hebbal",
      "Marathahalli",
      "Sarjapur",
      "Bellandur",
      "Yelahanka",
      "Kengeri",
      "Bommanahalli",
      "Vijayanagar",
      "Sahakarnagar",
      "Sahakaranagara",
      "Jakkur",
      "Yelahanka New Town",
      "Thanisandra",
      "Bagalur",
      "Devanahalli",
    ],
    Chennai: [
      "T Nagar",
      "Anna Nagar",
      "Adyar",
      "Velachery",
      "OMR",
      "Porur",
      "Tambaram",
      "Chrompet",
      "Mylapore",
      "Nungambakkam",
      "Egmore",
      "Guindy",
      "Sholinganallur",
      "Perungudi",
      "Thoraipakkam",
      "Pallikaranai",
      "Medavakkam",
      "Perumbakkam",
      "Navalur",
      "Siruseri",
    ],
    Hyderabad: [
      "Banjara Hills",
      "Jubilee Hills",
      "Gachibowli",
      "Hitech City",
      "Kondapur",
      "Madhapur",
      "Kukatpally",
      "Miyapur",
      "Begumpet",
      "Secunderabad",
      "Ameerpet",
      "Somajiguda",
      "Punjagutta",
      "Lakdi Ka Pul",
      "Abids",
      "Nampally",
      "Charminar",
      "Malakpet",
      "Dilsukhnagar",
      "LB Nagar",
    ],
    Pune: [
      "Koregaon Park",
      "Viman Nagar",
      "Kharadi",
      "Wakad",
      "Hinjewadi",
      "Baner",
      "Aundh",
      "Shivaji Nagar",
      "Camp",
      "Deccan",
      "Kothrud",
      "Karve Nagar",
      "Warje",
      "Katraj",
      "Kondhwa",
      "Undri",
      "Wagholi",
      "Mundhwa",
      "Hadapsar",
      "Magarpatta",
    ],
    Kolkata: [
      "Salt Lake",
      "New Town",
      "Park Street",
      "Ballygunge",
      "Alipore",
      "Tollygunge",
      "Behala",
      "Jadavpur",
      "Garia",
      "Rajarhat",
      "Dum Dum",
      "Barasat",
      "Howrah",
      "Belgharia",
      "Madhyamgram",
      "Barrackpore",
      "Serampore",
      "Chandannagar",
      "Bally",
      "Rishra",
    ],
    Ahmedabad: [
      "Satellite",
      "Vastrapur",
      "Bodakdev",
      "Prahlad Nagar",
      "Ambawadi",
      "Navrangpura",
      "CG Road",
      "SG Highway",
      "Bopal",
      "Gota",
      "Shela",
      "Thaltej",
      "Maninagar",
      "Naranpura",
      "Paldi",
      "Ellis Bridge",
      "Ashram Road",
      "Gurukul",
      "Memnagar",
      "Chandkheda",
    ],
    Jaipur: [
      "Malviya Nagar",
      "Vaishali Nagar",
      "Mansarovar",
      "Jagatpura",
      "Tonk Road",
      "Ajmer Road",
      "C Scheme",
      "MI Road",
      "Bani Park",
      "Civil Lines",
      "Sodala",
      "Gopalpura",
      "Pratap Nagar",
      "Shyam Nagar",
      "Raja Park",
      "Lal Kothi",
      "Bapu Nagar",
      "Kardhani",
      "Sitapura",
      "Kukas",
    ],
    Kochi: [
      "Marine Drive",
      "Kakkanad",
      "Edapally",
      "Palarivattom",
      "Kaloor",
      "Panampilly Nagar",
      "Vyttila",
      "Kadavanthra",
      "Thevara",
      "Willingdon Island",
      "Fort Kochi",
      "Mattancherry",
      "Ernakulam South",
      "Ernakulam North",
      "Aluva",
      "Angamaly",
      "Perumbavoor",
      "Muvattupuzha",
      "Kothamangalam",
      "Piravom",
    ],
  };

  // Get dynamic localities based on selected city
  const getLocalitiesForCity = (city: string): string[] => {
    return citiesData[city as keyof typeof citiesData] || [];
  };

  // City coordinates for map display
  const cityCoordinates = {
    Mumbai: { lat: 19.076, lng: 72.8777 },
    Delhi: { lat: 28.7041, lng: 77.1025 },
    Bangalore: { lat: 12.9716, lng: 77.5946 },
    Chennai: { lat: 13.0827, lng: 80.2707 },
    Hyderabad: { lat: 17.385, lng: 78.4867 },
    Pune: { lat: 18.5204, lng: 73.8567 },
    Kolkata: { lat: 22.5726, lng: 88.3639 },
    Ahmedabad: { lat: 23.0225, lng: 72.5714 },
    Jaipur: { lat: 26.9124, lng: 75.7873 },
    Kochi: { lat: 9.9312, lng: 76.2673 },
  };

  // Generate coordinates for properties without lat/lng
  const generateCoordinates = (city: string) => {
    const cityCoord = cityCoordinates[city as keyof typeof cityCoordinates];
    if (cityCoord) {
      // Add small random offset to avoid overlapping markers
      return {
        latitude: cityCoord.lat + (Math.random() - 0.5) * 0.1,
        longitude: cityCoord.lng + (Math.random() - 0.5) * 0.1,
      };
    }
    // Default to Bangalore if city not found
    return {
      latitude: 12.9716 + (Math.random() - 0.5) * 0.1,
      longitude: 77.5946 + (Math.random() - 0.5) * 0.1,
    };
  };

  // Price ranges based on listing type
  const getPriceRanges = (listingType: string[]) => {
    const isRent = listingType.includes("rent");
    if (isRent) {
      return {
        min: 0,
        max: 200000,
        step: 5000,
        label: "Monthly Rent",
      };
    }
    return {
      min: 0,
      max: 50000000,
      step: 100000,
      label: "Sale Price",
    };
  };

  const [mapCollapsed, setMapCollapsed] = useState(false);
  const [showFilters, setShowFilters] = useState(false);
  const [filterTimeout, setFilterTimeout] = useState<number | null>(null);
  const [showSuggestions, setShowSuggestions] = useState(false);
  const [searchInputRef, setSearchInputRef] = useState<HTMLInputElement | null>(
    null
  );
  const [showLocationModal, setShowLocationModal] = useState(false);
  const [isListening, setIsListening] = useState(false);
  const [recognition, setRecognition] = useState<any>(null);
  const [lastStoredSearch, setLastStoredSearch] = useState<string>("");
  const [searchDebounceTimer, setSearchDebounceTimer] = useState<NodeJS.Timeout | null>(
    null
  );
  const [viewMode, setViewMode] = useState<"all" | "recent">("all");
  const [recentlyViewedProperties, setRecentlyViewedProperties] = useState<
    Property[]
  >([]);
  const [loadingRecentViews, setLoadingRecentViews] = useState(false);
  const [userRole, setUserRole] = useState<
    "buyer" | "tenant" | "seller" | "agent" | null
  >(null);

  const [filters, setFilters] = useState<Filters>({
    search: "",
    bedrooms: [],
    bathrooms: [],
    property_type: [],
    min_price: "",
    max_price: "",
    city: "",
    sort: "latest",
    furnishing: [],
    monthly_rent: "",
    loan_available: "",
    amenities: [],
    listing_type: [],
    locality: [],
    // Tenant-specific filters
    rent_range_min: "",
    rent_range_max: "",
    deposit_amount: "",
    maintenance_cost: "",
    tenant_preference: "",
    lease_duration: "",
    // Buyer-specific filters
    possession_status: "",
    possession_date: "",
    rera_approved: "",
    // Common filters
    verified_only: false,
  });

  // Get current price range settings
  const priceRange = getPriceRanges(filters.listing_type);
  const isForSale =
    filters.listing_type.includes("sell") ||
    filters.listing_type.includes("urgent-sale") ||
    filters.listing_type.includes("bidding");

  useEffect(() => {
    // Set user ID if not exists
    if (!localStorage.getItem("user_id")) {
      const user = JSON.parse(localStorage.getItem("user") || "{}");
      const userId = user.id || `guest_${Date.now()}`;
      localStorage.setItem("user_id", userId);
    }

    // Detect user role from localStorage
    const user = JSON.parse(localStorage.getItem("user") || "{}");
    const role = user.user_role || user.role || "buyer"; // Default to buyer
    setUserRole(role);

    // Reset view mode to 'all' if user is not authenticated
    if (!isAuthenticated() && viewMode === "recent") {
      console.log(
        "🔄 User not authenticated, switching from recent to all view"
      );
      setViewMode("all");
    }

    // Check for search parameter from URL
    const searchQuery = searchParams.get("search");
    const cityParam = searchParams.get("city");

    if (searchQuery) {
      setFilters((prev) => ({
        ...prev,
        search: decodeURIComponent(searchQuery),
      }));
    }

    if (cityParam) {
      const cityName = decodeURIComponent(cityParam);
      setFilters((prev) => ({ ...prev, city: cityName }));
      // Geocode the city to center the map
      geocodeSearchLocation(cityName);
    }

    // Check for location parameters from URL (from hero section current location)
    const lat = searchParams.get("lat");
    const lng = searchParams.get("lng");
    const location = searchParams.get("location");

    if (lat && lng) {
      const locationData = {
        lat: parseFloat(lat),
        lng: parseFloat(lng),
        name: location ? decodeURIComponent(location) : "Current Location",
      };
      setCurrentLocation(locationData);
      setFilters((prev) => ({ ...prev, search: locationData.name }));
      // Set map center to current location
      setMapCenter({ lat: locationData.lat, lng: locationData.lng });
      console.log("🌍 Location from hero section:", locationData);
    }

    loadProperties();
    loadRecommendations();

    // Load recently viewed properties if user is logged in
    if (isAuthenticated()) {
      const user = JSON.parse(localStorage.getItem("user") || "{}");
      // Try multiple possible user ID fields
      const userId = user.user_id || user.id || user.uuid || localStorage.getItem("user_id");
      console.log("🔍 Initial user check for recently viewed:", {
        user,
        userId,
        isAuthenticated: isAuthenticated(),
        userKeys: Object.keys(user),
        userIdType: typeof userId,
        localStorage_user_id: localStorage.getItem("user_id"),
        access_token: localStorage.getItem("access_token") ? "present" : "missing"
      });
      if (userId && userId !== "anonymous" && !userId.startsWith("guest_")) {
        console.log("🔍 Loading recently viewed for user:", userId);
        loadRecentlyViewedProperties(userId);
      } else {
        console.log("❌ User ID not valid for recently viewed:", userId);
      }
    } else {
      console.log("❌ User not authenticated for recently viewed");
    }
  }, [searchParams, viewMode]);

  // Debug effect to monitor authentication state changes
  useEffect(() => {
    console.log("🔍 Auth state changed:", {
      isAuthenticated: isAuthenticated(),
      viewMode,
      recentViewsCount: recentlyViewedProperties.length,
    });
  }, [viewMode, recentlyViewedProperties.length]);

  // Trigger search when filters change (except search field) - without storing history
  useEffect(() => {
    if (
      filters.property_type.length > 0 ||
      filters.city ||
      filters.listing_type.length > 0 ||
      filters.bedrooms.length > 0 ||
      filters.bathrooms.length > 0 ||
      filters.furnishing.length > 0 ||
      filters.sort !== "latest"
    ) {
      loadPropertiesWithoutHistory();
    }
  }, [
    filters.property_type,
    filters.city,
    filters.listing_type,
    filters.bedrooms,
    filters.bathrooms,
    filters.furnishing,
    filters.sort,
    filters.min_price,
    filters.max_price,
    filters.monthly_rent,
    filters.loan_available,
    filters.amenities,
    filters.locality,
  ]);

  // Auto-trigger search after typing stops - optimized (without storing search history)
  useEffect(() => {
    const timeoutId = setTimeout(() => {
      loadPropertiesWithoutHistory();
      if (filters.search.trim()) {
        geocodeSearchLocation(filters.search.trim());
      }
    }, 300);

    return () => clearTimeout(timeoutId);
  }, [filters.search]);

  // Geocode search location to center map
  const geocodeSearchLocation = async (searchQuery: string) => {
    try {
      const response = await fetch(
        `https://nominatim.openstreetmap.org/search?format=json&q=${encodeURIComponent(
          searchQuery
        )}&limit=1&countrycodes=in`
      );
      const data = await response.json();
      if (data && data.length > 0) {
        const location = data[0];
        setMapCenter({
          lat: parseFloat(location.lat),
          lng: parseFloat(location.lon),
        });
      }
    } catch (error) {
      console.error("Error geocoding search location:", error);
    }
  };

  // Store search - only called on explicit user actions (Search button, Apply Filters)
  const storeSearch = async (searchQuery: string, appliedFilters: Filters) => {
    console.log("🔍 storeSearch called:", { searchQuery, appliedFilters });

    // Skip empty searches
    if (
      !searchQuery.trim() &&
      !appliedFilters.city &&
      appliedFilters.property_type.length === 0 &&
      appliedFilters.listing_type.length === 0
    ) {
      console.log("❌ Empty search, skipping");
      return;
    }

    // Prevent duplicate calls
    const searchKey = `${searchQuery}|${JSON.stringify(appliedFilters)}`;
    if (searchKey === lastStoredSearch) {
      console.log("❌ Duplicate search, skipping");
      return;
    }

    // Store in backend with user-specific ID
    try {
      const user = JSON.parse(localStorage.getItem("user") || "{}");
      const userId = user.user_id || user.id || localStorage.getItem("user_id");

      console.log("👤 User ID:", userId);

      if (!userId || userId === "anonymous" || userId.startsWith("guest_")) {
        console.log("❌ User not logged in, skipping backend storage");
        return;
      }

      const searchData = {
        user_id: userId,
        query: searchQuery,
        filters: {
          property_type: appliedFilters.property_type || [],
          bedrooms:
            appliedFilters.bedrooms && appliedFilters.bedrooms.length > 0
              ? parseInt(appliedFilters.bedrooms[0])
              : undefined,
          city: appliedFilters.city,
          listing_type: appliedFilters.listing_type || [],
          min_price: appliedFilters.min_price
            ? parseInt(appliedFilters.min_price)
            : undefined,
          max_price: appliedFilters.max_price
            ? parseInt(appliedFilters.max_price)
            : undefined,
          furnishing: appliedFilters.furnishing || [],
          amenities: appliedFilters.amenities || [],
          locality: appliedFilters.locality || [],
          state: "Karnataka",
        },
        location: appliedFilters.city || "All Cities",
      };

      console.log("📤 Sending to backend:", searchData);

      // Update last stored search BEFORE making the call to prevent duplicates
      setLastStoredSearch(searchKey);
      
      const result = await searchHistoryAPI.storeSearch(searchData);
      console.log("✅ Search saved successfully:", result);
    } catch (error) {
      console.error("❌ Backend storage failed:", error);
      // Reset lastStoredSearch on error so user can retry
      setLastStoredSearch("");
    }
  };

  // Load recommendations
  const loadRecommendations = async () => {
    const searchHistory = JSON.parse(
      localStorage.getItem("search_history") || "[]"
    );
    if (searchHistory.length === 0) return;

    // Get user preferences from last search
    const lastSearch = searchHistory[0];
    const city = lastSearch.filters.city;
    const propertyType = Array.isArray(lastSearch.filters.property_type)
      ? lastSearch.filters.property_type[0]
      : lastSearch.filters.property_type;

    if (city || propertyType) {
      const allProperties = await api.fetchPropertiesWithFilters({
        city,
        property_type: propertyType,
      });
      const recommended = allProperties.slice(0, 3).map((property: any) => ({
        id: property.id,
        title: property.title,
        location: `${property.locality}, ${property.city}`,
        beds: property.bedrooms || "N/A",
        baths: property.bathrooms || "N/A",
        area: property.built_up_area
          ? `${property.built_up_area} sq ft`
          : "N/A",
        price: property.monthly_rent
          ? `${formatPrice(parseFloat(property.monthly_rent))}/month`
          : formatPrice(
              parseFloat(
                property.sale_price ||
                  property.starting_bid_price ||
                  property.per_bed_rent ||
                  0
              )
            ),
        imageUrl:
          property.primary_image_url ||
          property.primary_image ||
          property.photo ||
          undefined,
        urgent: property.listing_type === "urgent-sale",
        property_type: property.property_type,
        city: property.city,
        locality: property.locality,
        sale_price: property.sale_price,
      }));
      setRecommendations(recommended);
    }
  };

  // Enhanced search query parser
  const parseSearchQuery = (query: string) => {
    const parsed = {
      bedrooms: "",
      location: "",
      propertyType: "",
      minPrice: "",
      maxPrice: "",
      originalQuery: query,
    };

    const queryLower = query.toLowerCase().trim();

    // Simplified price range parsing patterns
    const priceRangePatterns = [
      // Between patterns - most specific first
      /between\s*(\d+)\s*(?:lakh|lakhs)\s*(?:to|and)\s*(\d+)\s*(?:lakh|lakhs)/i,
      /between\s*(\d+)\s*l\s*(?:to|and)\s*(\d+)\s*l/i,
      /between\s*(\d+)\s*(?:to|and)\s*(\d+)\s*(?:lakh|lakhs|l)/i,
      /between\s*(\d+)\s*(?:to|and)\s*(\d+)/i,
      // Direct range patterns
      /(\d+)\s*(?:lakh|lakhs)\s*(?:to|-|till)\s*(\d+)\s*(?:lakh|lakhs)/i,
      /(\d+)\s*l\s*(?:to|-|till)\s*(\d+)\s*l/i,
      /(\d+)\s*(?:to|-|till)\s*(\d+)\s*(?:lakh|lakhs|l)/i,
      /(\d+)\s*(?:to|-|till)\s*(\d+)/i,
      // Single number patterns
      /^(\d+)\s*(?:lakh|lakhs|l)$/i,
      /^(\d+)$/,
    ];

    // Extract all numbers from query for fallback
    const numbers = queryLower.match(/\d+/g);

    for (const pattern of priceRangePatterns) {
      const match = queryLower.match(pattern);
      if (match) {
        let min = match[1] ? parseInt(match[1]) : null;
        let max = match[2] ? parseInt(match[2]) : null;

        // Skip if no valid numbers found
        if (!min || (match[0].includes("to") && !max)) {
          continue;
        }

        // Handle single number patterns
        if (queryLower.includes("under") || queryLower.includes("below")) {
          max = min;
          min = null;
        } else if (
          queryLower.includes("above") ||
          queryLower.includes("over")
        ) {
          max = null;
        } else if (queryLower.includes("around") && !max) {
          // For "around 80", create a range ±20%
          const center = min || 0;
          min = Math.floor(center * 0.8);
          max = Math.floor(center * 1.2);
        } else if (match[0].match(/^\d+\s*(?:lakh|lakhs|l)?$/i) && !max) {
          // Single number - create range ±5 lakhs
          const center = min || 0;
          min = Math.max(0, center - 5);
          max = center + 5;
        }

        // Ensure min <= max
        if (min && max && min > max) {
          [min, max] = [max, min];
        }

        // Simplified price conversion
        const convertPrice = (price: number | null, matchedText: string) => {
          if (!price) return null;
          const text = matchedText.toLowerCase();

          if (text.includes("cr") || text.includes("crore")) {
            return price * 10000000; // Crores
          } else if (
            text.includes("lakh") ||
            text.includes("l") ||
            text.includes("between")
          ) {
            return price * 100000; // Lakhs
          } else if (price >= 10 && price <= 500) {
            return price * 100000; // Assume lakhs for reasonable range
          } else {
            return price * 100000; // Default to lakhs
          }
        };

        // Always treat as lakhs unless explicitly crores
        const text = match[0].toLowerCase();
        const isLakhPattern = !text.includes("cr") && !text.includes("crore");

        if (min) {
          if (isLakhPattern) {
            parsed.minPrice = (min * 100000).toString();
          } else {
            parsed.minPrice = convertPrice(min, match[0])?.toString() || "";
          }
        }
        if (max) {
          if (isLakhPattern) {
            parsed.maxPrice = (max * 100000).toString();
          } else {
            parsed.maxPrice = convertPrice(max, match[0])?.toString() || "";
          }
        }

        console.log("🔍 Price range detected:", {
          original: match[0],
          min: parsed.minPrice,
          max: parsed.maxPrice,
        });

        // Auto-sync with filter panel
        if (parsed.minPrice || parsed.maxPrice) {
          setTimeout(() => {
            setFilters((prev) => ({
              ...prev,
              min_price: parsed.minPrice || prev.min_price,
              max_price: parsed.maxPrice || prev.max_price,
            }));
          }, 0);
        }
        break;
      }
    }

    // Single number price detection
    if (
      !parsed.minPrice &&
      !parsed.maxPrice &&
      numbers &&
      numbers.length === 1
    ) {
      const num = parseInt(numbers[0]);
      if (num >= 10 && num <= 500) {
        // Create ±5 lakh range for single numbers
        parsed.minPrice = ((num - 5) * 100000).toString();
        parsed.maxPrice = ((num + 5) * 100000).toString();
        console.log("🔍 Single number price range:", {
          original: num,
          min: parsed.minPrice,
          max: parsed.maxPrice,
        });

        // Auto-sync with filter panel
        setTimeout(() => {
          setFilters((prev) => ({
            ...prev,
            min_price: parsed.minPrice,
            max_price: parsed.maxPrice,
          }));
        }, 0);
      }
    }

    // Fallback for multiple numbers
    if (
      !parsed.minPrice &&
      !parsed.maxPrice &&
      numbers &&
      numbers.length >= 2
    ) {
      const nums = numbers
        .map((n) => parseInt(n))
        .filter((n) => n >= 10 && n <= 500)
        .sort((a, b) => a - b);
      if (nums.length >= 2 && nums[0] !== nums[1]) {
        parsed.minPrice = (nums[0] * 100000).toString();
        parsed.maxPrice = (nums[1] * 100000).toString();
        console.log("🔍 Fallback price range:", {
          min: parsed.minPrice,
          max: parsed.maxPrice,
        });

        // Auto-sync with filter panel
        setTimeout(() => {
          setFilters((prev) => ({
            ...prev,
            min_price: parsed.minPrice,
            max_price: parsed.maxPrice,
          }));
        }, 0);
      }
    }

    // Check for possession status queries (ready to move, under construction, etc.)
    const possessionKeywords = [
      "ready to move",
      "under construction",
      "new launch",
      "immediate possession",
    ];
    const hasPossessionStatus = possessionKeywords.some((keyword) =>
      queryLower.includes(keyword)
    );

    // If it's a possession status query, don't try to parse it - let backend handle it
    if (hasPossessionStatus) {
      console.log("🔍 Possession status query detected:", query);
      return parsed; // Return with only originalQuery
    }

    // Enhanced BHK/bedroom parsing - more flexible patterns
    const bedroomPatterns = [
      /(\d+)\s*bhk/i, // "2 bhk", "2bhk"
      /(\d+)\s*b\s*h\s*k/i, // "2 b h k", "2BHK"
      /(\d+)\s*(?:bed|bedroom)s?/i, // "2 bed", "2 bedroom", "2 bedrooms"
      /(\d+)\s*(?:br|bdr)/i, // "2 br", "2 bdr"
      /(\d+)\s*room/i, // "2 room"
      /(?:^|\s)(\d+)(?=\s|$)/, // standalone number (fallback)
    ];

    let bhkMatch = null;
    let bedroomMatch = null;

    for (const pattern of bedroomPatterns) {
      const match = queryLower.match(pattern);
      if (match && parseInt(match[1]) <= 10) {
        // reasonable bedroom count
        parsed.bedrooms = match[1];
        // Store matches for later use
        if (pattern.source.includes("bhk")) {
          bhkMatch = match;
        } else {
          bedroomMatch = match;
        }
        console.log("🏠 Bedroom count detected:", match[1]);
        break;
      }
    }

    // Extract location information with road name handling
    const locationMatch = queryLower.match(/(?:in|at|near)\s+([a-zA-Z\s]+)/);
    if (locationMatch) {
      parsed.location = locationMatch[1].trim();
    } else {
      // If no preposition, check if the remaining text (after removing BHK) is a location
      let remainingText = queryLower;
      if (bhkMatch) {
        remainingText = remainingText.replace(bhkMatch[0], "").trim();
      }
      if (bedroomMatch) {
        remainingText = remainingText.replace(bedroomMatch[0], "").trim();
      }

      // Only remove common words if it's not a possession status query
      remainingText = remainingText
        .replace(
          /\b(apartment|flat|house|villa|property|properties|in|at|near)\b/g,
          ""
        )
        .trim();

      if (remainingText) {
        parsed.location = remainingText;
      }
    }

    // Handle road names - extract locality from road names like "Jakkur Road" -> "Jakkur"
    if (parsed.location) {
      const roadMatch = parsed.location.match(
        /^([a-zA-Z\s]+?)\s+(?:road|main\s+road|cross|street)$/i
      );
      if (roadMatch) {
        parsed.location = roadMatch[1].trim();
        console.log("🔍 Extracted locality from road:", roadMatch[1].trim());
      }
    }

    // Also handle direct road name extraction from original query
    if (!parsed.location && queryLower.includes("road")) {
      const directRoadMatch = queryLower.match(
        /^([a-zA-Z\s]+?)\s+(?:road|main\s+road|cross|street)/
      );
      if (directRoadMatch) {
        parsed.location = directRoadMatch[1].trim();
        console.log("🔍 Direct road extraction:", directRoadMatch[1].trim());
      }
    }

    // Normalize location using fuzzy matching
    if (parsed.location) {
      parsed.location = normalizeLocation(parsed.location);
    }

    // If no location found yet, check if the entire query is a location
    if (!parsed.location && !hasPossessionStatus) {
      const queryWords = queryLower.split(" ");
      for (const word of queryWords) {
        if (
          word.length > 3 &&
          (word.includes("sahakara") ||
            word.includes("nagar") ||
            word.includes("bangalore") ||
            word.includes("bengaluru"))
        ) {
          parsed.location = query.trim();
          console.log("🔍 Found location in query words:", parsed.location);
          break;
        }
      }
    }

    console.log("🔍 Query parsing:", {
      original: query,
      parsed: parsed,
    });

    // Extract property type
    if (queryLower.includes("apartment") || queryLower.includes("flat")) {
      parsed.propertyType = "apartment";
    } else if (queryLower.includes("villa")) {
      parsed.propertyType = "villa";
    } else if (queryLower.includes("house")) {
      parsed.propertyType = "house";
    }

    // Handle "under" price queries like "under 70L"
    const underPriceMatch = queryLower.match(
      /under\s*(\d+)\s*([lL]|lakh|lakhs|cr|crore)?/
    );
    if (underPriceMatch) {
      const num = parseInt(underPriceMatch[1]);
      const unit = underPriceMatch[2]?.toLowerCase() || "l";
      const maxPrice =
        unit.includes("cr") || unit.includes("crore")
          ? num * 10000000
          : num * 100000;
      parsed.maxPrice = maxPrice.toString();
      console.log("🔍 Under price detected:", {
        original: underPriceMatch[0],
        maxPrice,
      });
    }

    return parsed;
  };

  // Strict search function that requires ALL search terms to match
  const searchProperties = (properties: any[], searchTerm: string) => {
    if (!searchTerm.trim()) return properties;

    const searchWords = searchTerm.toLowerCase().trim().split(/\s+/);

    return properties.filter((property: any) => {
      // Create searchable text from all property fields
      const searchableFields = [
        property.title,
        property.locality,
        property.city,
        property.address,
        property.description,
        property.property_type,
        property.listing_type,
        property.amenities,
        property.furnishing,
        property.builder_name,
        property.project_name,
        property.area_name,
        property.landmark,
        property.bedrooms?.toString(),
        property.bathrooms?.toString(),
        property.built_up_area?.toString(),
      ]
        .filter(Boolean)
        .join(" ")
        .toLowerCase();

      // ALL search words must match - if even one doesn't match, exclude property
      return searchWords.every((word) => {
        // Check variations for common terms
        const variations = {
          bhk: ["bedroom", "bed"],
          apt: ["apartment"],
          villa: ["independent"],
          bangalore: ["bengaluru", "blr"],
          mumbai: ["bombay"],
          delhi: ["new delhi", "ncr"],
          chennai: ["madras"],
          hyderabad: ["hyd", "secunderabad"],
          pune: ["poona"],
          kolkata: ["calcutta"],
        };

        // Direct substring match
        if (searchableFields.includes(word)) return true;

        // Check if word appears in searchable fields
        const fieldWords = searchableFields.split(/\s+/);
        if (fieldWords.some((field) => field.includes(word))) return true;

        // Check variations
        for (const [key, values] of Object.entries(variations)) {
          if (word === key && values.some((v) => searchableFields.includes(v)))
            return true;
          if (values.includes(word) && searchableFields.includes(key))
            return true;
        }

        return false;
      });
    });
  };

  // Simple similarity calculation for fuzzy matching
  const calculateSimilarity = (str1: string, str2: string): number => {
    const longer = str1.length > str2.length ? str1 : str2;
    const shorter = str1.length > str2.length ? str2 : str1;

    if (longer.length === 0) return 1.0;

    const editDistance = levenshteinDistance(longer, shorter);
    return (longer.length - editDistance) / longer.length;
  };

  // Levenshtein distance calculation
  const levenshteinDistance = (str1: string, str2: string): number => {
    const matrix = [];

    for (let i = 0; i <= str2.length; i++) {
      matrix[i] = [i];
    }

    for (let j = 0; j <= str1.length; j++) {
      matrix[0][j] = j;
    }

    for (let i = 1; i <= str2.length; i++) {
      for (let j = 1; j <= str1.length; j++) {
        if (str2.charAt(i - 1) === str1.charAt(j - 1)) {
          matrix[i][j] = matrix[i - 1][j - 1];
        } else {
          matrix[i][j] = Math.min(
            matrix[i - 1][j - 1] + 1,
            matrix[i][j - 1] + 1,
            matrix[i - 1][j] + 1
          );
        }
      }
    }

    return matrix[str2.length][str1.length];
  };

  // Load properties without storing search history (for auto-search)
  const loadPropertiesWithoutHistory = async () => {
    try {
      setLoading(true);

      let fetchedProperties;

      // Check if we have current location from URL parameters (hero section)
      if (currentLocation) {
        console.log(
          "🌍 Loading properties near current location:",
          currentLocation
        );
        try {
          // Try nearby properties API first
          fetchedProperties = await api.fetchNearbyProperties(
            currentLocation.lat,
            currentLocation.lng
          );
          console.log("📍 Nearby properties found:", fetchedProperties.length);

          // If no nearby properties found, try broader search
          if (fetchedProperties.length === 0) {
            console.log("📍 No nearby properties, trying city-based search");
            fetchedProperties = await api.fetchPropertiesWithFilters({
              city: currentLocation.name,
              ordering:
                filters.sort === "price_low"
                  ? "sale_price"
                  : filters.sort === "price_high"
                  ? "-sale_price"
                  : "-id",
            });

            // If still no results, try broader location search
            if (fetchedProperties.length === 0) {
              console.log("📍 No city results, trying location search");
              fetchedProperties = await api.fetchPropertiesWithFilters({
                search: currentLocation.name,
                ordering:
                  filters.sort === "price_low"
                    ? "sale_price"
                    : filters.sort === "price_high"
                    ? "-sale_price"
                    : "-id",
              });
            }
          }
        } catch (error) {
          console.error("Error fetching nearby properties:", error);
          // Fallback to city-based search
          fetchedProperties = await api.fetchPropertiesWithFilters({
            city: currentLocation.name,
            ordering:
              filters.sort === "price_low"
                ? "sale_price"
                : filters.sort === "price_high"
                ? "-sale_price"
                : "-id",
          });
        }
      } else if (filters.search.trim()) {
        console.log("🔍 SEARCH MODE: Searching for:", filters.search.trim());

        const searchTerm = filters.search.trim();
        console.log("🔍 Checking for price range in:", searchTerm);

        const priceRangeMatch = searchTerm.match(
          /(\d+)\s*([lL]|lakh|lakhs|cr|crore)?\s*(?:to|-|till)\s*(\d+)\s*([lL]|lakh|lakhs|cr|crore)?/i
        );
        console.log("🔍 Price range match result:", priceRangeMatch);

        if (priceRangeMatch) {
          // Price range search detected
          const minNum = parseInt(priceRangeMatch[1]);
          const minUnit = priceRangeMatch[2]?.toLowerCase() || "l";
          const maxNum = parseInt(priceRangeMatch[3]);
          const maxUnit = priceRangeMatch[4]?.toLowerCase() || "l";

          const minPrice =
            minUnit.includes("cr") || minUnit.includes("crore")
              ? minNum * 10000000
              : minNum * 100000;
          const maxPrice =
            maxUnit.includes("cr") || maxUnit.includes("crore")
              ? maxNum * 10000000
              : maxNum * 100000;

          console.log("🔍 Price range detected:", {
            original: searchTerm,
            minNum,
            minUnit,
            maxNum,
            maxUnit,
            minPrice,
            maxPrice,
            minFormatted: `₹${minPrice / 100000}L`,
            maxFormatted: `₹${maxPrice / 100000}L`,
          });

          const searchFilters = {
            min_price: minPrice,
            max_price: maxPrice,
            ordering:
              filters.sort === "price_low"
                ? "sale_price"
                : filters.sort === "price_high"
                ? "-sale_price"
                : "-id",
          };

          console.log("🔍 Sending price range filters:", searchFilters);
          fetchedProperties = await api.fetchPropertiesWithFilters(
            searchFilters
          );
        } else {
          // First try backend search
          console.log(
            "🔍 No price range detected, using text search for:",
            searchTerm
          );
          fetchedProperties = await api.fetchPropertiesWithFilters({
            search: searchTerm,
            ordering:
              filters.sort === "price_low"
                ? "sale_price"
                : filters.sort === "price_high"
                ? "-sale_price"
                : "-id",
          });

          // If backend search returns no results, try client-side search on all properties
          if (fetchedProperties.length === 0) {
            console.log(
              "🔍 Backend search returned no results, trying client-side search..."
            );
            const allProperties = await api.fetchPropertiesWithFilters({
              ordering:
                filters.sort === "price_low"
                  ? "sale_price"
                  : filters.sort === "price_high"
                  ? "-sale_price"
                  : "-id",
            });
            fetchedProperties = searchProperties(allProperties, searchTerm);
            console.log(
              "🔍 Client-side search found:",
              fetchedProperties.length,
              "properties"
            );
          }
        }

        console.log(
          "🔍 Search API returned:",
          fetchedProperties.length,
          "properties"
        );

        console.log(
          "🔍 Search Results:",
          fetchedProperties.length,
          "properties found"
        );
        console.log(
          "🔍 Sample properties:",
          fetchedProperties
            .slice(0, 3)
            .map((p: any) => ({
              id: p.id,
              title: p.title,
              city: p.city,
              locality: p.locality,
            }))
        );

        // Generic fallback search with fuzzy matching
        if (fetchedProperties.length === 0) {
          console.log("🔍 No results found, trying fallback searches...");
          const fallbackSearches = generateFallbackSearches(
            filters.search.trim()
          );

          for (const fallbackTerm of fallbackSearches) {
            console.log(`🔍 Trying fallback search: ${fallbackTerm}`);
            const fallbackResults = await api.fetchPropertiesWithFilters({
              search: fallbackTerm,
            });
            console.log(
              `🔍 Fallback '${fallbackTerm}' returned:`,
              fallbackResults.length,
              "properties"
            );

            if (fallbackResults.length > 0) {
              fetchedProperties = fallbackResults;
              console.log("🔍 Using fallback results from:", fallbackTerm);
              break;
            }
          }
        }
      } else {
        console.log("🔍 FILTER MODE: No search term, applying filters only");
        // When no search term, show all properties with current filters
        // Use filter API when no search term - show all properties
        const filterParams = {
          listing_type:
            filters.listing_type.length > 0
              ? filters.listing_type.join(",")
              : undefined,
          property_type:
            filters.property_type.length > 0
              ? filters.property_type.join(",")
              : undefined,
          min_price:
            filters.min_price && filters.min_price.trim() !== ""
              ? parseInt(filters.min_price)
              : undefined,
          max_price:
            filters.max_price && filters.max_price.trim() !== ""
              ? parseInt(filters.max_price)
              : undefined,
          city: filters.city || undefined,
          locality:
            filters.locality && filters.locality.length > 0
              ? filters.locality.join(",")
              : undefined,
          bedrooms:
            filters.bedrooms.length > 0
              ? filters.bedrooms.join(",")
              : undefined,
          bathrooms:
            filters.bathrooms.length > 0
              ? filters.bathrooms.join(",")
              : undefined,
          furnishing:
            filters.furnishing.length > 0
              ? filters.furnishing.join(",")
              : undefined,
          monthly_rent: filters.monthly_rent
            ? parseInt(filters.monthly_rent)
            : undefined,
          loan_available: filters.loan_available
            ? filters.loan_available === "true"
            : undefined,
          amenities:
            filters.amenities.length > 0
              ? filters.amenities.join(",")
              : undefined,
          ordering:
            filters.sort === "price_low"
              ? "sale_price"
              : filters.sort === "price_high"
              ? "-sale_price"
              : "-id",
        };

        console.log("🔍 Price Filter Debug:", {
          min_price_raw: filters.min_price,
          max_price_raw: filters.max_price,
          min_price_parsed: filterParams.min_price,
          max_price_parsed: filterParams.max_price,
          min_price_empty:
            !filters.min_price || filters.min_price.trim() === "",
          max_price_empty:
            !filters.max_price || filters.max_price.trim() === "",
          will_send_price_filters: !!(
            filterParams.min_price || filterParams.max_price
          ),
        });

        // Remove undefined values from filterParams
        const cleanFilterParams = Object.fromEntries(
          Object.entries(filterParams).filter(
            ([_, value]) => value !== undefined
          )
        );

        console.log("🔍 Clean Filter Params:", cleanFilterParams);

        fetchedProperties = await api.fetchPropertiesWithFilters(
          cleanFilterParams
        );
      }

      const formattedProperties = fetchedProperties.map((property: any) => {
        // Generate coordinates if not available
        const coords =
          property.latitude && property.longitude
            ? { latitude: property.latitude, longitude: property.longitude }
            : generateCoordinates(property.city);

        // Debug logging for coordinate generation
        if (!property.latitude || !property.longitude) {
          console.log(
            `🗺️ Generated coordinates for ${property.title} in ${property.city}:`,
            coords
          );
        }

        return {
          id: property.id,
          title: property.title,
          location: `${property.locality}, ${property.city}`,
          beds: property.bedrooms || "N/A",
          baths: property.bathrooms || "N/A",
          area: property.built_up_area
            ? `${property.built_up_area} sq ft`
            : "N/A",
          price: property.monthly_rent
            ? `${formatPrice(parseFloat(property.monthly_rent))}/month`
            : formatPrice(
                parseFloat(
                  property.sale_price ||
                    property.starting_bid_price ||
                    property.per_bed_rent ||
                    0
                )
              ),
          imageUrl:
            property.primary_image_url ||
            property.primary_image ||
            property.photo ||
            undefined,
          urgent: property.listing_type === "urgent-sale",
          property_type: property.property_type,
          city: property.city,
          locality: property.locality,
          sale_price: property.sale_price,
          latitude: coords.latitude,
          longitude: coords.longitude,
          listing_type: property.listing_type,
          priceNegotiable: property.price_negotiable || false,
          loanAvailable: property.loan_available || false,
        };
      });

      // Log coordinate summary
      const propertiesWithCoords = formattedProperties.filter(
        (p: Property) => p.latitude && p.longitude
      );
      console.log(
        `🗺️ Coordinate Summary: ${propertiesWithCoords.length}/${formattedProperties.length} properties have coordinates`
      );
      console.log(
        `🗺️ Properties for map:`,
        formattedProperties.map((p: Property) => ({
          id: p.id,
          title: p.title,
          city: p.city,
          hasCoords: !!(p.latitude && p.longitude),
          coords: { lat: p.latitude, lng: p.longitude },
        }))
      );

      // Apply client-side search if search term exists and we have properties
      let finalProperties = formattedProperties;
      if (filters.search.trim() && formattedProperties.length > 0) {
        const searchFiltered = searchProperties(
          formattedProperties,
          filters.search.trim()
        );
        if (searchFiltered.length > 0) {
          finalProperties = searchFiltered;
          console.log(
            "🔍 Applied client-side search filter:",
            searchFiltered.length,
            "properties match"
          );
        }
      }

      console.log(
        "🔍 FINAL RESULT: Setting",
        finalProperties.length,
        "properties in state"
      );
      console.log("🔍 Search term was:", filters.search.trim() || "(empty)");

      setProperties(finalProperties);
      setFilteredProperties(finalProperties);
    } catch (error) {
      console.error("Error loading properties:", error);
      setProperties([]);
      setFilteredProperties([]);
    } finally {
      setLoading(false);
    }
  };

  // Load properties WITH search history storage (for manual search/filter actions)
  const loadProperties = async () => {
    try {
      setLoading(true);

      let fetchedProperties;

      // Check if we have current location from URL parameters (hero section)
      if (currentLocation) {
        console.log(
          "🌍 Loading properties near current location:",
          currentLocation
        );
        try {
          // Try nearby properties API first
          fetchedProperties = await api.fetchNearbyProperties(
            currentLocation.lat,
            currentLocation.lng
          );
          console.log("📍 Nearby properties found:", fetchedProperties.length);

          // If no nearby properties found, try broader search
          if (fetchedProperties.length === 0) {
            console.log("📍 No nearby properties, trying city-based search");
            fetchedProperties = await api.fetchPropertiesWithFilters({
              city: currentLocation.name,
              ordering:
                filters.sort === "price_low"
                  ? "sale_price"
                  : filters.sort === "price_high"
                  ? "-sale_price"
                  : "-id",
            });

            // If still no results, try broader location search
            if (fetchedProperties.length === 0) {
              console.log("📍 No city results, trying location search");
              fetchedProperties = await api.fetchPropertiesWithFilters({
                search: currentLocation.name,
                ordering:
                  filters.sort === "price_low"
                    ? "sale_price"
                    : filters.sort === "price_high"
                    ? "-sale_price"
                    : "-id",
              });
            }
          }
        } catch (error) {
          console.error("Error fetching nearby properties:", error);
          // Fallback to city-based search
          fetchedProperties = await api.fetchPropertiesWithFilters({
            city: currentLocation.name,
            ordering:
              filters.sort === "price_low"
                ? "sale_price"
                : filters.sort === "price_high"
                ? "-sale_price"
                : "-id",
          });
        }
      } else if (filters.search.trim()) {
        console.log("🔍 SEARCH MODE: Searching for:", filters.search.trim());

        const searchTerm = filters.search.trim();
        console.log("🔍 Checking for price range in:", searchTerm);

        const priceRangeMatch = searchTerm.match(
          /(\d+)\s*([lL]|lakh|lakhs|cr|crore)?\s*(?:to|-|till)\s*(\d+)\s*([lL]|lakh|lakhs|cr|crore)?/i
        );
        console.log("🔍 Price range match result:", priceRangeMatch);

        if (priceRangeMatch) {
          // Price range search detected
          const minNum = parseInt(priceRangeMatch[1]);
          const minUnit = priceRangeMatch[2]?.toLowerCase() || "l";
          const maxNum = parseInt(priceRangeMatch[3]);
          const maxUnit = priceRangeMatch[4]?.toLowerCase() || "l";

          const minPrice =
            minUnit.includes("cr") || minUnit.includes("crore")
              ? minNum * 10000000
              : minNum * 100000;
          const maxPrice =
            maxUnit.includes("cr") || maxUnit.includes("crore")
              ? maxNum * 10000000
              : maxNum * 100000;

          console.log("🔍 Price range detected:", {
            original: searchTerm,
            minNum,
            minUnit,
            maxNum,
            maxUnit,
            minPrice,
            maxPrice,
            minFormatted: `₹${minPrice / 100000}L`,
            maxFormatted: `₹${maxPrice / 100000}L`,
          });

          const searchFilters = {
            min_price: minPrice,
            max_price: maxPrice,
            ordering:
              filters.sort === "price_low"
                ? "sale_price"
                : filters.sort === "price_high"
                ? "-sale_price"
                : "-id",
          };

          console.log("🔍 Sending price range filters:", searchFilters);
          fetchedProperties = await api.fetchPropertiesWithFilters(
            searchFilters
          );
        } else {
          // First try backend search
          console.log(
            "🔍 No price range detected, using text search for:",
            searchTerm
          );
          fetchedProperties = await api.fetchPropertiesWithFilters({
            search: searchTerm,
            ordering:
              filters.sort === "price_low"
                ? "sale_price"
                : filters.sort === "price_high"
                ? "-sale_price"
                : "-id",
          });

          // If backend search returns no results, try client-side search on all properties
          if (fetchedProperties.length === 0) {
            console.log(
              "🔍 Backend search returned no results, trying client-side search..."
            );
            const allProperties = await api.fetchPropertiesWithFilters({
              ordering:
                filters.sort === "price_low"
                  ? "sale_price"
                  : filters.sort === "price_high"
                  ? "-sale_price"
                  : "-id",
            });
            fetchedProperties = searchProperties(allProperties, searchTerm);
            console.log(
              "🔍 Client-side search found:",
              fetchedProperties.length,
              "properties"
            );
          }
        }

        console.log(
          "🔍 Search API returned:",
          fetchedProperties.length,
          "properties"
        );

        console.log(
          "🔍 Search Results:",
          fetchedProperties.length,
          "properties found"
        );
        console.log(
          "🔍 Sample properties:",
          fetchedProperties
            .slice(0, 3)
            .map((p: any) => ({
              id: p.id,
              title: p.title,
              city: p.city,
              locality: p.locality,
            }))
        );

        // Generic fallback search with fuzzy matching
        if (fetchedProperties.length === 0) {
          console.log("🔍 No results found, trying fallback searches...");
          const fallbackSearches = generateFallbackSearches(
            filters.search.trim()
          );

          for (const fallbackTerm of fallbackSearches) {
            console.log(`🔍 Trying fallback search: ${fallbackTerm}`);
            const fallbackResults = await api.fetchPropertiesWithFilters({
              search: fallbackTerm,
            });
            console.log(
              `🔍 Fallback '${fallbackTerm}' returned:`,
              fallbackResults.length,
              "properties"
            );

            if (fallbackResults.length > 0) {
              fetchedProperties = fallbackResults;
              console.log("🔍 Using fallback results from:", fallbackTerm);
              break;
            }
          }
        }

        // Note: Search history is stored only on explicit user actions (Search button, Apply Filters)
      } else {
        console.log("🔍 FILTER MODE: No search term, applying filters only");
        // When no search term, show all properties with current filters
        // Use filter API when no search term - show all properties
        const filterParams = {
          listing_type:
            filters.listing_type.length > 0
              ? filters.listing_type.join(",")
              : undefined,
          property_type:
            filters.property_type.length > 0
              ? filters.property_type.join(",")
              : undefined,
          min_price:
            filters.min_price && filters.min_price.trim() !== ""
              ? parseInt(filters.min_price)
              : undefined,
          max_price:
            filters.max_price && filters.max_price.trim() !== ""
              ? parseInt(filters.max_price)
              : undefined,
          city: filters.city || undefined,
          locality:
            filters.locality && filters.locality.length > 0
              ? filters.locality.join(",")
              : undefined,
          bedrooms:
            filters.bedrooms.length > 0
              ? filters.bedrooms.join(",")
              : undefined,
          bathrooms:
            filters.bathrooms.length > 0
              ? filters.bathrooms.join(",")
              : undefined,
          furnishing:
            filters.furnishing.length > 0
              ? filters.furnishing.join(",")
              : undefined,
          monthly_rent: filters.monthly_rent
            ? parseInt(filters.monthly_rent)
            : undefined,
          loan_available: filters.loan_available
            ? filters.loan_available === "true"
            : undefined,
          amenities:
            filters.amenities.length > 0
              ? filters.amenities.join(",")
              : undefined,
          ordering:
            filters.sort === "price_low"
              ? "sale_price"
              : filters.sort === "price_high"
              ? "-sale_price"
              : "-id",
        };

        console.log("🔍 Price Filter Debug:", {
          min_price_raw: filters.min_price,
          max_price_raw: filters.max_price,
          min_price_parsed: filterParams.min_price,
          max_price_parsed: filterParams.max_price,
          min_price_empty:
            !filters.min_price || filters.min_price.trim() === "",
          max_price_empty:
            !filters.max_price || filters.max_price.trim() === "",
          will_send_price_filters: !!(
            filterParams.min_price || filterParams.max_price
          ),
        });

        // Remove undefined values from filterParams
        const cleanFilterParams = Object.fromEntries(
          Object.entries(filterParams).filter(
            ([_, value]) => value !== undefined
          )
        );

        console.log("🔍 Clean Filter Params:", cleanFilterParams);

        fetchedProperties = await api.fetchPropertiesWithFilters(
          cleanFilterParams
        );
        // Note: Search history is stored only on explicit user actions (Search button, Apply Filters)
      }

      const formattedProperties = fetchedProperties.map((property: any) => {
        // Generate coordinates if not available
        const coords =
          property.latitude && property.longitude
            ? { latitude: property.latitude, longitude: property.longitude }
            : generateCoordinates(property.city);

        // Debug logging for coordinate generation
        if (!property.latitude || !property.longitude) {
          console.log(
            `🗺️ Generated coordinates for ${property.title} in ${property.city}:`,
            coords
          );
        }

        return {
          id: property.id,
          title: property.title,
          location: `${property.locality}, ${property.city}`,
          beds: property.bedrooms || "N/A",
          baths: property.bathrooms || "N/A",
          area: property.built_up_area
            ? `${property.built_up_area} sq ft`
            : "N/A",
          price: property.monthly_rent
            ? `${formatPrice(parseFloat(property.monthly_rent))}/month`
            : formatPrice(
                parseFloat(
                  property.sale_price ||
                    property.starting_bid_price ||
                    property.per_bed_rent ||
                    0
                )
              ),
          imageUrl:
            property.primary_image_url ||
            property.primary_image ||
            property.photo ||
            undefined,
          urgent: property.listing_type === "urgent-sale",
          property_type: property.property_type,
          city: property.city,
          locality: property.locality,
          sale_price: property.sale_price,
          latitude: coords.latitude,
          longitude: coords.longitude,
          listing_type: property.listing_type,
          priceNegotiable: property.price_negotiable || false,
          loanAvailable: property.loan_available || false,
        };
      });

      // Log coordinate summary
      const propertiesWithCoords = formattedProperties.filter(
        (p: Property) => p.latitude && p.longitude
      );
      console.log(
        `🗺️ Coordinate Summary: ${propertiesWithCoords.length}/${formattedProperties.length} properties have coordinates`
      );
      console.log(
        `🗺️ Properties for map:`,
        formattedProperties.map((p: Property) => ({
          id: p.id,
          title: p.title,
          city: p.city,
          hasCoords: !!(p.latitude && p.longitude),
          coords: { lat: p.latitude, lng: p.longitude },
        }))
      );

      // Apply client-side search if search term exists and we have properties
      let finalProperties = formattedProperties;
      if (filters.search.trim() && formattedProperties.length > 0) {
        const searchFiltered = searchProperties(
          formattedProperties,
          filters.search.trim()
        );
        if (searchFiltered.length > 0) {
          finalProperties = searchFiltered;
          console.log(
            "🔍 Applied client-side search filter:",
            searchFiltered.length,
            "properties match"
          );
        }
      }

      console.log(
        "🔍 FINAL RESULT: Setting",
        finalProperties.length,
        "properties in state"
      );
      console.log("🔍 Search term was:", filters.search.trim() || "(empty)");

      setProperties(finalProperties);
      setFilteredProperties(finalProperties);
    } catch (error) {
      console.error("Error loading properties:", error);
      setProperties([]);
      setFilteredProperties([]);
    } finally {
      setLoading(false);
    }
  };

  const formatPrice = (price: number) => {
    if (price >= 10000000) return `₹${(price / 10000000).toFixed(1)} Cr`;
    if (price >= 100000) return `₹${(price / 100000).toFixed(0)} L`;
    return `₹${price.toLocaleString()}`;
  };

  const handleFilterChange = (
    key: keyof Filters,
    value: string | string[] | boolean
  ) => {
    setFilters((prev) => {
      const newFilters = { ...prev, [key]: value };
      // Log role-specific filter changes
      if (
        [
          "rent_range_min",
          "rent_range_max",
          "deposit_amount",
          "maintenance_cost",
          "tenant_preference",
          "lease_duration",
        ].includes(key)
      ) {
        console.log("🏠 Tenant filter changed:", key, value);
      }
      if (
        ["possession_status", "possession_date", "rera_approved"].includes(key)
      ) {
        console.log("🏡 Buyer filter changed:", key, value);
      }
      if (key === "verified_only") {
        console.log("✅ Verified filter changed:", value);
      }
      
      // Track sort change
      if (key === "sort") {
        import("../utils/behaviorTracker").then(({ behaviorTracker }) => {
          behaviorTracker.trackSortChange({ sort_by: value });
        });
      }
      
      return newFilters;
    });
  };

  const clearFilters = () => {
    setFilters({
      search: "",
      bedrooms: [],
      bathrooms: [],
      property_type: [],
      min_price: "",
      max_price: "",
      city: "",
      sort: "latest",
      furnishing: [],
      monthly_rent: "",
      loan_available: "",
      amenities: [],
      listing_type: [],
      locality: [],
      // Tenant-specific filters
      rent_range_min: "",
      rent_range_max: "",
      deposit_amount: "",
      maintenance_cost: "",
      tenant_preference: "",
      lease_duration: "",
      // Buyer-specific filters
      possession_status: "",
      possession_date: "",
      rera_approved: "",
      // Common filters
      verified_only: false,
    });
  };

  const handleFavorite = async (propertyId: number) => {
    setClickedHearts((prev) => new Set(prev).add(propertyId));

    const token = localStorage.getItem("access_token");
    if (token) {
      try {
        await fetch("http://localhost:8000/api/add-to-wishlist/", {
          method: "POST",
          headers: {
            "Content-Type": "application/json",
            Authorization: `Bearer ${token}`,
          },
          body: JSON.stringify({ property_id: propertyId }),
        });
      } catch (error) {
        console.error("Error adding to wishlist:", error);
      }
    } else {
      localStorage.setItem("pending_wishlist", propertyId.toString());
      setTimeout(() => navigate("/login"), 300);
    }
  };

  const handleSuggestionSelect = (suggestion: string) => {
    setFilters((prev) => ({ ...prev, search: suggestion }));
    setShowSuggestions(false);
  };

  const handleSearchInputChange = (value: string) => {
    handleFilterChange("search", value);

    // Show suggestions when typing 2+ characters or when clicking on empty search
    setShowSuggestions(
      value.length >= 2 ||
        (value.length === 0 && document.activeElement === searchInputRef)
    );

    // Auto-apply smart filters when patterns are detected
    if (value.trim().length >= 3) {
      const parsedQuery = parseSearchQuery(value.trim());
      console.log("🔍 Auto-parsing query:", value, parsedQuery);

      // Apply filters with debouncing
      if (searchDebounceTimer) {
        clearTimeout(searchDebounceTimer);
      }

      const timer = setTimeout(() => {
        setFilters((prev) => {
          const newFilters = { ...prev };

          // Apply price filters
          if (parsedQuery.minPrice) newFilters.min_price = parsedQuery.minPrice;
          if (parsedQuery.maxPrice) newFilters.max_price = parsedQuery.maxPrice;

          // Apply bedroom filter
          if (
            parsedQuery.bedrooms &&
            !prev.bedrooms.includes(parsedQuery.bedrooms)
          ) {
            newFilters.bedrooms = [parsedQuery.bedrooms];
          }

          // Apply city filter
          if (parsedQuery.location && parsedQuery.location !== prev.city) {
            newFilters.city = parsedQuery.location;
          }

          // Apply property type filter
          if (
            parsedQuery.propertyType &&
            !prev.property_type.includes(parsedQuery.propertyType)
          ) {
            newFilters.property_type = [parsedQuery.propertyType];
          }

          return newFilters;
        });
      }, 300);

      setSearchDebounceTimer(timer);
    }
  };

  const handleSearchSubmit = async () => {
    if (filters.search.trim()) {
      setShowSuggestions(false);
      loadProperties();

      // Track search event
      console.log('🔍 About to track search event:', {
        query: filters.search.trim(),
        location: filters.city,
        filters: {
          city: filters.city,
          property_type: filters.property_type,
          listing_type: filters.listing_type,
          bedrooms: filters.bedrooms,
          min_price: filters.min_price,
          max_price: filters.max_price,
        },
      });
      const { behaviorTracker } = await import("../utils/behaviorTracker");
      behaviorTracker.trackSearch({
        query: filters.search.trim(),
        location: filters.city,
        filters: {
          city: filters.city,
          property_type: filters.property_type,
          listing_type: filters.listing_type,
          bedrooms: filters.bedrooms,
          min_price: filters.min_price,
          max_price: filters.max_price,
        },
      });
      console.log('✅ trackSearch called successfully');
    }
    console.log("🔍 Search button clicked");
    setShowSuggestions(false);
    await storeSearch(filters.search.trim(), filters);
    loadPropertiesWithoutHistory();
  };

  const handleSearchKeyPress = (e: React.KeyboardEvent) => {
    if (e.key === "Enter") {
      handleSearchSubmit();
    }
  };

  // Load recently viewed properties
  const loadRecentlyViewedProperties = async (userId: string) => {
    try {
      setLoadingRecentViews(true);
      console.log("🔍 Loading recently viewed properties for user:", userId);
      
      if (!userId || userId === "anonymous" || userId.startsWith("guest_")) {
        console.log("❌ Invalid user ID for recently viewed:", userId);
        setRecentlyViewedProperties([]);
        return;
      }
      
      // Use the property views API to get user's view history
      const { propertyViewsAPI } = await import('../api/propertyViews');
      const viewHistory = await propertyViewsAPI.getUserHistory(userId, 10); // Get last 10 viewed properties
      
      console.log("📊 View history response:", viewHistory);
      
      // Handle different possible response structures
      let properties = [];
      if (viewHistory) {
        if (viewHistory.properties && Array.isArray(viewHistory.properties)) {
          properties = viewHistory.properties;
        } else if (viewHistory.views && Array.isArray(viewHistory.views)) {
          properties = viewHistory.views;
        } else if (Array.isArray(viewHistory)) {
          properties = viewHistory;
        }
      }
      
      console.log("📊 Extracted properties:", properties);
      
      if (properties && properties.length > 0) {
        // Convert view history to property format
        const recentProperties: Property[] = properties.map((property: any) => {
          // Handle both direct property objects and view objects with property data
          const propData = property.property || property;
          
          return {
            id: propData.id || property.property_id,
            title: propData.title || `Property ${propData.id || property.property_id}`,
            location: `${propData.locality || 'Locality'}, ${propData.city || 'City'}`,
            beds: propData.bedrooms || 'N/A',
            baths: propData.bathrooms || 'N/A',
            area: propData.built_up_area ? `${propData.built_up_area} sq ft` : 'N/A',
            price: propData.monthly_rent
              ? `${formatPrice(parseFloat(propData.monthly_rent))}/month`
              : formatPrice(
                  parseFloat(
                    propData.sale_price ||
                      propData.starting_bid_price ||
                      propData.per_bed_rent ||
                      0
                  )
                ),
            imageUrl: propData.primary_image_url || propData.primary_image || propData.photo || undefined,
            urgent: propData.listing_type === 'urgent-sale',
            property_type: propData.property_type || 'apartment',
            city: propData.city || 'City',
            locality: propData.locality || 'Locality',
            sale_price: propData.sale_price ? parseFloat(propData.sale_price) : 0,
            latitude: propData.latitude,
            longitude: propData.longitude,
            listing_type: propData.listing_type
          };
        });
        
        console.log("✅ Recently viewed properties loaded:", recentProperties.length);
        setRecentlyViewedProperties(recentProperties);
      } else {
        console.log("📊 No view history found for user:", userId);
        setRecentlyViewedProperties([]);
      }
    } catch (error) {
      console.error("❌ Error loading recently viewed properties:", error);
      setRecentlyViewedProperties([]);
    } finally {
      setLoadingRecentViews(false);
    }
  };

  const handleSearchInputFocus = () => {
    // Always show suggestions on focus, even for empty search
    setShowSuggestions(true);
    console.log("🔍 Search input focused, showing suggestions");
  };

  // Enhanced popular search terms with smart suggestions
  const popularSearches = [
    "2 BHK in Bangalore",
    "3 BHK apartment in Mumbai",
    "Villa in Pune under 2 Cr",
    "Apartment in Delhi",
    "Furnished flat ready to move",
    "Properties under 1 Cr",
    "New launch projects",
    "Gated community properties",
    "Properties near metro station",
    "IT corridor Bangalore",
    "Family friendly localities",
    "Investment properties",
    "Koramangala 2 BHK",
    "Whitefield apartments",
    "Bandra West flats",
    "Andheri East properties",
    "Gurgaon sector 14",
    "Noida expressway",
    "3 BHK between 80L to 1.5Cr",
    "Luxury villas above 3 Cr",
  ];

  const handleSearchInputBlur = () => {
    // Delay hiding suggestions to allow for clicks
    setTimeout(() => setShowSuggestions(false), 300);
  };

  const getAccurateLocation = async (lat: number, lng: number) => {
    const GEOAPIFY_API_KEY = '9b3807944bbe44d2a0b29a5706c15bf9';
    
    const geocodingServices = [
      async () => {
        const response = await fetch(
          `https://api.geoapify.com/v1/geocode/reverse?lat=${lat}&lon=${lng}&apiKey=${GEOAPIFY_API_KEY}&format=json&lang=en`
        );
        const data = await response.json();
        if (data?.results?.[0]) {
          const result = data.results[0];
          return {
            locality: result.suburb || result.district || result.neighbourhood,
            city: result.city || result.county,
            state: result.state,
            street: result.street,
            formatted: result.formatted,
            confidence: result.rank?.confidence || 0.8
          };
        }
        return null;
      },
      
      async () => {
        const response = await fetch(
          `https://nominatim.openstreetmap.org/reverse?format=json&lat=${lat}&lon=${lng}&zoom=16&addressdetails=1`
        );
        const data = await response.json();
        if (data?.address) {
          return {
            locality: data.address.suburb || data.address.neighbourhood || data.address.village,
            city: data.address.city || data.address.town,
            state: data.address.state,
            street: data.address.road,
            formatted: data.display_name,
            confidence: 0.6
          };
        }
        return null;
      }
    ];

    for (const service of geocodingServices) {
      try {
        const result = await service();
        if (result && (result.locality || result.city)) {
          return result;
        }
      } catch (error) {
        console.warn('Geocoding service failed:', error);
        continue;
      }
    }
    
    return null;
  };

  const handleCurrentLocation = async () => {
    if (!navigator.geolocation) {
      toast.error("Geolocation is not supported by this browser.");
      return;
    }

    // Check for cached location first
    const cachedLocation = localStorage.getItem('cached_location');
    const cacheTime = localStorage.getItem('location_cache_time');
    const CACHE_DURATION = 10 * 60 * 1000; // 10 minutes
    
    if (cachedLocation && cacheTime && (Date.now() - parseInt(cacheTime)) < CACHE_DURATION) {
      const location = JSON.parse(cachedLocation);
      const locationData = {
        lat: location.lat,
        lng: location.lng,
        name: location.name,
      };
      setCurrentLocation(locationData);
      setFilters((prev) => ({ ...prev, search: location.name }));
      setMapCenter({ lat: location.lat, lng: location.lng });
      toast.success(`Location: ${location.name}`);
      console.log('🌍 Using cached location:', location.name);
      return;
    }

    toast.info('Getting your precise location...');
    console.log('🌍 Getting fresh location...');

    try {
      const position = await new Promise<GeolocationPosition>(
        (resolve, reject) => {
          navigator.geolocation.getCurrentPosition(resolve, reject, {
            enableHighAccuracy: true,
            timeout: 20000,
            maximumAge: 60000,
          });
        }
      );

      const { latitude, longitude, accuracy } = position.coords;
      console.log(`🌍 GPS accuracy: ${accuracy}m`);
      
      const locationData = await getAccurateLocation(latitude, longitude);
      
      if (locationData) {
        let locationName = 'Current Location';
        
        if (locationData.locality) {
          locationName = locationData.locality;
          // Handle road names - extract locality from road names like "Jakkur Road" -> "Jakkur"
          const roadMatch = locationName.match(/^([a-zA-Z\s]+?)\s+(?:road|main\s+road|cross|street)$/i);
          if (roadMatch) {
            locationName = roadMatch[1].trim();
          }
        } else if (locationData.city) {
          locationName = locationData.city;
        }
        
        // Cache the location
        localStorage.setItem('cached_location', JSON.stringify({
          name: locationName,
          lat: latitude,
          lng: longitude,
          accuracy: accuracy
        }));
        localStorage.setItem('location_cache_time', Date.now().toString());
        
        // Update current location and filters
        const currentLocationData = {
          lat: latitude,
          lng: longitude,
          name: locationName,
        };
        setCurrentLocation(currentLocationData);
        setFilters((prev) => ({ ...prev, search: locationName }));
        setMapCenter({ lat: latitude, lng: longitude });
        
        toast.success(`Location: ${locationName} (±${Math.round(accuracy)}m)`);
        console.log('🌍 Accurate location set:', locationName, `(±${Math.round(accuracy)}m)`);
      } else {
        throw new Error('No location data available');
      }
    } catch (error: any) {
      console.error('🌍 Location error:', error);
      
      let errorMessage = "Unable to get your precise location.";
      if (error.code === 1) {
        errorMessage = "Location access denied. Please enable location access in your browser settings and try again.";
      } else if (error.code === 2) {
        errorMessage = "Location unavailable. Please check your GPS settings and try again.";
      } else if (error.code === 3) {
        errorMessage = "Location request timed out. Please try again.";
      }
      
      toast.error(errorMessage);
    }
  };

  const handleLocationAllow = async () => {
    setShowLocationModal(false);
    await handleCurrentLocation();
  };

  const handleLocationCancel = () => {
    setShowLocationModal(false);
  };

  const handleSaveSearch = async () => {
    try {
      const user = JSON.parse(localStorage.getItem("user") || "{}");
      const userId = user.user_id || user.id || localStorage.getItem("user_id");

      if (!userId || userId === "anonymous" || userId.startsWith("guest_")) {
        alert("Please login to save searches for email alerts.");
        return;
      }

      const searchData = {
        user_id: userId,
        query: filters.search || "",
        filters: {
          city: filters.city || "",
          locality: Array.isArray(filters.locality) ? filters.locality : [],
          property_type: filters.property_type || [],
          listing_type: filters.listing_type || [],
          bedrooms: filters.bedrooms || [],
          state: "Karnataka",
        },
        location: filters.city || "",
      };

      const response = await fetch(
        "http://localhost:8000/api/search-history/",
        {
          method: "POST",
          headers: { "Content-Type": "application/json" },
          body: JSON.stringify(searchData),
        }
      );

      if (response.ok) {
        alert(
          "Search saved! You will receive email notifications when matching properties are posted."
        );
      } else {
        console.error("Failed to save search:", await response.text());
        alert("Failed to save search. Please try again.");
      }
    } catch (error) {
      console.error("Error saving search:", error);
      alert("Error saving search. Please try again.");
    }
  };

  // Initialize speech recognition and cleanup
  useEffect(() => {
    if ("SpeechRecognition" in window || "webkitSpeechRecognition" in window) {
      const SpeechRecognition =
        window.SpeechRecognition || window.webkitSpeechRecognition;
      const recognitionInstance = new SpeechRecognition();

      recognitionInstance.continuous = false;
      recognitionInstance.interimResults = false;
      recognitionInstance.lang = "en-US";

      recognitionInstance.onresult = (event) => {
        const transcript = event.results[0][0].transcript;
        setFilters((prev) => ({ ...prev, search: transcript }));
        setIsListening(false);
      };

      recognitionInstance.onerror = (event) => {
        console.error("Speech recognition error:", event.error);
        setIsListening(false);
        if (event.error === "not-allowed") {
          alert(
            "Microphone access denied. Please allow microphone access and try again."
          );
        }
      };

      recognitionInstance.onend = () => {
        setIsListening(false);
      };

      setRecognition(recognitionInstance as any);
    }

    // Cleanup debounce timer on unmount
    return () => {
      if (searchDebounceTimer) {
        clearTimeout(searchDebounceTimer);
      }
    };
  }, [searchDebounceTimer]);

  const handleVoiceSearch = () => {
    if (!recognition) {
      alert("Speech recognition is not supported in your browser.");
      return;
    }

    if (isListening) {
      recognition.stop();
      setIsListening(false);
    } else {
      try {
        recognition.start();
        setIsListening(true);
      } catch (error) {
        console.error("Error starting speech recognition:", error);
        alert("Error starting voice recognition. Please try again.");
      }
    }
  };

  const handleImageSearch = async () => {
    const input = document.createElement("input");
    input.type = "file";
    input.accept = "image/*";
    input.multiple = false;
    
    input.onchange = async (e) => {
      const file = (e.target as HTMLInputElement).files?.[0];
      if (!file) return;
      
      // Validate file size (max 10MB)
      if (file.size > 10 * 1024 * 1024) {
        toast.error("Image file is too large. Please select an image under 10MB.");
        return;
      }
      
      // Validate file type
      if (!file.type.startsWith('image/')) {
        toast.error("Please select a valid image file (JPG, PNG, GIF, etc.).");
        return;
      }
      
      try {
        setLoading(true);
        toast.info("Analyzing image and searching for similar properties...");
        
        // Import the image search service
        const { imageSearchService } = await import('../services/imageSearchService');
        
        // Get user preferences for better matching
        const userPreferences = {
          city: filters.city,
          propertyType: filters.property_type.length > 0 ? filters.property_type[0] : undefined,
          maxPrice: filters.max_price ? parseInt(filters.max_price) : undefined,
          minBedrooms: filters.bedrooms.length > 0 ? parseInt(filters.bedrooms[0]) : undefined
        };
        
        // Search for similar properties
        const result = await imageSearchService.intelligentPropertySearch(file, userPreferences);
        
        if (result.success && result.similarProperties && result.similarProperties.length > 0) {
          // Format properties for display
          const formattedProperties = result.similarProperties.map((property: any) => {
            const coords = property.latitude && property.longitude
              ? { latitude: property.latitude, longitude: property.longitude }
              : generateCoordinates(property.city);
            
            return {
              id: property.id,
              title: property.title,
              location: `${property.locality}, ${property.city}`,
              beds: property.bedrooms || "N/A",
              baths: property.bathrooms || "N/A",
              area: property.built_up_area ? `${property.built_up_area} sq ft` : "N/A",
              price: property.monthly_rent
                ? `${formatPrice(parseFloat(property.monthly_rent))}/month`
                : formatPrice(parseFloat(property.sale_price || 0)),
              imageUrl: property.primary_image_url || property.primary_image || property.photo || undefined,
              urgent: property.listing_type === "urgent-sale",
              property_type: property.property_type,
              city: property.city,
              locality: property.locality,
              sale_price: property.sale_price,
              latitude: coords.latitude,
              longitude: coords.longitude,
              listing_type: property.listing_type,
              priceNegotiable: property.price_negotiable || false,
              loanAvailable: property.loan_available || false,
              similarityScore: property.similarityScore
            };
          });
          
          // Update the properties display
          setProperties(formattedProperties);
          setFilteredProperties(formattedProperties);
          
          // Update search term to indicate image search
          const fileName = file.name.length > 20 ? file.name.substring(0, 20) + '...' : file.name;
          setFilters(prev => ({ ...prev, search: `📷 Image search: ${fileName}` }));
          
          // Show success message
          const confidence = Math.round((result.confidence || 0) * 100);
          toast.success(
            `Found ${formattedProperties.length} similar properties (${confidence}% confidence match)`
          );
          
          console.log('🎯 Image search results:', {
            totalFound: formattedProperties.length,
            confidence: confidence,
            topScores: formattedProperties.slice(0, 3).map(p => p.similarityScore)
          });
          
        } else {
          toast.error(
            result.error || "No similar properties found for this image. Try uploading a clearer property photo or use text search."
          );
          console.log('❌ Image search failed:', result.error);
        }
        
      } catch (error) {
        console.error('❌ Image search error:', error);
        toast.error("Failed to analyze image. Please try again or use text search.");
      } finally {
        setLoading(false);
      }
    };
    
    input.click();
  };

  return (
    <div className="bg-gray-50 h-screen overflow-hidden">
      {/* Header with Search and Filters */}
      <div className="bg-white shadow-sm border-b">
        <div className="max-w-7xl mx-auto px-4 py-4">
          {/* Search Bar with Filter Button */}
          <div className="flex items-center gap-4">

            <div className="flex-1 relative">
              <input
                ref={(el) => {
                  setSearchInputRef(el);
                }}
                type="text"
                placeholder="Search by location, landmark, property type etc"
                value={filters.search}
                onChange={(e) => {
                  handleSearchInputChange(e.target.value);
                }}
                onFocus={handleSearchInputFocus}
                onBlur={handleSearchInputBlur}
                onKeyPress={handleSearchKeyPress}
                className="w-full pl-4 pr-20 py-3 border border-gray-300 rounded-xl focus:ring-2 focus:ring-[#0056D2] focus:border-transparent text-gray-900 bg-white"
              />
              <SearchSuggestions
                query={filters.search}
                onSuggestionSelect={handleSuggestionSelect}
                onQueryChange={handleSearchInputChange}
                isVisible={
                  showSuggestions &&
                  (filters.search.length >= 2 || filters.search.length === 0)
                }
                onClose={() => setShowSuggestions(false)}
                popularSearches={popularSearches}
              />
              <div className="absolute right-2 top-1/2 transform -translate-y-1/2 flex gap-2">
                <button
                  onClick={handleImageSearch}
                  disabled={loading}
                  className="bg-white/10 border border-gray-300 text-gray-700 px-3 py-2 rounded-lg hover:bg-gray-50 transition-colors flex items-center text-sm disabled:opacity-50 disabled:cursor-not-allowed"
                  title="Search by image - Upload a property photo to find similar properties"
                >
                  <Camera className="w-4 h-4" />
                </button>
                <button
                  onClick={handleVoiceSearch}
                  className={`border border-gray-300 px-3 py-2 rounded-lg transition-all flex items-center text-sm ${
                    isListening
                      ? "bg-red-50 text-red-600 border-red-300 animate-pulse shadow-lg shadow-red-200"
                      : "bg-white/10 text-gray-700 hover:bg-gray-50"
                  }`}
                  title={
                    isListening ? "Listening... Click to stop" : "Voice search"
                  }
                >
                  <Mic
                    className={`w-4 h-4 ${isListening ? "animate-pulse" : ""}`}
                  />
                </button>
                <button
                  onClick={handleCurrentLocation}
                  className="bg-white/10 border border-gray-300 text-gray-700 px-3 py-2 rounded-lg hover:bg-gray-50 transition-colors flex items-center text-sm"
                  title="Search by current location"
                >
                  <Navigation className="w-4 h-4" />
                </button>
                <button
                  onClick={handleSearchSubmit}
                  className="bg-[#0056D2] text-white px-4 py-2 rounded-lg hover:bg-[#0056D2]/90 transition-colors font-medium text-sm"
                >
                  Search
                </button>
                <button
                  onClick={handleSaveSearch}
                  className="bg-green-600 text-white px-4 py-2 rounded-lg hover:bg-green-700 transition-colors font-medium text-sm"
                  title="Save search for email alerts"
                >
                  Save
                </button>
              </div>
            </div>

            {/* Filters Dropdown */}
            <div className="relative">
              <button
                onClick={() => setShowFilters(!showFilters)}
                className="flex items-center gap-2 px-6 py-3 border border-gray-300 rounded-xl hover:bg-gray-50 transition-colors"
              >
                <svg
                  className="w-4 h-4"
                  fill="none"
                  stroke="currentColor"
                  viewBox="0 0 24 24"
                >
                  <path
                    strokeLinecap="round"
                    strokeLinejoin="round"
                    strokeWidth={2}
                    d="M3 4a1 1 0 011-1h16a1 1 0 011 1v2.586a1 1 0 01-.293.707l-6.414 6.414a1 1 0 00-.293.707V17l-4 4v-6.586a1 1 0 00-.293-.707L3.293 7.207A1 1 0 013 6.5V4z"
                  />
                </svg>
                Filters
              </button>

              {/* Dropdown Panel */}
              {showFilters && (
                <div className="fixed top-20 left-1/2 transform -translate-x-1/2 w-[85vw] max-w-[1200px] bg-white border border-gray-200 rounded-xl shadow-xl z-50">
                  <div
                    className="p-4 overflow-y-auto"
                    style={{ maxHeight: "calc(100vh - 140px)" }}
                  >
                    <div className="flex items-center justify-between mb-6">
                      <h3 className="text-lg font-semibold text-gray-900">
                        {userRole === "tenant"
                          ? "Tenant Filters"
                          : userRole === "buyer"
                          ? "Buyer Filters"
                          : "All Filters"}
                      </h3>

                    </div>

                    {/* Location Section */}
                    <div className="mb-6">
                      <h4 className="text-md font-medium text-gray-800 mb-4">
                        Location
                      </h4>
                      <div className="grid grid-cols-2 gap-6">
                        <div>
                          <label className="block text-sm font-medium text-gray-700 mb-2">
                            City
                          </label>
                          <select
                            value={filters.city}
                            onChange={(e) => {
                              handleFilterChange("city", e.target.value);
                              // Reset locality when city changes
                              setFilters((prev) => ({ ...prev, locality: [] }));
                            }}
                            className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-[#0056D2] focus:border-transparent text-sm"
                          >
                            <option value="">All Cities</option>
                            {Object.keys(citiesData).map((city) => (
                              <option key={city} value={city}>
                                {city}
                              </option>
                            ))}
                          </select>
                        </div>
                        <div>
                          <label className="block text-sm font-medium text-gray-700 mb-2">
                            Locality (Multiple Selection)
                          </label>
                          <div className="max-h-32 overflow-y-auto border border-gray-300 rounded-lg p-2 bg-white">
                            {!filters.city ? (
                              <p className="text-sm text-gray-500 p-2">
                                Select a city first
                              </p>
                            ) : (
                              <div className="space-y-1">
                                {getLocalitiesForCity(filters.city).map(
                                  (locality) => (
                                    <label
                                      key={locality}
                                      className="flex items-center text-sm p-1 hover:bg-gray-50 rounded cursor-pointer"
                                    >
                                      <input
                                        type="checkbox"
                                        checked={
                                          filters.locality?.includes(
                                            locality
                                          ) || false
                                        }
                                        onChange={(e) => {
                                          const newLocalities = e.target.checked
                                            ? [
                                                ...(filters.locality || []),
                                                locality,
                                              ]
                                            : (filters.locality || []).filter(
                                                (l) => l !== locality
                                              );
                                          handleFilterChange(
                                            "locality",
                                            newLocalities
                                          );
                                        }}
                                        className="mr-2 w-4 h-4 rounded text-[#0056D2] focus:ring-[#0056D2]"
                                      />
                                      {locality}
                                    </label>
                                  )
                                )}
                                {filters.locality &&
                                  filters.locality.length > 0 && (
                                    <div className="mt-2 pt-2 border-t border-gray-200">
                                      <button
                                        onClick={() =>
                                          handleFilterChange("locality", [])
                                        }
                                        className="text-xs text-red-600 hover:text-red-800"
                                      >
                                        Clear All ({filters.locality.length}{" "}
                                        selected)
                                      </button>
                                    </div>
                                  )}
                              </div>
                            )}
                          </div>
                        </div>
                      </div>
                    </div>

                    {/* Property Details Section */}
                    <div className="mb-6">
                      <h4 className="text-md font-medium text-gray-800 mb-4">
                        Property Details
                      </h4>
                      <div className="grid grid-cols-4 gap-6">
                        <div>
                          <label className="block text-sm font-medium text-gray-700 mb-2">
                            Property Type
                          </label>
                          <div className="space-y-2">
                            {["apartment", "villa", "house", "commercial"].map(
                              (type) => (
                                <label
                                  key={type}
                                  className="flex items-center text-sm"
                                >
                                  <input
                                    type="checkbox"
                                    checked={filters.property_type.includes(
                                      type
                                    )}
                                    onChange={(e) => {
                                      const newTypes = e.target.checked
                                        ? [...filters.property_type, type]
                                        : filters.property_type.filter(
                                            (t) => t !== type
                                          );
                                      handleFilterChange(
                                        "property_type",
                                        newTypes
                                      );
                                    }}
                                    className="mr-2 w-4 h-4 rounded"
                                  />
                                  {type.charAt(0).toUpperCase() + type.slice(1)}
                                </label>
                              )
                            )}
                          </div>
                        </div>
                        <div>
                          <label className="block text-sm font-medium text-gray-700 mb-2">
                            Listing Type
                          </label>
                          <div className="space-y-2">
                            {[
                              "sell",
                              "rent",
                              "urgent-sale",
                              "bidding",
                              "lease",
                            ].map((type) => (
                              <label
                                key={type}
                                className="flex items-center text-sm"
                              >
                                <input
                                  type="checkbox"
                                  checked={filters.listing_type.includes(type)}
                                  onChange={(e) => {
                                    const newTypes = e.target.checked
                                      ? [...filters.listing_type, type]
                                      : filters.listing_type.filter(
                                          (t) => t !== type
                                        );
                                    handleFilterChange(
                                      "listing_type",
                                      newTypes
                                    );
                                  }}
                                  className="mr-2 w-4 h-4 rounded"
                                />
                                {type === "sell"
                                  ? "For Sale"
                                  : type === "rent"
                                  ? "For Rent"
                                  : type === "urgent-sale"
                                  ? "Urgent Sale"
                                  : type === "bidding"
                                  ? "Bidding/Auction"
                                  : type === "lease"
                                  ? "Lease"
                                  : type}
                              </label>
                            ))}
                          </div>
                        </div>
                        <div>
                          <label className="block text-sm font-medium text-gray-700 mb-2">
                            Bedrooms
                          </label>
                          <div className="space-y-2">
                            {["1", "2", "3", "4"].map((bedroom) => (
                              <label
                                key={bedroom}
                                className="flex items-center text-sm"
                              >
                                <input
                                  type="checkbox"
                                  checked={filters.bedrooms.includes(bedroom)}
                                  onChange={(e) => {
                                    const newBedrooms = e.target.checked
                                      ? [...filters.bedrooms, bedroom]
                                      : filters.bedrooms.filter(
                                          (b) => b !== bedroom
                                        );
                                    handleFilterChange("bedrooms", newBedrooms);
                                  }}
                                  className="mr-2 w-4 h-4 rounded"
                                />
                                {bedroom === "4"
                                  ? "4+ Bedrooms"
                                  : `${bedroom} Bedroom${
                                      bedroom !== "1" ? "s" : ""
                                    }`}
                              </label>
                            ))}
                          </div>
                        </div>
                        <div>
                          <label className="block text-sm font-medium text-gray-700 mb-2">
                            Bathrooms
                          </label>
                          <div className="space-y-2">
                            {["1", "2", "3", "4"].map((bathroom) => (
                              <label
                                key={bathroom}
                                className="flex items-center text-sm"
                              >
                                <input
                                  type="checkbox"
                                  checked={filters.bathrooms.includes(bathroom)}
                                  onChange={(e) => {
                                    const newBathrooms = e.target.checked
                                      ? [...filters.bathrooms, bathroom]
                                      : filters.bathrooms.filter(
                                          (b) => b !== bathroom
                                        );
                                    handleFilterChange(
                                      "bathrooms",
                                      newBathrooms
                                    );
                                  }}
                                  className="mr-2 w-4 h-4 rounded"
                                />
                                {bathroom === "4"
                                  ? "4+ Bathrooms"
                                  : `${bathroom} Bathroom${
                                      bathroom !== "1" ? "s" : ""
                                    }`}
                              </label>
                            ))}
                          </div>
                        </div>
                      </div>
                    </div>

                    {/* Price Section */}
                    <div className="mb-6">
                      <h4 className="text-md font-medium text-gray-800 mb-4">
                        Price Range
                      </h4>

                      {/* Direct Price Input */}
                      <div className="grid grid-cols-2 gap-4 mb-4">
                        <div>
                          <label className="block text-sm font-medium text-gray-700 mb-1">
                            Min {priceRange.label} (₹)
                          </label>
                          <input
                            type="text"
                            placeholder="Enter min price (e.g., 50L, 2Cr)"
                            value={filters.min_price || ""}
                            onChange={(e) => {
                              const value = e.target.value.trim();
                              if (value === "") {
                                handleFilterChange("min_price", "");
                              } else {
                                const parsed = parsePrice(value);
                                if (parsed !== null) {
                                  handleFilterChange(
                                    "min_price",
                                    parsed.toString()
                                  );
                                }
                              }
                            }}
                            className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-[#0056D2] focus:border-transparent text-sm"
                          />
                        </div>
                        <div>
                          <label className="block text-sm font-medium text-gray-700 mb-1">
                            Max {priceRange.label} (₹)
                          </label>
                          <input
                            type="text"
                            placeholder="Enter max price (e.g., 1Cr, 80L)"
                            value={filters.max_price || ""}
                            onChange={(e) => {
                              const value = e.target.value.trim();
                              if (value === "") {
                                handleFilterChange("max_price", "");
                              } else {
                                const parsed = parsePrice(value);
                                if (parsed !== null) {
                                  handleFilterChange(
                                    "max_price",
                                    parsed.toString()
                                  );
                                }
                              }
                            }}
                            className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-[#0056D2] focus:border-transparent text-sm"
                          />
                        </div>
                      </div>

                      <div className="grid grid-cols-2 gap-6">
                        <div>
                          <label className="block text-sm font-medium text-gray-700 mb-2">
                            Min {priceRange.label} (₹)
                          </label>
                          <input
                            type="range"
                            min={priceRange.min}
                            max={filters.max_price || priceRange.max}
                            step={priceRange.step}
                            value={filters.min_price || priceRange.min}
                            onChange={(e) =>
                              handleFilterChange("min_price", e.target.value)
                            }
                            className="w-full h-2 bg-gray-200 rounded-lg appearance-none cursor-pointer slider"
                          />
                          <div className="flex justify-between text-xs text-gray-500 mt-1">
                            <span>₹0</span>
                            <span className="font-medium text-[#0056D2]">
                              {filters.min_price
                                ? `₹${(
                                    parseInt(filters.min_price) / 100000
                                  ).toFixed(0)}L`
                                : "₹0"}
                            </span>
                            <span>
                              {priceRange.max > 1000000
                                ? `₹${(priceRange.max / 10000000).toFixed(0)}Cr`
                                : `₹${(priceRange.max / 1000).toFixed(0)}K`}
                            </span>
                          </div>
                        </div>
                        <div>
                          <label className="block text-sm font-medium text-gray-700 mb-2">
                            Max {priceRange.label} (₹)
                          </label>
                          <input
                            type="range"
                            min={filters.min_price || priceRange.min}
                            max={priceRange.max}
                            step={priceRange.step}
                            value={filters.max_price || priceRange.max}
                            onChange={(e) =>
                              handleFilterChange("max_price", e.target.value)
                            }
                            className="w-full h-2 bg-gray-200 rounded-lg appearance-none cursor-pointer slider"
                          />
                          <div className="flex justify-between text-xs text-gray-500 mt-1">
                            <span>₹0</span>
                            <span className="font-medium text-[#0056D2]">
                              {filters.max_price
                                ? `₹${(
                                    parseInt(filters.max_price) / 100000
                                  ).toFixed(0)}L`
                                : `₹${(priceRange.max / 10000000).toFixed(
                                    0
                                  )}Cr`}
                            </span>
                            <span>
                              {priceRange.max > 1000000
                                ? `₹${(priceRange.max / 10000000).toFixed(0)}Cr`
                                : `₹${(priceRange.max / 1000).toFixed(0)}K`}
                            </span>
                          </div>
                        </div>
                      </div>
                    </div>

                    {/* Tenant-specific Filters */}
                    {userRole === "tenant" && (
                      <div className="mb-6">
                        <h4 className="text-md font-medium text-gray-800 mb-4">
                          Tenant Preferences
                        </h4>
                        <div className="grid grid-cols-2 gap-6">
                          <div>
                            <label className="block text-sm font-medium text-gray-700 mb-2">
                              Rent Range (₹/month)
                            </label>
                            <div className="grid grid-cols-2 gap-2">
                              <input
                                type="text"
                                placeholder="Min rent"
                                value={filters.rent_range_min}
                                onChange={(e) =>
                                  handleFilterChange(
                                    "rent_range_min",
                                    e.target.value
                                  )
                                }
                                className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-[#0056D2] focus:border-transparent text-sm"
                              />
                              <input
                                type="text"
                                placeholder="Max rent"
                                value={filters.rent_range_max}
                                onChange={(e) =>
                                  handleFilterChange(
                                    "rent_range_max",
                                    e.target.value
                                  )
                                }
                                className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-[#0056D2] focus:border-transparent text-sm"
                              />
                            </div>
                          </div>
                          <div>
                            <label className="block text-sm font-medium text-gray-700 mb-2">
                              Deposit Amount (₹)
                            </label>
                            <input
                              type="text"
                              placeholder="Max deposit"
                              value={filters.deposit_amount}
                              onChange={(e) =>
                                handleFilterChange(
                                  "deposit_amount",
                                  e.target.value
                                )
                              }
                              className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-[#0056D2] focus:border-transparent text-sm"
                            />
                          </div>
                          <div>
                            <label className="block text-sm font-medium text-gray-700 mb-2">
                              Maintenance Cost (₹/month)
                            </label>
                            <input
                              type="text"
                              placeholder="Max maintenance"
                              value={filters.maintenance_cost}
                              onChange={(e) =>
                                handleFilterChange(
                                  "maintenance_cost",
                                  e.target.value
                                )
                              }
                              className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-[#0056D2] focus:border-transparent text-sm"
                            />
                          </div>
                          <div>
                            <label className="block text-sm font-medium text-gray-700 mb-2">
                              Tenant Preference
                            </label>
                            <select
                              value={filters.tenant_preference}
                              onChange={(e) =>
                                handleFilterChange(
                                  "tenant_preference",
                                  e.target.value
                                )
                              }
                              className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-[#0056D2] focus:border-transparent text-sm"
                            >
                              <option value="">Any</option>
                              <option value="bachelors">Bachelors Only</option>
                              <option value="family">Family Only</option>
                              <option value="both">Both Allowed</option>
                            </select>
                          </div>
                          <div>
                            <label className="block text-sm font-medium text-gray-700 mb-2">
                              Lease Duration
                            </label>
                            <select
                              value={filters.lease_duration}
                              onChange={(e) =>
                                handleFilterChange(
                                  "lease_duration",
                                  e.target.value
                                )
                              }
                              className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-[#0056D2] focus:border-transparent text-sm"
                            >
                              <option value="">Any Duration</option>
                              <option value="6months">6 Months</option>
                              <option value="11months">11 Months</option>
                              <option value="1year">1 Year</option>
                              <option value="2years">2 Years</option>
                            </select>
                          </div>
                        </div>
                      </div>
                    )}

                    {/* Buyer-specific Filters */}
                    {userRole === "buyer" && (
                      <div className="mb-6">
                        <h4 className="text-md font-medium text-gray-800 mb-4">
                          Buyer Preferences
                        </h4>
                        <div className="grid grid-cols-2 gap-6">
                          <div>
                            <label className="block text-sm font-medium text-gray-700 mb-2">
                              Possession Status
                            </label>
                            <select
                              value={filters.possession_status}
                              onChange={(e) =>
                                handleFilterChange(
                                  "possession_status",
                                  e.target.value
                                )
                              }
                              className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-[#0056D2] focus:border-transparent text-sm"
                            >
                              <option value="">Any Status</option>
                              <option value="ready-to-move">
                                Ready to Move
                              </option>
                              <option value="under-construction">
                                Under Construction
                              </option>
                              <option value="new-launch">New Launch</option>
                            </select>
                          </div>
                          <div>
                            <label className="block text-sm font-medium text-gray-700 mb-2">
                              Possession Date
                            </label>
                            <input
                              type="date"
                              value={filters.possession_date}
                              onChange={(e) =>
                                handleFilterChange(
                                  "possession_date",
                                  e.target.value
                                )
                              }
                              className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-[#0056D2] focus:border-transparent text-sm"
                            />
                          </div>
                          <div>
                            <label className="flex items-center text-sm cursor-pointer">
                              <input
                                type="checkbox"
                                checked={filters.rera_approved === "true"}
                                onChange={(e) =>
                                  handleFilterChange(
                                    "rera_approved",
                                    e.target.checked ? "true" : ""
                                  )
                                }
                                className="mr-2 w-4 h-4 rounded text-[#0056D2] focus:ring-[#0056D2]"
                              />
                              RERA Approved Only
                            </label>
                          </div>
                        </div>
                      </div>
                    )}

                    {/* Verified Listings Toggle - Common for buyers and tenants */}
                    {(userRole === "buyer" || userRole === "tenant") && (
                      <div className="mb-6">
                        <h4 className="text-md font-medium text-gray-800 mb-4">
                          Listing Quality
                        </h4>
                        <div>
                          <label className="flex items-center text-sm cursor-pointer">
                            <input
                              type="checkbox"
                              checked={filters.verified_only}
                              onChange={(e) =>
                                handleFilterChange(
                                  "verified_only",
                                  e.target.checked
                                )
                              }
                              className="mr-2 w-4 h-4 rounded text-[#0056D2] focus:ring-[#0056D2]"
                            />
                            Show verified listings only
                          </label>
                        </div>
                      </div>
                    )}

                    {/* Additional Options Section */}
                    <div className="mb-6">
                      <h4 className="text-md font-medium text-gray-800 mb-4">
                        Additional Options
                      </h4>
                      <div className="grid grid-cols-3 gap-6">
                        <div>
                          <label className="block text-sm font-medium text-gray-700 mb-2">
                            Furnishing
                          </label>
                          <div className="space-y-2">
                            {["furnished", "semi-furnished", "unfurnished"].map(
                              (furnish) => (
                                <label
                                  key={furnish}
                                  className="flex items-center text-sm"
                                >
                                  <input
                                    type="checkbox"
                                    checked={filters.furnishing.includes(
                                      furnish
                                    )}
                                    onChange={(e) => {
                                      const newFurnishing = e.target.checked
                                        ? [...filters.furnishing, furnish]
                                        : filters.furnishing.filter(
                                            (f) => f !== furnish
                                          );
                                      handleFilterChange(
                                        "furnishing",
                                        newFurnishing
                                      );
                                    }}
                                    className="mr-2 w-4 h-4 rounded"
                                  />
                                  {furnish.charAt(0).toUpperCase() +
                                    furnish.slice(1).replace("-", "-")}
                                </label>
                              )
                            )}
                          </div>
                        </div>
                        {/* Conditional Loan Availability - Only show for buyers */}
                        {userRole === "buyer" && isForSale && (
                          <div>
                            <label className="block text-sm font-medium text-gray-700 mb-2">
                              Loan Available
                            </label>
                            <div className="space-y-2">
                              {[
                                { value: "true", label: "Yes" },
                                { value: "false", label: "No" },
                              ].map((option) => (
                                <label
                                  key={option.value}
                                  className="flex items-center text-sm"
                                >
                                  <input
                                    type="radio"
                                    name="loan_available"
                                    value={option.value}
                                    checked={
                                      filters.loan_available === option.value
                                    }
                                    onChange={(e) =>
                                      handleFilterChange(
                                        "loan_available",
                                        e.target.value
                                      )
                                    }
                                    className="mr-2 w-4 h-4 text-[#0056D2] focus:ring-[#0056D2]"
                                  />
                                  {option.label}
                                </label>
                              ))}
                            </div>
                          </div>
                        )}
                        <div>
                          <label className="block text-sm font-medium text-gray-700 mb-2">
                            Sort By
                          </label>
                          <select
                            value={filters.sort}
                            onChange={(e) =>
                              handleFilterChange("sort", e.target.value)
                            }
                            className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-[#0056D2] focus:border-transparent text-sm"
                          >
                            <option value="latest">Latest</option>
                            <option value="price_low">
                              Price: Low to High
                            </option>
                            <option value="price_high">
                              Price: High to Low
                            </option>
                          </select>
                        </div>
                      </div>
                    </div>

                    {/* Amenities Section */}
                    <div className="mb-6">
                      <h4 className="text-md font-medium text-gray-800 mb-4">
                        Amenities & Features
                      </h4>
                      <div className="grid grid-cols-3 gap-4">
                        {[
                          "Lift",
                          "Power Backup",
                          "Parking",
                          "Security",
                          "Gym",
                          "Swimming Pool",
                          "CCTV",
                          "Gated Community",
                          "Water Availability",
                        ].map((amenity) => (
                          <label
                            key={amenity}
                            className="flex items-center text-sm cursor-pointer hover:bg-gray-50 p-1 rounded"
                          >
                            <input
                              type="checkbox"
                              checked={filters.amenities.includes(amenity)}
                              onChange={(e) => {
                                const newAmenities = e.target.checked
                                  ? [...filters.amenities, amenity]
                                  : filters.amenities.filter(
                                      (a) => a !== amenity
                                    );
                                handleFilterChange("amenities", newAmenities);
                              }}
                              className="mr-2 w-4 h-4 rounded text-[#0056D2] focus:ring-[#0056D2]"
                            />
                            <span
                              className={
                                amenity === "Gated Community" ||
                                amenity === "Water Availability"
                                  ? "font-medium"
                                  : ""
                              }
                            >
                              {amenity}
                            </span>
                          </label>
                        ))}
                      </div>
                    </div>

                    {/* Action Buttons */}
                    <div className="flex gap-3 pt-4 border-t border-gray-200">
                      <button
                        onClick={() => setShowFilters(false)}
                        className="px-4 py-2 border border-gray-300 rounded-lg hover:bg-gray-50 transition-colors text-sm font-medium"
                      >
                        Cancel
                      </button>
                      <button
                        onClick={clearFilters}
                        className="flex-1 px-4 py-2 border border-gray-300 rounded-lg hover:bg-gray-50 transition-colors text-sm font-medium"
                      >
                        Clear All
                      </button>
                      <button
                        onClick={async () => {
                          // Store search history on explicit Apply Filters click
                          await storeSearch(filters.search.trim(), filters);
                          loadPropertiesWithoutHistory();
                          setShowFilters(false);
                          
                          // Track filter apply event
                          const { behaviorTracker } = await import("../utils/behaviorTracker");
                          behaviorTracker.trackFilterApply({
                            filters: {
                              city: filters.city,
                              property_type: filters.property_type,
                              listing_type: filters.listing_type,
                              bedrooms: filters.bedrooms,
                              min_price: filters.min_price,
                              max_price: filters.max_price,
                            }
                          });
                        }}
                        className="flex-1 bg-[#0056D2] text-white px-4 py-2 rounded-lg hover:bg-[#0056D2]/90 transition-colors text-sm font-medium"
                      >
                        Apply Filters
                      </button>
                    </div>
                  </div>
                </div>
              )}
            </div>
          </div>
        </div>
      </div>

      {/* Location Permission Modal */}
      <LocationPermissionModal
        isOpen={showLocationModal}
        onAllow={handleLocationAllow}
        onCancel={handleLocationCancel}
      />

      {/* Main Content - Fixed Map + Scrollable Properties */}
      <div
        className="flex flex-col md:flex-row"
        style={{ height: "calc(100vh - 80px)" }}
      >
        {/* LEFT SIDE - Fixed Google Map (Zillow-style) */}
        <div className="w-full md:w-1/2 h-[40vh] md:h-[600px] md:max-h-[600px] md:min-h-[600px]">
          <div className="w-full h-full md:fixed md:w-1/2 md:h-[600px] md:max-h-[600px] md:min-h-[600px] md:top-[80px] relative">
            <div className="bg-white overflow-hidden h-full">
              <div className="absolute top-4 left-4 z-20 bg-white/90 backdrop-blur-sm px-3 py-2 rounded-lg shadow-md">
                <h3 className="text-sm font-semibold text-gray-800">
                  Property Locations
                </h3>
              </div>
              <div className="h-full">
                <GoogleMap
                  properties={filteredProperties.map((p) => ({
                    id: p.id,
                    title: p.title,
                    location: p.location,
                    price: p.price,
                    latitude: p.latitude,
                    longitude: p.longitude,
                    imageUrl: p.imageUrl || undefined,
                    listing_type: p.listing_type,
                  }))}
                  center={mapCenter || currentLocation}
                  onPropertySelect={(property) => {
                    console.log("Selected property:", property);
                  }}
                />
              </div>
            </div>
          </div>
        </div>

        {/* RIGHT SIDE - Scrollable Property Listings */}
        <div className="w-full md:w-1/2 md:ml-[50%] flex flex-col overflow-hidden bg-white">
          {/* Recommendations Section */}
          {recommendations.length > 0 && (
            <div className="p-4 pb-2 border-b bg-blue-50 flex-shrink-0">
              <h3 className="text-md font-semibold text-gray-900 mb-3">
                Recommended for You
              </h3>
              <div className="flex gap-3 overflow-x-auto pb-2">
                {recommendations.map((property) => (
                  <div
                    key={property.id}
                    className="flex-shrink-0 w-48 bg-white rounded-lg border p-3 cursor-pointer hover:shadow-md transition-shadow"
                    onClick={() => navigate(`/property/${property.id}`)}
                  >
                    <div className="w-full h-24 bg-gray-200 rounded mb-2 overflow-hidden">
                      {property.imageUrl ? (
                        <img
                          src={property.imageUrl}
                          alt={property.title}
                          className="w-full h-full object-cover"
                        />
                      ) : (
                        <div className="w-full h-full flex items-center justify-center text-gray-400 text-xs">
                          No Image
                        </div>
                      )}
                    </div>
                    <h4 className="text-sm font-medium text-gray-900 truncate">
                      {property.title}
                    </h4>
                    <p className="text-xs text-gray-600 truncate">
                      {property.location}
                    </p>
                    <p className="text-sm font-semibold text-[#0056D2] mt-1">
                      {property.price}
                    </p>
                  </div>
                ))}
              </div>
            </div>
          )}

          {/* Header Section */}
          <div className="p-4 pb-2 flex-shrink-0">
            <div className="flex items-center justify-between mb-2">
              <div className="flex items-center gap-4">
                <button
                  onClick={() => {
                    console.log("🔍 All Properties tab clicked");
                    setViewMode("all");
                  }}
                  className={`px-4 py-2 rounded-lg font-medium text-sm transition-colors ${
                    viewMode === "all"
                      ? "bg-[#0056D2] text-white"
                      : "bg-gray-100 text-gray-700 hover:bg-gray-200"
                  }`}
                >
                  All Properties
                </button>
                {isAuthenticated() && (
                  <button
                    onClick={() => {
                      console.log("🔍 Recently Viewed tab clicked");
                      setViewMode("recent");
                      // Reload recent views when switching to recent tab
                      const user = JSON.parse(
                        localStorage.getItem("user") || "{}"
                      );
                      // Try multiple possible user ID fields
                      const userId = user.user_id || user.id || user.uuid || localStorage.getItem("user_id");
                      console.log("🔍 Tab click - user data:", {
                        user,
                        userId,
                        userKeys: Object.keys(user),
                        localStorage: {
                          user: localStorage.getItem("user"),
                          token: localStorage.getItem("token"),
                          access_token: localStorage.getItem("access_token"),
                        },
                      });
                      if (
                        userId &&
                        userId !== "anonymous" &&
                        !userId.startsWith("guest_")
                      ) {
                        console.log(
                          "🔍 Tab click - loading recently viewed for:",
                          userId
                        );
                        console.log(
                          "🔍 Tab click - current recent views count:",
                          recentlyViewedProperties.length
                        );
                        loadRecentlyViewedProperties(userId);
                      } else {
                        console.log("❌ Tab click - invalid user ID:", userId);
                        console.log("❌ Tab click - user validation failed:", {
                          userId,
                          isAnonymous: userId === "anonymous",
                          startsWithGuest: userId?.startsWith("guest_"),
                          isFalsy: !userId,
                        });
                      }
                    }}
                    className={`px-4 py-2 rounded-lg font-medium text-sm transition-colors ${
                      viewMode === "recent"
                        ? "bg-[#0056D2] text-white"
                        : "bg-gray-100 text-gray-700 hover:bg-gray-200"
                    }`}
                  >
                    Recently Viewed ({recentlyViewedProperties.length})
                  </button>
                )}
              </div>
            </div>
            <h2 className="text-lg font-semibold text-gray-900">
              {viewMode === "recent" ? (
                <>
                  Recently Viewed Properties
                  <span className="text-gray-500 font-normal text-sm ml-2">
                    ({recentlyViewedProperties.length} properties)
                  </span>
                </>
              ) : filters.search.trim() ? (
                <>
                  Search results for "{filters.search.trim()}"
                  <span className="text-gray-500 font-normal text-sm ml-2">
                    ({filteredProperties.length}{" "}
                    {filteredProperties.length === 1
                      ? "property"
                      : "properties"}{" "}
                    found)
                  </span>
                </>
              ) : currentLocation ? (
                <>
                  Properties near {currentLocation.name}
                  <span className="text-gray-500 font-normal text-sm ml-2">
                    ({filteredProperties.length} properties)
                  </span>
                </>
              ) : (
                <>
                  All Properties
                  <span className="text-gray-500 font-normal text-sm ml-2">
                    ({filteredProperties.length} properties)
                  </span>
                </>
              )}
            </h2>
            {currentLocation && !filters.search.trim() && (
              <div className="flex items-center gap-1 mt-1">
                <MapPin className="w-4 h-4 text-blue-600" />
                <p className="text-sm text-blue-600">
                  Showing properties within 10km of your location
                </p>
              </div>
            )}
            {filters.search.trim() && (
              <div className="flex items-center gap-1 mt-1">
                <svg
                  className="w-4 h-4 text-green-600"
                  fill="none"
                  stroke="currentColor"
                  viewBox="0 0 24 24"
                >
                  <path
                    strokeLinecap="round"
                    strokeLinejoin="round"
                    strokeWidth={2}
                    d="M21 21l-6-6m2-5a7 7 0 11-14 0 7 7 0 0114 0z"
                  />
                </svg>
                <p className="text-sm text-green-600">
                  {filteredProperties.length > 0
                    ? `Found ${filteredProperties.length} matching ${
                        filteredProperties.length === 1
                          ? "property"
                          : "properties"
                      }`
                    : "No properties match your search. Try different keywords or check spelling."}
                </p>
              </div>
            )}
          </div>

          {/* Properties Grid Container - Fixed Structure */}
          <div className="flex-1 overflow-y-auto px-4 pb-4 relative">
            {viewMode === "recent" && isAuthenticated() ? (
              loadingRecentViews ? (
                <div className="flex items-center justify-center h-full">
                  <div className="text-center">
                    <div className="animate-spin rounded-full h-8 w-8 border-b-2 border-[#0056D2] mx-auto"></div>
                    <p className="text-gray-500 mt-2 text-sm">
                      Loading recently viewed properties...
                    </p>
                  </div>
                </div>
              ) : recentlyViewedProperties.length > 0 ? (
                <div className="grid gap-4 grid-cols-1 lg:grid-cols-2">
                  {recentlyViewedProperties.map((property) => (
                    <PropertyCard
                      key={property.id}
                      id={property.id}
                      title={property.title}
                      location={property.location}
                      beds={property.beds}
                      baths={property.baths}
                      area={property.area}
                      price={property.price}
                      imageUrl={property.imageUrl || undefined}
                      isUrgent={property.urgent}
                      onFavorite={handleFavorite}
                      isFavorited={clickedHearts.has(property.id)}
                      priceNegotiable={property.priceNegotiable}
                      loanAvailable={property.loanAvailable}
                      propertyData={{
                        id: property.id.toString(),
                        title: property.title,
                        location: property.location,
                        beds: typeof property.beds === 'string' ? parseInt(property.beds) || 0 : property.beds,
                        baths: typeof property.baths === 'string' ? parseInt(property.baths) || 0 : property.baths,
                        area: property.area,
                        price: property.price,
                        image: property.imageUrl || '',
                        images: property.imageUrl ? [property.imageUrl] : [],
                        verified: true,
                        riblScore: 'A+',
                        description: '',
                        amenities: [],
                        nearbyPlaces: [],
                        priceHistory: [],
                        agent: {
                          name: '',
                          rating: 0,
                          experience: '',
                          phone: '',
                          image: ''
                        },
                        specifications: {
                          propertyType: property.property_type || 'apartment',
                          facing: '',
                          floor: '',
                          totalFloors: '',
                          age: '',
                          furnishing: '',
                          parking: 0
                        },
                        legalInfo: {
                          approvals: [],
                          rera: '',
                          ownership: ''
                        }
                      }}
                    />
                  ))}
                </div>
              ) : (
                <div className="absolute inset-0 flex items-center justify-center">
                  <div className="text-center max-w-md mx-auto px-4">
                    <svg
                      className="w-16 h-16 text-gray-300 mx-auto mb-4"
                      fill="none"
                      stroke="currentColor"
                      viewBox="0 0 24 24"
                    >
                      <path
                        strokeLinecap="round"
                        strokeLinejoin="round"
                        strokeWidth="1"
                        d="M12 8v4l3 3m6-3a9 9 0 11-18 0 9 9 0 0118 0z"
                      />
                    </svg>
                    <h3 className="text-lg font-medium text-gray-900 mb-2">
                      No Recently Viewed Properties
                    </h3>
                    <p className="text-gray-500 mb-4">
                      Properties you view will appear here for quick access.
                    </p>
                    <button
                      onClick={() => {
                        console.log(
                          "🔍 Browse All Properties clicked from empty state"
                        );
                        setViewMode("all");
                      }}
                      className="bg-[#0056D2] text-white px-4 py-2 rounded-lg hover:bg-[#0056D2]/90 transition-colors text-sm font-medium"
                    >
                      Browse All Properties
                    </button>
                  </div>
                </div>
              )
            ) : loading ? (
              <div className="flex items-center justify-center h-full">
                <div className="text-center">
                  <div className="animate-spin rounded-full h-8 w-8 border-b-2 border-[#0056D2] mx-auto"></div>
                  <p className="text-gray-500 mt-2 text-sm">
                    Loading properties...
                  </p>
                </div>
              </div>
            ) : filteredProperties.length > 0 ? (
              <div className="grid gap-4 grid-cols-1 lg:grid-cols-2">
                {filteredProperties.map((property) => (
                  <PropertyCard
                    key={property.id}
                    id={property.id}
                    title={property.title}
                    location={property.location}
                    beds={property.beds}
                    baths={property.baths}
                    area={property.area}
                    price={property.price}
                    imageUrl={property.imageUrl || undefined}
                    isUrgent={property.urgent}
                    onFavorite={handleFavorite}
                    isFavorited={clickedHearts.has(property.id)}
                    priceNegotiable={property.priceNegotiable}
                    loanAvailable={property.loanAvailable}
                    propertyData={{
                      id: property.id.toString(),
                      title: property.title,
                      location: property.location,
                      beds: typeof property.beds === 'string' ? parseInt(property.beds) || 0 : property.beds,
                      baths: typeof property.baths === 'string' ? parseInt(property.baths) || 0 : property.baths,
                      area: property.area,
                      price: property.price,
                      image: property.imageUrl || '',
                      images: property.imageUrl ? [property.imageUrl] : [],
                      verified: true,
                      riblScore: 'A+',
                      description: '',
                      amenities: [],
                      nearbyPlaces: [],
                      priceHistory: [],
                      agent: {
                        name: '',
                        rating: 0,
                        experience: '',
                        phone: '',
                        image: ''
                      },
                      specifications: {
                        propertyType: property.property_type || 'apartment',
                        facing: '',
                        floor: '',
                        totalFloors: '',
                        age: '',
                        furnishing: '',
                        parking: 0
                      },
                      legalInfo: {
                        approvals: [],
                        rera: '',
                        ownership: ''
                      }
                    }}
                  />
                ))}
              </div>
            ) : (
              <div className="absolute inset-0 flex items-center justify-center">
                <div className="text-center max-w-md mx-auto px-4">
                  {filters.search.trim() ? (
                    <>
                      <svg
                        className="w-16 h-16 text-gray-300 mx-auto mb-4"
                        fill="none"
                        stroke="currentColor"
                        viewBox="0 0 24 24"
                      >
                        <path
                          strokeLinecap="round"
                          strokeLinejoin="round"
                          strokeWidth="1"
                          d="M21 21l-6-6m2-5a7 7 0 11-14 0 7 7 0 0114 0z"
                        />
                      </svg>
                      <h3 className="text-lg font-medium text-gray-900 mb-2">
                        No properties found for "{filters.search.trim()}"
                      </h3>
                      <p className="text-gray-500 mb-4">
                        We couldn't find any properties matching your search.
                        Try:
                      </p>
                      <ul className="text-sm text-gray-400 text-left space-y-1 mb-4">
                        <li>• Checking your spelling</li>
                        <li>• Using different keywords</li>
                        <li>• Searching for nearby areas</li>
                        <li>• Removing some filters</li>
                      </ul>
                      <button
                        onClick={() => {
                          setFilters((prev) => ({ ...prev, search: "" }));
                          loadPropertiesWithoutHistory();
                        }}
                        className="bg-[#0056D2] text-white px-4 py-2 rounded-lg hover:bg-[#0056D2]/90 transition-colors text-sm font-medium"
                      >
                        Clear Search & Show All Properties
                      </button>
                    </>
                  ) : (
                    <>
                      <svg
                        className="w-12 h-12 text-gray-300 mx-auto mb-3"
                        fill="none"
                        stroke="currentColor"
                        viewBox="0 0 24 24"
                      >
                        <path
                          strokeLinecap="round"
                          strokeLinejoin="round"
                          strokeWidth="1"
                          d="M19 21V5a2 2 0 00-2-2H7a2 2 0 00-2 2v16m14 0h2m-2 0h-5m-9 0H3m2 0h5M9 7h1m-1 4h1m4-4h1m-1 4h1m-5 10v-5a1 1 0 011-1h2a1 1 0 011 1v5m-4 0h4"
                        />
                      </svg>
                      <p className="text-gray-500">No properties found</p>
                      <p className="text-gray-400 text-sm">
                        Try adjusting your filters
                      </p>
                    </>
                  )}
                </div>
              </div>
            )}
          </div>
        </div>
      </div>
    </div>
  );
}

