import React, { useState, useEffect } from "react";
import { useNavigate } from "react-router-dom";
import { motion } from "framer-motion";
import { MapPin, Bed, Bath, Maximize, CheckCircle, Star, Clock, Gavel, Home } from "lucide-react";
import { SavePropertyButton } from "./SavePropertyButton";
import { WishlistHeartButton } from "./WishlistHeartButton";
import { useAzureBlobMedia } from '../hooks/useAzureBlobMedia';

import { getPropertyMedia, generateDownloadURL } from "../api/mediaUpload";
import { getRandomPropertyImages } from "../utils/sampleImages";
import { calculateListingScore, getScoreColor, getScoreBgColor } from "../utils/listingScore";
import { Property } from "../data/properties";


interface PropertyCardProps {
  id: number | string;
  title: string;
  location: string;
  beds: string | number;
  baths: string | number;
  area: string;
  price: string;
  imageUrl?: string;
  isUrgent?: boolean;
  isBidding?: boolean;
  urgentSaleEndDate?: string;
  urgentSaleEndTime?: string;
  auctionEndDate?: string;
  auctionEndTime?: string;
  onFavorite?: (id: number) => void;
  isFavorited?: boolean;
  views?: number;
  showCompareButton?: boolean;
  status?: string;
  qualityScore?: number;
  autoTags?: string[];
  amenities?: string[];
  priceNegotiable?: boolean;
  loanAvailable?: boolean;
  isCompleted?: boolean;
  winnerName?: string;
  finalBidAmount?: number;
  // Add property data for score calculation
  propertyData?: Property;
  description?: string;
  images?: string[];
  specifications?: any;
  legalInfo?: any;
}

function formatTimeLeft(timeLeftMs: number) {
  if (timeLeftMs <= 0) return "Expired";

  const totalSeconds = Math.floor(timeLeftMs / 1000);
  const days = Math.floor(totalSeconds / 86400);
  const hours = Math.floor((totalSeconds % 86400) / 3600);
  const minutes = Math.floor((totalSeconds % 3600) / 60);
  const seconds = totalSeconds % 60;

  if (days > 0) return `${days}d ${hours}h ${minutes}m`;
  if (hours > 0) return `${hours}h ${minutes}m ${seconds}s`;
  if (minutes > 0) return `${minutes}m ${seconds}s`;
  return `${seconds}s`;
}

export const PropertyCard: React.FC<PropertyCardProps> = ({
  id,
  title,
  location,
  beds,
  baths,
  area,
  price,
  imageUrl,
  isUrgent = false,
  isBidding = false,
  urgentSaleEndDate,
  urgentSaleEndTime,
  auctionEndDate,
  auctionEndTime,
  onFavorite,
  isFavorited = false,
  views,
  showCompareButton = false,
  qualityScore = 0,
  autoTags = [],
  amenities = [],
  priceNegotiable = false,
  loanAvailable = false,
  isCompleted = false,
  winnerName,
  finalBidAmount,
  propertyData,
  description,
  images,
  specifications,
  legalInfo,
}) => {
  const navigate = useNavigate();
  const [timeLeft, setTimeLeft] = useState<string>("");
  
  // Calculate dynamic listing score
  const listingScore = React.useMemo(() => {
    if (propertyData) {
      return calculateListingScore(propertyData);
    }
    // Fallback calculation using available props
    const mockProperty: Partial<Property> = {
      id: id.toString(),
      title,
      location,
      beds: typeof beds === 'string' ? parseInt(beds) : beds,
      baths: typeof baths === 'string' ? parseInt(baths) : baths,
      area,
      price,
      description: description || '',
      amenities: amenities || [],
      images: images || (imageUrl ? [imageUrl] : []),
      specifications,
      legalInfo
    };
    return calculateListingScore(mockProperty as Property);
  }, [propertyData, id, title, location, beds, baths, area, price, description, amenities, images, imageUrl, specifications, legalInfo]);
  
  // Use Azure Blob media integration (same as PropertyDetail)
  const {
    images: azureImages,
    loading: mediaLoading,
    hasMedia
  } = useAzureBlobMedia(id.toString());

  console.log(`PropertyCard ${id} - Azure images:`, azureImages);
  console.log(`PropertyCard ${id} - Azure images length:`, azureImages?.length);

  // Get the first available image from Azure Blob or fallback to imageUrl
  const propertyImage = (() => {
    if (azureImages && azureImages.length > 0) {
      console.log(`PropertyCard ${id} - Using Azure image URL:`, azureImages[0].url);
      return azureImages[0].url;
    }
    console.log(`PropertyCard ${id} - Using fallback image`);
    return imageUrl || "https://images.unsplash.com/photo-1586023492125-27b2c045efd7?ixlib=rb-4.0.3&auto=format&fit=crop&w=400&h=250";
  })();

  useEffect(() => {
    if (
      (isUrgent && urgentSaleEndDate && urgentSaleEndTime) ||
      (isBidding && auctionEndDate && auctionEndTime)
    ) {
      const interval = setInterval(() => {
        let endDateTime: Date;
        let dateString: string;

        if (isUrgent && urgentSaleEndDate && urgentSaleEndTime) {
          dateString = `${urgentSaleEndDate}T${urgentSaleEndTime}`;
          endDateTime = new Date(dateString);
        } else if (isBidding && auctionEndDate && auctionEndTime) {
          dateString = `${auctionEndDate}T${auctionEndTime}`;
          endDateTime = new Date(dateString);
        } else {
          return;
        }

        const now = new Date();
        const diff = endDateTime.getTime() - now.getTime();

        if (diff > 0) {
          setTimeLeft(formatTimeLeft(diff));
        } else {
          setTimeLeft("Expired");
        }
      }, 1000);

      return () => clearInterval(interval);
    }
  }, [
    isUrgent,
    isBidding,
    urgentSaleEndDate,
    urgentSaleEndTime,
    auctionEndDate,
    auctionEndTime,
  ]);

  const handleViewMore = () => {
    // Track property click
    import("../utils/behaviorTracker").then(({ behaviorTracker }) => {
      behaviorTracker.trackPropertyClick(typeof id === 'string' ? parseInt(id) : id, {
        title,
        location,
        price
      });
    });
    navigate(isBidding ? `/bidding-property/${id}` : `/property/${id}`);
  };

  const handleFavorite = (e: React.MouseEvent) => {
    e.stopPropagation();
    if (onFavorite) onFavorite(typeof id === 'string' ? parseInt(id) : id);
  };

  return (
    <motion.div
      whileHover={{ scale: 1.02 }}
      className="bg-white rounded-lg shadow-md overflow-hidden relative border border-gray-200 hover:shadow-lg transition-all duration-300 cursor-pointer w-full max-w-sm mx-auto min-h-[400px] flex flex-col"
      onClick={handleViewMore}
    >
      {/* === Image Section === */}
      <div className="relative h-48 w-full overflow-hidden">
        <img 
          src={propertyImage} 
          alt={title} 
          className="w-full h-full object-cover transition-transform duration-500 group-hover:scale-105"
          onError={(e) => {
            console.error('PropertyCard image failed to load:', {
              propertyId: id,
              imageUrl: propertyImage,
              azureImagesCount: azureImages?.length || 0,
              hasAzureImages: !!(azureImages && azureImages.length > 0),
              fallbackUrl: imageUrl
            });
            // Try fallback image
            const fallbackUrl = "https://images.unsplash.com/photo-1586023492125-27b2c045efd7?ixlib=rb-4.0.3&auto=format&fit=crop&w=400&h=250";
            if (e.currentTarget.src !== fallbackUrl) {
              e.currentTarget.src = fallbackUrl;
            } else {
              // Even fallback failed, show placeholder
              e.currentTarget.style.display = 'none';
              const placeholder = document.createElement('div');
              placeholder.className = 'w-full h-full bg-gradient-to-br from-gray-50 to-gray-100 flex items-center justify-center';
              placeholder.innerHTML = `
                <div class="text-center">
                  <svg class="w-16 h-16 text-gray-300 mx-auto mb-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="1.5" d="M4 16l4.586-4.586a2 2 0 012.828 0L16 16m-2-2l1.586-1.586a2 2 0 012.828 0L20 14" />
                  </svg>
                  <p class="text-gray-400 text-sm font-medium">No image available</p>
                </div>
              `;
              e.currentTarget.parentNode?.appendChild(placeholder);
            }
          }}
          onLoad={() => {
            console.log('PropertyCard image loaded successfully:', {
              propertyId: id,
              imageUrl: propertyImage
            });
          }}
        />

        {/* === Top Left Badges (Side by Side) === */}
        <div className="absolute top-3 left-3 flex gap-1">
          <span className="bg-green-500 text-white text-xs font-bold px-2 py-1 rounded flex items-center gap-1">
            <CheckCircle className="w-3 h-3" />
            Verified
          </span>
          {isUrgent ? (
            <span className="bg-red-500 text-white text-xs font-bold px-2 py-1 rounded flex items-center gap-1">
              <Clock className="w-3 h-3" />
              Urgent Sale
            </span>
          ) : isBidding ? (
            <span className="bg-yellow-500 text-white text-xs font-bold px-2 py-1 rounded flex items-center gap-1">
              <Gavel className="w-3 h-3" />
              Auction
            </span>
          ) : (
            <span className="bg-orange-500 text-white text-xs font-bold px-2 py-1 rounded flex items-center gap-1">
              <Star className="w-3 h-3" />
              Featured
            </span>
          )}
        </div>

        {/* === Bottom Left Price and Heart (Same Line) === */}
        <div className="absolute bottom-3 left-3 right-3 flex items-center justify-between">
          <div className="flex items-center gap-2">
            <div className="bg-black/70 text-white px-2 py-1 rounded text-lg font-bold">
              {price}
            </div>
            <span className="bg-blue-500 text-white text-xs font-bold px-2 py-1 rounded">
              {priceNegotiable ? 'Negotiable' : 'Fixed'}
            </span>
          </div>
          <WishlistHeartButton propertyId={typeof id === 'string' ? parseInt(id) : id} />
        </div>
      </div>

      {/* === Details Section === */}
      <div className="p-4 flex-1 flex flex-col justify-between">
        <div>
          {/* Title */}
          <h3 className="text-lg font-bold text-gray-900 mb-2 line-clamp-1">
            {title}
          </h3>
          
          {/* Location */}
          <p className="text-gray-600 text-sm mb-3 flex items-center">
            <MapPin className="w-4 h-4 mr-1 text-gray-400" />
            {location}
          </p>

          {/* Property Details Row */}
          <div className="flex items-center justify-between text-sm text-gray-600 mb-4">
            <div className="flex items-center gap-1">
              <Home className="w-4 h-4 text-blue-600" />
              <span>{beds} Beds</span>
            </div>
            <div className="flex items-center gap-1">
              <Bath className="w-4 h-4 text-blue-600" />
              <span>{baths} Baths</span>
            </div>
            <div className="flex items-center gap-1">
              <Maximize className="w-4 h-4 text-blue-600" />
              <span>{area}</span>
            </div>
          </div>
        </div>

        {/* View Details Button */}
        <button
          onClick={(e) => {
            e.stopPropagation();
            handleViewMore();
          }}
          className="w-full py-2 bg-blue-600 text-white rounded-lg font-medium hover:bg-blue-700 transition-colors duration-200 mt-auto relative z-20 shadow-sm border border-blue-600 text-sm"
          style={{ 
            backgroundColor: '#2563eb',
            borderColor: '#2563eb',
            color: '#ffffff',
            display: 'block',
            visibility: 'visible',
            opacity: 1
          }}
        >
          View Details
        </button>
      </div>
    </motion.div>
  );
};
