import React, { useState, useEffect } from "react";
import { useParams, useNavigate } from "react-router-dom";
import { Shield, Waves } from "lucide-react";
import { toast } from "sonner";
import { WishlistButton } from "./WishlistButton";
import { WishlistHeartButton } from "./WishlistHeartButton";
import { SavedPropertiesButton } from "./SavedPropertiesButton";
import { ShareButton } from "./ShareButton";
import PropertyDetailsDocuments from "./PropertyDetailsDocuments";
import { AzureBlobMediaGallery } from "./AzureBlobMediaGallery";
import { useAzureBlobMedia } from "../hooks/useAzureBlobMedia";
import { useBehaviorTracker } from "../utils/behaviorTracker";
import { trackPropertyView } from "../utils/propertyViewTracker";
import AirQualityScore from "./AirQualityScore";
import { CrimeHeatmap } from "./CrimeHeatmap";
import { FloodRiskCard } from "./FloodRiskCard";
import { Activity, AlertTriangle, TrendingUp, Clock } from "lucide-react";
// import { getPropertyImageByType } from '../utils/sampleImages';
import { fetchNearbySchools } from "../services/schoolsService";
import { SchoolsGoogleMap } from "./SchoolsGoogleMap";
import { getPropertyImageByType } from "../utils/sampleImages";
import { HyperlocalInsights } from "./HyperlocalInsights";
import { PaymentModal } from "./transactions/PaymentModal";
import { TransactionType } from "../types/transaction";
import { PropertyLegalDocuments } from "./legal/PropertyLegalDocuments";
import { AuctionRegistrationModal } from "./AuctionRegistrationModal";
import { useVoiceAssistant } from "../hooks/useVoiceAssistant";
import { PropertyAssistant } from "../utils/propertyAssistant";
import { Character3D } from "./Character3D";
import "./VoiceAssistant.css";
import { getRecommendedProperties } from "../api/properties";
import { BuyerMarketSnapshot } from "./buyer-dashboard/BuyerMarketSnapshot";
import { FacilitiesMap } from "./geo_demographic";

interface Property {
  id: number;
  // Personal Details
  owner_name: string;
  company_name?: string;
  mobile: string;
  email: string;
  user_id?: number; // Add missing user_id property
  owner_id?: number; // Add missing owner_id property

  // Property Info
  title: string;
  description?: string;
  category?: string;
  price?: number;
  specifications?: string[];

  // Listing & Property Type
  listing_type: "sell" | "rent" | "urgent-sale" | "bidding";
  property_type: "apartment" | "villa" | "house" | "commercial" | "pg-coliving";

  // Property Details
  built_up_area?: number;
  carpet_area?: number;
  plot_area?: number;
  super_area?: number;
  bedrooms?: number;
  bathrooms?: number;
  balconies?: number;
  floor_number?: string;
  total_floors?: string;
  facing: string;
  furnishing: string;
  property_age: string;
  lease_terms?: string;
  common_facilities?: string[];

  // Pricing
  sale_price?: string;
  price_per_sqft?: number;
  monthly_rent?: string;
  security_deposit?: string;
  price_negotiable: boolean;
  loan_availability: boolean;
  possession_status: string;

  // Bidding specific
  starting_bidding_price?: number;
  minimum_increment?: number;
  reserve_price?: number;
  auction_start_date?: string;
  auction_start_time?: string;
  auction_end_date?: string;
  auction_end_time?: string;
  timezone?: string;
  urgent_sale_start_date?: string;
  urgent_sale_start_time?: string;
  urgent_sale_end_date?: string;
  urgent_sale_end_time?: string;
  actual_price?: number;
  deposit_requirement?: number;
  bidder_registration_fee?: number;
  auction_eligibility?: string;

  // Backend Analytics & Activity
  views?: number;
  views_today?: number;
  inquiries?: number;
  active_bidders?: number;
  bidding_history?: {
    bidder_id: string;
    bidder_name?: string;
    bid_amount: number;
    bid_time: string;
  }[];
  urgency_reason?: string;
  urgency_reasons?: string[];
  call_to_action_title?: string;
  call_to_action_message?: string;
  savings_amount?: number;
  discount_percentage?: number;

  // Location
  city: string;
  locality: string;
  address: string;
  landmark?: string;
  pincode?: string;
  latitude?: number;
  longitude?: number;
  nearby_places?: string[];

  // Features
  amenities: string[];
  features: string[];
  highlights: string[];

  // Media
  images: string[];
  image_urls?: string[];
  primary_image?: string;
  virtual_tour_url?: string;
  videos?: string[];
  house_plan?: string;
  three_d_plan?: string;
  three_sixty_view?: string;
  virtual_tours?: string[];

  // Documents
  documents?: {
    name: string;
    file_url: string;
    file_size?: string;
    file_type?: string;
    document_type?: string;
  }[];

  // Legal Compliance
  rera_number?: string;
  rera_state?: string;
  rera_verified?: boolean;

  // Contact & Verification
  contact_type: "owner" | "agent";
  kyc_verified: boolean;
  phone_verified: boolean;
  email_verified: boolean;

  // System fields
  photo: string;
  state: string;
  created_at: string;
}

// Similar Property Card Component with Azure Blob integration
function SimilarPropertyCard({ property }: { property: any }) {
  const navigate = useNavigate();
  const { images: azureImages } = useAzureBlobMedia(property.id.toString());
  
  // Get the first available image from Azure Blob or fallback
  const propertyImage = (() => {
    if (azureImages && azureImages.length > 0) {
      return azureImages[0].url;
    }
    return property.primary_image || getPropertyImageByType(property.property_type || 'apartment');
  })();

  const formatPrice = (price: string | number) => {
    const numPrice = typeof price === "string" ? parseFloat(price) : price;
    if (isNaN(numPrice) || numPrice === 0) return "Price on request";
    if (numPrice >= 10000000) return `₹${(numPrice / 10000000).toFixed(1)} Cr`;
    if (numPrice >= 100000) return `₹${(numPrice / 100000).toFixed(0)} L`;
    return `₹${numPrice.toLocaleString()}`;
  };

  return (
    <div 
      className="bg-white rounded-lg shadow-md overflow-hidden hover:shadow-lg transition-all duration-300 cursor-pointer border border-gray-200 max-w-sm mx-auto min-h-[400px] flex flex-col"
      onClick={() => navigate(`/property/${property.id}`)}
    >
      <div className="relative h-48 w-full overflow-hidden">
        <img 
          src={propertyImage}
          alt={property.title}
          className="w-full h-full object-cover transition-transform duration-500 hover:scale-105"
          onError={(e) => {
            const fallbackUrl = getPropertyImageByType(property.property_type || 'apartment');
            if (e.currentTarget.src !== fallbackUrl) {
              e.currentTarget.src = fallbackUrl;
            }
          }}
        />
        
        {/* Top Left Badges */}
        <div className="absolute top-3 left-3 flex gap-1">
          <span className="bg-green-500 text-white text-xs font-bold px-2 py-1 rounded flex items-center gap-1">
            <svg className="w-3 h-3" fill="currentColor" viewBox="0 0 20 20">
              <path fillRule="evenodd" d="M16.707 5.293a1 1 0 010 1.414l-8 8a1 1 0 01-1.414 0l-4-4a1 1 0 011.414-1.414L8 12.586l7.293-7.293a1 1 0 011.414 0z" clipRule="evenodd" />
            </svg>
            Verified
          </span>
          <span className="bg-orange-500 text-white text-xs font-bold px-2 py-1 rounded flex items-center gap-1">
            <svg className="w-3 h-3" fill="currentColor" viewBox="0 0 20 20">
              <path d="M9.049 2.927c.3-.921 1.603-.921 1.902 0l1.07 3.292a1 1 0 00.95.69h3.462c.969 0 1.371 1.24.588 1.81l-2.8 2.034a1 1 0 00-.364 1.118l1.07 3.292c.3.921-.755 1.688-1.54 1.118l-2.8-2.034a1 1 0 00-1.175 0l-2.8 2.034c-.784.57-1.838-.197-1.539-1.118l1.07-3.292a1 1 0 00-.364-1.118L2.98 8.72c-.783-.57-.38-1.81.588-1.81h3.461a1 1 0 00.951-.69l1.07-3.292z" />
            </svg>
            Featured
          </span>
        </div>
        
        {/* Bottom Left Price and Heart */}
        <div className="absolute bottom-3 left-3 right-3 flex items-center justify-between">
          <div className="bg-black/70 text-white px-2 py-1 rounded text-lg font-bold">
            {formatPrice(property.sale_price || property.price || 0)}
          </div>
          <button className="bg-white/90 p-2 rounded-full shadow-md hover:scale-110 transition-all">
            <svg className="w-5 h-5 text-gray-600" fill="none" stroke="currentColor" viewBox="0 0 24 24">
              <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M4.318 6.318a4.5 4.5 0 000 6.364L12 20.364l7.682-7.682a4.5 4.5 0 00-6.364-6.364L12 7.636l-1.318-1.318a4.5 4.5 0 00-6.364 0z" />
            </svg>
          </button>
        </div>
      </div>
      
      {/* Details Section */}
      <div className="p-4 flex-1 flex flex-col justify-between">
        <div>
          {/* Title */}
          <h3 className="text-lg font-bold text-gray-900 mb-2 line-clamp-1">
            {property.title || `Premium ${property.bedrooms}BHK ${property.property_type || 'Apartment'}`}
          </h3>
          
          {/* Location */}
          <p className="text-gray-600 text-sm mb-3 flex items-center">
            <svg className="w-4 h-4 mr-1 text-gray-400" fill="currentColor" viewBox="0 0 20 20">
              <path fillRule="evenodd" d="M5.05 4.05a7 7 0 119.9 9.9L10 18.9l-4.95-4.95a7 7 0 010-9.9zM10 11a2 2 0 100-4 2 2 0 000 4z" clipRule="evenodd" />
            </svg>
            {property.locality}, {property.city}
          </p>
          
          {/* Property Details Row */}
          <div className="flex items-center justify-between text-sm text-gray-600 mb-4">
            <div className="flex items-center gap-1">
              <svg className="w-4 h-4 text-blue-600" fill="currentColor" viewBox="0 0 20 20">
                <path d="M10.707 2.293a1 1 0 00-1.414 0l-7 7a1 1 0 001.414 1.414L4 10.414V17a1 1 0 001 1h2a1 1 0 001-1v-2h2v2a1 1 0 001 1h2a1 1 0 001-1v-6.586l.293.293a1 1 0 001.414-1.414l-7-7z" />
              </svg>
              <span>{property.bedrooms || 'N/A'} Beds</span>
            </div>
            <div className="flex items-center gap-1">
              <svg className="w-4 h-4 text-blue-600" fill="currentColor" viewBox="0 0 20 20">
                <path d="M3 4a1 1 0 011-1h12a1 1 0 011 1v2a1 1 0 01-1 1H4a1 1 0 01-1-1V4zM3 10a1 1 0 011-1h6a1 1 0 011 1v6a1 1 0 01-1 1H4a1 1 0 01-1-1v-6zM14 9a1 1 0 00-1 1v6a1 1 0 001 1h2a1 1 0 001-1v-6a1 1 0 00-1-1h-2z" />
              </svg>
              <span>{property.bathrooms || 'N/A'} Baths</span>
            </div>
            <div className="flex items-center gap-1">
              <svg className="w-4 h-4 text-blue-600" fill="currentColor" viewBox="0 0 20 20">
                <path fillRule="evenodd" d="M3 3a1 1 0 000 2v8a2 2 0 002 2h2.586l-1.293 1.293a1 1 0 101.414 1.414L10 15.414l2.293 2.293a1 1 0 001.414-1.414L12.414 15H15a2 2 0 002-2V5a1 1 0 100-2H3z" clipRule="evenodd" />
              </svg>
              <span>{property.built_up_area || property.carpet_area || 'N/A'} sq.ft</span>
            </div>
          </div>
        </div>
        
        {/* View Details Button */}
        <button className="w-full py-2 bg-blue-600 text-white rounded-lg font-medium hover:bg-blue-700 transition-colors duration-200 text-sm">
          View Details
        </button>
      </div>
    </div>
  );
}



export function PropertyDetail() {
  const { id } = useParams();
  const navigate = useNavigate();
  const [property, setProperty] = useState<Property | null>(null);
  const [loading, setLoading] = useState(true);
  const [activeTab, setActiveTab] = useState("overview");
  const [currentImageIndex, setCurrentImageIndex] = useState(0);
  const [currentVideoIndex, setCurrentVideoIndex] = useState(0);
  const [showVideoPlayer, setShowVideoPlayer] = useState(false);
  const [autoScroll, setAutoScroll] = useState(true);
  const [timeLeft, setTimeLeft] = useState<string>("");
  const [showAuctionModal, setShowAuctionModal] = useState(false);
  const [showRegistrationModal, setShowRegistrationModal] = useState(false);
  const [registrationLoading, setRegistrationLoading] = useState(false);
  const [paymentLoading, setPaymentLoading] = useState(false);
  const [registrationStatus, setRegistrationStatus] = useState<
    "idle" | "payment" | "registered" | "pending"
  >("idle");
  const [isRegistered, setIsRegistered] = useState(false);
  const [showPaymentModal, setShowPaymentModal] = useState(false);
  const [paymentType, setPaymentType] = useState<"buy" | "urgent">("buy");
  const [creatingAgreement, setCreatingAgreement] = useState(false);
  const [showBiddingModal, setShowBiddingModal] = useState(false);
  const [bidAmount, setBidAmount] = useState("");
  const [placingBid, setPlacingBid] = useState(false);
  const [schools, setSchools] = useState<any[]>([]);
  const [schoolsLoading, setSchoolsLoading] = useState(false);
  const [auctionStatus, setAuctionStatus] = useState<any>(null);
  const [auctionLoading, setAuctionLoading] = useState(false);
  const [showCharacterModal, setShowCharacterModal] = useState(false);
  const [showConversationHistory, setShowConversationHistory] = useState(false);
  const [facilities, setFacilities] = useState<any[]>([]);
  const [facilitiesLoading, setFacilitiesLoading] = useState(false);

  // Voice Assistant Integration
  const voiceAssistant = useVoiceAssistant();
  const [propertyAssistant, setPropertyAssistant] =
    useState<PropertyAssistant | null>(null);
  const [conversationHistory, setConversationHistory] = useState<
    Array<{ type: "user" | "assistant"; message: string }>
  >([]);
  const [hasGreeted, setHasGreeted] = useState(false);

  // Initialize Property Assistant when property loads
  useEffect(() => {
    console.log("📊 Schools state changed:", {
      count: schools.length,
      schools: schools.map((s) => ({ name: s.name, distance: s.distance })),
      loading: schoolsLoading,
    });
  }, [schools, schoolsLoading]);

  useEffect(() => {
    const loadFacilities = async () => {
      if (activeTab === "nearby-facilities" && property) {
        console.log("🏢 Loading facilities for property:", property.id);
        setFacilitiesLoading(true);
        try {
          const lat = property.latitude || 12.9716;
          const lon = property.longitude || 77.5946;
          console.log("📍 Coordinates:", { lat, lon });
          
          const url = `http://localhost:8090/api/geo-demographic/nearby-facilities?lat=${lat}&lon=${lon}&radius_km=5`;
          console.log("🌐 Fetching from:", url);
          
          const response = await fetch(url);
          console.log("📡 Response status:", response.status);
          
          if (!response.ok) {
            throw new Error(`HTTP ${response.status}: ${response.statusText}`);
          }
          
          const data = await response.json();
          console.log("✅ Facilities data:", data);
          console.log("📊 Facilities count:", data.facilities?.length);
          
          setFacilities(data.facilities || []);
        } catch (error) {
          console.error("❌ Error fetching facilities:", error);
          setFacilities([]);
        } finally {
          setFacilitiesLoading(false);
        }
      }
    };
    loadFacilities();
  }, [activeTab, property?.id]);

  useEffect(() => {
    if (property) {
      setPropertyAssistant(
        new PropertyAssistant(property, voiceAssistant.language),
      );
    }
  }, [property, voiceAssistant.language]);

  // Handle voice recognition results
  useEffect(() => {
    if (voiceAssistant?.transcript && propertyAssistant) {
      const userMessage = voiceAssistant.transcript;
      console.log("User said:", userMessage);
      const assistantResponse = propertyAssistant.processQuery(userMessage);
      console.log("Assistant response:", assistantResponse);

      setConversationHistory((prev) => {
        const newHistory: Array<{
          type: "user" | "assistant";
          message: string;
        }> = [
          ...prev,
          { type: "user" as const, message: userMessage },
          { type: "assistant" as const, message: assistantResponse },
        ];
        console.log("Updated conversation history:", newHistory);
        return newHistory;
      });

      // Speak the response immediately
      if (voiceAssistant.speak) voiceAssistant.speak(assistantResponse);
      if (voiceAssistant.resetTranscript) voiceAssistant.resetTranscript();
    }
  }, [voiceAssistant?.transcript, propertyAssistant]);

  // Auto-greet when modal opens
  useEffect(() => {
    if (
      showCharacterModal &&
      propertyAssistant &&
      !hasGreeted &&
      voiceAssistant?.isSupported
    ) {
      const greeting = propertyAssistant.getGreeting();
      setConversationHistory([
        { type: "assistant" as const, message: greeting },
      ]);
      setHasGreeted(true);

      // Speak greeting immediately
      if (voiceAssistant.speak) voiceAssistant.speak(greeting);
    }
  }, [
    showCharacterModal,
    propertyAssistant,
    hasGreeted,
    voiceAssistant.isSupported,
    voiceAssistant,
  ]);

  // Speak welcome message when language changes
  useEffect(() => {
    if (
      showCharacterModal &&
      propertyAssistant &&
      voiceAssistant.isSupported &&
      hasGreeted
    ) {
      const greeting = propertyAssistant.getGreeting();
      voiceAssistant.speak(greeting);
    }
  }, [voiceAssistant.language, propertyAssistant]);

  const [showTourModal, setShowTourModal] = useState(false);
  const [availableProperties, setAvailableProperties] = useState<any[]>([]);
  const [selectedProperties, setSelectedProperties] = useState<number[]>([]);
  const [loadingTourProperties, setLoadingTourProperties] = useState(false);

  const [formData, setFormData] = useState({
    name: "",
    email: "",
    phone: "",
    message: "I'm interested in this property...",
  });

  // Azure Blob media integration
  const {
    images: azureImages,
    videos: azureVideos,
    documents: azureDocuments,
    loading: mediaLoading,
    hasMedia,
  } = useAzureBlobMedia(id);

  // Debug: Log Azure images
  useEffect(() => {
    console.log("Azure Images:", azureImages);
    console.log("Property ID:", id);
  }, [azureImages, id]);

  const formatTimeLeft = (timeLeftMs: number) => {
    if (timeLeftMs <= 0) return "Ended";
    const totalSeconds = Math.floor(timeLeftMs / 1000);
    const days = Math.floor(totalSeconds / 86400);
    const hours = Math.floor((totalSeconds % 86400) / 3600);
    const minutes = Math.floor((totalSeconds % 3600) / 60);
    const seconds = totalSeconds % 60;

    if (days > 0) return `${days}d ${hours}h ${minutes}m ${seconds}s`;
    if (hours > 0) return `${hours}h ${minutes}m ${seconds}s`;
    return `${minutes}m ${seconds}s`;
  };

  useEffect(() => {
    const fetchProperty = async () => {
      try {
        console.log("Fetching property with ID:", id);
        // Import the api utility
        const { api } = await import("../utils/api");
        const data = await api.fetchPropertyById(id!);

        if (data) {
          console.log("Property data received:", data);

          // CRITICAL FIX: Add default coordinates if missing
          if (!data.latitude || !data.longitude) {
            console.log(
              "⚠️ Property missing coordinates, adding default Bangalore coordinates",
            );
            data.latitude = 12.9716; // Bangalore latitude
            data.longitude = 77.5946; // Bangalore longitude
          }

          setProperty(data);

          // Track property view using dedicated view tracker
          const user = JSON.parse(localStorage.getItem("user") || "{}");
          const userId = user.user_id || user.id;

          console.log("🔍 Tracking property view:", {
            propertyId: parseInt(id!),
            userId,
            userExists: !!userId,
          });

          // Use the dedicated property view tracker
          const tracked = await trackPropertyView(
            parseInt(id!),
            userId ? String(userId) : undefined,
          );
          console.log("📊 View tracking result:", tracked);

          // Also store property data in user activities for fallback
          if (userId && tracked) {
            try {
              const activityResponse = await fetch(
                "http://localhost:8090/api/user-activities/",
                {
                  method: "POST",
                  headers: {
                    "Content-Type": "application/json",
                    Authorization: `Bearer ${localStorage.getItem("access_token")}`,
                  },
                  body: JSON.stringify({
                    user_id: String(userId),
                    activity_type: "property_view",
                    property_id: parseInt(id!),
                    activity_data: {
                      property_title: data.title,
                      property_location: `${data.locality}, ${data.city}`,
                      property_price: data.sale_price || data.monthly_rent,
                      property_image: data.primary_image || data.photo,
                      property_type: data.property_type,
                      property_bedrooms: data.bedrooms,
                      property_bathrooms: data.bathrooms,
                      property_area: data.built_up_area || data.carpet_area,
                      property_city: data.city,
                      property_locality: data.locality,
                      property_latitude: data.latitude,
                      property_longitude: data.longitude,
                      property_listing_type: data.listing_type,
                      timestamp: new Date().toISOString(),
                      source: "property_detail_page",
                    },
                  }),
                },
              );

              if (activityResponse.ok) {
                console.log("✅ Property data stored in user activities");
              } else {
                console.log("⚠️ Failed to store property data in activities");
              }
            } catch (error) {
              console.error("Error storing property activity:", error);
            }
          }

          // Also track using behavior tracker for analytics
          const { behaviorTracker } = await import("../utils/behaviorTracker");
          behaviorTracker.trackPropertyView(parseInt(id!), {
            title: data.title,
            property_type: data.property_type,
            location: data.city,
            price: data.sale_price,
          });
        } else {
          console.error("Property not found or API error");
          setProperty(null);
        }
      } catch (error) {
        console.error("Error fetching property:", error);
        setProperty(null);
      } finally {
        setLoading(false);
      }
    };
    fetchProperty();
  }, [id]);

  // Fetch nearby schools when school-distance tab is active
  useEffect(() => {
    const loadSchools = async () => {
      if (activeTab === "school-distance" && property) {
        console.log("🏫 Loading schools for property:", property.id);
        setSchoolsLoading(true);
        setSchools([]);

        try {
          // Use property coordinates or fallback to Bangalore
          const lat = property.latitude || 12.9716;
          const lon = property.longitude || 77.5946;

          console.log("📍 Using coordinates:", {
            lat,
            lon,
            hasOriginal: !!(property.latitude && property.longitude),
          });

          const schoolsResponse = await fetchNearbySchools(lat, lon, 5);
          console.log("✅ Schools loaded:", schoolsResponse.schools.length);
          setSchools(schoolsResponse.schools);
        } catch (error) {
          console.error("❌ School fetch failed:", error);
          setSchools([]);
        } finally {
          setSchoolsLoading(false);
        }
      }
    };
    loadSchools();
  }, [activeTab, property?.id]);

  // Timeline countdown effect
  useEffect(() => {
    if (!property) return;

    const isUrgentSale = property.listing_type === "urgent-sale";
    const isBidding = property.listing_type === "bidding";

    if (
      (isUrgentSale &&
        property.urgent_sale_end_date &&
        property.urgent_sale_end_time) ||
      (isBidding && property.auction_end_date && property.auction_end_time)
    ) {
      const interval = setInterval(() => {
        let endDateTime: Date;

        if (
          isUrgentSale &&
          property.urgent_sale_end_date &&
          property.urgent_sale_end_time
        ) {
          endDateTime = new Date(
            `${property.urgent_sale_end_date}T${property.urgent_sale_end_time}`,
          );
        } else if (
          isBidding &&
          property.auction_end_date &&
          property.auction_end_time
        ) {
          endDateTime = new Date(
            `${property.auction_end_date}T${property.auction_end_time}`,
          );
        } else {
          return;
        }

        const diff = endDateTime.getTime() - new Date().getTime();
        setTimeLeft(formatTimeLeft(diff));
      }, 1000);

      return () => clearInterval(interval);
    }
  }, [property]);

  // Prepare image and video arrays with Azure Blob integration
  const allImages = (() => {
    const images = [];

    // Debug: Always check Azure Blob images first
    console.log("Processing images - Azure count:", azureImages?.length || 0);

    // Prioritize Azure Blob images if available
    if (azureImages && azureImages.length > 0) {
      azureImages.forEach((img) => {
        console.log("Adding Azure image:", img.url);
        images.push(img.url);
      });
    }

    // Fallback to legacy images if no Azure Blob images
    if (images.length === 0) {
      console.log("No Azure images, using legacy images");

      // Helper function to process image URLs
      const processImageUrl = (url: string) => {
        if (!url) return null;

        // If it's already a full URL, use it
        if (url.startsWith("http")) return url;

        // For database-stored images, use the backend API endpoint
        // The backend should serve binary data from the database
        return `http://localhost:8090${url.startsWith("/") ? url : "/" + url}`;
      };

      // Add primary image/photo
      if (property?.photo) {
        const photoUrl = processImageUrl(property.photo);
        if (photoUrl) images.push(photoUrl);
      }

      // Add additional images from different possible fields
      if (
        property?.images &&
        Array.isArray(property.images) &&
        property.images.length > 0
      ) {
        property.images.forEach((img) => {
          if (typeof img === "string") {
            const imgUrl = processImageUrl(img);
            if (imgUrl) images.push(imgUrl);
          } else if (img && typeof img === "object" && "url" in img) {
            // Handle image objects with url property
            const imgUrl = processImageUrl((img as any).url);
            if (imgUrl) images.push(imgUrl);
          }
        });
      }

      // Add image_urls if available
      if (
        property?.image_urls &&
        Array.isArray(property.image_urls) &&
        property.image_urls.length > 0
      ) {
        property.image_urls.forEach((img) => {
          const imgUrl = processImageUrl(img);
          if (imgUrl) images.push(imgUrl);
        });
      }

      // Add primary_image if different from photo
      if (
        property?.primary_image &&
        property.primary_image !== property?.photo
      ) {
        const imgUrl = processImageUrl(property.primary_image);
        if (imgUrl) images.push(imgUrl);
      }

      // No sample images - only show real images
    }

    // Remove duplicates
    return [...new Set(images)];
  })();

  const allVideos = (() => {
    const videos: string[] = [];

    // Prioritize Azure Blob videos if available
    if (azureVideos && azureVideos.length > 0) {
      azureVideos.forEach((video) => {
        videos.push(video.url);
      });
    }

    // Fallback to legacy videos if no Azure Blob videos
    if (videos.length === 0) {
      // Handle videos array
      if (
        property?.videos &&
        Array.isArray(property.videos) &&
        property.videos.length > 0
      ) {
        property.videos.forEach((video) => {
          if (typeof video === "string") {
            const videoUrl = video.startsWith("http")
              ? video
              : `http://localhost:8090${video}`;
            videos.push(videoUrl);
          } else if (video && typeof video === "object" && "url" in video) {
            // Handle video objects with url property
            const videoUrl = (video as any).url.startsWith("http")
              ? (video as any).url
              : `http://localhost:8090${(video as any).url}`;
            videos.push(videoUrl);
          }
        });
      }

      // Handle virtual_tour_url as video
      if (property?.virtual_tour_url) {
        const tourUrl = property.virtual_tour_url.startsWith("http")
          ? property.virtual_tour_url
          : `http://localhost:8090${property.virtual_tour_url}`;
        videos.push(tourUrl);
      }

      // Handle virtual_tours array
      if (
        property?.virtual_tours &&
        Array.isArray(property.virtual_tours) &&
        property.virtual_tours.length > 0
      ) {
        property.virtual_tours.forEach((tour) => {
          const tourUrl = tour.startsWith("http")
            ? tour
            : `http://localhost:8090${tour}`;
          videos.push(tourUrl);
        });
      }
    }

    // Remove duplicates
    return [...new Set(videos)];
  })();

  // Auto-scroll effect for images - MUST be after allImages declaration
  useEffect(() => {
    if (!showVideoPlayer && allImages.length > 1 && autoScroll) {
      const interval = setInterval(() => {
        setCurrentImageIndex((prev) => (prev + 1) % allImages.length);
      }, 4000);

      return () => clearInterval(interval);
    }
  }, [allImages.length, showVideoPlayer, autoScroll]);

  const formatPrice = (price: string | number) => {
    const numPrice = typeof price === "string" ? parseFloat(price) : price;
    if (isNaN(numPrice) || numPrice === 0) return "Price on request";
    if (numPrice >= 10000000) return `₹${(numPrice / 10000000).toFixed(1)} Cr`;
    if (numPrice >= 100000) return `₹${(numPrice / 100000).toFixed(0)} L`;
    return `₹${numPrice.toLocaleString()}`;
  };

  const handleFormSubmit = (e: React.FormEvent) => {
    e.preventDefault();
    console.log("Form submitted:", formData);
  };

  const fetchTourProperties = async () => {
    setLoadingTourProperties(true);
    try {
      const user = JSON.parse(localStorage.getItem("user") || "{}");
      const userEmail = user.email;

      const response = await fetch("http://localhost:8090/api/properties/");
      if (response.ok) {
        const result = await response.json();
        const allProperties = result.properties || [];
        const filteredProperties = allProperties.filter(
          (p: any) => p.email !== userEmail,
        );
        setAvailableProperties(filteredProperties);
        setSelectedProperties([parseInt(id!)]);
      }
    } catch (error) {
      console.error("Failed to load properties:", error);
    } finally {
      setLoadingTourProperties(false);
    }
  };

  const togglePropertySelection = (propertyId: number) => {
    setSelectedProperties((prev) =>
      prev.includes(propertyId)
        ? prev.filter((id) => id !== propertyId)
        : [...prev, propertyId],
    );
  };

  const handleProceedToBooking = () => {
    if (selectedProperties.length === 0) {
      alert("Please select at least one property");
      return;
    }
    navigate(
      `/property/${selectedProperties[0]}/book-visit?properties=${selectedProperties.join(",")}`,
    );
  };

  const handleContactSubmit = async () => {
    try {
      const token = localStorage.getItem("access_token");
      const headers: any = {
        "Content-Type": "application/json",
      };

      if (token) {
        headers["Authorization"] = `Bearer ${token}`;
      }

      const response = await fetch(
        "http://localhost:8090/api/contact-inquiry/",
        {
          method: "POST",
          headers,
          body: JSON.stringify({
            property: property?.id,
            name: formData.name,
            email: formData.email,
            phone: formData.phone,
            message: formData.message,
          }),
        },
      );

      if (response.ok) {
        alert("Contact inquiry submitted successfully!");
        setFormData({
          name: "",
          email: "",
          phone: "",
          message: "I'm interested in this property...",
        });
      } else {
        alert("Failed to submit inquiry. Please try again.");
      }
    } catch (error) {
      console.error("Error submitting contact inquiry:", error);
      alert("Error submitting inquiry. Please try again.");
    }
  };

  // Check auction status for bidding properties
  useEffect(() => {
    const checkAuctionStatus = async () => {
      if (!property || property.listing_type !== "bidding") return;

      setAuctionLoading(true);
      try {
        const statusResponse = await fetch(
          `http://localhost:8090/api/auctions/${property.id}/current-status`,
          {
            headers: {
              "Content-Type": "application/json",
            },
          },
        );

        if (statusResponse.ok) {
          const statusData = await statusResponse.json();
          setAuctionStatus(statusData);
        } else if (statusResponse.status === 404) {
          // No auction session exists - silently ignore
          setAuctionStatus(null);
        }
      } catch (error) {
        // Silently handle errors
        setAuctionStatus(null);
      } finally {
        setAuctionLoading(false);
      }
    };

    checkAuctionStatus();

    // Auto-refresh every 5 seconds only if auction exists
    const interval = setInterval(() => {
      if (property?.listing_type === "bidding" && auctionStatus) {
        checkAuctionStatus();
      }
    }, 5000);

    return () => clearInterval(interval);
  }, [property?.id, property?.listing_type, auctionStatus]);

  useEffect(() => {
    const checkRegistrationStatus = async () => {
      if (!property?.id) return;

      const accessToken = localStorage.getItem("access_token");
      if (!accessToken) {
        console.log("No access token available for registration status check");
        return;
      }

      try {
        console.log("=== REGISTRATION STATUS CHECK ===");
        console.log("Property ID:", property.id);
        console.log("Access token exists:", !!accessToken);

        // First, find ALL sessions for this property (not just active ones)
        const auctionsResponse = await fetch(
          "http://localhost:8090/api/auctions/",
          {
            headers: {
              Authorization: `Bearer ${accessToken}`,
              "Content-Type": "application/json",
            },
          },
        );

        console.log("Auctions response status:", auctionsResponse.status);

        if (!auctionsResponse.ok) {
          console.log("Failed to fetch auctions:", auctionsResponse.status);
          return;
        }

        const auctionsData = await auctionsResponse.json();
        const sessions = auctionsData.sessions || [];
        console.log("Found sessions:", sessions.length);

        // Find ANY session for this property (active, scheduled, etc.)
        const propertySession = sessions.find(
          (session: any) => session.property_id === property.id,
        );

        if (!propertySession) {
          console.log("No auction session found for property:", property.id);
          console.log(
            "Available sessions:",
            sessions.map((s) => ({ id: s.id, property_id: s.property_id })),
          );
          return;
        }

        console.log("Found session for property:", propertySession.id);

        const response = await fetch(
          `http://localhost:8090/api/auctions/${propertySession.id}/registration-status`,
          {
            headers: {
              Authorization: `Bearer ${accessToken}`,
              "Content-Type": "application/json",
            },
          },
        );

        console.log("Registration status response status:", response.status);

        if (response.ok) {
          const data = await response.json();
          console.log("Registration status response:", data);
          // Check for both 'pending' and 'approved' status, and also check registration_fee_paid
          if (
            data &&
            (data.registration_status === "pending" ||
              data.registration_status === "approved" ||
              data.registration_fee_paid === true)
          ) {
            setIsRegistered(true);
            console.log(
              "✅ User is registered for auction - Status:",
              data.registration_status,
              "Fee paid:",
              data.registration_fee_paid,
            );
          } else {
            console.log(
              "❌ User is NOT registered - Status:",
              data.registration_status,
              "Fee paid:",
              data.registration_fee_paid,
            );
            setIsRegistered(false);
          }
        } else if (response.status === 404) {
          // 404 means no registration found - user is not registered
          console.log(
            "❌ No registration found (404) - User is NOT registered",
          );
          setIsRegistered(false);
        } else {
          const errorText = await response.text();
          console.log(
            "Registration status check failed:",
            response.status,
            response.statusText,
            errorText,
          );
          // If error contains "registration fee already paid", user is registered
          if (errorText.includes("registration fee already paid")) {
            console.log("✅ Error indicates user is already registered");
            setIsRegistered(true);
          } else {
            setIsRegistered(false);
          }
        }
        console.log("=================================");
      } catch (error) {
        console.log("Registration status check failed:", error);
        setIsRegistered(false);
      }
    };

    checkRegistrationStatus();
  }, [property?.id]);

  // Helper function for auction registration details submission
  const handleRegistrationSubmit = async (details: any) => {
    // Store details temporarily and proceed to payment
    sessionStorage.setItem(
      "auction_registration_details",
      JSON.stringify(details),
    );
    setShowRegistrationModal(false);
    setShowAuctionModal(true); // Show payment modal
  };

  const handlePayRegistrationFee = async () => {
    const accessToken = localStorage.getItem("access_token");

    if (!accessToken) {
      alert("Please login to register for auction");
      navigate("/login");
      return;
    }

    try {
      setPaymentLoading(true);

      // First, find the session ID for this property
      const auctionsResponse = await fetch(
        "http://localhost:8090/api/auctions/active",
        {
          headers: {
            Authorization: `Bearer ${accessToken}`,
            "Content-Type": "application/json",
          },
        },
      );

      if (!auctionsResponse.ok) {
        throw new Error(
          `Failed to fetch auction sessions: ${auctionsResponse.status} ${auctionsResponse.statusText}`,
        );
      }

      const auctionsData = await auctionsResponse.json();
      const sessions = auctionsData.sessions || [];

      // Find the session for this property
      const propertySession = sessions.find(
        (session: any) => session.property_id === property?.id,
      );

      if (!propertySession) {
        throw new Error("No active auction session found for this property");
      }

      // Generate unique payment reference
      const paymentReference = `PAY_${Date.now()}_${Math.random().toString(36).substr(2, 9).toUpperCase()}`;

      // Get stored registration details
      const storedDetails = sessionStorage.getItem(
        "auction_registration_details",
      );
      const registrationDetails = storedDetails
        ? JSON.parse(storedDetails)
        : null;

      // Simulate payment processing
      await new Promise((resolve) => setTimeout(resolve, 2000));

      // Call the pay-registration-fee endpoint with details
      const registerResponse = await fetch(
        `http://localhost:8090/api/auctions/${propertySession.id}/pay-registration-fee`,
        {
          method: "POST",
          headers: {
            "Content-Type": "application/json",
            Authorization: `Bearer ${accessToken}`,
          },
          body: JSON.stringify({
            payment_reference: paymentReference,
            registration_details: registrationDetails,
          }),
        },
      );

      if (!registerResponse.ok) {
        const errorText = await registerResponse.text();
        let errorMessage = "Registration failed";
        try {
          const errorData = JSON.parse(errorText);
          errorMessage = errorData.detail || errorData.message || errorMessage;
        } catch (e) {
          errorMessage = `Registration failed: ${registerResponse.status} ${registerResponse.statusText}`;
        }
        throw new Error(errorMessage);
      }

      const registrationResult = await registerResponse.json();

      if (registrationResult.registration_status === "pending") {
        setRegistrationStatus("pending");
        setIsRegistered(true);
      } else {
        setRegistrationStatus("registered");
        setIsRegistered(true);
      }

      // Clear stored details after successful registration
      sessionStorage.removeItem("auction_registration_details");
    } catch (error) {
      console.error("Registration error:", error);

      // If the error is "registration fee already paid", it means user is already registered
      if (
        error.message &&
        error.message.includes("registration fee already paid")
      ) {
        setRegistrationStatus("registered");
        setIsRegistered(true);
        alert("You are already registered for this auction!");
        setShowAuctionModal(false);
      } else {
        alert(`Registration failed: ${error.message}`);
      }
    } finally {
      setPaymentLoading(false);
    }
  };

  if (loading) {
    return (
      <div className="max-w-4xl mx-auto p-8 text-center">
        <div className="animate-spin rounded-full h-8 w-8 border-b-2 border-blue-600 mx-auto mb-4"></div>
        <p className="text-gray-500">Loading property details...</p>
      </div>
    );
  }

  if (!property) {
    return (
      <div className="max-w-4xl mx-auto p-8">
        <h2 className="text-2xl font-bold mb-4">Property Not Found</h2>
        <p className="text-gray-600 mb-4">Property ID: {id}</p>
        <p className="text-gray-600 mb-4">
          Unable to load property details. This could be due to:
        </p>
        <ul className="list-disc list-inside text-gray-600 mb-4">
          <li>Property doesn't exist</li>
          <li>Network connection issue</li>
          <li>Server error</li>
        </ul>
        <button
          onClick={() => navigate(-1)}
          className="mt-4 bg-green-600 text-white px-4 py-2 rounded hover:bg-green-700 mr-4"
        >
          Go Back
        </button>
        <button
          onClick={() => window.location.reload()}
          className="mt-4 bg-gray-600 text-white px-4 py-2 rounded hover:bg-gray-700"
        >
          Retry
        </button>
      </div>
    );
  }

  return (
    <div className="bg-white min-h-screen">
      {/* Media Gallery - Smaller and centered */}
      <div className="max-w-4xl mx-auto px-4 py-6">
        <div className="relative h-80 bg-gray-200 rounded-lg overflow-hidden">
          {showVideoPlayer && allVideos.length > 0 ? (
            <video
              src={allVideos[currentVideoIndex]}
              controls
              className="w-full h-full object-cover"
              onError={(e) => {
                console.error(
                  "Video failed to load:",
                  allVideos[currentVideoIndex],
                );
                console.error("Video element:", e.currentTarget);
                // Hide the video and show an error message
                e.currentTarget.style.display = "none";
                const errorDiv = document.createElement("div");
                errorDiv.className =
                  "w-full h-full flex items-center justify-center bg-gray-100";
                errorDiv.innerHTML =
                  '<div class="text-center"><svg class="w-16 h-16 text-gray-400 mx-auto mb-4" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M15 10l4.553-2.276A1 1 0 0121 8.618v6.764a1 1 0 01-1.447.894L15 14M5 18h8a2 2 0 002-2V8a2 2 0 00-2-2H5a2 2 0 00-2 2v8a2 2 0 002 2z"></path></svg><p class="text-gray-500">Video failed to load</p></div>';
                e.currentTarget.parentNode?.appendChild(errorDiv);
              }}
              onLoadStart={() =>
                console.log(
                  "Video loading started:",
                  allVideos[currentVideoIndex],
                )
              }
              onLoadedData={() =>
                console.log(
                  "Video loaded successfully:",
                  allVideos[currentVideoIndex],
                )
              }
            />
          ) : allImages.length > 0 ? (
            <img
              src={allImages[currentImageIndex]}
              alt={property?.title}
              className="w-full h-full object-cover"
              onError={(e) => {
                console.error("Image failed to load:", e.currentTarget.src);
                console.error("Available images:", allImages);

                // Hide the failed image and show placeholder
                e.currentTarget.style.display = "none";
                const placeholder = document.createElement("div");
                placeholder.className =
                  "w-full h-full bg-gradient-to-br from-gray-50 to-gray-100 flex items-center justify-center";
                placeholder.innerHTML = `
                <div class="text-center">
                  <svg class="w-16 h-16 text-gray-300 mx-auto mb-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="1.5" d="M4 16l4.586-4.586a2 2 0 012.828 0L16 16m-2-2l1.586-1.586a2 2 0 012.828 0L20 14" />
                  </svg>
                  <p class="text-gray-400 text-sm font-medium">No image available</p>
                </div>
              `;
                e.currentTarget.parentNode?.appendChild(placeholder);
              }}
              onLoad={() =>
                console.log(
                  "Image loaded successfully:",
                  allImages[currentImageIndex],
                )
              }
            />
          ) : (
            <div className="w-full h-full flex items-center justify-center bg-gray-100">
              <div className="text-center">
                <svg
                  className="w-16 h-16 text-gray-400 mx-auto mb-4"
                  fill="none"
                  stroke="currentColor"
                  viewBox="0 0 24 24"
                >
                  <path
                    strokeLinecap="round"
                    strokeLinejoin="round"
                    strokeWidth={2}
                    d="M4 16l4.586-4.586a2 2 0 012.828 0L16 16m-2-2l1.586-1.586a2 2 0 012.828 0L20 14m-6-6h.01M6 20h12a2 2 0 002-2V6a2 2 0 00-2-2H6a2 2 0 00-2 2v14a2 2 0 002 2z"
                  />
                </svg>
                <p className="text-gray-500">
                  No images available for this property
                </p>
              </div>
            </div>
          )}

          {/* Media Type Toggle */}
          <div className="absolute top-4 left-4 flex space-x-2">
            <button
              onClick={() => setShowVideoPlayer(false)}
              className={`px-3 py-1 rounded-full text-sm font-medium ${
                !showVideoPlayer
                  ? "bg-white text-gray-900"
                  : "bg-white/60 text-white"
              }`}
            >
              Photos ({allImages.length})
            </button>
            {allVideos.length > 0 && (
              <button
                onClick={() => setShowVideoPlayer(true)}
                className={`px-3 py-1 rounded-full text-sm font-medium ${
                  showVideoPlayer
                    ? "bg-white text-gray-900"
                    : "bg-white/60 text-white"
                }`}
              >
                Videos ({allVideos.length})
              </button>
            )}
          </div>

          {/* Auto-scroll control and image indicators */}
          {!showVideoPlayer && allImages.length > 1 && (
            <>
              {/* Auto-scroll toggle */}
              <div className="absolute top-4 right-4">
                <button
                  onClick={() => setAutoScroll(!autoScroll)}
                  className="bg-white/80 p-2 rounded-full hover:bg-white transition-colors"
                  title={
                    autoScroll ? "Pause auto-scroll" : "Resume auto-scroll"
                  }
                >
                  {autoScroll ? (
                    <svg
                      className="w-4 h-4"
                      fill="currentColor"
                      viewBox="0 0 20 20"
                    >
                      <path
                        fillRule="evenodd"
                        d="M18 10a8 8 0 11-16 0 8 8 0 0116 0zM7 8a1 1 0 012 0v4a1 1 0 11-2 0V8zm5-1a1 1 0 00-1 1v4a1 1 0 102 0V8a1 1 0 00-1-1z"
                        clipRule="evenodd"
                      />
                    </svg>
                  ) : (
                    <svg
                      className="w-4 h-4"
                      fill="currentColor"
                      viewBox="0 0 20 20"
                    >
                      <path
                        fillRule="evenodd"
                        d="M10 18a8 8 0 100-16 8 8 0 000 16zM9.555 7.168A1 1 0 008 8v4a1 1 0 001.555.832l3-2a1 1 0 000-1.664l-3-2z"
                        clipRule="evenodd"
                      />
                    </svg>
                  )}
                </button>
              </div>

              {/* Image indicators */}
              <div className="absolute bottom-4 left-1/2 transform -translate-x-1/2 flex space-x-2">
                {allImages.map((_, index) => (
                  <button
                    key={index}
                    onClick={() => {
                      setCurrentImageIndex(index);
                      setAutoScroll(false);
                    }}
                    className={`w-2 h-2 rounded-full transition-colors ${
                      index === currentImageIndex ? "bg-white" : "bg-white/50"
                    }`}
                  />
                ))}
              </div>
            </>
          )}

          {/* Navigation Arrows */}
          {showVideoPlayer && allVideos.length > 1 ? (
            <>
              <button
                onClick={() =>
                  setCurrentVideoIndex((prev) =>
                    prev > 0 ? prev - 1 : allVideos.length - 1,
                  )
                }
                className="absolute left-4 top-1/2 transform -translate-y-1/2 bg-white/80 p-2 rounded-full hover:bg-white"
              >
                <svg
                  className="w-6 h-6"
                  fill="none"
                  stroke="currentColor"
                  viewBox="0 0 24 24"
                >
                  <path
                    strokeLinecap="round"
                    strokeLinejoin="round"
                    strokeWidth={2}
                    d="M15 19l-7-7 7-7"
                  />
                </svg>
              </button>
              <button
                onClick={() =>
                  setCurrentVideoIndex((prev) =>
                    prev < allVideos.length - 1 ? prev + 1 : 0,
                  )
                }
                className="absolute right-4 top-1/2 transform -translate-y-1/2 bg-white/80 p-2 rounded-full hover:bg-white"
              >
                <svg
                  className="w-6 h-6"
                  fill="none"
                  stroke="currentColor"
                  viewBox="0 0 24 24"
                >
                  <path
                    strokeLinecap="round"
                    strokeLinejoin="round"
                    strokeWidth={2}
                    d="M9 5l7 7-7 7"
                  />
                </svg>
              </button>
            </>
          ) : !showVideoPlayer && allImages.length > 1 ? (
            <>
              <button
                onClick={() => {
                  setCurrentImageIndex((prev) =>
                    prev > 0 ? prev - 1 : allImages.length - 1,
                  );
                  setAutoScroll(false);
                }}
                className="absolute left-4 top-1/2 transform -translate-y-1/2 bg-white/80 p-2 rounded-full hover:bg-white"
              >
                <svg
                  className="w-6 h-6"
                  fill="none"
                  stroke="currentColor"
                  viewBox="0 0 24 24"
                >
                  <path
                    strokeLinecap="round"
                    strokeLinejoin="round"
                    strokeWidth={2}
                    d="M15 19l-7-7 7-7"
                  />
                </svg>
              </button>
              <button
                onClick={() => {
                  setCurrentImageIndex((prev) =>
                    prev < allImages.length - 1 ? prev + 1 : 0,
                  );
                  setAutoScroll(false);
                }}
                className="absolute right-4 top-1/2 transform -translate-y-1/2 bg-white/80 p-2 rounded-full hover:bg-white"
              >
                <svg
                  className="w-6 h-6"
                  fill="none"
                  stroke="currentColor"
                  viewBox="0 0 24 24"
                >
                  <path
                    strokeLinecap="round"
                    strokeLinejoin="round"
                    strokeWidth={2}
                    d="M9 5l7 7-7 7"
                  />
                </svg>
              </button>
            </>
          ) : null}
        </div>
      </div>

      {/* Thumbnail Gallery */}
      <div className="max-w-4xl mx-auto px-4 py-2">
        <div className="flex space-x-2 overflow-x-auto">
          {!showVideoPlayer
            ? allImages.map((image, index) => (
                <img
                  key={`img-${index}`}
                  src={image}
                  alt={`${property?.title} ${index + 1}`}
                  className={`w-20 h-16 object-cover rounded cursor-pointer border-2 ${
                    index === currentImageIndex
                      ? "border-blue-500"
                      : "border-gray-200"
                  }`}
                  onClick={() => setCurrentImageIndex(index)}
                  onError={(e) => {
                    console.error(
                      "Thumbnail image failed to load:",
                      e.currentTarget.src,
                    );
                    // Hide failed thumbnail
                    e.currentTarget.style.display = "none";
                  }}
                />
              ))
            : allVideos.map((video, index) => (
                <div
                  key={`vid-${index}`}
                  className={`relative w-20 h-16 bg-gray-800 rounded cursor-pointer border-2 ${
                    index === currentVideoIndex
                      ? "border-blue-500"
                      : "border-gray-200"
                  }`}
                  onClick={() => setCurrentVideoIndex(index)}
                >
                  <video
                    src={video}
                    className="w-full h-full object-cover rounded"
                    muted
                  />
                  <div className="absolute inset-0 flex items-center justify-center">
                    <svg
                      className="w-6 h-6 text-white"
                      fill="currentColor"
                      viewBox="0 0 20 20"
                    >
                      <path
                        fillRule="evenodd"
                        d="M10 18a8 8 0 100-16 8 8 0 000 16zM9.555 7.168A1 1 0 008 8v4a1 1 0 001.555.832l3-2a1 1 0 000-1.664l-3-2z"
                        clipRule="evenodd"
                      />
                    </svg>
                  </div>
                </div>
              ))}
        </div>
      </div>

      <div className="max-w-7xl mx-auto px-4 py-6">
        {/* Two Column Layout: 70% Left, 30% Right */}
        <div className="flex flex-col lg:flex-row gap-8">
          {/* Left Column (70%) */}
          <div className="lg:w-[70%]">
            {/* Property Header */}
            <div className="mb-6">
              <div className="flex items-start justify-between mb-4">
                <div className="flex-1">
                  <h1 className="text-3xl font-bold text-gray-900 mb-2">
                    {property?.title}
                  </h1>
                  <p className="text-gray-600 mb-4">
                    {property?.locality}, {property?.city}, {property?.state}
                  </p>
                </div>

                {/* Timeline Badge for Urgent Sale and Bidding */}
                {(property?.listing_type === "urgent-sale" ||
                  property?.listing_type === "bidding") &&
                  timeLeft && (
                    <div
                      className={`px-4 py-3 rounded-lg border-2 text-center min-w-[200px] ${
                        property.listing_type === "urgent-sale"
                          ? "bg-red-50 border-red-200 text-red-800"
                          : "bg-blue-50 border-blue-200 text-blue-800"
                      }`}
                    >
                      <div className="text-sm font-medium mb-1">
                        {property.listing_type === "urgent-sale"
                          ? " Urgent Sale Ends"
                          : " Auction Ends"}
                      </div>
                      <div className="text-lg font-bold font-mono">
                        {timeLeft}
                      </div>
                    </div>
                  )}
              </div>

              <div className="flex items-center justify-between mb-4">
                <div className="flex items-center gap-4">
                  <span className="text-3xl font-bold text-blue-600">
                    {property?.listing_type === "bidding"
                      ? `Starting: ${formatPrice(property?.starting_bidding_price || 0)}`
                      : formatPrice(property?.sale_price || 0)}
                  </span>
                </div>

                {/* Action Buttons */}
                <div className="flex items-center gap-2">
                  {/* Wishlist Button */}
                  <WishlistHeartButton propertyId={property.id} />

                  {/* Save Button */}
                  <SavedPropertiesButton propertyId={property.id} />

                  {/* Share Button */}
                  <ShareButton
                    title={property.title}
                    text={`Check out this ${property.property_type} in ${property.locality}, ${property.city} - ${formatPrice(property.sale_price || 0)}`}
                  />
                </div>
              </div>

              {/* Buyer Market Snapshot - Show after price */}
              <BuyerMarketSnapshot
                propertyPrice={parseFloat(
                  property?.sale_price || property?.monthly_rent || "0",
                )}
                locality={property?.locality || ""}
                city={property?.city}
                variant="full"
              />

              <div className="flex items-center gap-4">
                {/* Property Type Badge */}
                <span
                  className={`px-3 py-1 rounded-full text-sm font-medium ${
                    property?.listing_type === "urgent-sale"
                      ? "bg-red-100 text-red-800"
                      : property?.listing_type === "bidding"
                        ? "bg-blue-100 text-blue-800"
                        : "bg-green-100 text-green-800"
                  }`}
                >
                  {property?.listing_type === "urgent-sale"
                    ? "Urgent Sale"
                    : property?.listing_type === "bidding"
                      ? "Live Auction"
                      : "For Sale"}
                </span>

                {/* Edit Property Button */}
                {(() => {
                  // Get current user email from localStorage
                  const currentUser = JSON.parse(
                    localStorage.getItem("user") || "{}",
                  );
                  const userEmail =
                    currentUser.email || localStorage.getItem("user_email");
                  // Get user identification from localStorage
                  const accessToken = localStorage.getItem("access_token");
                  const userDataStr = localStorage.getItem("user");
                  const loginEmail = localStorage.getItem("login_email");

                  let currentUserEmail = null;
                  let userId = null;

                  // Try to get user email from stored user data
                  if (userDataStr) {
                    try {
                      const userData = JSON.parse(userDataStr);
                      currentUserEmail = userData.email;
                      userId = userData.user_id;
                    } catch (e) {
                      console.error("Error parsing user data:", e);
                    }
                  }

                  // Fallback to login_email if no email in user data
                  if (!currentUserEmail && loginEmail) {
                    currentUserEmail = loginEmail;
                  }

                  // Check if current user owns this property
                  const canEdit =
                    currentUserEmail &&
                    property?.email &&
                    currentUserEmail.toLowerCase() ===
                      property.email.toLowerCase();

                  if (!canEdit) return null;

                  return (
                    <button
                      onClick={() =>
                        navigate(`/list-property?edit=${property.id}`)
                      }
                      className="px-3 py-1 rounded-full text-sm font-medium bg-blue-100 text-blue-800 hover:bg-blue-200 transition-colors flex items-center gap-1"
                    >
                      <svg
                        className="w-3 h-3"
                        fill="none"
                        stroke="currentColor"
                        viewBox="0 0 24 24"
                      >
                        <path
                          strokeLinecap="round"
                          strokeLinejoin="round"
                          strokeWidth={2}
                          d="M11 5H6a2 2 0 00-2 2v11a2 2 0 002 2h11a2 2 0 002-2v-5m-1.414-9.414a2 2 0 112.828 2.828L11.828 15H9v-2.828l8.586-8.586z"
                        />
                      </svg>
                      Edit Property
                    </button>
                  );
                })()}

                {/* RERA Verification Badge */}
                {property?.rera_verified && (
                  <span className="px-3 py-1 rounded-full text-sm font-medium bg-green-100 text-green-800 flex items-center">
                    <svg
                      className="w-4 h-4 mr-1"
                      fill="currentColor"
                      viewBox="0 0 20 20"
                    >
                      <path
                        fillRule="evenodd"
                        d="M10 18a8 8 0 100-16 8 8 0 000 16zm3.707-9.293a1 1 0 00-1.414-1.414L9 10.586 7.707 9.293a1 1 0 00-1.414 1.414l2 2a1 1 0 001.414 0l4-4z"
                        clipRule="evenodd"
                      />
                    </svg>
                    RERA Verified
                  </span>
                )}
              </div>

              {/* Bidding Information */}
              {property?.listing_type === "bidding" && (
                <div className="bg-gradient-to-r from-blue-50 to-indigo-50 border border-blue-200 rounded-lg p-6 mb-4">
                  {/* Winner Announcement for Completed Auctions */}
                  {auctionStatus?.session?.session_status === "completed" && (
                    <div className="bg-gradient-to-r from-green-50 to-emerald-50 border-2 border-green-200 rounded-xl p-6 mb-6 shadow-lg">
                      <div className="text-center">
                        <div className="text-5xl mb-4 animate-bounce">🏆</div>
                        <h2 className="text-3xl font-bold text-green-800 mb-3">
                          Auction Completed!
                        </h2>
                        <div className="text-xl text-green-700 mb-4">
                          <span className="font-semibold">SOLD</span> to{" "}
                          <span className="font-bold text-2xl">
                            {auctionStatus?.winner_name ||
                              auctionStatus?.current_highest_bid?.full_name ||
                              "Winner"}
                          </span>
                        </div>
                        <div className="text-4xl font-bold text-green-600 mb-2">
                          {formatPrice(
                            auctionStatus?.current_highest_bid?.bid_amount ||
                              auctionStatus?.session?.current_highest_bid ||
                              0,
                          )}
                        </div>
                        <p className="text-sm text-green-600 font-medium">
                          Final winning bid amount
                        </p>
                        <div className="mt-4 pt-4 border-t border-green-200">
                          <p className="text-sm text-green-700">
                            Auction ended on{" "}
                            {auctionStatus?.session?.actual_end_time
                              ? new Date(
                                  auctionStatus.session.actual_end_time,
                                ).toLocaleString()
                              : "N/A"}
                          </p>
                        </div>
                      </div>
                    </div>
                  )}

                  {/* Live Auction Header - Only show if not completed */}
                  {auctionStatus?.session?.session_status !== "completed" && (
                    <div className="flex items-center justify-between mb-4">
                      <h3 className="font-bold text-blue-900 text-lg flex items-center">
                        <svg
                          className="w-6 h-6 mr-2 text-blue-600"
                          fill="currentColor"
                          viewBox="0 0 20 20"
                        >
                          <path
                            fillRule="evenodd"
                            d="M12 7a1 1 0 110-2h5a1 1 0 011 1v5a1 1 0 11-2 0V8.414l-4.293 4.293a1 1 0 01-1.414 0L8 10.414l-4.293 4.293a1 1 0 01-1.414-1.414l5-5a1 1 0 011.414 0L11 10.586 14.586 7H12z"
                            clipRule="evenodd"
                          />
                        </svg>
                        Live Auction Details
                      </h3>
                      <span className="bg-blue-600 text-white px-3 py-1 rounded-full text-sm font-medium animate-pulse">
                        🔴 LIVE
                      </span>
                    </div>
                  )}

                  {/* Auction Timeline */}
                  <div className="bg-white rounded-lg p-4 mb-4 border border-blue-100">
                    <h4 className="font-semibold text-blue-900 mb-3 flex items-center">
                      <svg
                        className="w-5 h-5 mr-2"
                        fill="currentColor"
                        viewBox="0 0 20 20"
                      >
                        <path
                          fillRule="evenodd"
                          d="M10 18a8 8 0 100-16 8 8 0 000 16zm1-12a1 1 0 10-2 0v4a1 1 0 00.293.707l2.828 2.829a1 1 0 101.415-1.415L11 9.586V6z"
                          clipRule="evenodd"
                        />
                      </svg>
                      Auction Timeline
                    </h4>
                    <div className="space-y-2 text-sm">
                      {property.auction_start_date && (
                        <div className="flex justify-between">
                          <span className="text-blue-700 font-medium">
                            Started:
                          </span>
                          <span className="text-gray-900">
                            {new Date(
                              `${property.auction_start_date}T${property.auction_start_time || "00:00:00"}`,
                            ).toLocaleString()}
                          </span>
                        </div>
                      )}
                      {property.auction_end_date && (
                        <div className="flex justify-between">
                          <span className="text-red-700 font-medium">
                            Ends:
                          </span>
                          <span className="text-gray-900 font-semibold">
                            {new Date(
                              `${property.auction_end_date}T${property.auction_end_time || "23:59:59"}`,
                            ).toLocaleString()}
                          </span>
                        </div>
                      )}
                      {property.timezone && (
                        <div className="flex justify-between">
                          <span className="text-blue-700 font-medium">
                            Timezone:
                          </span>
                          <span className="text-gray-900">
                            {property.timezone}
                          </span>
                        </div>
                      )}
                    </div>
                  </div>

                  {/* Bidding Rules */}
                  <div className="grid grid-cols-1 md:grid-cols-2 gap-4 text-sm">
                    {property.starting_bidding_price && (
                      <div className="bg-white p-3 rounded-lg border border-blue-100">
                        <span className="text-blue-700 font-medium block">
                          Starting Price:
                        </span>
                        <span className="font-bold text-lg text-blue-900">
                          {formatPrice(
                            property.starting_bidding_price.toString(),
                          )}
                        </span>
                      </div>
                    )}
                    {property.reserve_price && (
                      <div className="bg-white p-3 rounded-lg border border-blue-100">
                        <span className="text-blue-700 font-medium block">
                          Reserve Price:
                        </span>
                        <span className="font-bold text-lg text-blue-900">
                          {formatPrice(property.reserve_price.toString())}
                        </span>
                      </div>
                    )}
                    {property.minimum_increment && (
                      <div className="bg-white p-3 rounded-lg border border-blue-100">
                        <span className="text-blue-700 font-medium block">
                          Min. Increment:
                        </span>
                        <span className="font-bold text-lg text-blue-900">
                          {formatPrice(property.minimum_increment.toString())}
                        </span>
                      </div>
                    )}
                    {property.bidder_registration_fee && (
                      <div className="bg-white p-3 rounded-lg border border-blue-100">
                        <span className="text-blue-700 font-medium block">
                          Registration Fee:
                        </span>
                        <span className="font-bold text-lg text-blue-900">
                          {formatPrice(
                            property.bidder_registration_fee.toString(),
                          )}
                        </span>
                      </div>
                    )}
                  </div>

                  {/* Additional Auction Info */}
                  <div className="mt-4 grid grid-cols-1 md:grid-cols-2 gap-4 text-sm">
                    {property.auction_eligibility && (
                      <div>
                        <span className="text-blue-700 font-medium">
                          Eligibility:
                        </span>
                        <span className="ml-2 capitalize bg-blue-100 px-2 py-1 rounded text-blue-800">
                          {property.auction_eligibility.replace("-", " ")}
                        </span>
                      </div>
                    )}
                    {property.deposit_requirement && (
                      <div>
                        <span className="text-blue-700 font-medium">
                          Deposit Required:
                        </span>
                        <span className="ml-2 font-semibold">
                          {formatPrice(property.deposit_requirement.toString())}
                        </span>
                      </div>
                    )}
                  </div>

                  {/* Bidding Activity from Backend */}
                  {property.bidding_history &&
                    property.bidding_history.length > 0 && (
                      <div className="mt-4 bg-white rounded-lg p-4 border border-blue-100">
                        <h4 className="font-semibold text-blue-900 mb-3 flex items-center">
                          <svg
                            className="w-5 h-5 mr-2"
                            fill="currentColor"
                            viewBox="0 0 20 20"
                          >
                            <path d="M9 12l2 2 4-4m6 2a9 9 0 11-18 0 9 9 0 0118 0z" />
                          </svg>
                          Recent Bidding Activity
                        </h4>
                        <div className="space-y-2 text-sm max-h-32 overflow-y-auto">
                          {property.bidding_history
                            .slice(0, 5)
                            .map((bid, index) => (
                              <div
                                key={index}
                                className="flex justify-between items-center py-1 border-b border-gray-100"
                              >
                                <span className="text-gray-600">
                                  {bid.bidder_name ||
                                    `Bidder #${bid.bidder_id}`}
                                </span>
                                <span
                                  className={`font-semibold ${
                                    index === 0
                                      ? "text-green-600"
                                      : "text-blue-600"
                                  }`}
                                >
                                  {formatPrice(bid.bid_amount)}
                                </span>
                                <span className="text-xs text-gray-500">
                                  {bid.bid_time}
                                </span>
                              </div>
                            ))}
                        </div>
                      </div>
                    )}
                </div>
              )}

              {/* Urgent Sale Information */}
              {property?.listing_type === "urgent-sale" && (
                <div className="bg-gradient-to-r from-red-50 to-orange-50 border border-red-200 rounded-lg p-6 mb-4">
                  <div className="flex items-center justify-between mb-4">
                    <h3 className="font-bold text-red-900 text-lg flex items-center">
                      <svg
                        className="w-6 h-6 mr-2 text-red-600"
                        fill="currentColor"
                        viewBox="0 0 20 20"
                      >
                        <path
                          fillRule="evenodd"
                          d="M12.395 2.553a1 1 0 00-1.45-.385c-.345.23-.614.558-.822.88-.214.33-.403.713-.57 1.116-.334.804-.614 1.768-.84 2.734a31.365 31.365 0 00-.613 3.58 2.64 2.64 0 01-.945-1.067c-.328-.68-.398-1.534-.398-2.654A1 1 0 005.05 6.05 6.981 6.981 0 003 11a7 7 0 1011.95-4.95c-.592-.591-.98-.985-1.348-1.467-.363-.476-.724-1.063-1.207-2.03zM12.12 15.12A3 3 0 017 13s.879.5 2.5.5c0-1 .5-4 1.25-4.5.5 1 .786 1.293 1.371 1.879A2.99 2.99 0 0113 13a2.99 2.99 0 01-.879 2.121z"
                          clipRule="evenodd"
                        />
                      </svg>
                      Urgent Sale - Limited Time
                    </h3>
                    <span className="bg-red-600 text-white px-3 py-1 rounded-full text-sm font-medium animate-pulse">
                      🔥 URGENT
                    </span>
                  </div>

                  {/* Sale Timeline */}
                  <div className="bg-white rounded-lg p-4 mb-4 border border-red-100">
                    <h4 className="font-semibold text-red-900 mb-3 flex items-center">
                      <svg
                        className="w-5 h-5 mr-2"
                        fill="currentColor"
                        viewBox="0 0 20 20"
                      >
                        <path
                          fillRule="evenodd"
                          d="M10 18a8 8 0 100-16 8 8 0 000 16zm1-12a1 1 0 10-2 0v4a1 1 0 00.293.707l2.828 2.829a1 1 0 101.415-1.415L11 9.586V6z"
                          clipRule="evenodd"
                        />
                      </svg>
                      Sale Timeline
                    </h4>
                    <div className="space-y-2 text-sm">
                      {property.urgent_sale_start_date && (
                        <div className="flex justify-between">
                          <span className="text-red-700 font-medium">
                            Sale Started:
                          </span>
                          <span className="text-gray-900">
                            {new Date(
                              `${property.urgent_sale_start_date}T${property.urgent_sale_start_time || "00:00:00"}`,
                            ).toLocaleString()}
                          </span>
                        </div>
                      )}
                      {property.urgent_sale_end_date && (
                        <div className="flex justify-between">
                          <span className="text-red-700 font-medium">
                            Sale Ends:
                          </span>
                          <span className="text-gray-900 font-semibold">
                            {new Date(
                              `${property.urgent_sale_end_date}T${property.urgent_sale_end_time || "23:59:59"}`,
                            ).toLocaleString()}
                          </span>
                        </div>
                      )}
                      {property.timezone && (
                        <div className="flex justify-between">
                          <span className="text-red-700 font-medium">
                            Timezone:
                          </span>
                          <span className="text-gray-900">
                            {property.timezone}
                          </span>
                        </div>
                      )}
                    </div>
                  </div>

                  {/* Price Comparison */}
                  <div className="grid grid-cols-1 md:grid-cols-2 gap-4 mb-4">
                    {property.actual_price && (
                      <div className="bg-white p-4 rounded-lg border border-red-100">
                        <span className="text-red-700 font-medium block mb-1">
                          Original Price:
                        </span>
                        <span className="font-bold text-lg text-gray-500 line-through">
                          {formatPrice(property.actual_price.toString())}
                        </span>
                      </div>
                    )}
                    <div className="bg-white p-4 rounded-lg border border-red-100">
                      <span className="text-red-700 font-medium block mb-1">
                        Urgent Sale Price:
                      </span>
                      <span className="font-bold text-xl text-red-600">
                        {formatPrice(property.sale_price || 0)}
                      </span>
                      {property.actual_price && property.sale_price && (
                        <div className="text-sm text-green-600 font-medium mt-1">
                          {property.savings_amount
                            ? `Save ${formatPrice(property.savings_amount)}`
                            : `Save ₹${((parseFloat(property.actual_price.toString()) - parseFloat(property.sale_price)) / 100000).toFixed(1)}L`}
                          {property.discount_percentage
                            ? ` (${property.discount_percentage}% OFF)`
                            : ` (${(((parseFloat(property.actual_price.toString()) - parseFloat(property.sale_price)) / parseFloat(property.actual_price.toString())) * 100).toFixed(0)}% OFF)`}
                        </div>
                      )}
                    </div>
                  </div>

                  {/* Urgency Reasons from Backend */}
                  {property.urgency_reason && (
                    <div className="bg-white rounded-lg p-4 border border-red-100">
                      <h4 className="font-semibold text-red-900 mb-3 flex items-center">
                        <svg
                          className="w-5 h-5 mr-2"
                          fill="currentColor"
                          viewBox="0 0 20 20"
                        >
                          <path
                            fillRule="evenodd"
                            d="M18 10a8 8 0 11-16 0 8 8 0 0116 0zm-7-4a1 1 0 11-2 0 1 1 0 012 0zM9 9a1 1 0 000 2v3a1 1 0 001 1h1a1 1 0 100-2v-3a1 1 0 00-1-1H9z"
                            clipRule="evenodd"
                          />
                        </svg>
                        Why This Urgent Sale?
                      </h4>
                      <div className="text-sm text-red-700 space-y-2">
                        {property.urgency_reasons &&
                        property.urgency_reasons.length > 0 ? (
                          property.urgency_reasons.map((reason, index) => (
                            <div
                              key={index}
                              className="flex items-start space-x-2"
                            >
                              <span className="w-2 h-2 bg-red-500 rounded-full mt-2 flex-shrink-0"></span>
                              <span>{reason}</span>
                            </div>
                          ))
                        ) : (
                          <div className="flex items-start space-x-2">
                            <span className="w-2 h-2 bg-red-500 rounded-full mt-2 flex-shrink-0"></span>
                            <span>{property.urgency_reason}</span>
                          </div>
                        )}
                      </div>
                    </div>
                  )}

                  {/* Call to Action with Backend Data */}
                  <div className="mt-4 bg-gradient-to-r from-red-600 to-orange-600 text-white p-4 rounded-lg">
                    <div className="flex items-center justify-between">
                      <div>
                        <p className="font-semibold">
                          ⚡{" "}
                          {property.call_to_action_title ||
                            "Act Fast - Limited Time Offer!"}
                        </p>
                        <p className="text-sm opacity-90">
                          {property.call_to_action_message ||
                            "This deal won't last long. Contact us immediately!"}
                        </p>
                      </div>
                      <div className="text-right">
                        <div className="text-2xl font-bold">{timeLeft}</div>
                        <div className="text-xs opacity-90">Time Left</div>
                      </div>
                    </div>
                  </div>
                </div>
              )}
            </div>

            {/* Verified Visit Badge Section */}
            <div className="bg-gradient-to-br from-blue-600 to-indigo-700 rounded-xl p-6 mb-6 shadow-lg">
              <div className="flex items-start justify-between">
                <div className="flex-1">
                  <div className="flex items-center gap-2 mb-3">
                    <svg
                      className="w-6 h-6 text-yellow-400"
                      fill="currentColor"
                      viewBox="0 0 20 20"
                    >
                      <path d="M9.049 2.927c.3-.921 1.603-.921 1.902 0l1.07 3.292a1 1 0 00.95.69h3.462c.969 0 1.371 1.24.588 1.81l-2.8 2.034a1 1 0 00-.364 1.118l1.07 3.292c.3.921-.755 1.688-1.54 1.118l-2.8-2.034a1 1 0 00-1.175 0l-2.8 2.034c-.784.57-1.838-.197-1.539-1.118l1.07-3.292a1 1 0 00-.364-1.118L2.98 8.72c-.783-.57-.38-1.81.588-1.81h3.461a1 1 0 00.951-.69l1.07-3.292z" />
                    </svg>
                    <h3 className="text-xl font-bold text-gray-900">
                      Earn Verified Visit Badge
                    </h3>
                  </div>
                  <p className="text-gray-900 text-sm mb-2 font-semibold">
                    ✓ 45 people have visited this property
                  </p>
                  <p className="text-blue-100 text-sm mb-4">
                    Schedule a visit and earn a verified badge when you check-in
                    at the property!
                  </p>
                  <button
                    onClick={() => {
                      const user = JSON.parse(
                        localStorage.getItem("user") || "{}",
                      );
                      const userEmail = user.email;
                      const isOwner =
                        userEmail &&
                        property?.email &&
                        userEmail.toLowerCase() ===
                          property.email.toLowerCase();

                      if (isOwner) {
                        alert("You cannot earn a badge for your own property.");
                        return;
                      }

                      navigate("/user/dashboard?tab=badges");
                    }}
                    className="bg-white hover:bg-gray-100 text-blue-700 px-6 py-3 rounded-lg font-semibold transition-colors flex items-center gap-2 shadow-md"
                  >
                    <svg
                      className="w-5 h-5"
                      fill="currentColor"
                      viewBox="0 0 20 20"
                    >
                      <path
                        fillRule="evenodd"
                        d="M6.267 3.455a3.066 3.066 0 001.745-.723 3.066 3.066 0 013.976 0 3.066 3.066 0 001.745.723 3.066 3.066 0 012.812 2.812c.051.643.304 1.254.723 1.745a3.066 3.066 0 010 3.976 3.066 3.066 0 00-.723 1.745 3.066 3.066 0 01-2.812 2.812 3.066 3.066 0 00-1.745.723 3.066 3.066 0 01-3.976 0 3.066 3.066 0 00-1.745-.723 3.066 3.066 0 01-2.812-2.812 3.066 3.066 0 00-.723-1.745 3.066 3.066 0 010-3.976 3.066 3.066 0 00.723-1.745 3.066 3.066 0 012.812-2.812zm7.44 5.252a1 1 0 00-1.414-1.414L9 10.586 7.707 9.293a1 1 0 00-1.414 1.414l2 2a1 1 0 001.414 0l4-4z"
                        clipRule="evenodd"
                      />
                    </svg>
                    Earn Badge Now
                  </button>
                </div>
                <div className="ml-4">
                  <div className="w-24 h-24 bg-white/10 rounded-lg flex items-center justify-center backdrop-blur-sm">
                    <svg
                      className="w-16 h-16 text-yellow-400"
                      fill="currentColor"
                      viewBox="0 0 20 20"
                    >
                      <path
                        fillRule="evenodd"
                        d="M6.267 3.455a3.066 3.066 0 001.745-.723 3.066 3.066 0 013.976 0 3.066 3.066 0 001.745.723 3.066 3.066 0 012.812 2.812c.051.643.304 1.254.723 1.745a3.066 3.066 0 010 3.976 3.066 3.066 0 00-.723 1.745 3.066 3.066 0 01-2.812 2.812 3.066 3.066 0 00-1.745.723 3.066 3.066 0 01-3.976 0 3.066 3.066 0 00-1.745-.723 3.066 3.066 0 01-2.812-2.812 3.066 3.066 0 00-.723-1.745 3.066 3.066 0 010-3.976 3.066 3.066 0 00.723-1.745 3.066 3.066 0 012.812-2.812zm7.44 5.252a1 1 0 00-1.414-1.414L9 10.586 7.707 9.293a1 1 0 00-1.414 1.414l2 2a1 1 0 001.414 0l4-4z"
                        clipRule="evenodd"
                      />
                    </svg>
                  </div>
                </div>
              </div>
            </div>

            {/* Property Assistant Button */}
            <div className="mb-6">
              <button
                onClick={() => setShowCharacterModal(true)}
                style={{ backgroundColor: "#2563eb", color: "#ffffff" }}
                className="flex items-center space-x-1 px-4 py-2 rounded-lg hover:bg-blue-700 transition-colors shadow-sm font-medium text-sm disabled:bg-blue-400"
              >
                <div
                  className="w-8 h-8 bg-white/20 rounded-full flex items-center justify-center"
                  style={{
                    minWidth: "16px",
                    maxWidth: "16px",
                    minHeight: "16px",
                    maxHeight: "16px",
                  }}
                >
                  <img
                    src="/450.svg"
                    alt="Assistant Character"
                    className="voice-assistant-button-icon"
                  />
                </div>
                <span>Ask Property Assistant</span>
                <svg
                  className="w-4 h-4"
                  fill="none"
                  stroke="currentColor"
                  viewBox="0 0 24 24"
                >
                  <path
                    strokeLinecap="round"
                    strokeLinejoin="round"
                    strokeWidth={2}
                    d="M8.228 9c.549-1.165 2.03-2 3.772-2 2.21 0 4 1.343 4 3 0 1.4-1.278 2.575-3.006 2.907-.542.104-.994.54-.994 1.093m0 3h.01M21 12a9 9 0 11-18 0 9 9 0 0118 0z"
                  />
                </svg>
              </button>
            </div>

            {/* Enhanced Tabs */}
            <div className="border-b border-gray-200 mb-6">
              <nav className="flex space-x-1 bg-gray-50 p-1 rounded-lg">
                <button
                  onClick={() => setActiveTab("overview")}
                  className={`flex items-center space-x-2 px-4 py-2 rounded-md font-medium text-sm transition-all ${
                    activeTab === "overview"
                      ? "bg-white text-blue-600 shadow-sm"
                      : "text-gray-600 hover:text-gray-900 hover:bg-white/50"
                  }`}
                >
                  <svg
                    className="w-4 h-4"
                    fill="currentColor"
                    viewBox="0 0 20 20"
                  >
                    <path d="M3 4a1 1 0 011-1h12a1 1 0 011 1v2a1 1 0 01-1 1H4a1 1 0 01-1-1V4zM3 10a1 1 0 011-1h6a1 1 0 011 1v6a1 1 0 01-1 1H4a1 1 0 01-1-1v-6zM14 9a1 1 0 00-1 1v6a1 1 0 001 1h2a1 1 0 001-1v-6a1 1 0 00-1-1h-2z" />
                  </svg>
                  <span>Overview</span>
                </button>

                <button
                  onClick={() => setActiveTab("media")}
                  className={`flex items-center space-x-2 px-4 py-2 rounded-md font-medium text-sm transition-all ${
                    activeTab === "media"
                      ? "bg-white text-blue-600 shadow-sm"
                      : "text-gray-600 hover:text-gray-900 hover:bg-white/50"
                  }`}
                >
                  <svg
                    className="w-4 h-4"
                    fill="currentColor"
                    viewBox="0 0 20 20"
                  >
                    <path
                      fillRule="evenodd"
                      d="M4 3a2 2 0 00-2 2v10a2 2 0 002 2h12a2 2 0 002-2V5a2 2 0 00-2-2H4zm12 12H4l4-8 3 6 2-4 3 6z"
                      clipRule="evenodd"
                    />
                  </svg>
                  <span>Media</span>
                  <span className="bg-blue-100 text-blue-800 text-xs px-2 py-0.5 rounded-full">
                    {allImages.length + allVideos.length}
                  </span>
                </button>

                <button
                  onClick={() => setActiveTab("amenities")}
                  className={`flex items-center space-x-2 px-4 py-2 rounded-md font-medium text-sm transition-all ${
                    activeTab === "amenities"
                      ? "bg-white text-blue-600 shadow-sm"
                      : "text-gray-600 hover:text-gray-900 hover:bg-white/50"
                  }`}
                >
                  <svg
                    className="w-4 h-4"
                    fill="currentColor"
                    viewBox="0 0 20 20"
                  >
                    <path
                      fillRule="evenodd"
                      d="M3 6a3 3 0 013-3h10a1 1 0 01.8 1.6L14.25 8l2.55 3.4A1 1 0 0116 13H6a1 1 0 00-1 1v3a1 1 0 11-2 0V6z"
                      clipRule="evenodd"
                    />
                  </svg>
                  <span>Amenities</span>
                </button>

                <button
                  onClick={() => setActiveTab("nearby-facilities")}
                  className={`flex items-center space-x-2 px-4 py-2 rounded-md font-medium text-sm transition-all ${
                    activeTab === "nearby-facilities"
                      ? "bg-white text-blue-600 shadow-sm"
                      : "text-gray-600 hover:text-gray-900 hover:bg-white/50"
                  }`}
                >
                  <svg
                    className="w-4 h-4"
                    fill="currentColor"
                    viewBox="0 0 20 20"
                  >
                    <path
                      fillRule="evenodd"
                      d="M5.05 4.05a7 7 0 119.9 9.9L10 18.9l-4.95-4.95a7 7 0 010-9.9zM10 11a2 2 0 100-4 2 2 0 000 4z"
                      clipRule="evenodd"
                    />
                  </svg>
                  <span>Nearby Facilities</span>
                </button>

                <button
                  onClick={() => setActiveTab("location")}
                  className={`flex items-center space-x-2 px-4 py-2 rounded-md font-medium text-sm transition-all ${
                    activeTab === "location"
                      ? "bg-white text-blue-600 shadow-sm"
                      : "text-gray-600 hover:text-gray-900 hover:bg-white/50"
                  }`}
                >
                  <svg
                    className="w-4 h-4"
                    fill="currentColor"
                    viewBox="0 0 20 20"
                  >
                    <path
                      fillRule="evenodd"
                      d="M5.05 4.05a7 7 0 119.9 9.9L10 18.9l-4.95-4.95a7 7 0 010-9.9zM10 11a2 2 0 100-4 2 2 0 000 4z"
                      clipRule="evenodd"
                    />
                  </svg>
                  <span>Location</span>
                </button>

                <button
                  onClick={() => setActiveTab("documents")}
                  className={`flex items-center space-x-2 px-4 py-2 rounded-md font-medium text-sm transition-all ${
                    activeTab === "documents"
                      ? "bg-white text-blue-600 shadow-sm"
                      : "text-gray-600 hover:text-gray-900 hover:bg-white/50"
                  }`}
                >
                  <svg
                    className="w-4 h-4"
                    fill="currentColor"
                    viewBox="0 0 20 20"
                  >
                    <path
                      fillRule="evenodd"
                      d="M4 4a2 2 0 012-2h4.586A2 2 0 0112 2.586L15.414 6A2 2 0 0116 7.414V16a2 2 0 01-2 2H6a2 2 0 01-2-2V4zm2 6a1 1 0 011-1h6a1 1 0 110 2H7a1 1 0 01-1-1zm1 3a1 1 0 100 2h6a1 1 0 100-2H7z"
                      clipRule="evenodd"
                    />
                  </svg>
                  <span>Documents</span>
                </button>

                <button
                  onClick={() => setActiveTab("school-distance")}
                  className={`flex items-center space-x-2 px-4 py-2 rounded-md font-medium text-sm transition-all ${
                    activeTab === "school-distance"
                      ? "bg-white text-blue-600 shadow-sm"
                      : "text-gray-600 hover:text-gray-900 hover:bg-white/50"
                  }`}
                >
                  <svg
                    className="w-4 h-4"
                    fill="currentColor"
                    viewBox="0 0 20 20"
                  >
                    <path d="M10.394 2.08a1 1 0 00-.788 0l-7 3a1 1 0 000 1.84L5.25 8.051a.999.999 0 01.356-.257l4-1.714a1 1 0 11.788 1.838L7.667 9.088l1.94.831a1 1 0 00.787 0l7-3a1 1 0 000-1.838l-7-3zM3.31 9.397L5 10.12v4.102a8.969 8.969 0 00-1.05-.174 1 1 0 01-.89-.89 11.115 11.115 0 01.25-3.762zM9.3 16.573A9.026 9.026 0 007 14.935v-3.957l1.818.78a3 3 0 002.364 0l5.508-2.361a11.026 11.026 0 01.25 3.762 1 1 0 01-.89.89 8.968 8.968 0 00-5.35 2.524 1 1 0 01-1.4 0zM6 18a1 1 0 001-1v-2.065a8.935 8.935 0 00-2-.712V17a1 1 0 001 1z" />
                  </svg>
                  <span>School Distance</span>
                </button>

                <button
                  onClick={() => setActiveTab("safety")}
                  className={`flex items-center space-x-2 px-4 py-2 rounded-md font-medium text-sm transition-all ${
                    activeTab === "safety"
                      ? "bg-white text-blue-600 shadow-sm"
                      : "text-gray-600 hover:text-gray-900 hover:bg-white/50"
                  }`}
                >
                  <svg
                    className="w-4 h-4"
                    fill="currentColor"
                    viewBox="0 0 20 20"
                  >
                    <path
                      fillRule="evenodd"
                      d="M2.166 4.999A11.954 11.954 0 0010 1.944 11.954 11.954 0 0017.834 5c.11.65.166 1.32.166 2.001 0 5.225-3.34 9.67-8 11.317C5.34 16.67 2 12.225 2 7c0-.682.057-1.35.166-2.001zm11.541 3.708a1 1 0 00-1.414-1.414L9 10.586 7.707 9.293a1 1 0 00-1.414 1.414l2 2a1 1 0 001.414 0l4-4z"
                      clipRule="evenodd"
                    />
                  </svg>
                  <span>Safety & Crime</span>
                </button>
              </nav>
            </div>

            {/* Tab Content */}
            {activeTab === "overview" && (
              <div>
                <h2 className="text-xl font-bold mb-4">Property Details</h2>
                <div className="grid grid-cols-2 md:grid-cols-4 gap-4 mb-6">
                  <div className="text-center p-4 bg-gray-50 rounded-lg">
                    <div className="text-2xl font-bold text-blue-600">
                      {property?.bedrooms || "N/A"}
                    </div>
                    <div className="text-sm text-gray-600">BHK</div>
                  </div>
                  <div className="text-center p-4 bg-gray-50 rounded-lg">
                    <div className="text-2xl font-bold text-blue-600">
                      {property?.bathrooms || "N/A"}
                    </div>
                    <div className="text-sm text-gray-600">Bathrooms</div>
                  </div>
                  <div className="text-center p-4 bg-gray-50 rounded-lg">
                    <div className="text-2xl font-bold text-blue-600">
                      {property?.built_up_area ||
                        property?.carpet_area ||
                        "N/A"}
                    </div>
                    <div className="text-sm text-gray-600">sq.ft</div>
                  </div>
                  <div className="text-center p-4 bg-gray-50 rounded-lg">
                    <div className="text-2xl font-bold text-blue-600 capitalize">
                      {property?.property_type || "Property"}
                    </div>
                    <div className="text-sm text-gray-600">Type</div>
                  </div>
                </div>
                <div className="flex flex-wrap gap-4 mb-6">
                  {property?.listing_type === "bidding" ? (
                    <>
                      {/* Only show bidding buttons if auction is not completed */}
                      {auctionStatus?.session?.session_status !==
                      "completed" ? (
                        <>
                          <button
                            onClick={() => {
                              // Navigate to bidding interface
                              const accessToken =
                                localStorage.getItem("access_token");
                              if (!accessToken) {
                                alert("Please login to place bids");
                                navigate("/login");
                                return;
                              }

                              console.log("=== PLACE BID DEBUG ===");
                              console.log("isRegistered:", isRegistered);
                              console.log(
                                "registrationStatus:",
                                registrationStatus,
                              );
                              console.log("property.id:", property?.id);
                              console.log(
                                "property.listing_type:",
                                property?.listing_type,
                              );
                              console.log("=======================");

                              if (!isRegistered) {
                                alert("Please register for auction first");
                                setShowRegistrationModal(true);
                                return;
                              }
                              // Show bidding modal or navigate to bidding page
                              setShowBiddingModal(true);
                            }}
                            className="bg-blue-600 text-white px-6 py-3 rounded-lg flex items-center space-x-2 hover:bg-blue-700 transition-colors font-semibold shadow-lg"
                            style={{
                              backgroundColor: "#2563eb",
                              color: "white",
                            }}
                          >
                            <svg
                              className="w-5 h-5"
                              fill="none"
                              stroke="currentColor"
                              viewBox="0 0 24 24"
                            >
                              <path
                                strokeLinecap="round"
                                strokeLinejoin="round"
                                strokeWidth={2}
                                d="M13 10V3L4 14h7v7l9-11h-7z"
                              />
                            </svg>
                            <span>Place Bid</span>
                          </button>
                          <button
                            onClick={() => {
                              const accessToken =
                                localStorage.getItem("access_token");
                              if (!accessToken) {
                                alert("Please login to register for auction");
                                navigate("/login");
                                return;
                              }
                              if (!isRegistered) {
                                setShowRegistrationModal(true); // Show registration details modal first
                              } else {
                                window.location.reload();
                              }
                            }}
                            disabled={isRegistered}
                            className={`border px-6 py-2 rounded-lg flex items-center space-x-2 transition-colors ${
                              isRegistered
                                ? "border-green-300 bg-green-500 text-white cursor-not-allowed"
                                : "border-blue-300 bg-blue-500 text-white hover:bg-blue-600"
                            }`}
                          >
                            <svg
                              className="w-5 h-5"
                              fill="none"
                              stroke="currentColor"
                              viewBox="0 0 24 24"
                            >
                              <path
                                strokeLinecap="round"
                                strokeLinejoin="round"
                                strokeWidth={2}
                                d="M16 7a4 4 0 11-8 0 4 4 0 018 0zM12 14a7 7 0 00-7 7h14a7 7 0 00-7-7z"
                              />
                            </svg>
                            <span>
                              {isRegistered
                                ? "Registered ✓"
                                : "Register for Auction"}
                            </span>
                          </button>
                        </>
                      ) : (
                        /* Show auction completed message */
                        <div className="w-full bg-green-50 border-2 border-green-200 rounded-lg p-4">
                          <div className="text-center">
                            <h4 className="text-lg font-bold text-green-800 mb-2">
                              🎉 Auction Successfully Completed!
                            </h4>
                            <p className="text-green-700">
                              This property has been sold through auction.
                            </p>
                          </div>
                        </div>
                      )}
                    </>
                  ) : property?.listing_type === "urgent-sale" ? (
                    <>
                      <button
                        onClick={async () => {
                          const user = JSON.parse(
                            localStorage.getItem("user") || "{}",
                          );
                          if (!user.user_id) {
                            toast.error("Please login to initiate purchase");
                            navigate("/login");
                            return;
                          }

                          setCreatingAgreement(true);
                          try {
                            // Simulate agreement creation process
                            await new Promise((resolve) =>
                              setTimeout(resolve, 1500),
                            );

                            // Store purchase intent locally for now
                            const purchaseIntent = {
                              id: `purchase_${Date.now()}`,
                              property_id: property.id,
                              property_title: property.title,
                              property_address: `${property.address}, ${property.locality}, ${property.city}`,
                              sale_price: property.sale_price,
                              buyer_id: user.user_id,
                              buyer_name: user.name || user.email,
                              buyer_email: user.email,
                              seller_email: property.email,
                              status: "initiated",
                              created_at: new Date().toISOString(),
                            };

                            // Store in localStorage for now
                            const existingIntents = JSON.parse(
                              localStorage.getItem("purchase_intents") || "[]",
                            );
                            existingIntents.push(purchaseIntent);
                            localStorage.setItem(
                              "purchase_intents",
                              JSON.stringify(existingIntents),
                            );

                            toast.success(
                              "Purchase intent recorded successfully! You will be contacted by the seller soon.",
                            );
                            navigate("/user/dashboard");
                          } catch (error) {
                            console.error("Error:", error);
                            toast.error("Error initiating purchase");
                          } finally {
                            setCreatingAgreement(false);
                          }
                        }}
                        disabled={creatingAgreement}
                        className="bg-red-600 text-white px-6 py-2 rounded-lg flex items-center space-x-2 hover:bg-red-700 transition-colors disabled:opacity-50"
                      >
                        {creatingAgreement ? (
                          <div className="animate-spin rounded-full h-5 w-5 border-b-2 border-white"></div>
                        ) : (
                          <>
                            <svg
                              className="w-5 h-5"
                              fill="none"
                              stroke="currentColor"
                              viewBox="0 0 24 24"
                            >
                              <path
                                strokeLinecap="round"
                                strokeLinejoin="round"
                                strokeWidth={2}
                                d="M9 12h6m-6 4h6m2 5H7a2 2 0 01-2-2V5a2 2 0 012-2h5.586a1 1 0 01.707.293l5.414 5.414a1 1 0 01.293.707V19a2 2 0 01-2 2z"
                              />
                            </svg>
                            <span>Initiate Purchase</span>
                          </>
                        )}
                      </button>
                      <button
                        onClick={() =>
                          navigate(`/property/${id}/contact-agent`)
                        }
                        className="border border-red-300 text-red-700 px-6 py-2 rounded-lg flex items-center space-x-2 hover:bg-red-50 transition-colors"
                      >
                        <svg
                          className="w-5 h-5"
                          fill="none"
                          stroke="currentColor"
                          viewBox="0 0 24 24"
                        >
                          <path
                            strokeLinecap="round"
                            strokeLinejoin="round"
                            strokeWidth={2}
                            d="M3 5a2 2 0 012-2h3.28a1 1 0 01.948.684l1.498 4.493a1 1 0 01-.502 1.21l-2.257 1.13a11.042 11.042 0 005.516 5.516l1.13-2.257a1 1 0 011.21-.502l4.493 1.498a1 1 0 01.684.949V19a2 2 0 01-2 2h-1C9.716 21 3 14.284 3 6V5z"
                          />
                        </svg>
                        <span>Contact Agent/Owner</span>
                      </button>
                    </>
                  ) : (
                    <>
                      <button
                        onClick={async () => {
                          const user = JSON.parse(
                            localStorage.getItem("user") || "{}",
                          );
                          const userEmail = user.email;
                          const isOwner =
                            userEmail &&
                            property?.email &&
                            userEmail.toLowerCase() ===
                              property.email.toLowerCase();
                          if (isOwner) {
                            toast.error("You cannot buy your own property");
                            return;
                          }

                          if (!user.user_id) {
                            toast.error("Please login to initiate purchase");
                            navigate("/login");
                            return;
                          }

                          setCreatingAgreement(true);
                          try {
                            // Simulate agreement creation process
                            await new Promise((resolve) =>
                              setTimeout(resolve, 1500),
                            );

                            // Store purchase intent locally for now
                            const purchaseIntent = {
                              id: `purchase_${Date.now()}`,
                              property_id: property.id,
                              property_title: property.title,
                              property_address: `${property.address}, ${property.locality}, ${property.city}`,
                              sale_price: property.sale_price,
                              buyer_id: user.user_id,
                              buyer_name: user.name || user.email,
                              buyer_email: user.email,
                              seller_email: property.email,
                              status: "initiated",
                              created_at: new Date().toISOString(),
                            };

                            // Store in localStorage for now
                            const existingIntents = JSON.parse(
                              localStorage.getItem("purchase_intents") || "[]",
                            );
                            existingIntents.push(purchaseIntent);
                            localStorage.setItem(
                              "purchase_intents",
                              JSON.stringify(existingIntents),
                            );

                            toast.success(
                              "Purchase intent recorded successfully! You will be contacted by the seller soon.",
                            );
                            navigate("/user/dashboard");
                          } catch (error) {
                            console.error("Error:", error);
                            toast.error("Error initiating purchase");
                          } finally {
                            setCreatingAgreement(false);
                          }
                        }}
                        disabled={creatingAgreement}
                        className="px-6 py-3 rounded-lg flex items-center space-x-2 font-semibold transition-colors disabled:opacity-50"
                        style={{ backgroundColor: "#dc2626", color: "white" }}
                      >
                        {creatingAgreement ? (
                          <div className="animate-spin rounded-full h-5 w-5 border-b-2 border-white"></div>
                        ) : (
                          <>
                            <svg
                              className="w-5 h-5"
                              fill="none"
                              stroke="currentColor"
                              viewBox="0 0 24 24"
                            >
                              <path
                                strokeLinecap="round"
                                strokeLinejoin="round"
                                strokeWidth={2}
                                d="M9 12h6m-6 4h6m2 5H7a2 2 0 01-2-2V5a2 2 0 012-2h5.586a1 1 0 01.707.293l5.414 5.414a1 1 0 01.293.707V19a2 2 0 01-2 2z"
                              />
                            </svg>
                            <span>Initiate Purchase</span>
                          </>
                        )}
                      </button>
                      <button
                        onClick={() => {
                          const user = JSON.parse(
                            localStorage.getItem("user") || "{}",
                          );
                          const userEmail = user.email;
                          const isOwner =
                            userEmail &&
                            property?.email &&
                            userEmail.toLowerCase() ===
                              property.email.toLowerCase();

                          if (isOwner) {
                            toast.error(
                              "You cannot contact yourself for your own property",
                            );
                            return;
                          }
                          const accessToken =
                            localStorage.getItem("access_token");
                          if (!accessToken) {
                            toast.error(
                              "Please login to contact the property owner",
                            );
                            navigate("/login");
                            return;
                          }
                          navigate(`/property/${id}/contact-agent`);
                        }}
                        className="px-6 py-3 rounded-lg flex items-center space-x-2 font-semibold transition-colors"
                        style={{ backgroundColor: "#16a34a", color: "white" }}
                      >
                        <svg
                          className="w-5 h-5"
                          fill="none"
                          stroke="currentColor"
                          viewBox="0 0 24 24"
                        >
                          <path
                            strokeLinecap="round"
                            strokeLinejoin="round"
                            strokeWidth={2}
                            d="M3 5a2 2 0 012-2h3.28a1 1 0 01.948.684l1.498 4.493a1 1 0 01-.502 1.21l-2.257 1.13a11.042 11.042 0 005.516 5.516l1.13-2.257a1 1 0 011.21-.502l4.493 1.498a1 1 0 01.684.949V19a2 2 0 01-2 2h-1C9.716 21 3 14.284 3 6V5z"
                          />
                        </svg>
                        <span>Contact Agent/Owner</span>
                      </button>
                      <button
                        onClick={() => {
                          const user = JSON.parse(
                            localStorage.getItem("user") || "{}",
                          );
                          const userEmail = user.email;
                          const isOwner =
                            userEmail &&
                            property?.email &&
                            userEmail.toLowerCase() ===
                              property.email.toLowerCase();
                          if (isOwner) {
                            toast.error(
                              "You cannot schedule a visit for your own property",
                            );
                            return;
                          }
                          navigate(`/property/${id}/book-visit`);
                        }}
                        className="px-6 py-3 rounded-lg flex items-center space-x-2 font-semibold transition-colors"
                        style={{ backgroundColor: "#2563eb", color: "white" }}
                      >
                        <svg
                          className="w-5 h-5"
                          fill="none"
                          stroke="currentColor"
                          viewBox="0 0 24 24"
                        >
                          <path
                            strokeLinecap="round"
                            strokeLinejoin="round"
                            strokeWidth={2}
                            d="M8 7V3a1 1 0 011-1h6a1 1 0 011 1v4h3a1 1 0 011 1v9a1 1 0 01-1 1H5a1 1 0 01-1-1V8a1 1 0 011-1h3z"
                          />
                        </svg>
                        <span>Schedule a Tour</span>
                      </button>
                      <button
                        onClick={() => {
                          const user = JSON.parse(
                            localStorage.getItem("user") || "{}",
                          );
                          const userEmail = user.email;
                          const isOwner =
                            userEmail &&
                            property?.email &&
                            userEmail.toLowerCase() ===
                              property.email.toLowerCase();

                          if (isOwner) {
                            toast.error("Cannot Add Own Property", {
                              description:
                                "You cannot add your own property to itinerary",
                            });
                            return;
                          }

                          const itineraries = JSON.parse(
                            localStorage.getItem("user_itineraries") || "[]",
                          );
                          const isAlreadyAdded = itineraries.some(
                            (item: any) => item.id === property.id,
                          );

                          if (isAlreadyAdded) {
                            toast.error("Already in Itinerary", {
                              description:
                                "This property is already in your itinerary",
                            });
                            return;
                          }

                          itineraries.push({
                            id: property.id,
                            title: property.title,
                            address: property.address,
                            city: property.city,
                            photo: property.photo || property.primary_image,
                            sale_price: property.sale_price,
                            bedrooms: property.bedrooms,
                            built_up_area: property.built_up_area,
                            addedAt: new Date().toISOString(),
                          });
                          localStorage.setItem(
                            "user_itineraries",
                            JSON.stringify(itineraries),
                          );

                          toast.success("Added to Itinerary", {
                            description: "Property added to your itinerary",
                          });
                        }}
                        className="px-6 py-3 rounded-lg flex items-center space-x-2 font-semibold transition-colors"
                        style={{ backgroundColor: "#f59e0b", color: "white" }}
                      >
                        <svg
                          className="w-5 h-5"
                          fill="none"
                          stroke="currentColor"
                          viewBox="0 0 24 24"
                        >
                          <path
                            strokeLinecap="round"
                            strokeLinejoin="round"
                            strokeWidth={2}
                            d="M9 5H7a2 2 0 00-2 2v12a2 2 0 002 2h10a2 2 0 002-2V7a2 2 0 00-2-2h-2M9 5a2 2 0 002 2h2a2 2 0 002-2M9 5a2 2 0 012-2h2a2 2 0 012 2m-6 9l2 2 4-4"
                          />
                        </svg>
                        <span>Add to Itinerary</span>
                      </button>
                      <button
                        onClick={() => {
                          toast.success("Token booking initiated");
                        }}
                        className="px-6 py-3 rounded-lg flex items-center space-x-2 font-semibold transition-colors"
                        style={{ backgroundColor: "#8b5cf6", color: "white" }}
                      >
                        <svg
                          className="w-5 h-5"
                          fill="none"
                          stroke="currentColor"
                          viewBox="0 0 24 24"
                        >
                          <path
                            strokeLinecap="round"
                            strokeLinejoin="round"
                            strokeWidth={2}
                            d="M12 8c-1.657 0-3 .895-3 2s1.343 2 3 2 3 .895 3 2-1.343 2-3 2m0-8c1.11 0 2.08.402 2.599 1M12 8V7m0 1v8m0 0v1m0-1c-1.11 0-2.08-.402-2.599-1"
                          />
                        </svg>
                        <span>Book Token</span>
                      </button>
                      <button
                        onClick={async () => {
                          const user = JSON.parse(
                            localStorage.getItem("user") || "{}",
                          );
                          const userEmail = user.email;
                          const isOwner =
                            userEmail &&
                            property?.email &&
                            userEmail.toLowerCase() ===
                              property.email.toLowerCase();

                          if (isOwner) {
                            toast.error(
                              "You cannot negotiate on your own property",
                            );
                            return;
                          }

                          if (!property?.price_negotiable) {
                            toast.error(
                              "This property is not open for negotiation",
                            );
                            return;
                          }

                          const accessToken =
                            localStorage.getItem("access_token");
                          if (!accessToken) {
                            toast.error("Please login to start negotiation");
                            navigate("/login");
                            return;
                          }

                          try {
                            // Get seller ID from property data
                            const user = JSON.parse(
                              localStorage.getItem("user") || "{}",
                            );
                            const sellerId =
                              (property as any).listed_by_id ||
                              (property as any).user_id ||
                              (property as any).owner_id;

                            if (!sellerId) {
                              toast.error(
                                "Unable to start negotiation: Seller information not available",
                              );
                              return;
                            }

                            const response = await fetch(
                              "http://localhost:8090/api/negotiations",
                              {
                                method: "POST",
                                headers: {
                                  "Content-Type": "application/json",
                                  Authorization: `Bearer ${accessToken}`,
                                },
                                body: JSON.stringify({
                                  property_id: property.id,
                                  seller_id: sellerId,
                                  agent_id: (property as any).agent_id,
                                }),
                              },
                            );

                            if (response.ok) {
                              const data = await response.json();
                              toast.success(
                                "Negotiation started successfully!",
                              );
                              navigate(`/negotiations/${data.negotiation_id}`);
                            } else {
                              const error = await response.json();
                              toast.error(
                                error.detail || "Failed to start negotiation",
                              );
                            }
                          } catch (error) {
                            console.error("Negotiation error:", error);
                            toast.error("Failed to start negotiation");
                          }
                        }}
                        className="px-6 py-3 rounded-lg flex items-center space-x-2 font-semibold transition-colors"
                        style={{ backgroundColor: "#06b6d4", color: "white" }}
                      >
                        <svg
                          className="w-5 h-5"
                          fill="none"
                          stroke="currentColor"
                          viewBox="0 0 24 24"
                        >
                          <path
                            strokeLinecap="round"
                            strokeLinejoin="round"
                            strokeWidth={2}
                            d="M7 12l3-3 3 3 4-4M8 21l4-4 4 4M3 4h18M4 4h16v12a1 1 0 01-1 1H5a1 1 0 01-1-1V4z"
                          />
                        </svg>
                        <span>Price Negotiation</span>
                      </button>
                      <button
                        onClick={() => {
                          toast.success("Rent inquiry submitted");
                        }}
                        className="px-6 py-3 rounded-lg flex items-center space-x-2 font-semibold transition-colors"
                        style={{ backgroundColor: "#10b981", color: "white" }}
                      >
                        <svg
                          className="w-5 h-5"
                          fill="none"
                          stroke="currentColor"
                          viewBox="0 0 24 24"
                        >
                          <path
                            strokeLinecap="round"
                            strokeLinejoin="round"
                            strokeWidth={2}
                            d="M3 12l2-2m0 0l7-7 7 7M5 10v10a1 1 0 001 1h3m10-11l2 2m-2-2v10a1 1 0 01-1 1h-3m-6 0a1 1 0 001-1v-4a1 1 0 011-1h2a1 1 0 011 1v4a1 1 0 001 1m-6 0h6"
                          />
                        </svg>
                        <span>Rent Now</span>
                      </button>
                    </>
                  )}
                </div>
                {property?.description && (
                  <div>
                    <h3 className="text-lg font-semibold mb-3">Description</h3>
                    <p className="text-gray-700 leading-relaxed mb-6">
                      {property.description}
                    </p>
                  </div>
                )}

                {/* Air Quality Score */}
                <div className="mb-6">
                  <AirQualityScore propertyId={property.id} />
                </div>

                <div className="mb-6">
                  <h3 className="text-lg font-semibold mb-3">
                    Property Features
                  </h3>
                  <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                    {property?.facing && (
                      <div className="flex items-center space-x-2">
                        <span className="w-2 h-2 bg-blue-500 rounded-full"></span>
                        <span>Facing: {property.facing}</span>
                      </div>
                    )}
                    {property?.furnishing && (
                      <div className="flex items-center space-x-2">
                        <span className="w-2 h-2 bg-blue-500 rounded-full"></span>
                        <span>Furnishing: {property.furnishing}</span>
                      </div>
                    )}
                    {property?.property_age && (
                      <div className="flex items-center space-x-2">
                        <span className="w-2 h-2 bg-blue-500 rounded-full"></span>
                        <span>Property Age: {property.property_age}</span>
                      </div>
                    )}
                    {property?.carpet_area && (
                      <div className="flex items-center space-x-2">
                        <span className="w-2 h-2 bg-blue-500 rounded-full"></span>
                        <span>Carpet Area: {property.carpet_area} sq.ft</span>
                      </div>
                    )}
                    {property?.plot_area && (
                      <div className="flex items-center space-x-2">
                        <span className="w-2 h-2 bg-blue-500 rounded-full"></span>
                        <span>Plot Area: {property.plot_area} sq.ft</span>
                      </div>
                    )}
                    {property?.floor_number && (
                      <div className="flex items-center space-x-2">
                        <span className="w-2 h-2 bg-blue-500 rounded-full"></span>
                        <span>Floor: {property.floor_number}</span>
                      </div>
                    )}
                    {property?.total_floors && (
                      <div className="flex items-center space-x-2">
                        <span className="w-2 h-2 bg-blue-500 rounded-full"></span>
                        <span>Total Floors: {property.total_floors}</span>
                      </div>
                    )}
                    {property?.balconies && (
                      <div className="flex items-center space-x-2">
                        <span className="w-2 h-2 bg-blue-500 rounded-full"></span>
                        <span>Balconies: {property.balconies}</span>
                      </div>
                    )}
                  </div>
                </div>

                {/* Property Highlights */}
                {property?.highlights && property.highlights.length > 0 && (
                  <div className="mb-6">
                    <h3 className="text-lg font-semibold mb-3">
                      Property Highlights
                    </h3>
                    <div className="flex flex-wrap gap-2">
                      {property.highlights.map((highlight, index) => (
                        <span
                          key={index}
                          className="px-3 py-1 bg-green-100 text-green-800 text-sm rounded-full"
                        >
                          {highlight}
                        </span>
                      ))}
                    </div>
                  </div>
                )}

                {/* Comprehensive Property Information */}
                <div className="mb-6">
                  <h3 className="text-lg font-semibold mb-4">
                    Complete Property Information
                  </h3>

                  {/* Basic Details */}
                  <div className="bg-white border border-gray-200 rounded-lg p-4 mb-4">
                    <h4 className="font-semibold text-gray-900 mb-3 flex items-center">
                      <svg
                        className="w-5 h-5 mr-2 text-blue-600"
                        fill="currentColor"
                        viewBox="0 0 20 20"
                      >
                        <path d="M10.707 2.293a1 1 0 00-1.414 0l-7 7a1 1 0 001.414 1.414L4 10.414V17a1 1 0 001 1h2a1 1 0 001-1v-2h2v2a1 1 0 001 1h2a1 1 0 001-1v-6.586l.293.293a1 1 0 001.414-1.414l-7-7z" />
                      </svg>
                      Basic Details
                    </h4>
                    <div className="grid grid-cols-1 md:grid-cols-2 gap-3 text-sm">
                      <div>
                        <span className="font-medium text-gray-700">
                          Property Type:
                        </span>{" "}
                        <span className="capitalize">
                          {property?.property_type}
                        </span>
                      </div>
                      <div>
                        <span className="font-medium text-gray-700">
                          Listing Type:
                        </span>{" "}
                        <span className="capitalize">
                          {property?.listing_type?.replace("-", " ")}
                        </span>
                      </div>
                      <div>
                        <span className="font-medium text-gray-700">
                          Status:
                        </span>{" "}
                        <span className="capitalize">
                          {property?.possession_status || "Ready to move"}
                        </span>
                      </div>
                      <div>
                        <span className="font-medium text-gray-700">
                          Property Age:
                        </span>{" "}
                        <span className="capitalize">
                          {property?.property_age?.replace("-", " to ") ||
                            "Not specified"}
                        </span>
                      </div>
                    </div>
                  </div>

                  {/* Area Details */}
                  <div className="bg-white border border-gray-200 rounded-lg p-4 mb-4">
                    <h4 className="font-semibold text-gray-900 mb-3 flex items-center">
                      <svg
                        className="w-5 h-5 mr-2 text-green-600"
                        fill="currentColor"
                        viewBox="0 0 20 20"
                      >
                        <path
                          fillRule="evenodd"
                          d="M3 3a1 1 0 000 2v8a2 2 0 002 2h2.586l-1.293 1.293a1 1 0 101.414 1.414L10 15.414l2.293 2.293a1 1 0 001.414-1.414L12.414 15H15a2 2 0 002-2V5a1 1 0 100-2H3z"
                          clipRule="evenodd"
                        />
                      </svg>
                      Area Information
                    </h4>
                    <div className="grid grid-cols-1 md:grid-cols-2 gap-3 text-sm">
                      {property?.built_up_area && (
                        <div>
                          <span className="font-medium text-gray-700">
                            Built-up Area:
                          </span>{" "}
                          {property.built_up_area} sq.ft
                        </div>
                      )}
                      {property?.carpet_area && (
                        <div>
                          <span className="font-medium text-gray-700">
                            Carpet Area:
                          </span>{" "}
                          {property.carpet_area} sq.ft
                        </div>
                      )}
                      {property?.plot_area && (
                        <div>
                          <span className="font-medium text-gray-700">
                            Plot Area:
                          </span>{" "}
                          {property.plot_area} sq.ft
                        </div>
                      )}
                      {property?.super_area && (
                        <div>
                          <span className="font-medium text-gray-700">
                            Super Area:
                          </span>{" "}
                          {property.super_area} sq.ft
                        </div>
                      )}
                    </div>
                  </div>

                  {/* Room Configuration */}
                  <div className="bg-white border border-gray-200 rounded-lg p-4 mb-4">
                    <h4 className="font-semibold text-gray-900 mb-3 flex items-center">
                      <svg
                        className="w-5 h-5 mr-2 text-purple-600"
                        fill="currentColor"
                        viewBox="0 0 20 20"
                      >
                        <path d="M3 4a1 1 0 011-1h12a1 1 0 011 1v2a1 1 0 01-1 1H4a1 1 0 01-1-1V4zM3 10a1 1 0 011-1h6a1 1 0 011 1v6a1 1 0 01-1 1H4a1 1 0 01-1-1v-6z" />
                      </svg>
                      Room Configuration
                    </h4>
                    <div className="grid grid-cols-2 md:grid-cols-4 gap-3 text-sm">
                      <div>
                        <span className="font-medium text-gray-700">
                          Bedrooms:
                        </span>{" "}
                        {property?.bedrooms || "N/A"}
                      </div>
                      <div>
                        <span className="font-medium text-gray-700">
                          Bathrooms:
                        </span>{" "}
                        {property?.bathrooms || "N/A"}
                      </div>
                      <div>
                        <span className="font-medium text-gray-700">
                          Balconies:
                        </span>{" "}
                        {property?.balconies || "N/A"}
                      </div>
                      <div>
                        <span className="font-medium text-gray-700">
                          Facing:
                        </span>{" "}
                        <span className="capitalize">
                          {property?.facing?.replace("-", " ") || "N/A"}
                        </span>
                      </div>
                    </div>
                  </div>

                  {/* Floor Details */}
                  {(property?.floor_number || property?.total_floors) && (
                    <div className="bg-white border border-gray-200 rounded-lg p-4 mb-4">
                      <h4 className="font-semibold text-gray-900 mb-3 flex items-center">
                        <svg
                          className="w-5 h-5 mr-2 text-indigo-600"
                          fill="currentColor"
                          viewBox="0 0 20 20"
                        >
                          <path
                            fillRule="evenodd"
                            d="M5 9V7a5 5 0 0110 0v2a2 2 0 012 2v5a2 2 0 01-2 2H5a2 2 0 01-2-2v-5a2 2 0 012-2zm8-2v2H7V7a3 3 0 016 0z"
                            clipRule="evenodd"
                          />
                        </svg>
                        Floor Information
                      </h4>
                      <div className="grid grid-cols-1 md:grid-cols-2 gap-3 text-sm">
                        {property?.floor_number && (
                          <div>
                            <span className="font-medium text-gray-700">
                              Floor Number:
                            </span>{" "}
                            {property.floor_number}
                          </div>
                        )}
                        {property?.total_floors && (
                          <div>
                            <span className="font-medium text-gray-700">
                              Total Floors:
                            </span>{" "}
                            {property.total_floors}
                          </div>
                        )}
                      </div>
                    </div>
                  )}

                  {/* Pricing Details */}
                  <div className="bg-white border border-gray-200 rounded-lg p-4 mb-4">
                    <h4 className="font-semibold text-gray-900 mb-3 flex items-center">
                      <svg
                        className="w-5 h-5 mr-2 text-yellow-600"
                        fill="currentColor"
                        viewBox="0 0 20 20"
                      >
                        <path d="M8.433 7.418c.155-.103.346-.196.567-.267v1.698a2.305 2.305 0 01-.567-.267C8.07 8.34 8 8.114 8 8c0-.114.07-.34.433-.582zM11 12.849v-1.698c.22.071.412.164.567.267.364.243.433.468.433.582 0 .114-.07.34-.433.582a2.305 2.305 0 01-.567.267z" />
                        <path
                          fillRule="evenodd"
                          d="M10 18a8 8 0 100-16 8 8 0 000 16zm1-13a1 1 0 10-2 0v.092a4.535 4.535 0 00-1.676.662C6.602 6.234 6 7.009 6 8c0 .99.602 1.765 1.324 2.246.48.32 1.054.545 1.676.662v1.941c-.391-.127-.68-.317-.843-.504a1 1 0 10-1.51 1.31c.562.649 1.413 1.076 2.353 1.253V15a1 1 0 102 0v-.092a4.535 4.535 0 001.676-.662C13.398 13.766 14 12.991 14 12c0-.99-.602-1.765-1.324-2.246A4.535 4.535 0 0011 9.092V7.151c.391.127.68.317.843.504a1 1 0 101.511-1.31c-.563-.649-1.413-1.076-2.354-1.253V5z"
                          clipRule="evenodd"
                        />
                      </svg>
                      Pricing Information
                    </h4>
                    <div className="grid grid-cols-1 md:grid-cols-2 gap-3 text-sm">
                      {property?.sale_price && (
                        <div>
                          <span className="font-medium text-gray-700">
                            Sale Price:
                          </span>{" "}
                          {formatPrice(property.sale_price)}
                        </div>
                      )}
                      {property?.monthly_rent && (
                        <div>
                          <span className="font-medium text-gray-700">
                            Monthly Rent:
                          </span>{" "}
                          {formatPrice(property.monthly_rent)}
                        </div>
                      )}
                      {property?.security_deposit && (
                        <div>
                          <span className="font-medium text-gray-700">
                            Security Deposit:
                          </span>{" "}
                          {formatPrice(property.security_deposit)}
                        </div>
                      )}
                      <div>
                        <span className="font-medium text-gray-700">
                          Price Negotiable:
                        </span>{" "}
                        {property?.price_negotiable ? "Yes" : "No"}
                      </div>
                      <div>
                        <span className="font-medium text-gray-700">
                          Loan Available:
                        </span>{" "}
                        {property?.loan_availability ? "Yes" : "No"}
                      </div>
                      {property?.furnishing && (
                        <div>
                          <span className="font-medium text-gray-700">
                            Furnishing:
                          </span>{" "}
                          <span className="capitalize">
                            {property.furnishing.replace("-", " ")}
                          </span>
                        </div>
                      )}
                    </div>
                  </div>

                  {/* Enhanced Timeline Information */}
                  {((property?.listing_type === "urgent-sale" &&
                    property?.urgent_sale_end_date) ||
                    (property?.listing_type === "bidding" &&
                      property?.auction_end_date)) && (
                    <div
                      className={`border rounded-lg p-4 mb-4 ${
                        property.listing_type === "urgent-sale"
                          ? "bg-red-50 border-red-200"
                          : "bg-blue-50 border-blue-200"
                      }`}
                    >
                      <h4
                        className={`font-semibold mb-3 flex items-center ${
                          property.listing_type === "urgent-sale"
                            ? "text-red-900"
                            : "text-blue-900"
                        }`}
                      >
                        <svg
                          className={`w-5 h-5 mr-2 ${
                            property.listing_type === "urgent-sale"
                              ? "text-red-600"
                              : "text-blue-600"
                          }`}
                          fill="currentColor"
                          viewBox="0 0 20 20"
                        >
                          <path
                            fillRule="evenodd"
                            d="M10 18a8 8 0 100-16 8 8 0 000 16zm1-12a1 1 0 10-2 0v4a1 1 0 00.293.707l2.828 2.829a1 1 0 101.415-1.415L11 9.586V6z"
                            clipRule="evenodd"
                          />
                        </svg>
                        {property.listing_type === "urgent-sale"
                          ? "Urgent Sale Timeline"
                          : "Auction Timeline"}
                      </h4>

                      {/* Timeline Progress Bar */}
                      <div className="mb-4">
                        <div className="flex justify-between text-xs mb-1">
                          <span
                            className={
                              property.listing_type === "urgent-sale"
                                ? "text-red-700"
                                : "text-blue-700"
                            }
                          >
                            Started
                          </span>
                          <span
                            className={
                              property.listing_type === "urgent-sale"
                                ? "text-red-700"
                                : "text-blue-700"
                            }
                          >
                            Ends
                          </span>
                        </div>
                        <div className="w-full bg-gray-200 rounded-full h-2">
                          <div
                            className={`h-2 rounded-full ${
                              property.listing_type === "urgent-sale"
                                ? "bg-gradient-to-r from-red-500 to-orange-500"
                                : "bg-gradient-to-r from-blue-500 to-indigo-500"
                            }`}
                            style={{ width: "65%" }}
                          ></div>
                        </div>
                        <div className="text-center mt-2">
                          <span
                            className={`text-lg font-bold ${
                              property.listing_type === "urgent-sale"
                                ? "text-red-600"
                                : "text-blue-600"
                            }`}
                          >
                            {timeLeft} Remaining
                          </span>
                        </div>
                      </div>

                      <div className="space-y-3 text-sm">
                        {property?.listing_type === "urgent-sale" && (
                          <>
                            {property.urgent_sale_start_date && (
                              <div className="flex justify-between items-center p-2 bg-white rounded border">
                                <span className="font-medium text-red-700">
                                  Sale Started:
                                </span>
                                <span className="text-gray-900">
                                  {new Date(
                                    `${property.urgent_sale_start_date}T${property.urgent_sale_start_time || "00:00:00"}`,
                                  ).toLocaleString()}
                                </span>
                              </div>
                            )}
                            {property.urgent_sale_end_date && (
                              <div className="flex justify-between items-center p-2 bg-white rounded border border-red-200">
                                <span className="font-medium text-red-700">
                                  Sale Deadline:
                                </span>
                                <span className="text-gray-900 font-semibold">
                                  {new Date(
                                    `${property.urgent_sale_end_date}T${property.urgent_sale_end_time || "23:59:59"}`,
                                  ).toLocaleString()}
                                </span>
                              </div>
                            )}
                          </>
                        )}

                        {property?.listing_type === "bidding" && (
                          <>
                            {property.auction_start_date && (
                              <div className="flex justify-between items-center p-2 bg-white rounded border">
                                <span className="font-medium text-blue-700">
                                  Auction Started:
                                </span>
                                <span className="text-gray-900">
                                  {new Date(
                                    `${property.auction_start_date}T${property.auction_start_time || "00:00:00"}`,
                                  ).toLocaleString()}
                                </span>
                              </div>
                            )}
                            {property.auction_end_date && (
                              <div className="flex justify-between items-center p-2 bg-white rounded border border-blue-200">
                                <span className="font-medium text-blue-700">
                                  Auction Ends:
                                </span>
                                <span className="text-gray-900 font-semibold">
                                  {new Date(
                                    `${property.auction_end_date}T${property.auction_end_time || "23:59:59"}`,
                                  ).toLocaleString()}
                                </span>
                              </div>
                            )}
                          </>
                        )}

                        {/* Timeline Info from Backend */}
                        <div className="grid grid-cols-2 gap-2 mt-3">
                          <div className="text-center p-2 bg-white rounded border">
                            <div
                              className={`text-lg font-bold ${
                                property.listing_type === "urgent-sale"
                                  ? "text-red-600"
                                  : "text-blue-600"
                              }`}
                            >
                              {property.views_today || property.views || 0}
                            </div>
                            <div className="text-xs text-gray-600">
                              Views Today
                            </div>
                          </div>
                          <div className="text-center p-2 bg-white rounded border">
                            <div
                              className={`text-lg font-bold ${
                                property.listing_type === "urgent-sale"
                                  ? "text-red-600"
                                  : "text-blue-600"
                              }`}
                            >
                              {property.listing_type === "bidding"
                                ? property.active_bidders || 0
                                : property.inquiries || 0}
                            </div>
                            <div className="text-xs text-gray-600">
                              {property.listing_type === "bidding"
                                ? "Active Bidders"
                                : "Inquiries"}
                            </div>
                          </div>
                        </div>
                      </div>
                    </div>
                  )}

                  {/* System Information */}
                  <div className="bg-gray-50 border border-gray-200 rounded-lg p-4">
                    <h4 className="font-semibold text-gray-900 mb-3 flex items-center">
                      <svg
                        className="w-5 h-5 mr-2 text-gray-600"
                        fill="currentColor"
                        viewBox="0 0 20 20"
                      >
                        <path
                          fillRule="evenodd"
                          d="M18 10a8 8 0 11-16 0 8 8 0 0116 0zm-7-4a1 1 0 11-2 0 1 1 0 012 0zM9 9a1 1 0 000 2v3a1 1 0 001 1h1a1 1 0 100-2v-3a1 1 0 00-1-1H9z"
                          clipRule="evenodd"
                        />
                      </svg>
                      Listing Information
                    </h4>
                    <div className="grid grid-cols-1 md:grid-cols-2 gap-3 text-sm text-gray-600">
                      <div>
                        <span className="font-medium">Property ID:</span>{" "}
                        {property?.id}
                      </div>
                      <div>
                        <span className="font-medium">Listed On:</span>{" "}
                        {property?.created_at
                          ? new Date(property.created_at).toLocaleDateString()
                          : "N/A"}
                      </div>
                      <div>
                        <span className="font-medium">State:</span>{" "}
                        {property?.state}
                      </div>
                      <div>
                        <span className="font-medium">Pincode:</span>{" "}
                        {property?.pincode}
                      </div>
                    </div>
                  </div>

                  {/* Legal Documents Section */}
                  <PropertyLegalDocuments propertyId={property.id} />
                </div>


              </div>
            )}

            {activeTab === "media" && (
              <div>
                {/* Azure Blob Media Gallery */}
                {hasMedia ? (
                  <AzureBlobMediaGallery
                    propertyId={id!}
                    editable={(() => {
                      // Check if current user owns this property
                      const currentUser = JSON.parse(
                        localStorage.getItem("user") || "{}",
                      );
                      const userEmail =
                        currentUser.email || localStorage.getItem("user_email");
                      return (
                        userEmail &&
                        property?.email &&
                        userEmail.toLowerCase() === property.email.toLowerCase()
                      );
                    })()}
                    showUpload={false}
                    className="mb-6"
                  />
                ) : (
                  <div>
                    <h2 className="text-xl font-bold mb-4">Property Media</h2>

                    {/* Legacy Images Section */}
                    <div className="mb-8">
                      <h3 className="text-lg font-semibold mb-4">
                        Images ({allImages.length})
                      </h3>
                      {allImages.length > 0 ? (
                        <div className="grid grid-cols-2 md:grid-cols-3 lg:grid-cols-4 gap-4">
                          {allImages.map((image, index) => (
                            <div
                              key={`gallery-img-${index}`}
                              className="relative group"
                            >
                              <img
                                src={image}
                                alt={`${property?.title} ${index + 1}`}
                                className="w-full h-32 object-cover rounded-lg cursor-pointer transition-transform group-hover:scale-105"
                                onClick={() => {
                                  setCurrentImageIndex(index);
                                  setShowVideoPlayer(false);
                                  // Track image click
                                  import("../utils/behaviorTracker").then(({ behaviorTracker }) => {
                                    behaviorTracker.trackImageClick(parseInt(id!), {
                                      image_index: index,
                                      image_url: image
                                    });
                                  });
                                }}
                                onError={(e) => {
                                  console.error(
                                    "Gallery image failed to load:",
                                    e.currentTarget.src,
                                  );
                                  // Hide failed gallery image
                                  e.currentTarget.style.display = "none";
                                }}
                              />
                              <div className="absolute inset-0 bg-black bg-opacity-0 group-hover:bg-opacity-20 transition-opacity rounded-lg" />
                            </div>
                          ))}
                        </div>
                      ) : (
                        <div className="bg-gray-50 border-2 border-dashed border-gray-300 rounded-lg p-8 text-center">
                          <svg
                            className="w-12 h-12 text-gray-400 mx-auto mb-4"
                            fill="none"
                            stroke="currentColor"
                            viewBox="0 0 24 24"
                          >
                            <path
                              strokeLinecap="round"
                              strokeLinejoin="round"
                              strokeWidth={2}
                              d="M4 16l4.586-4.586a2 2 0 012.828 0L16 16m-2-2l1.586-1.586a2 2 0 012.828 0L20 14m-6-6h.01M6 20h12a2 2 0 002-2V6a2 2 0 00-2-2H6a2 2 0 00-2 2v14a2 2 0 002 2z"
                            />
                          </svg>
                          <p className="text-gray-500">
                            No images available for this property
                          </p>
                        </div>
                      )}
                    </div>

                    {/* Legacy Videos Section */}
                    <div className="mb-8">
                      <h3 className="text-lg font-semibold mb-4">
                        Videos ({allVideos.length})
                      </h3>
                      {allVideos.length > 0 ? (
                        <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-4">
                          {allVideos.map((video, index) => (
                            <div
                              key={`gallery-vid-${index}`}
                              className="relative group"
                            >
                              <video
                                src={video}
                                className="w-full h-48 object-cover rounded-lg cursor-pointer"
                                muted
                                onClick={() => {
                                  setCurrentVideoIndex(index);
                                  setShowVideoPlayer(true);
                                }}
                              />
                              <div className="absolute inset-0 flex items-center justify-center bg-black bg-opacity-30 group-hover:bg-opacity-50 transition-opacity rounded-lg">
                                <svg
                                  className="w-12 h-12 text-white"
                                  fill="currentColor"
                                  viewBox="0 0 20 20"
                                >
                                  <path
                                    fillRule="evenodd"
                                    d="M10 18a8 8 0 100-16 8 8 0 000 16zM9.555 7.168A1 1 0 008 8v4a1 1 0 001.555.832l3-2a1 1 0 000-1.664l-3-2z"
                                    clipRule="evenodd"
                                  />
                                </svg>
                              </div>
                            </div>
                          ))}
                        </div>
                      ) : (
                        <div className="bg-gray-50 border-2 border-dashed border-gray-300 rounded-lg p-8 text-center">
                          <svg
                            className="w-12 h-12 text-gray-400 mx-auto mb-4"
                            fill="none"
                            stroke="currentColor"
                            viewBox="0 0 24 24"
                          >
                            <path
                              strokeLinecap="round"
                              strokeLinejoin="round"
                              strokeWidth={2}
                              d="M15 10l4.553-2.276A1 1 0 0121 8.618v6.764a1 1 0 01-1.447.894L15 14M5 18h8a2 2 0 002-2V8a2 2 0 00-2-2H5a2 2 0 00-2 2v8a2 2 0 002 2z"
                            />
                          </svg>
                          <p className="text-gray-500">
                            No videos available for this property
                          </p>
                        </div>
                      )}
                    </div>
                  </div>
                )}

                {/* Virtual Tours & 3D Plans */}
                {(property?.three_d_plan ||
                  property?.three_sixty_view ||
                  property?.virtual_tours) && (
                  <div>
                    <h3 className="text-lg font-semibold mb-4">
                      Immersive Property Experience
                    </h3>
                    <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
                      {property?.three_d_plan && (
                        <div className="group relative bg-gradient-to-br from-blue-50 via-indigo-50 to-blue-100 p-8 rounded-2xl border-2 border-blue-200 hover:border-blue-400 transition-all duration-300 hover:shadow-2xl overflow-hidden">
                          {/* Animated Background Effect */}
                          <div className="absolute inset-0 bg-gradient-to-r from-blue-400/10 to-indigo-400/10 opacity-0 group-hover:opacity-100 transition-opacity duration-300"></div>
                          
                          {/* Icon */}
                          <div className="relative mb-4">
                            <div className="w-16 h-16 bg-blue-600 rounded-2xl flex items-center justify-center group-hover:scale-110 transition-transform duration-300 shadow-lg">
                              <svg className="w-8 h-8 text-white" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M19 11H5m14 0a2 2 0 012 2v6a2 2 0 01-2 2H5a2 2 0 01-2-2v-6a2 2 0 012-2m14 0V9a2 2 0 00-2-2M5 11V9a2 2 0 012-2m0 0V5a2 2 0 012-2h6a2 2 0 012 2v2M7 7h10" />
                              </svg>
                            </div>
                          </div>
                          
                          {/* Content */}
                          <h4 className="relative font-bold text-blue-900 mb-3 text-xl">
                            3D Floor Plan
                          </h4>
                          <p className="relative text-blue-700 text-sm mb-6 leading-relaxed">
                            Explore the property layout in stunning 3D visualization with interactive controls
                          </p>
                          
                          {/* Enhanced Button */}
                          <button className="relative w-full bg-gradient-to-r from-blue-600 to-indigo-600 text-white px-6 py-3.5 rounded-xl font-semibold text-sm hover:from-blue-700 hover:to-indigo-700 transition-all duration-300 shadow-lg hover:shadow-xl flex items-center justify-center gap-2 group/btn">
                            <span>View 3D Plan</span>
                            <svg className="w-5 h-5 group-hover/btn:translate-x-1 transition-transform duration-300" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                              <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M13 7l5 5m0 0l-5 5m5-5H6" />
                            </svg>
                          </button>
                        </div>
                      )}
                      {property?.three_sixty_view && (
                        <div className="group relative bg-gradient-to-br from-purple-50 via-pink-50 to-purple-100 p-8 rounded-2xl border-2 border-purple-200 hover:border-purple-400 transition-all duration-300 hover:shadow-2xl overflow-hidden">
                          {/* Animated Background Effect */}
                          <div className="absolute inset-0 bg-gradient-to-r from-purple-400/10 to-pink-400/10 opacity-0 group-hover:opacity-100 transition-opacity duration-300"></div>
                          
                          {/* Icon */}
                          <div className="relative mb-4">
                            <div className="w-16 h-16 bg-purple-600 rounded-2xl flex items-center justify-center group-hover:scale-110 group-hover:rotate-180 transition-all duration-500 shadow-lg">
                              <svg className="w-8 h-8 text-white" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M4 8V4m0 0h4M4 4l5 5m11-1V4m0 0h-4m4 0l-5 5M4 16v4m0 0h4m-4 0l5-5m11 5l-5-5m5 5v-4m0 4h-4" />
                              </svg>
                            </div>
                          </div>
                          
                          {/* Content */}
                          <h4 className="relative font-bold text-purple-900 mb-3 text-xl">
                            360° Virtual Tour
                          </h4>
                          <p className="relative text-purple-700 text-sm mb-6 leading-relaxed">
                            Immersive 360-degree experience - Walk through every corner from your device
                          </p>
                          
                          {/* Enhanced Button */}
                          <button className="relative w-full bg-gradient-to-r from-purple-600 to-pink-600 text-white px-6 py-3.5 rounded-xl font-semibold text-sm hover:from-purple-700 hover:to-pink-700 transition-all duration-300 shadow-lg hover:shadow-xl flex items-center justify-center gap-2 group/btn">
                            <svg className="w-5 h-5 group-hover/btn:rotate-90 transition-transform duration-300" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                              <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M14.752 11.168l-3.197-2.132A1 1 0 0010 9.87v4.263a1 1 0 001.555.832l3.197-2.132a1 1 0 000-1.664z" />
                              <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M21 12a9 9 0 11-18 0 9 9 0 0118 0z" />
                            </svg>
                            <span>Start 360° Tour</span>
                          </button>
                        </div>
                      )}
                    </div>
                  </div>
                )}
              </div>
            )}

            {activeTab === "amenities" && (
              <div>
                <h2 className="text-xl font-bold mb-4">Amenities</h2>
                <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-4">
                  {property?.amenities?.length > 0 ? (
                    property.amenities.map((amenity, index) => (
                      <div
                        key={index}
                        className="flex items-center space-x-2 p-3 bg-gray-50 rounded-lg"
                      >
                        <svg
                          className="w-5 h-5 text-green-500"
                          fill="currentColor"
                          viewBox="0 0 20 20"
                        >
                          <path
                            fillRule="evenodd"
                            d="M16.707 5.293a1 1 0 010 1.414l-8 8a1 1 0 01-1.414 0l-4-4a1 1 0 011.414-1.414L8 12.586l7.293-7.293a1 1 0 011.414 0z"
                            clipRule="evenodd"
                          />
                        </svg>
                        <span className="text-gray-700">{amenity}</span>
                      </div>
                    ))
                  ) : (
                    <>
                      <div className="flex items-center space-x-2 p-3 bg-gray-50 rounded-lg">
                        <svg
                          className="w-5 h-5 text-green-500"
                          fill="currentColor"
                          viewBox="0 0 20 20"
                        >
                          <path
                            fillRule="evenodd"
                            d="M16.707 5.293a1 1 0 010 1.414l-8 8a1 1 0 01-1.414 0l-4-4a1 1 0 011.414-1.414L8 12.586l7.293-7.293a1 1 0 011.414 0z"
                            clipRule="evenodd"
                          />
                        </svg>
                        <span>24/7 Security</span>
                      </div>
                      <div className="flex items-center space-x-2 p-3 bg-gray-50 rounded-lg">
                        <svg
                          className="w-5 h-5 text-green-500"
                          fill="currentColor"
                          viewBox="0 0 20 20"
                        >
                          <path
                            fillRule="evenodd"
                            d="M16.707 5.293a1 1 0 010 1.414l-8 8a1 1 0 01-1.414 0l-4-4a1 1 0 011.414-1.414L8 12.586l7.293-7.293a1 1 0 011.414 0z"
                            clipRule="evenodd"
                          />
                        </svg>
                        <span>Swimming Pool</span>
                      </div>
                      <div className="flex items-center space-x-2 p-3 bg-gray-50 rounded-lg">
                        <svg
                          className="w-5 h-5 text-green-500"
                          fill="currentColor"
                          viewBox="0 0 20 20"
                        >
                          <path
                            fillRule="evenodd"
                            d="M16.707 5.293a1 1 0 010 1.414l-8 8a1 1 0 01-1.414 0l-4-4a1 1 0 011.414-1.414L8 12.586l7.293-7.293a1 1 0 011.414 0z"
                            clipRule="evenodd"
                          />
                        </svg>
                        <span>Gym</span>
                      </div>
                      <div className="flex items-center space-x-2 p-3 bg-gray-50 rounded-lg">
                        <svg
                          className="w-5 h-5 text-green-500"
                          fill="currentColor"
                          viewBox="0 0 20 20"
                        >
                          <path
                            fillRule="evenodd"
                            d="M16.707 5.293a1 1 0 010 1.414l-8 8a1 1 0 01-1.414 0l-4-4a1 1 0 011.414-1.414L8 12.586l7.293-7.293a1 1 0 011.414 0z"
                            clipRule="evenodd"
                          />
                        </svg>
                        <span>Clubhouse</span>
                      </div>
                      <div className="flex items-center space-x-2 p-3 bg-gray-50 rounded-lg">
                        <svg
                          className="w-5 h-5 text-green-500"
                          fill="currentColor"
                          viewBox="0 0 20 20"
                        >
                          <path
                            fillRule="evenodd"
                            d="M16.707 5.293a1 1 0 010 1.414l-8 8a1 1 0 01-1.414 0l-4-4a1 1 0 011.414-1.414L8 12.586l7.293-7.293a1 1 0 011.414 0z"
                            clipRule="evenodd"
                          />
                        </svg>
                        <span>Indoor Games</span>
                      </div>
                      <div className="flex items-center space-x-2 p-3 bg-gray-50 rounded-lg">
                        <svg
                          className="w-5 h-5 text-green-500"
                          fill="currentColor"
                          viewBox="0 0 20 20"
                        >
                          <path
                            fillRule="evenodd"
                            d="M16.707 5.293a1 1 0 010 1.414l-8 8a1 1 0 01-1.414 0l-4-4a1 1 0 011.414-1.414L8 12.586l7.293-7.293a1 1 0 011.414 0z"
                            clipRule="evenodd"
                          />
                        </svg>
                        <span>Jogging Track</span>
                      </div>
                      <div className="flex items-center space-x-2 p-3 bg-gray-50 rounded-lg">
                        <svg
                          className="w-5 h-5 text-green-500"
                          fill="currentColor"
                          viewBox="0 0 20 20"
                        >
                          <path
                            fillRule="evenodd"
                            d="M16.707 5.293a1 1 0 010 1.414l-8 8a1 1 0 01-1.414 0l-4-4a1 1 0 011.414-1.414L8 12.586l7.293-7.293a1 1 0 011.414 0z"
                            clipRule="evenodd"
                          />
                        </svg>
                        <span>Covered Parking</span>
                      </div>
                      <div className="flex items-center space-x-2 p-3 bg-gray-50 rounded-lg">
                        <svg
                          className="w-5 h-5 text-green-500"
                          fill="currentColor"
                          viewBox="0 0 20 20"
                        >
                          <path
                            fillRule="evenodd"
                            d="M16.707 5.293a1 1 0 010 1.414l-8 8a1 1 0 01-1.414 0l-4-4a1 1 0 011.414-1.414L8 12.586l7.293-7.293a1 1 0 011.414 0z"
                            clipRule="evenodd"
                          />
                        </svg>
                        <span>Children's Play Area</span>
                      </div>
                    </>
                  )}
                </div>
              </div>
            )}

            {activeTab === "nearby-facilities" && property && (
              <div className="space-y-6">
                <div className="flex items-center justify-between">
                  <h2 className="text-xl font-bold">Nearby Facilities</h2>
                  <span className="text-sm text-gray-600">Within 5 km radius</span>
                </div>

                {facilitiesLoading ? (
                  <div className="flex items-center justify-center py-12">
                    <div className="animate-spin rounded-full h-8 w-8 border-b-2 border-blue-600 mr-3"></div>
                    <span className="text-gray-600">Loading nearby facilities...</span>
                  </div>
                ) : facilities.length === 0 ? (
                  <div className="text-center py-12 bg-gray-50 rounded-lg">
                    <svg className="w-16 h-16 text-gray-400 mx-auto mb-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                      <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M17.657 16.657L13.414 20.9a1.998 1.998 0 01-2.827 0l-4.244-4.243a8 8 0 1111.314 0z" />
                      <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M15 11a3 3 0 11-6 0 3 3 0 016 0z" />
                    </svg>
                    <h3 className="text-lg font-semibold text-gray-900 mb-2">No Facilities Found</h3>
                    <p className="text-gray-600 text-sm">Unable to load nearby facilities. Please check console for details.</p>
                  </div>
                ) : (
                  <FacilitiesMap
                    centerLat={property.latitude || 12.9716}
                    centerLon={property.longitude || 77.5946}
                    facilities={facilities}
                    radiusKm={5}
                  />
                )}
              </div>
            )}

            {activeTab === "location" && (
              <div>
                <h2 className="text-xl font-bold mb-4">Location</h2>
                <div className="mb-4">
                  <p className="text-gray-700 mb-2">
                    <strong>Address:</strong>{" "}
                    {property?.address || "Address not available"}
                  </p>
                  <p className="text-gray-700 mb-2">
                    <strong>Locality:</strong> {property?.locality}
                  </p>
                  <p className="text-gray-700 mb-2">
                    <strong>City:</strong> {property?.city}, {property?.state}
                  </p>
                  <p className="text-gray-700 mb-2">
                    <strong>Pincode:</strong> {property?.pincode}
                  </p>
                  {property?.landmark && (
                    <p className="text-gray-700 mb-2">
                      <strong>Landmark:</strong> {property.landmark}
                    </p>
                  )}
                  {property?.latitude && property?.longitude && (
                    <p className="text-gray-700 mb-4">
                      <strong>Coordinates:</strong> {property.latitude},{" "}
                      {property.longitude}
                    </p>
                  )}
                </div>

                {/* Nearby Places */}
                {property?.nearby_places &&
                  property.nearby_places.length > 0 && (
                    <div className="mb-4">
                      <h3 className="text-lg font-semibold mb-3">
                        Nearby Places
                      </h3>
                      <div className="flex flex-wrap gap-2">
                        {property.nearby_places.map((place, index) => (
                          <span
                            key={index}
                            className="px-3 py-1 bg-purple-100 text-purple-800 text-sm rounded-full"
                          >
                            {place}
                          </span>
                        ))}
                      </div>
                    </div>
                  )}

                <div className="bg-gray-100 h-64 rounded-lg flex items-center justify-center">
                  <p className="text-gray-500">Map integration coming soon</p>
                </div>
              </div>
            )}

            {activeTab === "documents" &&
              (() => {
                // Get user identification from localStorage
                const accessToken = localStorage.getItem("access_token");
                const userDataStr = localStorage.getItem("user");
                const loginEmail = localStorage.getItem("login_email");

                let userEmail = null;
                let userId = null;

                // Try to get user email from stored user data
                if (userDataStr) {
                  try {
                    const userData = JSON.parse(userDataStr);
                    userEmail = userData.email;
                    userId = userData.user_id;
                  } catch (e) {
                    console.error("Error parsing user data:", e);
                  }
                }

                // Fallback to login_email if no email in user data
                if (!userEmail && loginEmail) {
                  userEmail = loginEmail;
                }

                // Get property owner information
                const propertyEmail = property?.email;

                // Debug logging
                console.log("=== DOCUMENTS ACCESS DEBUG ===");
                console.log("Access token exists:", !!accessToken);
                console.log("User data from localStorage:", userDataStr);
                console.log("Login email from localStorage:", loginEmail);
                console.log("Extracted user email:", userEmail);
                console.log("Extracted user ID:", userId);
                console.log("Property email:", propertyEmail);
                console.log(
                  "Property object keys:",
                  property ? Object.keys(property) : "No property",
                );

                // Check if user is logged in
                const isLoggedIn = !!(accessToken && (userEmail || userId));

                // Check ownership - compare emails if both exist
                const isOwner =
                  userEmail &&
                  propertyEmail &&
                  userEmail.toLowerCase() === propertyEmail.toLowerCase();

                console.log("Is logged in:", isLoggedIn);
                console.log("Is owner (email match):", isOwner);
                console.log("==============================");

                if (!isLoggedIn) {
                  return (
                    <div className="text-center py-12">
                      <div className="bg-yellow-50 border border-yellow-200 rounded-lg p-6 max-w-md mx-auto">
                        <svg
                          className="w-12 h-12 text-yellow-600 mx-auto mb-4"
                          fill="none"
                          stroke="currentColor"
                          viewBox="0 0 24 24"
                        >
                          <path
                            strokeLinecap="round"
                            strokeLinejoin="round"
                            strokeWidth={2}
                            d="M12 15v2m-6 4h12a2 2 0 002-2v-6a2 2 0 00-2-2H6a2 2 0 00-2 2v6a2 2 0 002 2zm10-10V7a4 4 0 00-8 0v4h8z"
                          />
                        </svg>
                        <h3 className="text-lg font-semibold text-gray-900 mb-2">
                          Login Required
                        </h3>
                        <p className="text-gray-600 text-sm mb-4">
                          Please log in to view property documents.
                        </p>
                        <button
                          onClick={() => navigate("/login")}
                          className="bg-blue-600 text-white px-4 py-2 rounded-lg hover:bg-blue-700 transition-colors"
                        >
                          Login
                        </button>
                      </div>
                    </div>
                  );
                }

                // User is logged in - show documents with ownership info
                return (
                  <div>
                    {!isOwner && (
                      <div className="mb-4 p-3 bg-blue-50 border border-blue-200 rounded-lg">
                        <p className="text-sm text-blue-800">
                          <strong>Note:</strong> You are viewing documents as a
                          logged-in user. Property owners have full access to
                          manage documents.
                        </p>
                      </div>
                    )}
                    {isOwner && (
                      <div className="mb-4 p-3 bg-green-50 border border-green-200 rounded-lg">
                        <p className="text-sm text-green-800">
                          <strong>Property Owner:</strong> You have full access
                          to view and manage these documents.
                        </p>
                      </div>
                    )}
                    <PropertyDetailsDocuments
                      propertyId={Number(property.id)}
                    />
                  </div>
                );
              })()}

            {activeTab === "school-distance" && (
              <div>
                {/* Google Maps showing property and schools */}
                {property && (
                  <SchoolsGoogleMap
                    propertyLat={property.latitude || 12.9716}
                    propertyLon={property.longitude || 77.5946}
                    schools={schools}
                    propertyName={property.title}
                  />
                )}

                <div className="flex items-center justify-between mb-4">
                  <h2 className="text-xl font-bold">Nearby Schools</h2>
                  <div className="px-3 py-1 rounded-full text-sm font-medium bg-green-100 text-green-800">
                    {schools.length > 0 ? "Schools Found" : "Loading Schools"}
                  </div>
                </div>

                {schoolsLoading ? (
                  <div className="flex items-center justify-center py-8">
                    <div className="animate-spin rounded-full h-8 w-8 border-b-2 border-blue-600 mr-3"></div>
                    <span className="text-gray-600">
                      Loading nearby schools...
                    </span>
                  </div>
                ) : schools.length > 0 ? (
                  <div className="space-y-4">
                    {schools.map((school) => (
                      <div
                        key={school.id}
                        className="bg-white border border-gray-200 rounded-lg p-4 hover:shadow-md transition-shadow"
                      >
                        <div className="flex items-start justify-between">
                          <div className="flex-1">
                            <h3 className="font-semibold text-gray-900 mb-1">
                              {school.name}
                            </h3>
                            <p className="text-sm text-gray-600 mb-2">
                              {school.schoolType}
                            </p>
                            <div className="flex items-center text-sm text-gray-500">
                              <svg
                                className="w-4 h-4 mr-1"
                                fill="currentColor"
                                viewBox="0 0 20 20"
                              >
                                <path
                                  fillRule="evenodd"
                                  d="M5.05 4.05a7 7 0 119.9 9.9L10 18.9l-4.95-4.95a7 7 0 010-9.9zM10 11a2 2 0 100-4 2 2 0 000 4z"
                                  clipRule="evenodd"
                                />
                              </svg>
                              {school.address}
                            </div>
                          </div>
                          <div className="text-right">
                            <div className="text-lg font-bold text-blue-600">
                              {school.distance} km
                            </div>
                            <div className="text-xs text-gray-500">
                              {school.driveTime}
                            </div>
                          </div>
                        </div>
                      </div>
                    ))}
                  </div>
                ) : (
                  <div className="text-center py-8">
                    <div className="bg-yellow-50 border border-yellow-200 rounded-lg p-6">
                      <svg
                        className="w-12 h-12 text-yellow-600 mx-auto mb-4"
                        fill="none"
                        stroke="currentColor"
                        viewBox="0 0 24 24"
                      >
                        <path
                          strokeLinecap="round"
                          strokeLinejoin="round"
                          strokeWidth={2}
                          d="M12 9v2m0 4h.01m-6.938 4h13.856c1.54 0 2.502-1.667 1.732-2.5L13.732 4c-.77-.833-1.964-.833-2.732 0L3.732 16.5c-.77.833.192 2.5 1.732 2.5z"
                        />
                      </svg>
                      <h3 className="text-lg font-semibold text-gray-900 mb-2">
                        Schools Loading...
                      </h3>
                      <p className="text-gray-600 text-sm">
                        Setting up school data for this area. Please wait a
                        moment.
                      </p>
                    </div>
                  </div>
                )}
              </div>
            )}

            {activeTab === "safety" && (
              <div className="bg-gray-50 -m-6 p-6 rounded-2xl">
                {/* Page Header */}
                <div className="mb-8 text-center">
                  <h1 className="text-3xl font-bold tracking-tight text-gray-900 mb-2">
                    Safety & Crime
                  </h1>
                  <p className="text-gray-600 max-w-2xl mx-auto">
                    Comprehensive analysis of neighborhood safety and
                    environmental risks for informed property decisions
                  </p>
                </div>

                {property?.latitude && property?.longitude ? (
                  <>
                    {/* Responsive Layout */}
                    <div className="space-y-8 lg:space-y-0 lg:grid lg:grid-cols-2 lg:gap-8">
                      {/* Crime Heatmap Section */}
                      <div className="space-y-6">
                        <div className="flex items-center gap-3 mb-4">
                          <div className="p-2 bg-blue-600 rounded-lg">
                            <Shield className="w-6 h-6 text-white" />
                          </div>
                          <h2 className="text-2xl font-bold tracking-tight text-gray-900">
                            Crime & Safety Analysis
                          </h2>
                        </div>
                        <CrimeHeatmap
                          propertyLat={property.latitude}
                          propertyLon={property.longitude}
                          propertyName={property.title}
                        />
                      </div>

                      {/* Flood Risk Section */}
                      <div className="space-y-6">
                        <div className="flex items-center gap-3 mb-4">
                          <div className="p-2 bg-cyan-600 rounded-lg">
                            <Waves className="w-6 h-6 text-white" />
                          </div>
                          <h2 className="text-2xl font-bold tracking-tight text-gray-900">
                            Flood Risk Analysis
                          </h2>
                        </div>
                        <FloodRiskCard
                          lat={property.latitude}
                          lon={property.longitude}
                        />
                      </div>
                    </div>

                    {/* Show note if using default coordinates */}
                    {property.latitude === 12.9716 &&
                      property.longitude === 77.5946 && (
                        <div className="mt-6 bg-blue-50 border border-blue-200 rounded-2xl p-4">
                          <div className="flex items-start gap-3">
                            <AlertTriangle className="w-5 h-5 text-blue-600 mt-0.5 flex-shrink-0" />
                            <div>
                              <h4 className="text-sm font-semibold text-blue-800 mb-1">
                                Location Approximation
                              </h4>
                              <p className="text-sm text-blue-700 leading-relaxed">
                                Using approximate coordinates for{" "}
                                {property.city} area. For precise location-based
                                analysis, exact property coordinates are
                                recommended.
                              </p>
                            </div>
                          </div>
                        </div>
                      )}

                    {/* Bottom Disclaimer */}
                    <div className="mt-8 bg-white rounded-2xl shadow-sm p-6 border border-gray-200">
                      <div className="text-center">
                        <h3 className="text-lg font-semibold text-gray-900 mb-2">
                          Risk Assessment Methodology
                        </h3>
                        <p className="text-sm text-gray-600 leading-relaxed max-w-4xl mx-auto">
                          Our comprehensive risk assessment combines real-time
                          data from multiple sources including NCRB crime
                          statistics, Ambee Weather Intelligence, and advanced
                          geospatial analysis. Scores are calculated using
                          industry-standard algorithms and should be used as
                          guidance alongside professional consultation.
                        </p>
                      </div>
                    </div>
                  </>
                ) : (
                  /* Fallback when no coordinates */
                  <div className="space-y-8">
                    {/* Demo Crime Analysis */}
                    <div className="bg-white rounded-2xl shadow-xl p-6 space-y-6">
                      <div className="flex items-center gap-3 mb-4">
                        <div className="p-2 bg-blue-600 rounded-lg">
                          <Shield className="w-6 h-6 text-white" />
                        </div>
                        <h2 className="text-2xl font-bold tracking-tight text-gray-900">
                          Crime & Safety Analysis
                        </h2>
                      </div>

                      <div className="bg-gradient-to-br from-blue-50 to-indigo-50 rounded-2xl p-6 border border-blue-100">
                        <div className="flex items-center justify-between">
                          <div className="flex-1">
                            <div className="flex items-center gap-6 mb-4">
                              <div className="text-5xl font-bold text-gray-900">
                                7.2
                              </div>
                              <div className="flex flex-col gap-2">
                                <span className="text-lg text-gray-600">
                                  /10
                                </span>
                                <div className="px-4 py-1.5 rounded-full text-sm font-semibold bg-blue-100 text-blue-600 shadow-md">
                                  Safe
                                </div>
                              </div>
                            </div>
                            <p className="text-sm text-gray-500 italic">
                              Based on {property?.city || "area"} crime
                              statistics
                            </p>
                          </div>
                        </div>
                      </div>

                      <div className="grid grid-cols-1 md:grid-cols-3 gap-6">
                        <div className="bg-gradient-to-br from-green-50 to-emerald-50 rounded-xl shadow-md p-6 border border-green-100">
                          <div className="flex items-center gap-3 mb-3">
                            <div className="p-2 bg-green-500 rounded-full">
                              <Shield className="w-5 h-5 text-white" />
                            </div>
                            <span className="text-sm font-semibold text-green-700">
                              Low Risk Zones
                            </span>
                          </div>
                          <div className="text-3xl font-bold text-green-800 mb-1">
                            12
                          </div>
                          <p className="text-xs text-green-600 font-medium">
                            Within 5km radius
                          </p>
                        </div>

                        <div className="bg-gradient-to-br from-yellow-50 to-orange-50 rounded-xl shadow-md p-6 border border-yellow-100">
                          <div className="flex items-center gap-3 mb-3">
                            <div className="p-2 bg-yellow-500 rounded-full">
                              <AlertTriangle className="w-5 h-5 text-white" />
                            </div>
                            <span className="text-sm font-semibold text-yellow-700">
                              Medium Risk Zones
                            </span>
                          </div>
                          <div className="text-3xl font-bold text-yellow-800 mb-1">
                            3
                          </div>
                          <p className="text-xs text-yellow-600 font-medium">
                            Within 5km radius
                          </p>
                        </div>

                        <div className="bg-gradient-to-br from-red-50 to-rose-50 rounded-xl shadow-md p-6 border border-red-100">
                          <div className="flex items-center gap-3 mb-3">
                            <div className="p-2 bg-red-500 rounded-full">
                              <Activity className="w-5 h-5 text-white" />
                            </div>
                            <span className="text-sm font-semibold text-red-700">
                              High Risk Zones
                            </span>
                          </div>
                          <div className="text-3xl font-bold text-red-800 mb-1">
                            1
                          </div>
                          <p className="text-xs text-red-600 font-medium">
                            Within 5km radius
                          </p>
                        </div>
                      </div>
                    </div>

                    {/* Demo Flood Risk */}
                    <div className="bg-white rounded-2xl shadow-xl p-6 space-y-6">
                      <div className="flex items-center gap-3 mb-4">
                        <div className="p-2 bg-cyan-600 rounded-lg">
                          <Waves className="w-6 h-6 text-white" />
                        </div>
                        <h2 className="text-2xl font-bold tracking-tight text-gray-900">
                          Flood Risk Analysis
                        </h2>
                      </div>

                      <div className="bg-gradient-to-br from-green-50 to-emerald-50 rounded-2xl p-6 border border-green-200">
                        <div className="flex items-center justify-between">
                          <div className="flex-1">
                            <div className="flex items-center gap-6 mb-4">
                              <div className="text-5xl font-bold text-gray-900">
                                2.1
                              </div>
                              <div className="flex flex-col gap-2">
                                <span className="text-lg text-gray-600">
                                  /10
                                </span>
                                <div className="px-4 py-1.5 rounded-full text-sm font-semibold bg-green-100 text-green-600 shadow-md">
                                  Low Risk
                                </div>
                              </div>
                            </div>
                            <p className="text-gray-700 text-sm font-medium italic leading-relaxed">
                              This area has minimal flood risk based on
                              historical data and topographical analysis.
                            </p>
                          </div>
                          <div className="hidden md:flex flex-col items-center gap-2">
                            <Shield className="h-16 w-16 text-green-600 opacity-80" />
                            <Waves className="h-10 w-10 text-blue-500 opacity-60" />
                          </div>
                        </div>
                      </div>

                      <div className="grid grid-cols-1 md:grid-cols-3 gap-4">
                        <div className="bg-gradient-to-br from-blue-50 to-cyan-50 rounded-xl shadow-md p-4 border border-blue-100">
                          <div className="flex items-center gap-3 mb-2">
                            <div className="p-2 bg-blue-500 rounded-full">
                              <AlertTriangle className="w-4 h-4 text-white" />
                            </div>
                            <span className="text-sm font-semibold text-blue-700">
                              Flood Alerts
                            </span>
                          </div>
                          <div className="text-2xl font-bold text-blue-800">
                            0
                          </div>
                        </div>

                        <div className="bg-gradient-to-br from-emerald-50 to-green-50 rounded-xl shadow-md p-4 border border-emerald-100">
                          <div className="flex items-center gap-3 mb-2">
                            <div className="p-2 bg-emerald-500 rounded-full">
                              <TrendingUp className="w-4 h-4 text-white" />
                            </div>
                            <span className="text-sm font-semibold text-emerald-700">
                              Confidence
                            </span>
                          </div>
                          <div className="text-2xl font-bold text-emerald-800">
                            85%
                          </div>
                        </div>

                        <div className="bg-gradient-to-br from-purple-50 to-indigo-50 rounded-xl shadow-md p-4 border border-purple-100">
                          <div className="flex items-center gap-3 mb-2">
                            <div className="p-2 bg-purple-500 rounded-full">
                              <Clock className="w-4 h-4 text-white" />
                            </div>
                            <span className="text-sm font-semibold text-purple-700">
                              Last Updated
                            </span>
                          </div>
                          <div className="text-sm font-bold text-purple-800">
                            Real-time
                          </div>
                        </div>
                      </div>
                    </div>

                    {/* Note about coordinates */}
                    <div className="bg-amber-50 border border-amber-200 rounded-2xl p-6">
                      <div className="flex items-start gap-3">
                        <AlertTriangle className="w-5 h-5 text-amber-600 mt-0.5 flex-shrink-0" />
                        <div>
                          <h4 className="text-sm font-semibold text-amber-800 mb-1">
                            Demo Data
                          </h4>
                          <p className="text-sm text-amber-700 leading-relaxed">
                            Showing sample safety data. For precise
                            location-based analysis, property coordinates are
                            required.
                          </p>
                        </div>
                      </div>
                    </div>
                  </div>
                )}
              </div>
            )}
          </div>

          {/* Hyperlocal Insights Section */}
          <div className="mb-6">
            <HyperlocalInsights
              propertyId={id!}
              propertyLat={property?.latitude || 12.9716}
              propertyLon={property?.longitude || 77.5946}
            />
          </div>

          {/* Right Column (30%) - Property Visit Booking */}
          <div className="lg:w-[30%]">
            <div className="lg:sticky lg:top-6">
              {/* Book a Visit Section */}
              <div className="bg-gradient-to-br from-blue-50 to-indigo-100 border border-blue-200 rounded-xl p-6 shadow-lg">
                <div className="text-center mb-4">
                  <div className="inline-flex items-center justify-center w-16 h-16 bg-blue-600 rounded-full mb-3">
                    <svg
                      className="w-8 h-8 text-white"
                      fill="none"
                      stroke="currentColor"
                      viewBox="0 0 24 24"
                    >
                      <path
                        strokeLinecap="round"
                        strokeLinejoin="round"
                        strokeWidth={2}
                        d="M8 7V3a1 1 0 011-1h6a1 1 0 011 1v4h3a2 2 0 012 2v9a2 2 0 01-2 2H5a2 2 0 01-2-2V9a2 2 0 012-2h3z"
                      />
                    </svg>
                  </div>
                  <h3 className="text-xl font-bold text-gray-900 mb-2">
                    Interested in this property?
                  </h3>
                  <p className="text-gray-600 text-sm">
                    Schedule a visit to explore this amazing property in person
                  </p>
                </div>

                <div className="space-y-3 mb-4">
                  <div className="flex items-center text-sm text-gray-700">
                    <svg
                      className="w-4 h-4 mr-3 text-green-600"
                      fill="currentColor"
                      viewBox="0 0 20 20"
                    >
                      <path
                        fillRule="evenodd"
                        d="M16.707 5.293a1 1 0 010 1.414l-8 8a1 1 0 01-1.414 0l-4-4a1 1 0 011.414-1.414L8 12.586l7.293-7.293a1 1 0 011.414 0z"
                        clipRule="evenodd"
                      />
                    </svg>
                    Free property consultation
                  </div>
                  <div className="flex items-center text-sm text-gray-700">
                    <svg
                      className="w-4 h-4 mr-3 text-green-600"
                      fill="currentColor"
                      viewBox="0 0 20 20"
                    >
                      <path
                        fillRule="evenodd"
                        d="M16.707 5.293a1 1 0 010 1.414l-8 8a1 1 0 01-1.414 0l-4-4a1 1 0 011.414-1.414L8 12.586l7.293-7.293a1 1 0 011.414 0z"
                        clipRule="evenodd"
                      />
                    </svg>
                    Flexible timing options
                  </div>
                  <div className="flex items-center text-sm text-gray-700">
                    <svg
                      className="w-4 h-4 mr-3 text-green-600"
                      fill="currentColor"
                      viewBox="0 0 20 20"
                    >
                      <path
                        fillRule="evenodd"
                        d="M16.707 5.293a1 1 0 010 1.414l-8 8a1 1 0 01-1.414 0l-4-4a1 1 0 011.414-1.414L8 12.586l7.293-7.293a1 1 0 011.414 0z"
                        clipRule="evenodd"
                      />
                    </svg>
                    Expert guidance available
                  </div>
                </div>

                <div className="space-y-3">
                  <div className="flex gap-2">
                    <button
                      onClick={() => navigate(`/property/${id}/book-visit`)}
                      className="flex-1 px-4 py-3 bg-blue-600 text-white rounded-lg font-semibold hover:bg-blue-700 transition-colors flex items-center justify-center gap-2"
                    >
                      <svg
                        className="w-5 h-5"
                        fill="none"
                        stroke="currentColor"
                        viewBox="0 0 24 24"
                      >
                        <path
                          strokeLinecap="round"
                          strokeLinejoin="round"
                          strokeWidth={2}
                          d="M8 7V3a1 1 0 011-1h6a1 1 0 011 1v4h3a2 2 0 012 2v9a2 2 0 01-2 2H5a2 2 0 01-2-2V9a2 2 0 012-2h3z"
                        />
                      </svg>
                      Schedule Tour
                    </button>
                    <button
                      onClick={() => {
                        alert("Property added to your itinerary!");
                      }}
                      className="flex-1 px-4 py-3 bg-green-600 text-white rounded-lg font-semibold hover:bg-green-700 transition-colors flex items-center justify-center gap-2"
                    >
                      <svg
                        className="w-5 h-5"
                        fill="none"
                        stroke="currentColor"
                        viewBox="0 0 24 24"
                      >
                        <path
                          strokeLinecap="round"
                          strokeLinejoin="round"
                          strokeWidth={2}
                          d="M9 5H7a2 2 0 00-2 2v12a2 2 0 002 2h10a2 2 0 002-2V7a2 2 0 00-2-2h-2M9 5a2 2 0 002 2h2a2 2 0 002-2M9 5a2 2 0 012-2h2a2 2 0 012 2m-6 9l2 2 4-4"
                        />
                      </svg>
                      Add to Itinerary
                    </button>
                  </div>

                  <WishlistButton
                    propertyId={id!}
                    variant="detail"
                    className="w-full justify-center"
                  />
                </div>

                <div className="mt-6 pt-4 border-t border-gray-200">
                  <div className="flex items-center justify-center text-sm text-gray-600">
                    <svg
                      className="w-4 h-4 mr-2 text-blue-600"
                      fill="currentColor"
                      viewBox="0 0 20 20"
                    >
                      <path
                        fillRule="evenodd"
                        d="M2.166 4.999A11.954 11.954 0 0010 1.944 11.954 11.954 0 0017.834 5c.11.65.166 1.32.166 2.001 0 5.225-3.34 9.67-8 11.317C5.34 16.67 2 12.225 2 7c0-.682.057-1.35.166-2.001zm11.541 3.708a1 1 0 00-1.414-1.414L9 10.586 7.707 9.293a1 1 0 00-1.414 1.414l2 2a1 1 0 001.414 0l4-4z"
                        clipRule="evenodd"
                      />
                    </svg>
                    100% Safe & Secure
                  </div>
                </div>
              </div>
            </div>
          </div>
        </div>
      </div>

      {/* Tour Properties Selection Modal */}
      {showTourModal && (
        <div className="fixed inset-0 bg-black/50 backdrop-blur-sm flex items-center justify-center z-50 p-4">
          <div className="bg-white rounded-lg max-w-4xl w-full max-h-[90vh] overflow-hidden">
            <div className="p-6 border-b">
              <div className="flex items-center justify-between">
                <h2 className="text-2xl font-bold text-black">
                  Select Properties for Tour
                </h2>
                <button
                  onClick={() => setShowTourModal(false)}
                  className="text-gray-400 hover:text-gray-600"
                >
                  <svg
                    className="w-6 h-6"
                    fill="none"
                    stroke="currentColor"
                    viewBox="0 0 24 24"
                  >
                    <path
                      strokeLinecap="round"
                      strokeLinejoin="round"
                      strokeWidth={2}
                      d="M6 18L18 6M6 6l12 12"
                    />
                  </svg>
                </button>
              </div>
              <p className="text-gray-600 mt-2">
                Select one or more properties to schedule your tour
              </p>
            </div>

            <div className="p-6 overflow-y-auto max-h-[60vh]">
              {loadingTourProperties ? (
                <div className="text-center py-12">
                  <div className="animate-spin rounded-full h-12 w-12 border-b-2 border-blue-600 mx-auto"></div>
                  <p className="text-gray-600 mt-4">Loading properties...</p>
                </div>
              ) : availableProperties.length === 0 ? (
                <div className="text-center py-12">
                  <p className="text-gray-600">
                    No properties available for tour
                  </p>
                </div>
              ) : (
                <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
                  {availableProperties.map((prop) => (
                    <div
                      key={prop.id}
                      onClick={() => togglePropertySelection(prop.id)}
                      className={`relative rounded-xl p-5 cursor-pointer transition-all shadow-md hover:shadow-xl ${
                        selectedProperties.includes(prop.id)
                          ? "border-2 border-blue-600 bg-white"
                          : "border-2 border-gray-200 bg-white hover:border-blue-300"
                      }`}
                    >
                      <div className="flex gap-4">
                        <div className="relative flex-shrink-0">
                          {prop.primary_image ? (
                            <img
                              src={prop.primary_image}
                              alt={prop.title}
                              className="w-28 h-28 object-cover rounded-lg shadow-sm"
                            />
                          ) : (
                            <div className="w-28 h-28 bg-gray-100 rounded-lg flex items-center justify-center">
                              <svg
                                className="w-10 h-10 text-gray-400"
                                fill="none"
                                stroke="currentColor"
                                viewBox="0 0 24 24"
                              >
                                <path
                                  strokeLinecap="round"
                                  strokeLinejoin="round"
                                  strokeWidth={2}
                                  d="M3 12l2-2m0 0l7-7 7 7M5 10v10a1 1 0 001 1h3m10-11l2 2m-2-2v10a1 1 0 01-1 1h-3m-6 0a1 1 0 001-1v-4a1 1 0 011-1h2a1 1 0 011 1v4a1 1 0 001 1m-6 0h6"
                                />
                              </svg>
                            </div>
                          )}
                        </div>
                        <div className="flex-1 min-w-0">
                          <h3 className="font-bold text-black text-base mb-1 truncate">
                            {prop.title}
                          </h3>
                          <p className="text-sm text-gray-600 mb-2 line-clamp-1">
                            {prop.address}, {prop.city}
                          </p>
                          <p className="text-lg text-blue-600 font-bold mb-2">
                            ₹{prop.sale_price?.toLocaleString()}
                          </p>
                          <div className="flex gap-3 text-xs text-gray-600">
                            {prop.bedrooms && (
                              <span className="flex items-center gap-1 bg-gray-100 px-2 py-1 rounded">
                                <svg
                                  className="w-3 h-3"
                                  fill="currentColor"
                                  viewBox="0 0 20 20"
                                >
                                  <path d="M10.707 2.293a1 1 0 00-1.414 0l-7 7a1 1 0 001.414 1.414L4 10.414V17a1 1 0 001 1h2a1 1 0 001-1v-2a1 1 0 011-1h2a1 1 0 011 1v2a1 1 0 001 1h2a1 1 0 001-1v-6.586l.293.293a1 1 0 001.414-1.414l-7-7z" />
                                </svg>
                                {prop.bedrooms} BHK
                              </span>
                            )}
                            {prop.built_up_area && (
                              <span className="flex items-center gap-1 bg-gray-100 px-2 py-1 rounded">
                                <svg
                                  className="w-3 h-3"
                                  fill="currentColor"
                                  viewBox="0 0 20 20"
                                >
                                  <path
                                    fillRule="evenodd"
                                    d="M3 3a1 1 0 000 2v8a2 2 0 002 2h2.586l-1.293 1.293a1 1 0 101.414 1.414L10 15.414l2.293 2.293a1 1 0 001.414-1.414L12.414 15H15a2 2 0 002-2V5a1 1 0 100-2H3z"
                                    clipRule="evenodd"
                                  />
                                </svg>
                                {prop.built_up_area} sq.ft
                              </span>
                            )}
                          </div>
                        </div>
                      </div>
                    </div>
                  ))}
                </div>
              )}
            </div>

            <div className="p-6 border-t-2 bg-white">
              <div className="flex items-center justify-between">
                <div className="flex items-center gap-3">
                  <div className="w-12 h-12 bg-blue-600 rounded-full flex items-center justify-center">
                    <span className="text-white font-bold text-xl">
                      {selectedProperties.length}
                    </span>
                  </div>
                  <p className="text-black font-bold text-xl">
                    {selectedProperties.length}{" "}
                    {selectedProperties.length === 1
                      ? "Property"
                      : "Properties"}{" "}
                    Selected
                  </p>
                </div>
                <div className="flex gap-4">
                  <button
                    onClick={() => setShowTourModal(false)}
                    className="px-8 py-3 border-2 border-black rounded-lg hover:bg-gray-100 text-black font-bold text-base"
                  >
                    Cancel
                  </button>
                  <button
                    onClick={handleProceedToBooking}
                    disabled={selectedProperties.length === 0}
                    className="px-10 py-3 bg-black text-white rounded-lg hover:bg-gray-800 disabled:opacity-50 disabled:cursor-not-allowed font-bold text-base"
                  >
                    Proceed to Book →
                  </button>
                </div>
              </div>
            </div>
          </div>
        </div>
      )}

      {/* Registration Details Modal */}
      {showRegistrationModal && (
        <AuctionRegistrationModal
          isOpen={showRegistrationModal}
          onClose={() => setShowRegistrationModal(false)}
          onSubmit={handleRegistrationSubmit}
          loading={registrationLoading}
          registrationFee={
            auctionStatus?.session?.registration_fee ||
            property?.bidder_registration_fee ||
            50000
          }
        />
      )}

      {/* Auction Registration Modal */}
      {showAuctionModal && (
        <div className="fixed inset-0 !backdrop-blur-sm flex items-center justify-center z-50 p-4">
          <div className="bg-white rounded-lg max-w-md w-full p-6">
            <div className="flex justify-between items-center mb-4">
              <h3 className="text-lg font-bold">Register for Auction</h3>
              <button
                onClick={() => setShowAuctionModal(false)}
                className="text-gray-400 hover:text-gray-600"
              >
                <svg
                  className="w-6 h-6"
                  fill="none"
                  stroke="currentColor"
                  viewBox="0 0 24 24"
                >
                  <path
                    strokeLinecap="round"
                    strokeLinejoin="round"
                    strokeWidth={2}
                    d="M6 18L18 6M6 6l12 12"
                  />
                </svg>
              </button>
            </div>

            {registrationStatus === "idle" && (
              <div>
                <div className="mb-4 p-4 bg-blue-50 rounded-lg">
                  <h4 className="font-semibold text-blue-900 mb-2">
                    Registration Fee Required
                  </h4>
                  <p className="text-blue-700 text-sm mb-2">
                    To participate in this auction, you need to pay a
                    registration fee of:
                  </p>
                  <div className="text-2xl font-bold text-blue-600">
                    {formatPrice(
                      auctionStatus?.session?.registration_fee ||
                        property?.bidder_registration_fee ||
                        50000,
                    )}
                  </div>
                </div>

                <div className="mb-4">
                  <h5 className="font-semibold mb-2">What happens next?</h5>
                  <ul className="text-sm text-gray-600 space-y-1">
                    <li>• Pay the registration fee securely</li>
                    <li>• Registration will be processed for approval</li>
                    <li>• Fee is refundable if you don't win</li>
                    <li>• Receive auction updates via email/SMS</li>
                  </ul>
                </div>

                <button
                  onClick={handlePayRegistrationFee}
                  disabled={paymentLoading}
                  className="w-full bg-blue-600 text-white py-3 px-6 rounded-lg font-semibold hover:bg-blue-700 disabled:opacity-50 flex items-center justify-center transition-colors shadow-lg"
                  style={{
                    backgroundColor: paymentLoading ? "#9ca3af" : "#2563eb",
                    color: "white",
                    minHeight: "48px",
                  }}
                >
                  {paymentLoading ? (
                    <>
                      <div className="animate-spin rounded-full h-5 w-5 border-b-2 border-white mr-2"></div>
                      <span>Processing Payment...</span>
                    </>
                  ) : (
                    <>
                      <svg
                        className="w-5 h-5 mr-2"
                        fill="none"
                        stroke="currentColor"
                        viewBox="0 0 24 24"
                      >
                        <path
                          strokeLinecap="round"
                          strokeLinejoin="round"
                          strokeWidth={2}
                          d="M17 9V7a2 2 0 00-2-2H5a2 2 0 00-2 2v6a2 2 0 002 2h2m2 4h10a2 2 0 002-2v-6a2 2 0 00-2-2H9a2 2 0 00-2 2v2a2 2 0 002 2z"
                        />
                      </svg>
                      <span>Pay Registration Fee</span>
                    </>
                  )}
                </button>
              </div>
            )}

            {registrationStatus === "payment" && (
              <div className="text-center py-8">
                <div className="animate-spin rounded-full h-12 w-12 border-b-2 border-blue-500 mx-auto mb-4"></div>
                <p className="text-gray-600">Processing payment...</p>
              </div>
            )}

            {(registrationStatus === "registered" ||
              registrationStatus === "pending") && (
              <div className="text-center py-4">
                <div
                  className={`w-16 h-16 rounded-full flex items-center justify-center mx-auto mb-4 ${
                    registrationStatus === "pending"
                      ? "bg-orange-100"
                      : "bg-green-100"
                  }`}
                >
                  <svg
                    className={`w-8 h-8 ${
                      registrationStatus === "pending"
                        ? "text-orange-600"
                        : "text-green-600"
                    }`}
                    fill="none"
                    stroke="currentColor"
                    viewBox="0 0 24 24"
                  >
                    <path
                      strokeLinecap="round"
                      strokeLinejoin="round"
                      strokeWidth={2}
                      d="M5 13l4 4L19 7"
                    />
                  </svg>
                </div>
                <h4
                  className={`text-lg font-bold mb-2 ${
                    registrationStatus === "pending"
                      ? "text-orange-600"
                      : "text-green-600"
                  }`}
                >
                  {registrationStatus === "pending"
                    ? "Registration Pending!"
                    : "Registration Successful!"}
                </h4>
                <p className="text-gray-600 text-sm mb-4">
                  {registrationStatus === "pending"
                    ? "Your registration fee has been paid and is being processed. You will be notified once approved."
                    : "You are now registered for this auction. You can start placing bids."}
                </p>
                <button
                  onClick={() => {
                    setShowAuctionModal(false);
                    setRegistrationStatus("idle");
                  }}
                  className={`text-white px-6 py-2 rounded-lg ${
                    registrationStatus === "pending"
                      ? "bg-orange-600 hover:bg-orange-700"
                      : "bg-green-600 hover:bg-green-700"
                  }`}
                >
                  {registrationStatus === "pending" ? "Close" : "Start Bidding"}
                </button>
              </div>
            )}
          </div>
        </div>
      )}

      {/* Enhanced Bidding Modal */}
      {showBiddingModal && (
        <div className="fixed inset-0 bg-black/50 backdrop-blur-sm flex items-center justify-center z-50 p-4">
          <div className="bg-white rounded-xl max-w-lg w-full p-6 shadow-2xl">
            <div className="flex justify-between items-center mb-6">
              <div className="flex items-center gap-3">
                <div className="w-10 h-10 bg-blue-600 rounded-full flex items-center justify-center">
                  <svg
                    className="w-6 h-6 text-white"
                    fill="none"
                    stroke="currentColor"
                    viewBox="0 0 24 24"
                  >
                    <path
                      strokeLinecap="round"
                      strokeLinejoin="round"
                      strokeWidth={2}
                      d="M13 10V3L4 14h7v7l9-11h-7z"
                    />
                  </svg>
                </div>
                <h3 className="text-xl font-bold text-gray-900">
                  Place Your Bid
                </h3>
              </div>
              <button
                onClick={() => {
                  setShowBiddingModal(false);
                  setBidAmount("");
                }}
                className="text-gray-400 hover:text-gray-600 p-1 rounded-full hover:bg-gray-100"
              >
                <svg
                  className="w-6 h-6"
                  fill="none"
                  stroke="currentColor"
                  viewBox="0 0 24 24"
                >
                  <path
                    strokeLinecap="round"
                    strokeLinejoin="round"
                    strokeWidth={2}
                    d="M6 18L18 6M6 6l12 12"
                  />
                </svg>
              </button>
            </div>

            {/* Auction Status Card */}
            <div className="mb-6 p-4 bg-gradient-to-r from-blue-50 to-indigo-50 rounded-lg border border-blue-200">
              <h4 className="font-semibold text-blue-900 mb-3 flex items-center">
                <svg
                  className="w-5 h-5 mr-2"
                  fill="currentColor"
                  viewBox="0 0 20 20"
                >
                  <path
                    fillRule="evenodd"
                    d="M12 7a1 1 0 110-2h5a1 1 0 011 1v5a1 1 0 11-2 0V8.414l-4.293 4.293a1 1 0 01-1.414 0L8 10.414l-4.293 4.293a1 1 0 01-1.414-1.414l5-5a1 1 0 011.414 0L11 10.586 14.586 7H12z"
                    clipRule="evenodd"
                  />
                </svg>
                Current Auction Status
              </h4>
              <div className="grid grid-cols-1 gap-3 text-sm">
                <div className="flex justify-between items-center p-2 bg-white rounded border">
                  <span className="text-gray-600">Starting Bid:</span>
                  <span className="font-bold text-blue-600">
                    {formatPrice(property?.starting_bidding_price || 0)}
                  </span>
                </div>
                <div className="flex justify-between items-center p-2 bg-white rounded border">
                  <span className="text-gray-600">Minimum Increment:</span>
                  <span className="font-semibold text-gray-900">
                    {formatPrice(
                      auctionStatus?.session?.minimum_increment ||
                        property?.minimum_increment ||
                        50000,
                    )}
                  </span>
                </div>
                <div className="flex justify-between items-center p-2 bg-green-50 rounded border border-green-200">
                  <span className="text-green-700 font-medium">
                    Your Minimum Bid:
                  </span>
                  <span className="font-bold text-green-600 text-lg">
                    {formatPrice(
                      (auctionStatus?.session?.current_highest_bid ||
                        auctionStatus?.session?.starting_bid_price ||
                        property?.starting_bidding_price ||
                        0) +
                        (auctionStatus?.session?.minimum_increment ||
                          property?.minimum_increment ||
                          50000),
                    )}
                  </span>
                </div>
              </div>
            </div>

            {/* Bid Amount Input */}
            <div className="mb-6">
              <label className="block text-sm font-semibold text-gray-700 mb-3">
                Enter Your Bid Amount
              </label>
              <div className="relative">
                <div className="absolute inset-y-0 left-0 pl-3 flex items-center pointer-events-none">
                  <span className="text-gray-500 text-lg font-semibold">₹</span>
                </div>
                <input
                  type="number"
                  value={bidAmount}
                  onChange={(e) => setBidAmount(e.target.value)}
                  placeholder={`Minimum: ${((auctionStatus?.session?.current_highest_bid || auctionStatus?.session?.starting_bid_price || property?.starting_bidding_price || 0) + (auctionStatus?.session?.minimum_increment || property?.minimum_increment || 50000)).toLocaleString()}`}
                  className="w-full pl-8 pr-4 py-3 text-lg font-semibold border-2 border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-500 focus:border-blue-500 transition-colors"
                  min={
                    (auctionStatus?.session?.current_highest_bid ||
                      auctionStatus?.session?.starting_bid_price ||
                      property?.starting_bidding_price ||
                      0) +
                    (auctionStatus?.session?.minimum_increment ||
                      property?.minimum_increment ||
                      50000)
                  }
                  step={
                    auctionStatus?.session?.minimum_increment ||
                    property?.minimum_increment ||
                    50000
                  }
                />
              </div>
              {bidAmount &&
                parseFloat(bidAmount) >=
                  (auctionStatus?.session?.current_highest_bid ||
                    auctionStatus?.session?.starting_bid_price ||
                    property?.starting_bidding_price ||
                    0) +
                    (auctionStatus?.session?.minimum_increment ||
                      property?.minimum_increment ||
                      50000) && (
                  <div className="mt-2 p-2 bg-green-50 rounded border border-green-200">
                    <p className="text-sm text-green-700">
                      ✓ Your bid:{" "}
                      <span className="font-bold">
                        {formatPrice(parseFloat(bidAmount))}
                      </span>
                    </p>
                  </div>
                )}
              {bidAmount &&
                parseFloat(bidAmount) <
                  (auctionStatus?.session?.current_highest_bid ||
                    auctionStatus?.session?.starting_bid_price ||
                    property?.starting_bidding_price ||
                    0) +
                    (auctionStatus?.session?.minimum_increment ||
                      property?.minimum_increment ||
                      50000) && (
                  <div className="mt-2 p-2 bg-red-50 rounded border border-red-200">
                    <p className="text-sm text-red-700">
                      ⚠ Bid must be at least{" "}
                      {formatPrice(
                        (auctionStatus?.session?.current_highest_bid ||
                          auctionStatus?.session?.starting_bid_price ||
                          property?.starting_bidding_price ||
                          0) +
                          (auctionStatus?.session?.minimum_increment ||
                            property?.minimum_increment ||
                            50000),
                      )}
                    </p>
                  </div>
                )}
            </div>

            {/* Action Buttons */}
            <div className="flex gap-3">
              <button
                onClick={() => {
                  setShowBiddingModal(false);
                  setBidAmount("");
                }}
                className="flex-1 px-4 py-3 border-2 border-gray-300 text-gray-700 rounded-lg font-semibold hover:bg-gray-50 transition-colors"
              >
                Cancel
              </button>
              <button
                onClick={async () => {
                  const currentHighest =
                    auctionStatus?.session?.current_highest_bid ||
                    auctionStatus?.session?.starting_bid_price ||
                    property?.starting_bidding_price ||
                    0;
                  const minIncrement =
                    auctionStatus?.session?.minimum_increment ||
                    property?.minimum_increment ||
                    50000;
                  const minBid = currentHighest + minIncrement;

                  if (!bidAmount) {
                    alert("Please enter a bid amount");
                    return;
                  }

                  if (parseFloat(bidAmount) < minBid) {
                    alert(`Bid amount must be at least ${formatPrice(minBid)}`);
                    return;
                  }

                  setPlacingBid(true);
                  try {
                    // Find session ID for this property
                    const auctionsResponse = await fetch(
                      "http://localhost:8090/api/auctions/active",
                      {
                        headers: {
                          Authorization: `Bearer ${localStorage.getItem("access_token")}`,
                          "Content-Type": "application/json",
                        },
                      },
                    );

                    if (!auctionsResponse.ok) {
                      throw new Error("Failed to fetch auction sessions");
                    }

                    const auctionsData = await auctionsResponse.json();
                    const sessions = auctionsData.sessions || [];
                    const propertySession = sessions.find(
                      (session: any) => session.property_id === property?.id,
                    );

                    if (!propertySession) {
                      throw new Error(
                        "No active auction session found for this property",
                      );
                    }

                    // Place bid
                    const bidResponse = await fetch(
                      `http://localhost:8090/api/auctions/${propertySession.id}/place-bid`,
                      {
                        method: "POST",
                        headers: {
                          "Content-Type": "application/json",
                          Authorization: `Bearer ${localStorage.getItem("access_token")}`,
                        },
                        body: JSON.stringify({
                          bid_amount: parseFloat(bidAmount),
                          bid_type: "manual",
                        }),
                      },
                    );

                    if (!bidResponse.ok) {
                      const errorText = await bidResponse.text();
                      let errorMessage = "Failed to place bid";
                      try {
                        const errorData = JSON.parse(errorText);
                        errorMessage =
                          errorData.detail || errorData.message || errorMessage;
                      } catch (e) {
                        errorMessage = `Failed to place bid: ${bidResponse.status} ${bidResponse.statusText}`;
                      }
                      throw new Error(errorMessage);
                    }

                    const bidResult = await bidResponse.json();
                    alert(
                      `🎉 Bid placed successfully!\n\nYour bid: ${formatPrice(parseFloat(bidAmount))}\nBid ID: ${bidResult.bid_id || "N/A"}`,
                    );
                    setShowBiddingModal(false);
                    setBidAmount("");

                    // Optionally refresh the page to show updated auction status
                    // window.location.reload();
                  } catch (error) {
                    console.error("Bidding error:", error);
                    alert(`❌ Failed to place bid:\n\n${error.message}`);
                  } finally {
                    setPlacingBid(false);
                  }
                }}
                disabled={
                  placingBid ||
                  !bidAmount ||
                  parseFloat(bidAmount) <
                    (auctionStatus?.session?.current_highest_bid ||
                      auctionStatus?.session?.starting_bid_price ||
                      property?.starting_bidding_price ||
                      0) +
                      (auctionStatus?.session?.minimum_increment ||
                        property?.minimum_increment ||
                        50000)
                }
                className="flex-1 px-4 py-3 bg-blue-600 text-white rounded-lg font-semibold hover:bg-blue-700 disabled:opacity-50 disabled:cursor-not-allowed flex items-center justify-center gap-2 transition-colors"
                style={{ backgroundColor: placingBid ? "#9ca3af" : "#2563eb" }}
              >
                {placingBid ? (
                  <>
                    <div className="animate-spin rounded-full h-5 w-5 border-b-2 border-white"></div>
                    <span>Placing Bid...</span>
                  </>
                ) : (
                  <>
                    <svg
                      className="w-5 h-5"
                      fill="none"
                      stroke="currentColor"
                      viewBox="0 0 24 24"
                    >
                      <path
                        strokeLinecap="round"
                        strokeLinejoin="round"
                        strokeWidth={2}
                        d="M13 10V3L4 14h7v7l9-11h-7z"
                      />
                    </svg>
                    <span>
                      Place Bid{" "}
                      {bidAmount
                        ? `- ${formatPrice(parseFloat(bidAmount))}`
                        : ""}
                    </span>
                  </>
                )}
              </button>
            </div>

            {/* Quick Bid Buttons */}
            <div className="mt-4 pt-4 border-t border-gray-200">
              <p className="text-sm text-gray-600 mb-2">Quick bid amounts:</p>
              <div className="flex gap-2">
                {(() => {
                  const currentHighest =
                    auctionStatus?.session?.current_highest_bid ||
                    auctionStatus?.session?.starting_bid_price ||
                    property?.starting_bidding_price ||
                    0;
                  const minIncrement =
                    auctionStatus?.session?.minimum_increment ||
                    property?.minimum_increment ||
                    50000;
                  return [
                    currentHighest + minIncrement,
                    currentHighest + minIncrement * 2,
                    currentHighest + minIncrement * 3,
                  ];
                })().map((amount, index) => (
                  <button
                    key={index}
                    onClick={() => setBidAmount(amount.toString())}
                    className="flex-1 px-3 py-2 text-sm border border-blue-300 text-blue-600 rounded hover:bg-blue-50 transition-colors"
                  >
                    {formatPrice(amount)}
                  </button>
                ))}
              </div>
            </div>
          </div>
        </div>
      )}
      {/* Voice Assistant Modal */}
      {showCharacterModal && (
        <div className="voice-assistant-modal">
          <div className="voice-assistant-modal-content">
            {/* Close Button */}
            <div className="voice-assistant-close-btn-wrapper">
              <button
                onClick={() => {
                  voiceAssistant.stopSpeaking();
                  voiceAssistant.stopListening();
                  setShowCharacterModal(false);
                }}
                className="voice-assistant-close-btn"
              >
                <svg
                  className="w-6 h-6"
                  fill="none"
                  stroke="currentColor"
                  viewBox="0 0 24 24"
                >
                  <path
                    strokeLinecap="round"
                    strokeLinejoin="round"
                    strokeWidth={2}
                    d="M6 18L18 6M6 6l12 12"
                  />
                </svg>
              </button>
            </div>

            {!voiceAssistant.isSupported ? (
              <div className="text-center py-12 px-6">
                <div className="w-20 h-20 mx-auto mb-4 bg-blue-50 rounded-full flex items-center justify-center">
                  <svg
                    className="w-10 h-10 text-blue-600"
                    fill="none"
                    stroke="currentColor"
                    viewBox="0 0 24 24"
                  >
                    <path
                      strokeLinecap="round"
                      strokeLinejoin="round"
                      strokeWidth={2}
                      d="M12 8v4m0 4h.01M21 12a9 9 0 11-18 0 9 9 0 0118 0z"
                    />
                  </svg>
                </div>
                <h4 className="text-xl font-bold text-blue-900 mb-3">
                  Voice Not Supported
                </h4>
                <p className="text-gray-600 text-sm leading-relaxed">
                  Your browser doesn't support voice features. Please use a
                  modern browser like Chrome or Edge.
                </p>
              </div>
            ) : (
              <>
                <div className="voice-assistant-header">
                  <div
                    style={{
                      display: "flex",
                      justifyContent: "space-between",
                      alignItems: "center",
                      marginBottom: "0.5rem",
                    }}
                  >
                    <h3 className="text-2xl font-bold text-blue-900">
                      Property Assistant
                    </h3>
                    <select
                      value={voiceAssistant.language}
                      onChange={(e) =>
                        voiceAssistant.setLanguage(e.target.value)
                      }
                      style={{
                        padding: "0.5rem",
                        borderRadius: "0.5rem",
                        border: "2px solid #3b82f6",
                        backgroundColor: "white",
                        color: "#1e40af",
                        fontWeight: "600",
                        cursor: "pointer",
                        fontSize: "0.875rem",
                      }}
                    >
                      <option value="en">English</option>
                      <option value="kn">ಕನ್ನಡ</option>
                    </select>
                  </div>
                </div>

                {/* Assistant Avatar with Controls and Conversation */}
                <div className="voice-assistant-avatar-section">
                  <Character3D
                    isListening={voiceAssistant.isListening}
                    isSpeaking={voiceAssistant.isSpeaking}
                    language={voiceAssistant.language}
                  />
                </div>

                {/* Description */}
                <div className="px-6 pb-4 bg-white">
                  <div className="bg-blue-50 rounded-xl p-4 border border-blue-100">
                    <p className="text-sm text-blue-900 font-medium mb-3">
                      💬 I can help you with:
                    </p>
                    <div className="space-y-2 text-sm text-blue-700">
                      <div className="flex items-center gap-2">
                        <span className="w-1.5 h-1.5 bg-blue-500 rounded-full"></span>
                        <span>Property pricing & payment details</span>
                      </div>
                      <div className="flex items-center gap-2">
                        <span className="w-1.5 h-1.5 bg-blue-500 rounded-full"></span>
                        <span>Location & nearby amenities</span>
                      </div>
                      <div className="flex items-center gap-2">
                        <span className="w-1.5 h-1.5 bg-blue-500 rounded-full"></span>
                        <span>Room configuration & specifications</span>
                      </div>
                      <div className="flex items-center gap-2">
                        <span className="w-1.5 h-1.5 bg-blue-500 rounded-full"></span>
                        <span>Features & facilities</span>
                      </div>
                    </div>
                  </div>
                </div>

                {/* Conversation History */}
                {conversationHistory.length > 0 && (
                  <div style={{ display: "none" }}></div>
                )}

                {/* Error Display */}
                {voiceAssistant.error && (
                  <div className="px-6 pb-4 bg-white">
                    <div className="bg-red-50 border border-red-200 rounded-xl p-4">
                      <div className="flex items-start gap-3">
                        <svg
                          className="w-5 h-5 text-red-600 flex-shrink-0 mt-0.5"
                          fill="none"
                          stroke="currentColor"
                          viewBox="0 0 24 24"
                        >
                          <path
                            strokeLinecap="round"
                            strokeLinejoin="round"
                            strokeWidth={2}
                            d="M12 8v4m0 4h.01M21 12a9 9 0 11-18 0 9 9 0 0118 0z"
                          />
                        </svg>
                        <p className="text-sm text-red-700">
                          {voiceAssistant.error}
                        </p>
                      </div>
                    </div>
                  </div>
                )}

                {/* Current Transcript */}
                {voiceAssistant.transcript && (
                  <div className="mx-6 mb-4 p-3 bg-blue-50 border border-blue-200 rounded-lg">
                    <p className="text-sm text-blue-800">
                      <span className="font-medium">You said:</span>{" "}
                      {voiceAssistant.transcript}
                    </p>
                  </div>
                )}

                {/* Error Display */}
                {voiceAssistant.error && (
                  <div className="mx-6 mb-4 p-3 bg-red-50 border border-red-200 rounded-lg">
                    <p className="text-red-700 text-sm">
                      {voiceAssistant.error}
                    </p>
                  </div>
                )}

                {/* Bottom Controls */}
                <div className="flex items-center justify-center gap-4 pb-6 px-6">
                  {/* Conversation History Button */}
                  {conversationHistory.length > 0 ? (
                    <button
                      onClick={() => setShowConversationHistory(true)}
                      className="w-12 h-12 bg-blue-600 text-white rounded-full flex items-center justify-center hover:bg-blue-700 transition-colors relative"
                      title="View conversation history"
                    >
                      <span className="text-3xl">💬</span>
                      <span className="absolute -top-0.5 -right-0.5 bg-blue-500 text-white text-xs rounded-full w-4 h-4 flex items-center justify-center font-bold border border-white">
                        {conversationHistory.length}
                      </span>
                    </button>
                  ) : (
                    <button
                      className="w-12 h-12 bg-gray-300 text-white rounded-full flex items-center justify-center cursor-not-allowed"
                      disabled
                    >
                      <span className="text-3xl">💬</span>
                    </button>
                  )}

                  {/* Status Display Button */}
                  <button
                    disabled
                    className="flex-1 bg-gray-100 text-gray-700 py-3 px-6 rounded-full font-medium cursor-default"
                  >
                    {voiceAssistant.isListening
                      ? "🎤 Speak now..."
                      : voiceAssistant.isSpeaking
                        ? "🔊 Assistant is speaking..."
                        : "✨ Ready to help"}
                  </button>

                  {/* Microphone Button - Start/Stop Listening */}
                  <button
                    onClick={() => {
                      if (voiceAssistant.isListening) {
                        voiceAssistant.stopListening();
                      } else {
                        voiceAssistant.startListening();
                      }
                    }}
                    disabled={voiceAssistant.isSpeaking}
                    style={{
                      backgroundColor: voiceAssistant.isListening
                        ? "#1d4ed8"
                        : "#2563eb",
                      color: "white",
                    }}
                    className={`w-12 h-12 rounded-full flex items-center justify-center transition-colors disabled:opacity-50 ${
                      voiceAssistant.isListening ? "animate-pulse" : ""
                    }`}
                  >
                    <svg
                      className="w-6 h-6"
                      fill="currentColor"
                      viewBox="0 0 20 20"
                    >
                      <path
                        fillRule="evenodd"
                        d="M7 4a3 3 0 016 0v4a3 3 0 11-6 0V4zm4 10.93A7.001 7.001 0 0017 8a1 1 0 10-2 0A5 5 0 015 8a1 1 0 00-2 0 7.001 7.001 0 006 6.93V17H6a1 1 0 100 2h8a1 1 0 100-2h-3v-2.07z"
                        clipRule="evenodd"
                      />
                    </svg>
                  </button>

                  {/* Mute Button */}
                  <button
                    onClick={() => voiceAssistant.stopSpeaking()}
                    disabled={!voiceAssistant.isSpeaking}
                    style={{
                      backgroundColor: "#ef4444",
                      color: "white",
                    }}
                    className="w-12 h-12 rounded-full flex items-center justify-center transition-colors disabled:opacity-30"
                  >
                    <svg
                      className="w-6 h-6"
                      fill="currentColor"
                      viewBox="0 0 20 20"
                    >
                      <path
                        fillRule="evenodd"
                        d="M9.383 3.076A1 1 0 0110 4v12a1 1 0 01-1.707.707L4.586 13H2a1 1 0 01-1-1V8a1 1 0 011-1h2.586l3.707-3.707a1 1 0 011.09-.217zM12.293 7.293a1 1 0 011.414 0L15 8.586l1.293-1.293a1 1 0 111.414 1.414L16.414 10l1.293 1.293a1 1 0 01-1.414 1.414L15 11.414l-1.293 1.293a1 1 0 01-1.414-1.414L13.586 10l-1.293-1.293a1 1 0 010-1.414z"
                        clipRule="evenodd"
                      />
                    </svg>
                  </button>
                </div>

                {/* Footer */}
                <div className="text-center pb-4">
                  <p className="text-xs text-gray-500">
                    Property Assistant by NAL
                  </p>
                </div>
              </>
            )}
          </div>
        </div>
      )}

      {/* Conversation History Modal */}
      {showConversationHistory && (
        <div className="fixed inset-0 bg-black/50 backdrop-blur-sm flex items-center justify-center z-50 p-4">
          <div className="bg-white rounded-xl max-w-2xl w-full max-h-[80vh] overflow-hidden shadow-2xl">
            <div className="flex justify-between items-center p-6 border-b">
              <h3 className="text-xl font-bold text-gray-900">
                Conversation History
              </h3>
              <div className="flex items-center gap-2">
                <button
                  onClick={() => {
                    voiceAssistant.stopSpeaking();
                    setConversationHistory([]);
                    setHasGreeted(false);
                  }}
                  className="px-3 py-1 rounded-lg font-medium transition-colors"
                  style={{
                    backgroundColor: "#ef4444",
                    color: "white",
                    border: "none",
                    cursor: "pointer",
                    fontSize: "14px",
                  }}
                  onMouseEnter={(e) =>
                    (e.currentTarget.style.backgroundColor = "#dc2626")
                  }
                  onMouseLeave={(e) =>
                    (e.currentTarget.style.backgroundColor = "#ef4444")
                  }
                >
                  Clear
                </button>
                <button
                  onClick={() => setShowConversationHistory(false)}
                  className="p-2 rounded-full hover:bg-gray-100 transition-colors"
                  style={{
                    color: "#1f2937",
                    backgroundColor: "transparent",
                    border: "none",
                    cursor: "pointer",
                  }}
                >
                  <svg
                    className="w-6 h-6"
                    fill="none"
                    viewBox="0 0 24 24"
                    style={{ stroke: "#1f2937", strokeWidth: 2 }}
                  >
                    <path
                      strokeLinecap="round"
                      strokeLinejoin="round"
                      d="M6 18L18 6M6 6l12 12"
                    />
                  </svg>
                </button>
              </div>
            </div>

            <div
              className="p-6"
              style={{ maxHeight: "60vh", overflowY: "auto" }}
            >
              {conversationHistory.length === 0 ? (
                <div className="text-center py-12">
                  <span className="text-6xl mb-4 block">💬</span>
                  <p className="text-gray-500">No conversation yet</p>
                </div>
              ) : (
                <div className="space-y-4">
                  {conversationHistory.map((msg, idx) => (
                    <div
                      key={idx}
                      className={`flex ${msg.type === "user" ? "justify-end" : "justify-start"}`}
                    >
                      <div
                        className={`max-w-[80%] rounded-lg p-4`}
                        style={{
                          backgroundColor:
                            msg.type === "user" ? "#2563eb" : "#f3f4f6",
                          color: msg.type === "user" ? "white" : "#111827",
                        }}
                      >
                        <div className="flex items-center gap-2 mb-1">
                          <span
                            className="text-xs font-semibold"
                            style={{ opacity: 0.75 }}
                          >
                            {msg.type === "user" ? "You" : "Assistant"}
                          </span>
                        </div>
                        <p className="text-sm leading-relaxed">{msg.message}</p>
                      </div>
                    </div>
                  ))}
                </div>
              )}
            </div>

            <div className="p-4 border-t bg-gray-50">
              <button
                onClick={() => setShowConversationHistory(false)}
                className="w-full py-2 px-4 rounded-lg font-medium transition-colors"
                style={{
                  backgroundColor: "#2563eb",
                  color: "white",
                  border: "none",
                  cursor: "pointer",
                }}
                onMouseEnter={(e) =>
                  (e.currentTarget.style.backgroundColor = "#1d4ed8")
                }
                onMouseLeave={(e) =>
                  (e.currentTarget.style.backgroundColor = "#2563eb")
                }
              >
                Close
              </button>
            </div>
          </div>
        </div>
      )}

      {/* Payment Modal */}
      {showPaymentModal && (
        <PaymentModal
          isOpen={showPaymentModal}
          onClose={() => setShowPaymentModal(false)}
          amount={parseFloat(
            property?.listing_type === "rent"
              ? property?.monthly_rent || "0"
              : property?.sale_price || "0",
          )}
          propertyId={property?.id}
          transactionType={
            paymentType === "urgent"
              ? TransactionType.URGENT_SALE
              : TransactionType.PROPERTY_PURCHASE
          }
          description={`Payment for ${property?.title}`}
          onSuccess={(transactionId) => {
            toast.success(
              "Payment successful! Transaction ID: " + transactionId,
            );
            setShowPaymentModal(false);
            navigate("/user/dashboard");
          }}
        />
      )}
    </div>
  );
}
