import React, { useState, useEffect } from 'react';
import DocumentsList from './DocumentsList';
import { RefreshCw, Shield } from 'lucide-react';
import { api } from '../utils/api';

interface PropertyDetailsDocumentsProps {
  propertyId: number;
}

const PropertyDetailsDocuments: React.FC<PropertyDetailsDocumentsProps> = ({ propertyId }) => {
  const [refreshTrigger, setRefreshTrigger] = useState(0);
  const [isRefreshing, setIsRefreshing] = useState(false);
  const [isOwner, setIsOwner] = useState<boolean | null>(null);
  const [isLoading, setIsLoading] = useState(true);

  // Check ownership similar to property edit logic
  useEffect(() => {
    const checkOwnership = async () => {
      try {
        setIsLoading(true);
        const propertyData = await api.fetchPropertyById(propertyId.toString());
        
        if (propertyData) {
          // Same ownership check as property edit
          const currentUser = JSON.parse(localStorage.getItem('user') || '{}');
          const userEmail = currentUser.email || localStorage.getItem('user_email');
          
          if (!userEmail || propertyData.email !== userEmail) {
            setIsOwner(false);
          } else {
            setIsOwner(true);
          }
        } else {
          setIsOwner(false);
        }
      } catch (error) {
        console.error('Error checking property ownership:', error);
        setIsOwner(false);
      } finally {
        setIsLoading(false);
      }
    };

    checkOwnership();
  }, [propertyId]);

  if (isLoading) {
    return (
      <div className="property-documents-tab">
        <div className="text-center py-12">
          <div className="animate-spin rounded-full h-8 w-8 border-b-2 border-blue-600 mx-auto mb-4"></div>
          <p className="text-gray-600">Checking access permissions...</p>
        </div>
      </div>
    );
  }

  // Show RIBL score for everyone, but documents only for owners
  if (isOwner === false) {
    return (
      <div className="property-documents-tab">
        {/* RIBL Score - visible to everyone */}
        <div className="mb-6">
          <h3 className="text-lg font-semibold mb-4">Property Verification Score</h3>
          <DocumentsList 
            propertyId={propertyId}
            refreshTrigger={0}
            showOnlyScore={true}
          />
        </div>
        
        <div className="text-center py-8">
          <Shield className="w-16 h-16 text-gray-400 mx-auto mb-4" />
          <h3 className="text-lg font-medium text-gray-900 mb-2">Documents Not Available</h3>
          <p className="text-gray-600">Property documents are only visible to the property owner.</p>
        </div>
      </div>
    );
  }

  const handleRefresh = async () => {
    setIsRefreshing(true);
    setRefreshTrigger(prev => prev + 1);
    // Add a small delay for better UX
    setTimeout(() => setIsRefreshing(false), 1000);
  };

  return (
    <div className="property-documents-tab">
      <div className="mb-6 flex justify-between items-start">
        <div>
          <h3 className="text-xl font-semibold text-gray-900 mb-2">Property Documents</h3>
          <p className="text-gray-600">View and download property-related documents</p>
          
          {/* Ownership verification notice */}
          <div className="mt-3 flex items-center text-sm text-blue-600 bg-blue-50 px-3 py-2 rounded-md">
            <Shield className="w-4 h-4 mr-2" />
            <span>Documents are only visible to the property owner</span>
          </div>
        </div>
        
        <button
          onClick={handleRefresh}
          disabled={isRefreshing}
          className="flex items-center px-3 py-2 text-sm font-medium text-gray-700 bg-white border border-gray-300 rounded-md hover:bg-gray-50 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-blue-500 disabled:opacity-50"
        >
          <RefreshCw className={`w-4 h-4 mr-2 ${isRefreshing ? 'animate-spin' : ''}`} />
          {isRefreshing ? 'Refreshing...' : 'Refresh'}
        </button>
      </div>
      
      <DocumentsList 
        propertyId={propertyId}
        refreshTrigger={refreshTrigger}
      />
    </div>
  );
};

export default PropertyDetailsDocuments;