import React, { useState } from 'react';
import DocumentUpload from './DocumentUpload';
import DocumentsList from './DocumentsList';
import { PropertyDocument } from '../utils/documentsApi';

interface PropertyDocumentsProps {
  propertyId: number;
  userId: string;
  showUpload?: boolean;
}

const PropertyDocuments: React.FC<PropertyDocumentsProps> = ({ 
  propertyId, 
  userId, 
  showUpload = true 
}) => {
  const [refreshTrigger, setRefreshTrigger] = useState(0);
  const [message, setMessage] = useState<{ type: 'success' | 'error'; text: string } | null>(null);

  const handleUploadSuccess = (document: PropertyDocument) => {
    setMessage({ type: 'success', text: 'Document uploaded successfully!' });
    setRefreshTrigger(prev => prev + 1);
    
    // Clear message after 3 seconds
    setTimeout(() => setMessage(null), 3000);
  };

  const handleUploadError = (error: string) => {
    setMessage({ type: 'error', text: error });
    
    // Clear message after 5 seconds
    setTimeout(() => setMessage(null), 5000);
  };

  return (
    <div className="property-documents space-y-6">
      {/* Message Display */}
      {message && (
        <div className={`p-4 rounded-md ${
          message.type === 'success' 
            ? 'bg-green-50 border border-green-200 text-green-700' 
            : 'bg-red-50 border border-red-200 text-red-700'
        }`}>
          <div className="flex">
            <div className="flex-shrink-0">
              {message.type === 'success' ? (
                <svg className="h-5 w-5 text-green-400" fill="currentColor" viewBox="0 0 20 20">
                  <path fillRule="evenodd" d="M10 18a8 8 0 100-16 8 8 0 000 16zm3.707-9.293a1 1 0 00-1.414-1.414L9 10.586 7.707 9.293a1 1 0 00-1.414 1.414l2 2a1 1 0 001.414 0l4-4z" clipRule="evenodd" />
                </svg>
              ) : (
                <svg className="h-5 w-5 text-red-400" fill="currentColor" viewBox="0 0 20 20">
                  <path fillRule="evenodd" d="M10 18a8 8 0 100-16 8 8 0 000 16zM8.707 7.293a1 1 0 00-1.414 1.414L8.586 10l-1.293 1.293a1 1 0 101.414 1.414L10 11.414l1.293 1.293a1 1 0 001.414-1.414L11.414 10l1.293-1.293a1 1 0 00-1.414-1.414L10 8.586 8.707 7.293z" clipRule="evenodd" />
                </svg>
              )}
            </div>
            <div className="ml-3">
              <p className="text-sm font-medium">{message.text}</p>
            </div>
          </div>
        </div>
      )}

      {/* Upload Section */}
      {showUpload && (
        <div className="bg-white border border-gray-200 rounded-lg p-6">
          <h2 className="text-xl font-semibold mb-4">Upload Documents</h2>
          <DocumentUpload
            propertyId={propertyId}
            userId={userId}
            onUploadSuccess={handleUploadSuccess}
            onUploadError={handleUploadError}
          />
        </div>
      )}

      {/* Documents List */}
      <div className="bg-white border border-gray-200 rounded-lg p-6">
        <DocumentsList 
          propertyId={propertyId} 
          refreshTrigger={refreshTrigger}
        />
      </div>
    </div>
  );
};

export default PropertyDocuments;