import React, { useState, useEffect } from 'react';
import { documentsApi, PropertyDocument } from '../utils/documentsApi';
import { mediaApi, PropertyMedia } from '../utils/mediaApi';

interface PropertyFilesProps {
  propertyId: number;
}

export const PropertyFiles: React.FC<PropertyFilesProps> = ({ propertyId }) => {
  const [documents, setDocuments] = useState<PropertyDocument[]>([]);
  const [media, setMedia] = useState<PropertyMedia[]>([]);
  const [loading, setLoading] = useState(false);

  useEffect(() => {
    loadFiles();
  }, [propertyId]);

  const loadFiles = async () => {
    setLoading(true);
    try {
      const [docsResponse, mediaResponse] = await Promise.all([
        documentsApi.getPropertyDocuments(propertyId),
        mediaApi.getPropertyMedia(propertyId)
      ]);
      setDocuments(docsResponse.documents);
      setMedia(mediaResponse.media);
    } catch (error) {
      console.error('Error loading files:', error);
    } finally {
      setLoading(false);
    }
  };

  const handleDocumentUpload = async (file: File, documentType: string) => {
    try {
      await documentsApi.uploadDocument(propertyId, file, documentType);
      loadFiles();
    } catch (error) {
      console.error('Upload failed:', error);
    }
  };

  const handleMediaUpload = async (file: File, mediaType: 'photo' | 'video') => {
    try {
      await mediaApi.uploadMedia(propertyId, file, mediaType);
      loadFiles();
    } catch (error) {
      console.error('Upload failed:', error);
    }
  };

  const viewFile = async (fileId: number, type: 'document' | 'media') => {
    try {
      const { download_url } = type === 'document' 
        ? await documentsApi.getDownloadUrl(fileId)
        : await mediaApi.getDownloadUrl(fileId);
      window.open(download_url, '_blank');
    } catch (error) {
      console.error('Failed to get download URL:', error);
    }
  };

  if (loading) return <div>Loading files...</div>;

  return (
    <div className="space-y-6">
      {/* Document Upload */}
      <div>
        <h3 className="text-lg font-semibold mb-2">Documents</h3>
        <input
          type="file"
          accept=".pdf,.doc,.docx"
          onChange={(e) => {
            const file = e.target.files?.[0];
            if (file) handleDocumentUpload(file, 'contract');
          }}
          className="mb-4"
        />
        
        <div className="grid gap-2">
          {documents.map(doc => (
            <div key={doc.id} className="flex items-center justify-between p-2 border rounded">
              <span>{doc.doc_type}</span>
              <button
                onClick={() => viewFile(doc.id, 'document')}
                className="px-3 py-1 bg-blue-500 text-white rounded hover:bg-blue-600"
              >
                View
              </button>
            </div>
          ))}
        </div>
      </div>

      {/* Media Upload */}
      <div>
        <h3 className="text-lg font-semibold mb-2">Photos & Videos</h3>
        <input
          type="file"
          accept="image/*,video/*"
          onChange={(e) => {
            const file = e.target.files?.[0];
            if (file) {
              const mediaType = file.type.startsWith('video/') ? 'video' : 'photo';
              handleMediaUpload(file, mediaType);
            }
          }}
          className="mb-4"
        />
        
        <div className="grid grid-cols-3 gap-4">
          {media.map(item => (
            <div key={item.id} className="relative">
              {item.media_type === 'photo' ? (
                <img
                  src={item.thumbnail_url || item.file_url}
                  alt={item.filename}
                  className="w-full h-32 object-cover rounded cursor-pointer"
                  onClick={() => viewFile(item.id, 'media')}
                />
              ) : (
                <div
                  className="w-full h-32 bg-gray-200 rounded flex items-center justify-center cursor-pointer"
                  onClick={() => viewFile(item.id, 'media')}
                >
                  <span>▶️ Video</span>
                </div>
              )}
            </div>
          ))}
        </div>
      </div>
    </div>
  );
};