import { useState, useMemo, useEffect } from "react";

import { Button } from "./ui/button";
import { Input } from "./ui/input";
import {
  Select,
  SelectContent,
  SelectItem,
  SelectTrigger,
  SelectValue,
} from "./ui/select";
import { Card, CardContent } from "./ui/card";
import { Badge } from "./ui/badge";
import {
  Search,
  MapPin,
  Filter,
  Heart,
  Share2,
  Train,
  Building,
  MapIcon,
} from "lucide-react";

import { PropertyMap } from "./PropertyMap";
import { Link } from "react-router-dom";
import { getProperties } from "../api/properties";
import {
  fetchHyperlocalIndicators,
  HyperlocalIndicators,
} from "../api/hyperlocal";

interface Property {
  id: number;
  title: string;
  category: string;
  description: string;
  price: number;
  address: string;
  city: string;
  locality: string;
  property_age: string;
  facing: string;
  floor_number?: string;
  total_floors?: string;
  latitude?: number;
  longitude?: number;
  images?: { name: string; type: string; size: number; data: string }[];
  created_at: string;
}

export function PropertyListing() {
  const [properties, setProperties] = useState<Property[]>([]);
  const [loading, setLoading] = useState(true);
  const [searchQuery, setSearchQuery] = useState("");
  const [selectedCity, setSelectedCity] = useState("all");
  const [selectedBHK, setSelectedBHK] = useState("all");
  const [priceRange, setPriceRange] = useState("all");
  const [sortBy, setSortBy] = useState("relevance");
  const [currentPropertyIndex, setCurrentPropertyIndex] = useState(0);

  const [hyperlocalDataCache, setHyperlocalDataCache] = useState<{
    [key: number]: HyperlocalIndicators;
  }>({});

  useEffect(() => {
    fetchProperties();
  }, []);

  const fetchProperties = async () => {
    try {
      const response = await getProperties();

      const mapped: Property[] = response.properties.map((p: any) => ({
        id: p.id,
        title: p.title || "Property",
        category: p.listing_type || "",
        description: p.description || "",
        price: Number(p.sale_price || p.monthly_rent || 0),
        address: p.address || "",
        city: p.city || "",
        locality: p.locality || "",
        property_age: "",
        facing: p.furnishing || "",
        floor_number: undefined,
        total_floors: undefined,
        latitude: p.latitude ? Number(p.latitude) : undefined,
        longitude: p.longitude ? Number(p.longitude) : undefined,
        images: [],
        created_at: p.created_at || new Date().toISOString(),
      }));

      setProperties(mapped);
      fetchHyperlocalDataForProperties(mapped);
    } catch (error) {
      console.error("Error fetching properties:", error);
    } finally {
      setLoading(false);
    }
  };

  const fetchHyperlocalDataForProperties = async (propertyList: Property[]) => {
    const cache: { [key: number]: HyperlocalIndicators } = {};

    const propertiesWithCoords = propertyList
      .filter((p) => p.latitude && p.longitude)
      .slice(0, 10);

    await Promise.all(
      propertiesWithCoords.map(async (property) => {
        try {
          const data = await fetchHyperlocalIndicators(
            property.latitude!,
            property.longitude!
          );
          cache[property.id] = data;
        } catch {
          cache[property.id] = {
            metroLines: [],
            infrastructure: [],
            zoning: [],
          };
        }
      })
    );

    setHyperlocalDataCache(cache);
  };

  const cities = useMemo(() => {
    return Array.from(new Set(properties.map((p) => p.city)));
  }, [properties]);

  const filteredProperties = useMemo(() => {
    let filtered = properties.filter((property) => {
      const matchesSearch =
        property.title.toLowerCase().includes(searchQuery.toLowerCase()) ||
        property.city.toLowerCase().includes(searchQuery.toLowerCase()) ||
        property.locality.toLowerCase().includes(searchQuery.toLowerCase());

      const matchesCity =
        selectedCity === "all" ||
        property.city.toLowerCase() === selectedCity.toLowerCase();

      const matchesPrice =
        priceRange === "all" ||
        checkPriceRange(property.price.toString(), priceRange);

      return matchesSearch && matchesCity && matchesPrice;
    });

    switch (sortBy) {
      case "price-low":
        filtered.sort((a, b) => a.price - b.price);
        break;
      case "price-high":
        filtered.sort((a, b) => b.price - a.price);
        break;
      case "newest":
        filtered.sort((a, b) => b.id - a.id);
        break;
    }

    return filtered;
  }, [properties, searchQuery, selectedCity, priceRange, sortBy]);

  const checkPriceRange = (price: string, range: string): boolean => {
    const p = parseFloat(price);
    if (range === "under-50l") return p < 5000000;
    if (range === "50l-1cr") return p >= 5000000 && p < 10000000;
    if (range === "1cr-2cr") return p >= 10000000 && p < 20000000;
    if (range === "above-2cr") return p >= 20000000;
    return true;
  };

  return (
    <div className="bg-gray-50 min-h-full">
      <div className="w-full px-4 sm:px-6 lg:px-8 py-6">
        <div className="bg-white rounded-lg shadow-sm p-6 mb-6">
          <div className="relative">
            <Search className="absolute left-3 top-1/2 -translate-y-1/2 w-5 h-5 text-gray-400" />
            <Input
              placeholder="Search by property name, location, or builder..."
              value={searchQuery}
              onChange={(e) => setSearchQuery(e.target.value)}
              className="pl-10 h-12"
            />
          </div>
        </div>

        {loading ? (
          <div className="text-center py-12 text-gray-400">
            Loading properties...
          </div>
        ) : (
          <div className="grid grid-cols-1 lg:grid-cols-2 gap-6">
            <PropertyMap
              properties={filteredProperties}
              currentIndex={currentPropertyIndex}
            />

            <div className="space-y-4">
              {filteredProperties.map((property, index) => (
                <Card
                  key={property.id}
                  onClick={() => setCurrentPropertyIndex(index)}
                >
                  <CardContent className="p-4">
                    <h3 className="font-semibold text-lg">
                      {property.title}
                    </h3>
                    <div className="text-sm text-gray-600 flex items-center">
                      <MapPin className="w-4 h-4 mr-1" />
                      {property.locality}, {property.city}
                    </div>

                    {hyperlocalDataCache[property.id] && (
                      <div className="mt-2 flex gap-2 text-xs">
                        <span className="flex items-center gap-1">
                          <Train className="w-3 h-3" />
                          {hyperlocalDataCache[property.id].metroLines.length}
                        </span>
                        <span className="flex items-center gap-1">
                          <Building className="w-3 h-3" />
                          {
                            hyperlocalDataCache[property.id].infrastructure
                              .length
                          }
                        </span>
                        <span className="flex items-center gap-1">
                          <MapIcon className="w-3 h-3" />
                          {hyperlocalDataCache[property.id].zoning.length}
                        </span>
                      </div>
                    )}

                    <Link to={`/property/${property.id}`}>
                      <Button className="mt-3 w-full">
                        View Details
                      </Button>
                    </Link>
                  </CardContent>
                </Card>
              ))}
            </div>
          </div>
        )}
      </div>
    </div>
  );
}
