import React, { useState } from 'react';
import { MapPin, Info } from 'lucide-react';
import { LocationConnectivity } from './LocationConnectivity';

interface PropertyLocationInfoProps {
  property: {
    id: string;
    latitude?: number;
    longitude?: number;
    city?: string;
    locality?: string;
    address?: string;
  };
}

export const PropertyLocationInfo: React.FC<PropertyLocationInfoProps> = ({ property }) => {
  const [showConnectivity, setShowConnectivity] = useState(false);

  // Check if property has valid coordinates
  const hasCoordinates = property.latitude && property.longitude && 
                        !isNaN(property.latitude) && !isNaN(property.longitude);

  if (!hasCoordinates) {
    return (
      <div className="bg-yellow-50 border border-yellow-200 rounded-lg p-4">
        <div className="flex items-center">
          <Info className="w-5 h-5 text-yellow-600 mr-2" />
          <p className="text-yellow-800 text-sm">
            Location coordinates not available for connectivity analysis
          </p>
        </div>
      </div>
    );
  }

  const location = {
    latitude: property.latitude!,
    longitude: property.longitude!
  };

  return (
    <div className="space-y-4">
      {/* Location Summary */}
      <div className="bg-white border rounded-lg p-4">
        <div className="flex items-center justify-between">
          <div className="flex items-center">
            <MapPin className="w-5 h-5 text-blue-600 mr-2" />
            <div>
              <h4 className="font-medium text-gray-900">Property Location</h4>
              <p className="text-sm text-gray-600">
                {property.locality && `${property.locality}, `}
                {property.city}
              </p>
              <p className="text-xs text-gray-500">
                {location.latitude.toFixed(4)}, {location.longitude.toFixed(4)}
              </p>
            </div>
          </div>
          <button
            onClick={() => setShowConnectivity(!showConnectivity)}
            className="px-3 py-1 text-sm bg-blue-100 text-blue-700 rounded-md hover:bg-blue-200 transition-colors"
          >
            {showConnectivity ? 'Hide' : 'Show'} Connectivity
          </button>
        </div>
      </div>

      {/* Connectivity Analysis */}
      {showConnectivity && (
        <LocationConnectivity 
          location={location}
          useTestData={true} // Set to false when real API is available
        />
      )}
    </div>
  );
};