import { useState } from "react";
// Remove hardcoded Property import
import { Card, CardContent, CardHeader, CardTitle } from "./ui/card";
import { Badge } from "./ui/badge";
import { MapPin, Navigation, Layers, ChevronLeft, ChevronRight } from "lucide-react";
import { Button } from "./ui/button";

interface Property {
  id: number;
  title: string;
  category: string;
  price: number;
  city: string;
  locality: string;
  images?: { name: string; type: string; size: number; data: string }[];
}

interface PropertyMapProps {
  properties: Property[];
  currentIndex?: number;
}

export function PropertyMap({ properties, currentIndex = 0 }: PropertyMapProps) {
  const [selectedProperty, setSelectedProperty] = useState<Property | null>(null);
  const [mapView, setMapView] = useState<'satellite' | 'roadmap'>('roadmap');

  // Remove internal navigation functions as they're handled externally

  // Map coordinates for India map positioning
  const getIndiaMapCoordinates = (property: Property) => {
    const cityCoords = {
      "Devanahalli": { x: 56, y: 65 }, // Bangalore area
      "Whitefield": { x: 57, y: 66 }, // Bangalore area
      "Sector 106": { x: 37, y: 35 }, // Delhi area
      "Kolshet Road": { x: 25, y: 55 } // Mumbai area
    };
    
    const locationKey = Object.keys(cityCoords).find(key => 
      property.city?.includes(key) || property.locality?.includes(key)
    ) as keyof typeof cityCoords;
    
    const coords = cityCoords[locationKey] || { x: 50, y: 50 };
    
    // Add small random offset for multiple properties in same area
    return {
      x: coords.x + (Math.random() - 0.5) * 3,
      y: coords.y + (Math.random() - 0.5) * 3
    };
  };

  return (
    <Card className="h-full flex flex-col">
      <CardHeader className="pb-3 flex-shrink-0">
        <div className="flex items-center justify-between">
          <CardTitle className="flex items-center gap-2">
            <MapPin className="w-5 h-5" />
            Property Locations
          </CardTitle>
          <div className="text-xs text-gray-500">
            {properties.length > 0 && `${currentIndex + 1} / ${properties.length}`}
          </div>
        </div>
      </CardHeader>
      
      <CardContent className="p-0 flex-1 min-h-0">
        <div className="relative w-full h-full bg-gray-100 rounded overflow-hidden">
          {/* India Map */}
          <div className="absolute inset-0 bg-blue-50">
            <svg className="w-full h-full" viewBox="0 0 800 600">
              {/* India Map Outline */}
              <path d="M200 100 L250 80 L300 90 L350 85 L400 95 L450 100 L500 120 L520 140 L530 180 L540 220 L550 260 L560 300 L570 340 L580 380 L570 420 L550 450 L520 470 L480 480 L440 485 L400 490 L360 485 L320 480 L280 470 L240 450 L200 420 L180 380 L170 340 L160 300 L150 260 L140 220 L130 180 L140 140 L160 120 L180 110 Z" 
                    fill="#10b981" stroke="#059669" strokeWidth="2" opacity="0.8"/>
              
              {/* Major Cities */}
              <circle cx="300" cy="200" r="3" fill="#dc2626"/>
              <text x="305" y="205" fontSize="10" fill="#374151">Delhi</text>
              
              <circle cx="200" cy="300" r="3" fill="#dc2626"/>
              <text x="205" y="305" fontSize="10" fill="#374151">Mumbai</text>
              
              <circle cx="450" cy="350" r="3" fill="#dc2626"/>
              <text x="455" y="355" fontSize="10" fill="#374151">Bangalore</text>
              
              <circle cx="500" cy="300" r="3" fill="#dc2626"/>
              <text x="505" y="305" fontSize="10" fill="#374151">Chennai</text>
            </svg>
            
            {/* Property Markers on India Map */}
            {properties.map((property, index) => {
              const coords = getIndiaMapCoordinates(property);
              
              return (
                <div
                  key={property.id}
                  className="absolute transform -translate-x-1/2 -translate-y-1/2 cursor-pointer"
                  style={{
                    left: `${coords.x}%`,
                    top: `${coords.y}%`
                  }}
                  onClick={() => {
                    setSelectedProperty(property);
                  }}
                >
                  <div className={`relative ${selectedProperty?.id === property.id ? 'z-20' : 'z-10'}`}>
                    <div className={`w-8 h-8 rounded-full border-2 border-white shadow-lg flex items-center justify-center text-white text-xs font-bold transition-all ${
                      selectedProperty?.id === property.id 
                        ? 'bg-red-600 scale-125' 
                        : index === currentIndex
                        ? 'bg-orange-600'
                        : 'bg-blue-600 hover:bg-blue-700'
                    }`}>
                      {index + 1}
                    </div>
                    
                    {selectedProperty?.id === property.id && (
                      <div className="absolute top-10 left-1/2 transform -translate-x-1/2 w-64 bg-white rounded-lg shadow-xl border p-3 z-30">
                        <div className="absolute -top-2 left-1/2 transform -translate-x-1/2 w-4 h-4 bg-white border-l border-t rotate-45"></div>
                        
                        <div className="flex gap-3">
                          <img
                            src={property.images && property.images.length > 0 ? property.images[0].data : 'https://images.unsplash.com/photo-1560518883-ce09059eeffa?w=800&h=600&fit=crop'}
                            alt={property.title}
                            className="w-16 h-16 rounded object-cover"
                          />
                          <div className="flex-1 min-w-0">
                            <h4 className="font-semibold text-sm truncate">{property.title}</h4>
                            <p className="text-xs text-gray-600 mb-1">{property.locality}, {property.city}</p>
                            <div className="flex items-center gap-1 mb-1">
                              <Badge className="bg-green-600 text-white text-xs px-1 py-0">Verified</Badge>
                              <Badge className="bg-blue-600 text-white text-xs px-1 py-0">A+</Badge>
                            </div>
                            <div className="text-sm font-bold text-blue-600">₹{property.price.toLocaleString()}</div>
                          </div>
                        </div>
                        
                        <div className="flex justify-between items-center mt-2 pt-2 border-t">
                          <div className="text-xs text-gray-500">
                            {property.category}
                          </div>
                          <Button size="sm" className="text-xs px-2 py-1 h-6">
                            View
                          </Button>
                        </div>
                      </div>
                    )}
                  </div>
                </div>
              );
            })}
            
            {/* Map Controls */}
            <div className="absolute top-4 right-4 flex flex-col gap-2">
              <Button size="sm" variant="outline" className="bg-white shadow h-8 w-8 p-0">
                +
              </Button>
              <Button size="sm" variant="outline" className="bg-white shadow h-8 w-8 p-0">
                -
              </Button>
            </div>
            
            {/* Property Count */}
            <div className="absolute bottom-4 right-4 bg-white rounded-lg shadow px-3 py-2">
              <div className="text-xs font-semibold">{properties.length} Properties</div>
            </div>
            
            {/* Current Property Indicator */}
            {properties.length > 0 && (
              <div className="absolute bottom-4 left-4 bg-white rounded-lg shadow px-3 py-2">
                <div className="text-xs font-semibold">Viewing: {currentIndex + 1} / {properties.length}</div>
              </div>
            )}
          </div>
        </div>
      </CardContent>
    </Card>
  );
}