import { useState } from "react";
import { Button } from "./ui/button";
import { Input } from "./ui/input";
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "./ui/select";
import { X, Building, MapPin, AlertCircle, CheckCircle } from "lucide-react";

interface PropertyValuationModalProps {
  isOpen: boolean;
  onClose: () => void;
}

interface FormErrors {
  propertyType?: string;
  location?: string;
  area?: string;
  bedrooms?: string;
  submit?: string;
}

export function PropertyValuationModal({ isOpen, onClose }: PropertyValuationModalProps) {
  const [propertyType, setPropertyType] = useState("");
  const [location, setLocation] = useState("");
  const [area, setArea] = useState("");
  const [bedrooms, setBedrooms] = useState("");
  const [errors, setErrors] = useState<FormErrors>({});
  const [isSubmitting, setIsSubmitting] = useState(false);
  const [isSubmitted, setIsSubmitted] = useState(false);

  if (!isOpen) return null;

  const validateForm = (): FormErrors => {
    const newErrors: FormErrors = {};
    
    if (!propertyType) newErrors.propertyType = "Property type is required";
    if (!location.trim()) newErrors.location = "Location is required";
    if (!area || parseFloat(area) <= 0) newErrors.area = "Valid area is required";
    if (!bedrooms) newErrors.bedrooms = "Bedrooms selection is required";
    
    return newErrors;
  };

  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault();
    setErrors({});
    
    const formErrors = validateForm();
    if (Object.keys(formErrors).length > 0) {
      setErrors(formErrors);
      return;
    }
    
    setIsSubmitting(true);
    
    try {
      // Simulate API call - replace with real endpoint later
      await new Promise(resolve => setTimeout(resolve, 1500));
      
      // Store request data (you can integrate with backend later)
      const requestData = {
        propertyType,
        location: location.trim(),
        area: parseFloat(area),
        bedrooms,
        timestamp: new Date().toISOString()
      };
      
      // For now, just store in localStorage (replace with API call)
      const existingRequests = JSON.parse(localStorage.getItem('valuationRequests') || '[]');
      existingRequests.push({ ...requestData, id: Date.now() });
      localStorage.setItem('valuationRequests', JSON.stringify(existingRequests));
      
      setIsSubmitted(true);
    } catch (error) {
      setErrors({ submit: 'Failed to submit request. Please try again.' });
    } finally {
      setIsSubmitting(false);
    }
  };

  const handleClose = () => {
    setPropertyType("");
    setLocation("");
    setArea("");
    setBedrooms("");
    setErrors({});
    setIsSubmitting(false);
    setIsSubmitted(false);
    onClose();
  };

  if (isSubmitted) {
    return (
      <div className="fixed inset-0 bg-black/50 backdrop-blur-sm z-50 flex items-center justify-center p-4">
        <div className="bg-white rounded-xl shadow-2xl w-full max-w-md p-6 text-center">
          <CheckCircle className="w-16 h-16 text-green-500 mx-auto mb-4" />
          <h2 className="text-xl font-semibold mb-2">Request Submitted!</h2>
          <p className="text-gray-600 mb-6">
            Your valuation request has been received. Our team will contact you within 24-48 hours with a detailed report.
          </p>
          <Button onClick={handleClose} className="w-full">
            Close
          </Button>
        </div>
      </div>
    );
  }

  return (
    <div className="fixed inset-0 bg-black/50 backdrop-blur-sm z-50 flex items-center justify-center p-4">
      <div className="bg-white rounded-xl shadow-2xl w-full max-w-md">
        <div className="flex items-center justify-between p-6 border-b">
          <h2 className="text-xl font-semibold flex items-center">
            <Building className="w-5 h-5 mr-2" />
            Property Valuation
          </h2>
          <Button variant="ghost" size="sm" onClick={handleClose}>
            <X className="w-4 h-4" />
          </Button>
        </div>
        
        <form onSubmit={handleSubmit} className="p-6 space-y-4">
          {errors.submit && (
            <div className="flex items-center gap-2 p-3 bg-red-50 border border-red-200 rounded-lg text-red-700">
              <AlertCircle className="w-4 h-4" />
              <span className="text-sm">{errors.submit}</span>
            </div>
          )}
          
          <div className="space-y-2">
            <label className="text-sm font-medium">Property Type</label>
            <Select value={propertyType} onValueChange={setPropertyType}>
              <SelectTrigger className={errors.propertyType ? "border-red-300" : ""}>
                <SelectValue placeholder="Select property type" />
              </SelectTrigger>
              <SelectContent>
                <SelectItem value="apartment">Apartment</SelectItem>
                <SelectItem value="villa">Villa</SelectItem>
                <SelectItem value="plot">Plot</SelectItem>
                <SelectItem value="commercial">Commercial</SelectItem>
              </SelectContent>
            </Select>
            {errors.propertyType && (
              <p className="text-sm text-red-600">{errors.propertyType}</p>
            )}
          </div>
          
          <div className="space-y-2">
            <label className="text-sm font-medium">Location</label>
            <div className="relative">
              <MapPin className="absolute left-3 top-3 w-4 h-4 text-gray-400" />
              <Input
                placeholder="Enter property location (e.g., Koramangala, Bangalore)"
                value={location}
                onChange={(e) => setLocation(e.target.value)}
                className={`pl-10 ${errors.location ? "border-red-300" : ""}`}
              />
            </div>
            {errors.location && (
              <p className="text-sm text-red-600">{errors.location}</p>
            )}
          </div>
          
          <div className="space-y-2">
            <label className="text-sm font-medium">Area (sq ft)</label>
            <Input
              type="number"
              placeholder="Enter area in sq ft"
              value={area}
              onChange={(e) => setArea(e.target.value)}
              onWheel={(e) => e.preventDefault()}
              min="100"
              max="50000"
              className={errors.area ? "border-red-300" : ""}
            />
            {errors.area && (
              <p className="text-sm text-red-600">{errors.area}</p>
            )}
          </div>
          
          <div className="space-y-2">
            <label className="text-sm font-medium">Bedrooms</label>
            <Select value={bedrooms} onValueChange={setBedrooms}>
              <SelectTrigger className={errors.bedrooms ? "border-red-300" : ""}>
                <SelectValue placeholder="Select bedrooms" />
              </SelectTrigger>
              <SelectContent>
                <SelectItem value="1">1 BHK</SelectItem>
                <SelectItem value="2">2 BHK</SelectItem>
                <SelectItem value="3">3 BHK</SelectItem>
                <SelectItem value="4">4+ BHK</SelectItem>
              </SelectContent>
            </Select>
            {errors.bedrooms && (
              <p className="text-sm text-red-600">{errors.bedrooms}</p>
            )}
          </div>
          
          <Button 
            type="submit" 
            disabled={isSubmitting}
            className="w-full bg-purple-600 hover:bg-purple-600/90 disabled:opacity-50"
          >
            {isSubmitting ? "Submitting..." : "Get Property Valuation"}
          </Button>
          
          <p className="text-xs text-gray-500 text-center">
            Beta version - Indicative estimates only
          </p>
        </form>
      </div>
    </div>
  );
}