import React, { useState, useEffect } from 'react';
import { useParams, useNavigate } from 'react-router-dom';
import { createVisitSchedule, fileToBase64, VisitScheduleRequest, checkPendingVisit, VisitSchedule } from '../api/visitSchedules';
import { toast } from 'sonner';

interface Property {
  id: number;
  title: string;
  locality: string;
  city: string;
  state: string;
  sale_price?: string;
  photo: string;
  email?: string;
}

interface VisitRequest {
  id: number;
  property_id: number;
  property_title: string;
  property_address: string;
  property_city: string;
  full_name: string;
  email: string;
  mobile: string;
  preferred_date: string;
  preferred_time: string;
  alternate_date?: string;
  alternate_time?: string;
  number_of_visitors: number;
  visit_mode: string;
  status: string;
  comments?: string;
  created_at: string;
}

export function PropertyVisitBooking() {
  const { id } = useParams();
  const navigate = useNavigate();
  const [property, setProperty] = useState<Property | null>(null);
  const [loading, setLoading] = useState(true);
  const [submitting, setSubmitting] = useState(false);
  const [pendingVisit, setPendingVisit] = useState<VisitSchedule | null>(null);
  const [hasPendingVisit, setHasPendingVisit] = useState(false);
  const [bookingFormData, setBookingFormData] = useState({
    fullName: '',
    email: '',
    mobile: '',
    preferredContact: 'Call',
    buyerType: 'Individual Buyer',
    preferredDate: '',
    preferredTime: '',
    alternateDate: '',
    alternateTime: '',
    numberOfVisitors: 1,
    visitMode: 'In-person Visit',
    pickupAssistance: false,
    visitReason: 'Interested to Buy',
    comments: '',
    idDocument: null as File | null,
    videoCallEnabled: false,
    termsAccepted: false
  });

  useEffect(() => {
    const fetchProperty = async () => {
      try {
        const { api } = await import('../utils/api');
        const data = await api.fetchPropertyById(id!);
        if (data) {
          setProperty(data);
        }
      } catch (error) {
        console.error('Error fetching property:', error);
      } finally {
        setLoading(false);
      }
    };
    
    // Auto-populate user data
    const user = JSON.parse(localStorage.getItem('user') || '{}');
    if (user.first_name || user.email || user.mobile) {
      setBookingFormData(prev => ({
        ...prev,
        fullName: `${user.first_name || ''} ${user.last_name || ''}`.trim(),
        email: user.email || '',
        mobile: user.mobile || user.phone || ''
      }));
    }
    
    fetchProperty();
  }, [id]);

  useEffect(() => {
    const checkExistingVisit = async () => {
      try {
        const user = JSON.parse(localStorage.getItem('user') || '{}');
        if (user.user_id && id) {
          const result = await checkPendingVisit(user.user_id, parseInt(id));
          setHasPendingVisit(result.hasPending);
          if (result.visit) {
            setPendingVisit(result.visit);
          }
        }
      } catch (error) {
        console.error('Error checking pending visit:', error);
      }
    };
    
    if (!loading && property) {
      checkExistingVisit();
    }
  }, [id, loading, property]);

  // Check if current user owns this property
  const isOwnProperty = () => {
    const user = JSON.parse(localStorage.getItem('user') || '{}');
    const userEmail = user.email;
    return userEmail && property?.email && userEmail.toLowerCase() === property.email.toLowerCase();
  };

  const formatPrice = (price: string | number) => {
    const numPrice = typeof price === 'string' ? parseFloat(price) : price;
    if (isNaN(numPrice) || numPrice === 0) return 'Price on request';
    if (numPrice >= 10000000) return `₹${(numPrice / 10000000).toFixed(1)} Cr`;
    if (numPrice >= 100000) return `₹${(numPrice / 100000).toFixed(0)} L`;
    return `₹${numPrice.toLocaleString()}`;
  };

  const handleBookingFormSubmit = async (e: React.FormEvent) => {
    e.preventDefault();
    
    // Validation
    if (!bookingFormData.fullName.trim()) {
      toast.error('Please enter your full name');
      return;
    }
    if (!bookingFormData.email.trim()) {
      toast.error('Please enter your email address');
      return;
    }
    if (!bookingFormData.mobile.trim()) {
      toast.error('Please enter your mobile number');
      return;
    }
    if (!bookingFormData.preferredDate) {
      toast.error('Please select a preferred date');
      return;
    }
    if (!bookingFormData.termsAccepted) {
      toast.error('Please accept the terms and conditions');
      return;
    }
    
    setSubmitting(true);
    
    try {
      // Convert file to base64 string if present
      let idDocumentString = '';
      if (bookingFormData.idDocument) {
        idDocumentString = await fileToBase64(bookingFormData.idDocument);
      }
      
      // Get current user ID
      const user = JSON.parse(localStorage.getItem('user') || '{}');
      
      // Prepare request body according to API schema
      const requestBody: VisitScheduleRequest = {
        user_id: user.user_id || user.id || '',
        property_id: parseInt(property?.id?.toString() || '0'),
        full_name: bookingFormData.fullName,
        email: bookingFormData.email,
        mobile: bookingFormData.mobile,
        preferred_contact: bookingFormData.preferredContact as 'Call' | 'Email' | 'WhatsApp',
        buyer_type: bookingFormData.buyerType as 'Individual Buyer' | 'Investor' | 'Builder' | 'Agent',
        preferred_date: bookingFormData.preferredDate,
        preferred_time: bookingFormData.preferredTime || '10:00',
        alternate_date: bookingFormData.alternateDate || undefined,
        alternate_time: bookingFormData.alternateTime || undefined,
        number_of_visitors: bookingFormData.numberOfVisitors,
        visit_mode: bookingFormData.visitMode as 'In-person Visit' | 'Virtual Tour' | 'Video Call',
        pickup_assistance: bookingFormData.pickupAssistance,
        visit_reason: bookingFormData.visitReason,
        comments: bookingFormData.comments || undefined,
        id_document: idDocumentString || undefined,
        video_call_enabled: bookingFormData.videoCallEnabled
      };
      
      console.log('Submitting visit schedule request:', requestBody);
      console.log('User object:', user);
      console.log('Property ID type:', typeof property?.id, property?.id);
      
      const result = await createVisitSchedule(requestBody);
      
      // AUTO-CREATE VISIT AGREEMENT
      try {
        const { agreementsApi } = await import('../api/agreements');
        console.log('Fetching visit templates...');
        const templates = await agreementsApi.getTemplates('visit');
        console.log('Templates received:', templates);
        
        if (templates && templates.length > 0) {
          console.log('Creating visit agreement with template:', templates[0]);
          const agreement = await agreementsApi.create({
            template_id: templates[0].id,
            agreement_type: 'visit',
            property_id: parseInt(id!),
            parties: [{
              user_id: user.user_id,
              role: 'visitor',
              name: bookingFormData.fullName,
              email: bookingFormData.email
            }],
            content: templates[0].content
          });
          console.log('Visit agreement created:', agreement);
        } else {
          console.error('No visit templates found!');
        }
      } catch (err) {
        console.error('Agreement creation failed:', err);
      }
      
      toast.success('Visit Scheduled Successfully!', {
        description: `Booking ID: ${result.id}. Visit agreement created automatically.`,
        duration: 5000,
        action: {
          label: '✕',
          onClick: () => toast.dismiss()
        }
      });
      navigate(`/property/${id}`);
      
    } catch (error: any) {
      console.error('Error submitting booking:', error);
      toast.error('Booking Failed', {
        description: error.message || 'Error submitting booking. Please try again.',
        duration: 5000,
        action: {
          label: '✕',
          onClick: () => toast.dismiss()
        }
      });
    } finally {
      setSubmitting(false);
    }
  };

  if (loading) {
    return (
      <div className="max-w-4xl mx-auto p-8 text-center">
        <p className="text-gray-500">Loading...</p>
      </div>
    );
  }

  if (!property) {
    return (
      <div className="max-w-4xl mx-auto p-8 text-center">
        <p className="text-gray-500">Property not found</p>
        <button onClick={() => navigate(-1)} className="mt-4 bg-blue-600 text-white px-4 py-2 rounded">
          Go Back
        </button>
      </div>
    );
  }

  // Show message if user has pending visit for this property
  if (hasPendingVisit && pendingVisit) {
    return (
      <div className="bg-gray-50 min-h-screen py-8">
        <div className="max-w-2xl mx-auto px-4">
          <div className="bg-white rounded-lg shadow-sm border border-gray-200">
            {/* Header */}
            <div className="text-center p-6 border-b border-gray-200">
              <div className="w-16 h-16 bg-amber-100 rounded-full flex items-center justify-center mx-auto mb-4">
                <svg className="w-8 h-8 text-amber-600" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                  <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M12 8v4l3 3m6-3a9 9 0 11-18 0 9 9 0 0118 0z" />
                </svg>
              </div>
              <h2 className="text-2xl font-bold text-gray-900 mb-2">Visit Already Scheduled</h2>
              <p className="text-gray-600">
                You already have a {pendingVisit.status} visit request for this property. Please wait for the current request to be processed before scheduling another visit.
              </p>
            </div>

            {/* Visit Details Form */}
            <div className="p-6">
              <div className="space-y-4">
                <div className="flex justify-between items-center py-3 border-b border-gray-100">
                  <span className="text-gray-600 font-medium">Visit Status:</span>
                  <span className={`px-3 py-1 rounded-full text-sm font-medium capitalize ${
                    pendingVisit.status === 'pending' ? 'bg-yellow-100 text-yellow-800' :
                    pendingVisit.status === 'admin_approved' ? 'bg-blue-100 text-blue-800' :
                    pendingVisit.status === 'seller_approved' ? 'bg-green-100 text-green-800' :
                    'bg-gray-100 text-gray-800'
                  }`}>
                    {pendingVisit.status.replace('_', ' ')}
                  </span>
                </div>

                <div className="flex justify-between items-center py-3 border-b border-gray-100">
                  <span className="text-gray-600 font-medium">Preferred Date:</span>
                  <span className="font-semibold text-gray-900">{new Date(pendingVisit.preferred_date).toLocaleDateString('en-GB')}</span>
                </div>

                <div className="flex justify-between items-center py-3 border-b border-gray-100">
                  <span className="text-gray-600 font-medium">Preferred Time:</span>
                  <span className="font-semibold text-gray-900">{pendingVisit.preferred_time}</span>
                </div>

                <div className="flex justify-between items-center py-3 border-b border-gray-100">
                  <span className="text-gray-600 font-medium">Requested On:</span>
                  <span className="font-semibold text-gray-900">{new Date(pendingVisit.created_at).toLocaleDateString('en-GB')}</span>
                </div>

                <div className="flex justify-between items-center py-3 border-b border-gray-100">
                  <span className="text-gray-600 font-medium">Visit ID:</span>
                  <span className="font-semibold text-blue-600">#{pendingVisit.id}</span>
                </div>

                <div className="flex justify-between items-center py-3 border-b border-gray-100">
                  <span className="text-gray-600 font-medium">Visit Mode:</span>
                  <span className="font-semibold text-gray-900">{pendingVisit.visit_mode}</span>
                </div>

                <div className="flex justify-between items-center py-3 border-b border-gray-100">
                  <span className="text-gray-600 font-medium">Number of Visitors:</span>
                  <span className="font-semibold text-gray-900">{pendingVisit.number_of_visitors}</span>
                </div>

                <div className="flex justify-between items-center py-3 border-b border-gray-100">
                  <span className="text-gray-600 font-medium">Contact Person:</span>
                  <span className="font-semibold text-gray-900">{pendingVisit.full_name}</span>
                </div>

                <div className="flex justify-between items-center py-3 border-b border-gray-100">
                  <span className="text-gray-600 font-medium">Email:</span>
                  <span className="font-semibold text-gray-900">{pendingVisit.email}</span>
                </div>

                <div className="flex justify-between items-center py-3 border-b border-gray-100">
                  <span className="text-gray-600 font-medium">Mobile:</span>
                  <span className="font-semibold text-gray-900">{pendingVisit.mobile}</span>
                </div>

                {pendingVisit.alternate_date && (
                  <div className="flex justify-between items-center py-3 border-b border-gray-100">
                    <span className="text-gray-600 font-medium">Alternate Date:</span>
                    <span className="font-semibold text-gray-900">{new Date(pendingVisit.alternate_date).toLocaleDateString('en-GB')}</span>
                  </div>
                )}

                {pendingVisit.alternate_time && (
                  <div className="flex justify-between items-center py-3 border-b border-gray-100">
                    <span className="text-gray-600 font-medium">Alternate Time:</span>
                    <span className="font-semibold text-gray-900">{pendingVisit.alternate_time}</span>
                  </div>
                )}

                {pendingVisit.comments && (
                  <div className="py-3 border-b border-gray-100">
                    <span className="text-gray-600 font-medium block mb-2">Comments:</span>
                    <p className="text-gray-900 bg-gray-50 p-3 rounded">{pendingVisit.comments}</p>
                  </div>
                )}
              </div>
            </div>

            {/* Action Buttons */}
            <div className="px-6 py-8 bg-white border-t-4 border-blue-500">
              <div className="flex flex-col sm:flex-row gap-4">
                <button
                  onClick={() => navigate(`/property/${id}`)}
                  style={{backgroundColor: '#2563eb', color: 'white'}}
                  className="w-full sm:flex-1 px-8 py-4 text-lg font-bold rounded-lg hover:opacity-90 transition-all shadow-lg"
                >
                  View Property
                </button>
                <button
                  onClick={() => navigate('/user/dashboard')}
                  style={{backgroundColor: '#2563eb', color: 'white'}}
                  className="w-full sm:flex-1 px-8 py-4 text-lg font-bold rounded-lg hover:opacity-90 transition-all shadow-lg"
                >
                  Go to Dashboard
                </button>
              </div>
            </div>
          </div>
        </div>
      </div>
    );
  }

  // Show message if user tries to schedule visit for their own property
  if (isOwnProperty()) {
    return (
      <div className="bg-gray-50 min-h-screen py-8">
        <div className="max-w-4xl mx-auto px-4">
          <div className="bg-white rounded-lg shadow-sm p-8 text-center">
            <div className="w-16 h-16 bg-blue-100 rounded-full flex items-center justify-center mx-auto mb-4">
              <svg className="w-8 h-8 text-blue-600" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M13 16h-1v-4h-1m1-4h.01M21 12a9 9 0 11-18 0 9 9 0 0118 0z" />
              </svg>
            </div>
            <h2 className="text-2xl font-bold text-gray-900 mb-4">Cannot Schedule Visit</h2>
            <p className="text-gray-600 mb-6 max-w-md mx-auto">
              You cannot schedule a visit for your own property. As the property owner, you can manage visit requests from potential buyers in your dashboard.
            </p>
            <div className="flex gap-4 justify-center">
              <button
                onClick={() => navigate(`/property/${id}`)}
                className="px-6 py-2 border border-gray-300 text-gray-700 rounded-lg hover:bg-gray-50 transition-colors"
              >
                View Property
              </button>
              <button
                onClick={() => navigate('/user/dashboard')}
                className="px-6 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700 transition-colors"
              >
                Go to Dashboard
              </button>
            </div>
          </div>
        </div>
      </div>
    );
  }

  return (
    <div className="bg-gray-50 min-h-screen py-8">
      <div className="max-w-4xl mx-auto px-4">
        {/* Property Summary Header */}
        <div className="bg-white rounded-lg shadow-sm p-6 mb-6">
          <div className="flex items-center space-x-4">
            <img 
              src={property.photo?.startsWith('http') ? property.photo : `http://localhost:8090${property.photo}`}
              alt={property.title}
              className="w-20 h-20 object-cover rounded-lg"
            />
            <div className="flex-1">
              <h1 className="text-xl font-bold text-gray-900">{property.title}</h1>
              <p className="text-gray-600">{property.locality}, {property.city}, {property.state}</p>
              <p className="text-lg font-semibold text-blue-600">{formatPrice(property.sale_price || 0)}</p>
            </div>
            <button 
              onClick={() => navigate(`/property/${id}`)}
              className="text-gray-400 hover:text-gray-600"
            >
              <svg className="w-6 h-6" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M6 18L18 6M6 6l12 12" />
              </svg>
            </button>
          </div>
        </div>

        {/* Booking Form */}
        <div className="bg-white rounded-lg shadow-sm">
          <div className="border-b border-gray-200 px-6 py-4">
            <h2 className="text-2xl font-bold text-gray-900">Schedule Property Visit</h2>
            <p className="text-gray-600 mt-1">Fill out the form below to schedule your property visit</p>
          </div>
          
          <form onSubmit={handleBookingFormSubmit} className="p-6 space-y-6">
            {/* Personal Information */}
            <div className="bg-blue-50 border border-blue-200 rounded-lg p-4">
              <h3 className="text-lg font-semibold text-blue-900 mb-4">Personal Information</h3>
              <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-1">Full Name *</label>
                  <input
                    type="text"
                    required
                    value={bookingFormData.fullName}
                    onChange={(e) => setBookingFormData({...bookingFormData, fullName: e.target.value})}
                    className="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500"
                  />
                </div>
                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-1">Email Address *</label>
                  <input
                    type="email"
                    required
                    value={bookingFormData.email}
                    onChange={(e) => setBookingFormData({...bookingFormData, email: e.target.value})}
                    className="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500"
                  />
                </div>
                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-1">Mobile Number *</label>
                  <input
                    type="tel"
                    required
                    value={bookingFormData.mobile}
                    onChange={(e) => setBookingFormData({...bookingFormData, mobile: e.target.value})}
                    className="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500"
                  />
                </div>
                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-1">Preferred Contact</label>
                  <select
                    value={bookingFormData.preferredContact}
                    onChange={(e) => setBookingFormData({...bookingFormData, preferredContact: e.target.value})}
                    className="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500"
                  >
                    <option value="Call">Call</option>
                    <option value="WhatsApp">WhatsApp</option>
                    <option value="Email">Email</option>
                  </select>
                </div>
              </div>
              
              <div className="mt-4">
                <label className="block text-sm font-medium text-gray-700 mb-2">Buyer Type</label>
                <div className="flex space-x-4">
                  {['Individual Buyer', 'Investor', 'Tenant / Renter'].map((type) => (
                    <label key={type} className="flex items-center">
                      <input
                        type="radio"
                        name="buyerType"
                        value={type}
                        checked={bookingFormData.buyerType === type}
                        onChange={(e) => setBookingFormData({...bookingFormData, buyerType: e.target.value})}
                        className="mr-2"
                      />
                      <span className="text-sm">{type}</span>
                    </label>
                  ))}
                </div>
              </div>
            </div>

            {/* Visit Schedule */}
            <div className="bg-green-50 border border-green-200 rounded-lg p-4">
              <h3 className="text-lg font-semibold text-green-900 mb-4">Visit Schedule Preferences</h3>
              <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-1">Preferred Date *</label>
                  <input
                    type="date"
                    required
                    min={new Date().toISOString().split('T')[0]}
                    value={bookingFormData.preferredDate}
                    onChange={(e) => setBookingFormData({...bookingFormData, preferredDate: e.target.value})}
                    className="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-green-500"
                  />
                </div>
                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-1">Preferred Time</label>
                  <select
                    value={bookingFormData.preferredTime}
                    onChange={(e) => setBookingFormData({...bookingFormData, preferredTime: e.target.value})}
                    className="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-green-500"
                  >
                    <option value="">Select Time</option>
                    <option value="9–11 AM">9–11 AM</option>
                    <option value="11–1 PM">11–1 PM</option>
                    <option value="2–4 PM">2–4 PM</option>
                    <option value="4–6 PM">4–6 PM</option>
                  </select>
                </div>
                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-1">Alternate Date</label>
                  <input
                    type="date"
                    min={new Date().toISOString().split('T')[0]}
                    value={bookingFormData.alternateDate}
                    onChange={(e) => setBookingFormData({...bookingFormData, alternateDate: e.target.value})}
                    className="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-green-500"
                  />
                </div>
                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-1">Alternate Time</label>
                  <select
                    value={bookingFormData.alternateTime}
                    onChange={(e) => setBookingFormData({...bookingFormData, alternateTime: e.target.value})}
                    className="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-green-500"
                  >
                    <option value="">Select Time</option>
                    <option value="9–11 AM">9–11 AM</option>
                    <option value="11–1 PM">11–1 PM</option>
                    <option value="2–4 PM">2–4 PM</option>
                    <option value="4–6 PM">4–6 PM</option>
                  </select>
                </div>
                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-1">Number of Visitors</label>
                  <input
                    type="number"
                    min="1"
                    max="10"
                    value={bookingFormData.numberOfVisitors}
                    onChange={(e) => setBookingFormData({...bookingFormData, numberOfVisitors: parseInt(e.target.value)})}
                    className="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-green-500"
                  />
                </div>
                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-1">Visit Mode</label>
                  <select
                    value={bookingFormData.visitMode}
                    onChange={(e) => setBookingFormData({...bookingFormData, visitMode: e.target.value})}
                    className="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-green-500"
                  >
                    <option value="In-person Visit">In-person Visit</option>
                    <option value="Virtual Tour">Virtual Tour</option>
                  </select>
                </div>
              </div>
              
              <div className="mt-4 grid grid-cols-1 md:grid-cols-2 gap-4">
                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-1">Visit Reason</label>
                  <select
                    value={bookingFormData.visitReason}
                    onChange={(e) => setBookingFormData({...bookingFormData, visitReason: e.target.value})}
                    className="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-green-500"
                  >
                    <option value="Interested to Buy">Interested to Buy</option>
                    <option value="Investment Purpose">Investment Purpose</option>
                    <option value="Rental Purpose">Rental Purpose</option>
                    <option value="Just Exploring">Just Exploring</option>
                    <option value="Comparison">Comparison</option>
                  </select>
                </div>
                <div className="flex items-center">
                  <label className="flex items-center">
                    <input
                      type="checkbox"
                      checked={bookingFormData.pickupAssistance}
                      onChange={(e) => setBookingFormData({...bookingFormData, pickupAssistance: e.target.checked})}
                      className="mr-2"
                    />
                    <span className="text-sm">Pickup Assistance Required</span>
                  </label>
                </div>
                <div className="flex items-center">
                  <label className="flex items-center">
                    <input
                      type="checkbox"
                      checked={bookingFormData.videoCallEnabled}
                      onChange={(e) => setBookingFormData({...bookingFormData, videoCallEnabled: e.target.checked})}
                      className="mr-2"
                    />
                    <span className="text-sm">Enable Video Call Option</span>
                  </label>
                  <div className="ml-2 text-xs text-gray-500">
                    (Virtual meeting via Jitsi Meet)
                  </div>
                </div>
              </div>
            </div>

            {/* Additional Comments */}
            <div className="bg-purple-50 border border-purple-200 rounded-lg p-4">
              <h3 className="text-lg font-semibold text-purple-900 mb-4">Additional Comments</h3>
              <div>
                <label className="block text-sm font-medium text-gray-700 mb-1">Comments / Additional Notes</label>
                <textarea
                  value={bookingFormData.comments}
                  onChange={(e) => setBookingFormData({...bookingFormData, comments: e.target.value})}
                  rows={3}
                  className="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-purple-500"
                  placeholder="Any specific requirements or questions..."
                />
              </div>
            </div>

            {/* Verification & Security */}
            <div className="bg-orange-50 border border-orange-200 rounded-lg p-4">
              <h3 className="text-lg font-semibold text-orange-900 mb-4">Verification & Security</h3>
              <div>
                <label className="block text-sm font-medium text-gray-700 mb-1">ID Document Upload (Optional)</label>
                <input
                  type="file"
                  accept=".jpg,.jpeg,.png,.pdf"
                  onChange={(e) => setBookingFormData({...bookingFormData, idDocument: e.target.files?.[0] || null})}
                  className="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-orange-500"
                />
                <p className="text-xs text-gray-500 mt-1">Upload Government ID like Aadhar/DL/Passport for verification</p>
              </div>
            </div>

            {/* Terms & Conditions */}
            <div className="bg-gray-50 border border-gray-200 rounded-lg p-4">
              <label className="flex items-start">
                <input
                  type="checkbox"
                  required
                  checked={bookingFormData.termsAccepted}
                  onChange={(e) => setBookingFormData({...bookingFormData, termsAccepted: e.target.checked})}
                  className="mr-2 mt-1"
                />
                <span className="text-sm text-gray-700">
                  I agree to the <a href="#" className="text-blue-600 hover:underline">Terms of Service</a> and <a href="#" className="text-blue-600 hover:underline">Privacy Policy</a>. I consent to being contacted regarding this property visit.
                </span>
              </label>
            </div>

            {/* Submit Buttons */}
            <div className="flex space-x-4 pt-6">
              <button
                type="button"
                onClick={() => navigate(`/property/${id}`)}
                className="flex-1 px-6 py-3 border border-gray-300 text-gray-700 rounded-lg hover:bg-gray-50 transition-colors"
              >
                Cancel
              </button>
              <button
                type="submit"
                disabled={submitting}
                className="flex-1 px-8 py-4 rounded-lg font-bold text-lg transition-colors flex items-center justify-center gap-2 disabled:opacity-50 disabled:cursor-not-allowed"
                style={{backgroundColor: submitting ? '#9ca3af' : '#2563eb', color: 'white'}}
              >
                {submitting ? (
                  <>
                    <svg className="w-5 h-5 animate-spin" fill="none" viewBox="0 0 24 24">
                      <circle className="opacity-25" cx="12" cy="12" r="10" stroke="currentColor" strokeWidth="4"></circle>
                      <path className="opacity-75" fill="currentColor" d="m4 12a8 8 0 018-8V0C5.373 0 0 5.373 0 12h4zm2 5.291A7.962 7.962 0 014 12H0c0 3.042 1.135 5.824 3 7.938l3-2.647z"></path>
                    </svg>
                    Submitting...
                  </>
                ) : (
                  <>
                    <svg className="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                      <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M8 7V3a1 1 0 011-1h6a1 1 0 011 1v4h3a1 1 0 011 1v9a1 1 0 01-1 1H5a1 1 0 01-1-1V8a1 1 0 011-1h3z" />
                    </svg>
                    Schedule Visit
                  </>
                )}
              </button>
            </div>
          </form>
        </div>
      </div>
    </div>
  );
}