import { useEffect, useRef, useState } from 'react';

interface RecaptchaV2Props {
  onVerify: (token: string) => void;
}

export const RecaptchaV2: React.FC<RecaptchaV2Props> = ({ onVerify }) => {
  const containerRef = useRef<HTMLDivElement>(null);
  const [status, setStatus] = useState('loading');

  useEffect(() => {
    // Set global callback with timeout handling
    (window as any).handleRecaptcha = (token: string) => {
      setStatus('verified');
      onVerify(token);
    };

    // Set timeout after 10 seconds
    const timeout = setTimeout(() => {
      if (status === 'loading') {
        setStatus('timeout');
        onVerify('test-token'); // Use test token on timeout
      }
    }, 10000);

    // Check if reCAPTCHA is already loaded
    if ((window as any).grecaptcha) {
      renderRecaptcha();
      clearTimeout(timeout);
      return;
    }

    // Wait for script to load
    const checkRecaptcha = setInterval(() => {
      if ((window as any).grecaptcha && containerRef.current) {
        clearInterval(checkRecaptcha);
        clearTimeout(timeout);
        renderRecaptcha();
      }
    }, 100);

    return () => {
      clearInterval(checkRecaptcha);
      clearTimeout(timeout);
      delete (window as any).handleRecaptcha;
    };
  }, [onVerify, status]);

  const renderRecaptcha = () => {
    if (containerRef.current && (window as any).grecaptcha) {
      containerRef.current.innerHTML = '';
      
      try {
        (window as any).grecaptcha.render(containerRef.current, {
          sitekey: '6LeIxAcTAAAAAJcZVRqyHh71UMIEGNQ_MXjiZKhI',
          callback: 'handleRecaptcha',
          'error-callback': () => {
            setStatus('error');
            onVerify('test-token'); // Use test token on error
          }
        });
        setStatus('ready');
      } catch (error) {
        console.error('reCAPTCHA render error:', error);
        setStatus('error');
        onVerify('test-token'); // Use test token on error
      }
    }
  };

  const handleManualVerify = () => {
    setStatus('verified');
    onVerify('manual-test-token');
  };

  if (status === 'timeout' || status === 'error') {
    return (
      <div style={{ 
        minHeight: '78px',
        display: 'flex',
        flexDirection: 'column',
        justifyContent: 'center',
        alignItems: 'center',
        border: '1px solid #ddd',
        borderRadius: '4px',
        backgroundColor: '#f9f9f9',
        margin: '16px 0',
        padding: '16px'
      }}>
        <div style={{ color: '#666', fontSize: '14px', marginBottom: '8px' }}>
          reCAPTCHA failed to load
        </div>
        <button 
          onClick={handleManualVerify}
          style={{
            padding: '8px 16px',
            backgroundColor: '#667eea',
            color: 'white',
            border: 'none',
            borderRadius: '4px',
            cursor: 'pointer'
          }}
        >
          Continue anyway
        </button>
      </div>
    );
  }

  return (
    <div
      ref={containerRef}
      style={{ 
        minHeight: '78px',
        display: 'flex',
        justifyContent: 'center',
        alignItems: 'center',
        border: '1px solid #ddd',
        borderRadius: '4px',
        backgroundColor: '#f9f9f9',
        margin: '16px 0'
      }}
    >
      {status === 'loading' && (
        <div style={{ color: '#666', fontSize: '14px' }}>
          reCAPTCHA Loading...
        </div>
      )}
    </div>
  );
};