import React, { useState, useEffect } from 'react';
import { useNavigate } from 'react-router-dom';
import { getRecommendedProperties, Property } from '../api/properties';

const RecommendedProperties: React.FC = () => {
  const navigate = useNavigate();
  const [properties, setProperties] = useState<Property[]>([]);
  const [loading, setLoading] = useState(true);

  useEffect(() => {
    const fetchRecommendations = async () => {
      try {
        const data = await getRecommendedProperties();
        setProperties(data.slice(0, 3)); // Show only 3 recommendations
      } catch (error) {
        console.error('Failed to fetch recommendations:', error);
      } finally {
        setLoading(false);
      }
    };

    fetchRecommendations();
  }, []);

  const formatPrice = (price: number) => {
    if (price >= 10000000) return `₹${(price / 10000000).toFixed(1)} Cr`;
    if (price >= 100000) return `₹${(price / 100000).toFixed(1)} L`;
    return `₹${price.toLocaleString()}`;
  };

  if (loading) {
    return (
      <div className="mt-8">
        <h2 className="text-xl font-semibold mb-4">Recommended Properties</h2>
        <div className="grid grid-cols-1 md:grid-cols-3 gap-4">
          {[1, 2, 3].map((i) => (
            <div key={i} className="border rounded-lg p-4 animate-pulse">
              <div className="h-4 bg-gray-200 rounded mb-2"></div>
              <div className="h-3 bg-gray-200 rounded mb-2"></div>
              <div className="h-3 bg-gray-200 rounded w-1/2"></div>
            </div>
          ))}
        </div>
      </div>
    );
  }

  if (properties.length === 0) return null;

  return (
    <div className="mt-8">
      <div className="flex justify-between items-center mb-4">
        <h2 className="text-xl font-semibold">Recommended Properties</h2>
        <span className="text-sm text-gray-500">Based on your preferences</span>
      </div>
      
      <div className="grid grid-cols-1 md:grid-cols-3 gap-4">
        {properties.map((property) => (
          <div
            key={property.id}
            onClick={() => navigate(`/property/${property.id}`)}
            className="border rounded-lg p-4 cursor-pointer hover:shadow-md transition-shadow"
          >
            <h3 className="font-medium text-gray-900 mb-1 line-clamp-2">
              {property.bedrooms}BHK
            </h3>
            <p className="text-sm text-gray-600 mb-2">
              {property.locality}, {property.city}
            </p>
            <p className="font-semibold text-gray-900 mb-2">
              {property.sale_price ? formatPrice(property.sale_price) : 'Price on request'}
            </p>
            <p className="text-xs text-gray-500">
              {property.bedrooms} BHK • {property.built_up_area} sq.ft
            </p>
          </div>
        ))}
      </div>
    </div>
  );
};

export default RecommendedProperties;