import React, { useState, useEffect } from 'react';
import { Card, CardContent, CardHeader, CardTitle } from '@/components/ui/card';
import { Badge } from '@/components/ui/badge';
import { Progress } from '@/components/ui/progress';
import { Button } from '@/components/ui/button';
import { RefreshCw, TrendingUp, AlertTriangle, CheckCircle, XCircle } from 'lucide-react';
import { authenticatedFetch } from '../utils/auth';

interface RibilScore {
  property_id: string;
  ribil_score: number;
  ribil_grade: string;
  components: {
    document_verification: number;
    document_completeness: number;
    verification_quality: number;
  };
  insights: {
    current_status: {
      total_documents: number;
      verified_documents: number;
      pending_documents: number;
      failed_documents: number;
    };
    missing_documents: string[];
    improvement_suggestions: string[];
    potential_improvement: {
      potential_score: number;
      potential_grade: string;
      score_increase: number;
    };
    grade_meaning: string;
  };
  calculated_at: string;
}

interface RibilScorecardProps {
  propertyId: string;
  userId: string;
  sessionId?: string;
}

const RibilScorecard: React.FC<RibilScorecardProps> = ({ propertyId, userId, sessionId }) => {
  const [ribilScore, setRibilScore] = useState<RibilScore | null>(null);
  const [loading, setLoading] = useState(true);
  const [recalculating, setRecalculating] = useState(false);

  const fetchRibilScore = async () => {
    try {
      const url = sessionId 
        ? `http://localhost:8090/api/ribil/session/${sessionId}`
        : `http://localhost:8090/api/ribil/score/${propertyId}`;
      const response = await authenticatedFetch(url);
      const data = await response.json();
      if (data.success) {
        setRibilScore(data.data);
      }
    } catch (error) {
      console.error('Failed to fetch RIBIL score:', error);
    } finally {
      setLoading(false);
    }
  };

  const recalculateScore = async () => {
    setRecalculating(true);
    try {
      const url = sessionId 
        ? `http://localhost:8090/api/ribil/session/${sessionId}`
        : `http://localhost:8090/api/ribil/score/${propertyId}`;
      const response = await authenticatedFetch(url);
      const data = await response.json();
      if (data.success) {
        setRibilScore(data.data);
      }
    } catch (error) {
      console.error('Failed to recalculate RIBIL score:', error);
    } finally {
      setRecalculating(false);
    }
  };

  useEffect(() => {
    fetchRibilScore();
  }, [propertyId, userId, sessionId]);

  const getGradeColor = (grade: string) => {
    const colors = {
      'A+': 'bg-green-600', 'A': 'bg-green-500',
      'B+': 'bg-blue-600', 'B': 'bg-blue-500',
      'C+': 'bg-yellow-600', 'C': 'bg-yellow-500',
      'D+': 'bg-orange-600', 'D': 'bg-orange-500',
      'E': 'bg-red-500'
    };
    return colors[grade as keyof typeof colors] || 'bg-gray-500';
  };

  const getScoreColor = (score: number) => {
    if (score >= 85) return 'text-green-600';
    if (score >= 65) return 'text-blue-600';
    if (score >= 45) return 'text-yellow-600';
    if (score >= 25) return 'text-orange-600';
    return 'text-red-600';
  };

  if (loading) {
    return (
      <Card className="w-full">
        <CardContent className="p-6">
          <div className="animate-pulse space-y-4">
            <div className="h-8 bg-gray-200 rounded w-1/3"></div>
            <div className="h-4 bg-gray-200 rounded w-2/3"></div>
            <div className="h-20 bg-gray-200 rounded"></div>
          </div>
        </CardContent>
      </Card>
    );
  }

  if (!ribilScore) {
    return (
      <Card className="w-full">
        <CardContent className="p-6 text-center">
          <AlertTriangle className="mx-auto h-12 w-12 text-yellow-500 mb-4" />
          <h3 className="text-lg font-semibold mb-2">RIBIL Score Not Available</h3>
          <p className="text-gray-600 mb-4">
            {sessionId ? 
              'No documents found for this session. Upload documents to generate your property\'s trust score.' :
              'Upload documents to generate your property\'s trust score'
            }
          </p>
          <div className="text-xs text-gray-500 mb-4">
            Session: {sessionId || 'None'}<br/>
            User: {userId}<br/>
            Property: {propertyId}
          </div>
          <Button onClick={recalculateScore} disabled={recalculating}>
            {recalculating ? <RefreshCw className="animate-spin h-4 w-4 mr-2" /> : null}
            Calculate Score
          </Button>
        </CardContent>
      </Card>
    );
  }

  return (
    <div className="space-y-6">
      {/* Main Score Card */}
      <Card className="w-full">
        <CardHeader className="pb-4">
          <div className="flex items-center justify-between">
            <CardTitle className="text-xl font-bold">RIBIL Trust Score</CardTitle>
            <Button 
              variant="outline" 
              size="sm" 
              onClick={recalculateScore}
              disabled={recalculating}
            >
              {recalculating ? <RefreshCw className="animate-spin h-4 w-4 mr-2" /> : <RefreshCw className="h-4 w-4 mr-2" />}
              Refresh
            </Button>
          </div>
        </CardHeader>
        <CardContent className="space-y-6">
          {/* Score Display */}
          <div className="text-center">
            <div className="flex items-center justify-center space-x-4 mb-4">
              <div className="text-center">
                <div className={`text-4xl font-bold ${getScoreColor(ribilScore.ribil_score)}`}>
                  {ribilScore.ribil_score}
                </div>
                <div className="text-sm text-gray-500">Score</div>
              </div>
              <div className="text-center">
                <Badge className={`${getGradeColor(ribilScore.ribil_grade)} text-white text-2xl px-4 py-2`}>
                  {ribilScore.ribil_grade}
                </Badge>
                <div className="text-sm text-gray-500 mt-1">Grade</div>
              </div>
            </div>
            <p className="text-sm text-gray-600 max-w-md mx-auto">
              {ribilScore.insights.grade_meaning}
            </p>
          </div>

          {/* Progress Bar */}
          <div className="space-y-2">
            <div className="flex justify-between text-sm">
              <span>Trust Level</span>
              <span>{ribilScore.ribil_score}%</span>
            </div>
            <Progress value={ribilScore.ribil_score} className="h-3" />
          </div>

          {/* Component Breakdown */}
          <div className="grid grid-cols-1 md:grid-cols-3 gap-4">
            <div className="text-center p-4 bg-gray-50 rounded-lg">
              <div className="text-2xl font-bold text-blue-600">
                {ribilScore.components.document_verification}
              </div>
              <div className="text-sm text-gray-600">Document Verification</div>
              <div className="text-xs text-gray-500">50% Weight</div>
            </div>
            <div className="text-center p-4 bg-gray-50 rounded-lg">
              <div className="text-2xl font-bold text-green-600">
                {ribilScore.components.document_completeness}
              </div>
              <div className="text-sm text-gray-600">Completeness</div>
              <div className="text-xs text-gray-500">30% Weight</div>
            </div>
            <div className="text-center p-4 bg-gray-50 rounded-lg">
              <div className="text-2xl font-bold text-purple-600">
                {ribilScore.components.verification_quality}
              </div>
              <div className="text-sm text-gray-600">Quality</div>
              <div className="text-xs text-gray-500">20% Weight</div>
            </div>
          </div>
        </CardContent>
      </Card>

      {/* Document Status */}
      <Card>
        <CardHeader>
          <CardTitle className="text-lg">Document Status</CardTitle>
        </CardHeader>
        <CardContent>
          <div className="grid grid-cols-2 md:grid-cols-4 gap-4">
            <div className="text-center">
              <div className="text-2xl font-bold text-blue-600">
                {ribilScore.insights.current_status.total_documents}
              </div>
              <div className="text-sm text-gray-600">Total</div>
            </div>
            <div className="text-center">
              <div className="text-2xl font-bold text-green-600 flex items-center justify-center">
                <CheckCircle className="h-5 w-5 mr-1" />
                {ribilScore.insights.current_status.verified_documents}
              </div>
              <div className="text-sm text-gray-600">Verified</div>
            </div>
            <div className="text-center">
              <div className="text-2xl font-bold text-yellow-600">
                {ribilScore.insights.current_status.pending_documents}
              </div>
              <div className="text-sm text-gray-600">Pending</div>
            </div>
            <div className="text-center">
              <div className="text-2xl font-bold text-red-600 flex items-center justify-center">
                <XCircle className="h-5 w-5 mr-1" />
                {ribilScore.insights.current_status.failed_documents}
              </div>
              <div className="text-sm text-gray-600">Failed</div>
            </div>
          </div>
        </CardContent>
      </Card>

      {/* Improvement Suggestions */}
      {ribilScore.insights.improvement_suggestions.length > 0 && (
        <Card>
          <CardHeader>
            <CardTitle className="text-lg flex items-center">
              <TrendingUp className="h-5 w-5 mr-2" />
              Improvement Opportunities
            </CardTitle>
          </CardHeader>
          <CardContent className="space-y-4">
            {/* Potential Improvement */}
            <div className="bg-green-50 p-4 rounded-lg border border-green-200">
              <div className="flex items-center justify-between mb-2">
                <span className="font-semibold text-green-800">Potential Score</span>
                <div className="flex items-center space-x-2">
                  <span className="text-2xl font-bold text-green-600">
                    {ribilScore.insights.potential_improvement.potential_score}
                  </span>
                  <Badge className="bg-green-600 text-white">
                    {ribilScore.insights.potential_improvement.potential_grade}
                  </Badge>
                </div>
              </div>
              <p className="text-sm text-green-700">
                Increase by {ribilScore.insights.potential_improvement.score_increase} points
              </p>
            </div>

            {/* Missing Documents */}
            {ribilScore.insights.missing_documents.length > 0 && (
              <div>
                <h4 className="font-semibold mb-2">Missing Critical Documents:</h4>
                <div className="flex flex-wrap gap-2">
                  {ribilScore.insights.missing_documents.map((doc, index) => (
                    <Badge key={index} variant="destructive">
                      {doc.replace(/([A-Z])/g, ' $1').trim()}
                    </Badge>
                  ))}
                </div>
              </div>
            )}

            {/* Suggestions */}
            <div>
              <h4 className="font-semibold mb-2">Action Items:</h4>
              <ul className="space-y-1">
                {ribilScore.insights.improvement_suggestions.map((suggestion, index) => (
                  <li key={index} className="text-sm text-gray-700 flex items-start">
                    <span className="text-blue-500 mr-2">•</span>
                    {suggestion}
                  </li>
                ))}
              </ul>
            </div>
          </CardContent>
        </Card>
      )}

      {/* Last Updated */}
      <div className="text-center text-sm text-gray-500">
        Last updated: {new Date(ribilScore.calculated_at).toLocaleString()}
      </div>
    </div>
  );
};

export { RibilScorecard };
export default RibilScorecard;