import React from 'react';
import { hasRole, hasAnyRole, isAuthenticated, validateSession } from '../utils/auth';

interface RoleBasedAccessProps {
  children: React.ReactNode;
  requiredRole?: string;
  requiredRoles?: string[];
  requireAny?: boolean;
  fallback?: React.ReactNode;
  requireAuth?: boolean;
}

export const RoleBasedAccess: React.FC<RoleBasedAccessProps> = ({
  children,
  requiredRole,
  requiredRoles = [],
  requireAny = false,
  fallback = null,
  requireAuth = true
}) => {
  // Check authentication if required
  if (requireAuth && !isAuthenticated()) {
    return <>{fallback}</>;
  }

  // Only validate session if we have access_token (not for agent/seller tokens)
  const accessToken = localStorage.getItem('access_token');
  if (accessToken && !validateSession()) {
    return <>{fallback}</>;
  }

  // Check single role
  if (requiredRole && !hasRole(requiredRole)) {
    return <>{fallback}</>;
  }

  // Check multiple roles
  if (requiredRoles.length > 0) {
    const hasAccess = requireAny 
      ? hasAnyRole(requiredRoles)
      : requiredRoles.every(role => hasRole(role));
    
    if (!hasAccess) {
      return <>{fallback}</>;
    }
  }

  return <>{children}</>;
};

// Convenience components for common roles
export const AdminOnly: React.FC<{ children: React.ReactNode; fallback?: React.ReactNode }> = ({ 
  children, 
  fallback = null 
}) => (
  <RoleBasedAccess requiredRole="Admin" fallback={fallback}>
    {children}
  </RoleBasedAccess>
);

export const SellerOnly: React.FC<{ children: React.ReactNode; fallback?: React.ReactNode }> = ({ 
  children, 
  fallback = null 
}) => (
  <RoleBasedAccess requiredRole="Seller" fallback={fallback}>
    {children}
  </RoleBasedAccess>
);

export const AgentOnly: React.FC<{ children: React.ReactNode; fallback?: React.ReactNode }> = ({ 
  children, 
  fallback = null 
}) => (
  <RoleBasedAccess requiredRole="Agent" fallback={fallback}>
    {children}
  </RoleBasedAccess>
);

export const PropertyManagersOnly: React.FC<{ children: React.ReactNode; fallback?: React.ReactNode }> = ({ 
  children, 
  fallback = null 
}) => (
  <RoleBasedAccess requiredRoles={['Admin', 'Seller', 'Agent']} requireAny={true} fallback={fallback}>
    {children}
  </RoleBasedAccess>
);

export const AuthenticatedOnly: React.FC<{ children: React.ReactNode; fallback?: React.ReactNode }> = ({ 
  children, 
  fallback = null 
}) => (
  <RoleBasedAccess requireAuth={true} fallback={fallback}>
    {children}
  </RoleBasedAccess>
);