import React, { useState, useRef, useEffect } from 'react';
import { useNavigate } from 'react-router-dom';
import { LayoutDashboard, ChevronDown, Building2, Home, ShoppingBag, Check } from 'lucide-react';

interface RoleSwitcherProps {
  currentRole: 'agent' | 'seller' | 'buyer';
}

export function RoleSwitcher({ currentRole }: RoleSwitcherProps) {
  const [isOpen, setIsOpen] = useState(false);
  const dropdownRef = useRef<HTMLDivElement>(null);
  const navigate = useNavigate();

  useEffect(() => {
    const handleClickOutside = (event: MouseEvent) => {
      if (dropdownRef.current && !dropdownRef.current.contains(event.target as Node)) {
        setIsOpen(false);
      }
    };
    if (isOpen) {
      document.addEventListener('mousedown', handleClickOutside);
      return () => document.removeEventListener('mousedown', handleClickOutside);
    }
  }, [isOpen]);

  const handleRoleSwitch = (role: 'agent' | 'seller' | 'buyer') => {
    if (role === currentRole) {
      setIsOpen(false);
      return;
    }

    if (!sessionStorage.getItem('original_login_role')) {
      sessionStorage.setItem('original_login_role', currentRole);
      
      if (currentRole === 'agent') {
        const agentEmail = localStorage.getItem('agentEmail');
        if (agentEmail) sessionStorage.setItem('original_email', agentEmail);
      } else if (currentRole === 'seller') {
        const sellerEmail = localStorage.getItem('sellerEmail');
        if (sellerEmail) sessionStorage.setItem('original_email', sellerEmail);
      } else {
        const userEmail = localStorage.getItem('login_email') || localStorage.getItem('user_email');
        if (userEmail) sessionStorage.setItem('original_email', userEmail);
      }
    }

    switch (role) {
      case 'agent':
        navigate('/agent/dashboard');
        break;
      case 'seller':
        navigate('/seller/dashboard');
        break;
      case 'buyer':
        navigate('/user/dashboard');
        break;
    }
    setIsOpen(false);
  };

  const getRoleLabel = (role: string) => {
    switch (role) {
      case 'agent':
        return 'Agent';
      case 'seller':
        return 'Seller';
      case 'buyer':
        return 'Buyer';
      default:
        return role;
    }
  };

  const roles = [
    { 
      id: 'agent', 
      label: 'Agent Dashboard', 
      icon: Building2,
      description: 'Manage properties & clients'
    },
    { 
      id: 'seller', 
      label: 'Seller Dashboard', 
      icon: Home,
      description: 'List & sell properties'
    },
    { 
      id: 'buyer', 
      label: 'Buyer Dashboard', 
      icon: ShoppingBag,
      description: 'Browse & purchase properties'
    }
  ];

  return (
    <div className="relative" ref={dropdownRef}>
      <button
        onClick={() => setIsOpen(!isOpen)}
        className="flex items-center gap-2.5 px-4 py-2.5 rounded-xl transition-all duration-200 bg-white hover:bg-gray-50 border border-gray-200 hover:border-gray-300 shadow-sm hover:shadow-md group"
        title="Switch Dashboard"
      >
        <div className="p-1.5 rounded-lg bg-gray-100 group-hover:bg-gray-200 transition-all">
          <LayoutDashboard className="w-4 h-4 text-gray-700" />
        </div>
        <div className="flex flex-col items-start">
          <span className="text-sm font-semibold text-gray-900">
            {getRoleLabel(currentRole)}
          </span>
          <span className="text-xs text-gray-500">Dashboard</span>
        </div>
        <ChevronDown className={`w-4 h-4 text-gray-500 transition-transform duration-200 ${isOpen ? 'rotate-180' : ''}`} />
      </button>

      {isOpen && (
        <div className="absolute right-0 mt-3 w-72 bg-white rounded-2xl shadow-2xl border border-gray-200 overflow-hidden z-50 animate-in slide-in-from-top-2 duration-200">
          <div className="bg-gradient-to-br from-gray-50 to-white px-4 py-3 border-b border-gray-100">
            <h3 className="text-sm font-bold text-gray-900 flex items-center gap-2">
              <LayoutDashboard className="w-4 h-4 text-gray-600" />
              Switch Dashboard
            </h3>
            <p className="text-xs text-gray-500 mt-0.5">Choose your workspace</p>
          </div>
          <div className="p-2 space-y-1">
            {roles.map((role) => {
              const Icon = role.icon;
              const isActive = role.id === currentRole;
              return (
                <button
                  key={role.id}
                  onClick={() => handleRoleSwitch(role.id as 'agent' | 'seller' | 'buyer')}
                  className={`w-full flex items-center gap-3 px-3 py-3 rounded-xl transition-all duration-200 group ${
                    isActive
                      ? 'bg-gray-100 border border-gray-300 shadow-sm'
                      : 'hover:bg-gray-50 border border-transparent hover:border-gray-200'
                  }`}
                >
                  <div className={`p-2 rounded-lg ${isActive ? 'bg-gray-200' : 'bg-gray-100 group-hover:bg-gray-200'} transition-all duration-200`}>
                    <Icon className="w-5 h-5 text-gray-700" />
                  </div>
                  <div className="flex-1 text-left">
                    <div className="flex items-center gap-2">
                      <span className={`text-sm font-semibold text-gray-900`}>
                        {role.label}
                      </span>
                      {isActive && (
                        <div className="flex items-center gap-1 px-2 py-0.5 rounded-full bg-gray-700">
                          <Check className="w-3 h-3 text-white" />
                          <span className="text-xs font-medium text-white">Active</span>
                        </div>
                      )}
                    </div>
                    <p className="text-xs text-gray-500 mt-0.5">{role.description}</p>
                  </div>
                </button>
              );
            })}
          </div>
        </div>
      )}
    </div>
  );
}
