import React, { Component, ReactNode } from 'react';

interface SafeWrapperProps {
  children: ReactNode;
  fallback?: ReactNode;
  componentName?: string;
}

interface SafeWrapperState {
  hasError: boolean;
  error: Error | null;
}

export class SafeWrapper extends Component<SafeWrapperProps, SafeWrapperState> {
  constructor(props: SafeWrapperProps) {
    super(props);
    this.state = { hasError: false, error: null };
  }

  static getDerivedStateFromError(error: Error): SafeWrapperState {
    return { hasError: true, error };
  }

  componentDidCatch(error: Error, errorInfo: React.ErrorInfo) {
    const componentName = this.props.componentName || 'Unknown Component';
    console.error(`SafeWrapper caught error in ${componentName}:`, error);
    console.error('Error info:', errorInfo);
    
    // Log specific details for the 'cJ' error
    if (error.message.includes('cJ') || error.message.includes('Cannot read properties of undefined')) {
      console.error('🔍 Detected undefined property access error:', {
        message: error.message,
        stack: error.stack,
        component: componentName,
        errorInfo
      });
    }
  }

  render() {
    if (this.state.hasError) {
      if (this.props.fallback) {
        return this.props.fallback;
      }
      
      return (
        <div className="p-4 bg-red-50 border border-red-200 rounded-lg">
          <p className="text-red-600 text-sm">
            Component failed to load: {this.props.componentName || 'Unknown'}
          </p>
        </div>
      );
    }

    return this.props.children;
  }
}

// Higher-order component for safe rendering
export const withSafeWrapper = <P extends object>(
  WrappedComponent: React.ComponentType<P>,
  componentName?: string
) => {
  return React.forwardRef<any, P>((props, ref) => (
    <SafeWrapper componentName={componentName || WrappedComponent.name}>
      <WrappedComponent {...props} ref={ref} />
    </SafeWrapper>
  ));
};

// Hook for safe component rendering
export const useSafeRender = () => {
  const safeRender = (
    renderFn: () => ReactNode,
    fallback?: ReactNode,
    componentName?: string
  ): ReactNode => {
    try {
      return renderFn();
    } catch (error) {
      console.error(`Safe render failed for ${componentName}:`, error);
      return fallback || <div>Failed to render component</div>;
    }
  };

  return { safeRender };
};