import React, { useState } from 'react';
import { useSavedProperty } from '../hooks/usePropertyState';
import { toast } from 'sonner';

interface SavePropertyButtonProps {
  propertyId: number;
}

export const SavePropertyButton: React.FC<SavePropertyButtonProps> = ({ propertyId }) => {
  const [loading, setLoading] = useState(false);
  const { isSaved, toggleSaved } = useSavedProperty(propertyId);

  const handleSaveToggle = async (e: React.MouseEvent) => {
    e.stopPropagation();
    setLoading(true);
    
    try {
      const newState = await toggleSaved();
      if (newState) {
        toast.success('Property Saved', {
          description: 'Added to your saved properties collection',
          duration: 3000,
        });
      } else {
        toast.success('Property Removed', {
          description: 'Removed from your saved properties',
          duration: 3000,
        });
      }
    } catch (error) {
      console.error('Error toggling save status:', error);
      const message = error instanceof Error ? error.message : 'Unable to save property';
      toast.error('Action Failed', {
        description: message,
        duration: 4000,
      });
    } finally {
      setLoading(false);
    }
  };

  return (
    <button
      onClick={handleSaveToggle}
      disabled={loading}
      className="bg-white/90 p-2 rounded-full shadow-md hover:scale-110 transition-all disabled:opacity-50"
      title={isSaved ? "Remove from Saved" : "Save Property"}
    >
      <svg
        className="h-5 w-5"
        fill={isSaved ? "#3b82f6" : "none"}
        viewBox="0 0 24 24"
        stroke={isSaved ? "#3b82f6" : "#4b5563"}
        strokeWidth={2}
      >
        <path
          strokeLinecap="round"
          strokeLinejoin="round"
          d="M5 5a2 2 0 012-2h10a2 2 0 012 2v16l-7-3.5L5 21V5z"
        />
      </svg>
    </button>
  );
};