import React, { useState } from 'react';
import { useSavedProperty } from '../hooks/usePropertyState';

interface SavedPropertiesButtonProps {
  propertyId: number;
  className?: string;
}

export const SavedPropertiesButton: React.FC<SavedPropertiesButtonProps> = ({ 
  propertyId, 
  className = "" 
}) => {
  const [loading, setLoading] = useState(false);
  const { isSaved, toggleSaved } = useSavedProperty(propertyId);

  const handleSaveToggle = async (e: React.MouseEvent) => {
    e.stopPropagation();
    setLoading(true);
    
    try {
      const newSavedState = await toggleSaved();
      
      // Fetch property details for better tracking
      let propertyTitle = `Property #${propertyId}`;
      try {
        const response = await fetch(`http://localhost:8090/api/properties/${propertyId}`);
        if (response.ok) {
          const property = await response.json();
          propertyTitle = property.title || propertyTitle;
        }
      } catch (err) {
        console.warn('Could not fetch property title:', err);
      }
      
      alert(newSavedState ? `${propertyTitle} saved successfully!` : `${propertyTitle} removed from saved list!`);
      
      // Track save property event
      if (newSavedState) {
        const { behaviorTracker } = await import('../utils/behaviorTracker');
        behaviorTracker.trackSaveProperty(propertyId, { title: propertyTitle });
      }
    } catch (error) {
      console.error('Error saving property:', error);
      alert('Error saving property. Please try again.');
    } finally {
      setLoading(false);
    }
  };

  return (
    <button
      onClick={handleSaveToggle}
      disabled={loading}
      className={`bg-white/90 p-2 rounded-full shadow-md hover:scale-110 transition-all disabled:opacity-50 ${className}`}
      title={isSaved ? "Remove from Saved" : "Save Property"}
    >
      <svg
        className="h-5 w-5"
        fill={isSaved ? "#3b82f6" : "none"}
        viewBox="0 0 24 24"
        stroke={isSaved ? "#3b82f6" : "#4b5563"}
        strokeWidth={2}
      >
        <path
          strokeLinecap="round"
          strokeLinejoin="round"
          d="M5 5a2 2 0 012-2h10a2 2 0 012 2v16l-7-3.5L5 21V5z"
        />
      </svg>
    </button>
  );
};