import React, { useState, useEffect } from 'react';
import { Calendar, CheckCircle, User } from 'lucide-react';
import { checkDuplicateVisit, validateVisitScheduling } from '../utils/visitValidation';
import { getUserId } from '../utils/userUtils';

interface ScheduleVisitButtonProps {
  propertyId: number;
  propertyOwnerEmail?: string;
}

export const ScheduleVisitButton: React.FC<ScheduleVisitButtonProps> = ({ 
  propertyId, 
  propertyOwnerEmail 
}) => {
  const [hasScheduled, setHasScheduled] = useState(false);
  const [loading, setLoading] = useState(false);
  const [isOwner, setIsOwner] = useState(false);

  useEffect(() => {
    checkVisitStatus();
    checkOwnership();
  }, [propertyId]);

  const checkVisitStatus = async () => {
    const hasVisit = await checkDuplicateVisit(propertyId);
    setHasScheduled(hasVisit);
  };

  const checkOwnership = () => {
    const user = JSON.parse(localStorage.getItem('user') || '{}');
    const userEmail = user.email;
    setIsOwner(userEmail && propertyOwnerEmail && userEmail.toLowerCase() === propertyOwnerEmail.toLowerCase());
  };

  const handleScheduleClick = async () => {
    setLoading(true);
    
    try {
      const validation = await validateVisitScheduling(propertyId, propertyOwnerEmail);
      if (!validation.canSchedule) {
        setLoading(false);
        return;
      }
      
      // Track schedule visit
      import('../utils/behaviorTracker').then(({ behaviorTracker }) => {
        behaviorTracker.trackScheduleVisit(propertyId, {
          scheduled_at: new Date().toISOString()
        });
      });
      
      // Navigate to schedule visit form or open modal
      window.location.href = `/schedule-visit/${propertyId}`;
    } catch (error) {
      console.error('Error validating visit:', error);
      setLoading(false);
    }
  };

  if (isOwner) {
    return (
      <div className="bg-gray-50 border border-gray-200 rounded-lg p-4 flex items-center justify-center">
        <div className="text-center">
          <User className="w-6 h-6 text-gray-400 mx-auto mb-2" />
          <p className="text-sm font-medium text-gray-600">Property Owner</p>
          <p className="text-xs text-gray-500">You cannot schedule a visit to your own property</p>
        </div>
      </div>
    );
  }

  // Remove the hasScheduled UI block - let user click and show toast instead

  return (
    <button
      onClick={handleScheduleClick}
      disabled={loading}
      className="w-full bg-blue-600 hover:bg-blue-700 text-white font-semibold py-3 px-6 rounded-lg transition-colors flex items-center justify-center gap-2 disabled:opacity-50"
    >
      <Calendar className="w-5 h-5" />
      {loading ? 'Checking...' : 'Schedule Visit'}
    </button>
  );
};