import React, { useEffect, useRef, useState } from 'react';

interface School {
  id: string;
  name: string;
  lat: number;
  lon: number;
  distance: number;
  driveTime: string;
  address?: string;
  schoolType?: string;
}

interface SchoolsGoogleMapProps {
  propertyLat: number;
  propertyLon: number;
  schools: School[];
  propertyName: string;
}

declare global {
  interface Window {
    google: any;
  }
}

export const SchoolsGoogleMap: React.FC<SchoolsGoogleMapProps> = ({ 
  propertyLat, 
  propertyLon, 
  schools, 
  propertyName 
}) => {
  const mapRef = useRef<HTMLDivElement>(null);
  const mapInstanceRef = useRef<any>(null);
  const [selectedDistance, setSelectedDistance] = useState<number>(5);
  
  const filteredSchools = schools.filter(school => school.distance <= selectedDistance);
  
  // Debug logging for school filtering
  console.log(`📍 Schools: Total received: ${schools.length}`);
  console.log(`📍 Schools: Within ${selectedDistance}km: ${filteredSchools.length}`);
  console.log(`📍 Schools: Shown on map: ${filteredSchools.length}/${schools.length}`);

  useEffect(() => {
    const initMap = () => {
      if (!mapRef.current || !window.google) return;

      const map = new window.google.maps.Map(mapRef.current, {
        center: { lat: propertyLat, lng: propertyLon },
        zoom: 14,
        mapTypeId: 'roadmap',
      });

      mapInstanceRef.current = map;

      // Property marker (blue)
      new window.google.maps.Marker({
        position: { lat: propertyLat, lng: propertyLon },
        map: map,
        title: propertyName,
        icon: {
          url: 'data:image/svg+xml;charset=UTF-8,' + encodeURIComponent(`
            <svg width="32" height="32" viewBox="0 0 32 32" xmlns="http://www.w3.org/2000/svg">
              <circle cx="16" cy="16" r="12" fill="#2563eb" stroke="white" stroke-width="3"/>
              <text x="16" y="20" text-anchor="middle" fill="white" font-size="16">🏠</text>
            </svg>
          `),
          scaledSize: new window.google.maps.Size(32, 32),
        }
      });

      // School markers (green) - only filtered schools
      filteredSchools.forEach((school) => {
        const marker = new window.google.maps.Marker({
          position: { lat: school.lat, lng: school.lon },
          map: map,
          title: `${school.name} - ${school.distance}km`,
          icon: {
            url: 'data:image/svg+xml;charset=UTF-8,' + encodeURIComponent(`
              <svg width="28" height="28" viewBox="0 0 28 28" xmlns="http://www.w3.org/2000/svg">
                <circle cx="14" cy="14" r="10" fill="#16a34a" stroke="white" stroke-width="2"/>
                <text x="14" y="18" text-anchor="middle" fill="white" font-size="14">🏫</text>
              </svg>
            `),
            scaledSize: new window.google.maps.Size(28, 28),
          }
        });

        // Info window for school details
        const infoWindow = new window.google.maps.InfoWindow({
          content: `
            <div style="padding: 10px; min-width: 200px;">
              <h3 style="margin: 0 0 8px 0; color: #1f2937; font-size: 16px; font-weight: bold;">${school.name}</h3>
              <p style="margin: 4px 0; color: #4b5563; font-size: 14px;"><strong>Type:</strong> ${school.schoolType || 'School'}</p>
              <p style="margin: 4px 0; color: #4b5563; font-size: 14px;"><strong>Distance:</strong> ${school.distance}km</p>
              <p style="margin: 4px 0; color: #4b5563; font-size: 14px;"><strong>Drive Time:</strong> ${school.driveTime}</p>
              <p style="margin: 4px 0; color: #4b5563; font-size: 14px;"><strong>Address:</strong> ${school.address || 'Address not available'}</p>
              <div style="margin-top: 12px; text-align: center;">
                <a href="https://www.google.com/maps?q=${school.lat},${school.lon}" target="_blank" 
                   style="background-color: #2563eb; color: white; padding: 8px 16px; text-decoration: none; border-radius: 6px; font-size: 14px; font-weight: 500; display: inline-block;">
                  View Details
                </a>
              </div>
            </div>
          `
        });

        // Add click listener to show info window
        marker.addListener('click', () => {
          infoWindow.open(map, marker);
        });
      });

      // Fit bounds to show all markers
      const bounds = new window.google.maps.LatLngBounds();
      bounds.extend({ lat: propertyLat, lng: propertyLon });
      filteredSchools.forEach(school => {
        bounds.extend({ lat: school.lat, lng: school.lon });
      });
      if (filteredSchools.length > 0) {
        map.fitBounds(bounds);
      } else {
        map.setCenter({ lat: propertyLat, lng: propertyLon });
        map.setZoom(14);
      }
    };

    // Load Google Maps API if not already loaded
    if (!window.google) {
      const script = document.createElement('script');
      script.src = `https://maps.googleapis.com/maps/api/js?key=YOUR_API_KEY&libraries=places`;
      script.async = true;
      script.defer = true;
      script.onload = initMap;
      document.head.appendChild(script);
    } else {
      initMap();
    }
  }, [propertyLat, propertyLon, filteredSchools, propertyName]);

  return (
    <div className="bg-white border border-gray-200 rounded-lg p-4 mb-6">
      {/* Distance Filter */}
      <div className="mb-4">
        <h3 className="text-lg font-semibold mb-3">Property & Schools Location</h3>
        <div className="flex items-center gap-2 mb-4">
          <span className="text-sm font-medium text-gray-700">Show schools within:</span>
          {[1, 2, 3, 5].map((distance) => (
            <button
              key={distance}
              onClick={() => setSelectedDistance(distance)}
              className="px-4 py-2 rounded-full text-sm font-medium transition-all duration-200"
              style={{
                backgroundColor: selectedDistance === distance ? '#2563eb' : '#f3f4f6',
                color: selectedDistance === distance ? '#ffffff' : '#374151',
                boxShadow: selectedDistance === distance ? '0 4px 6px -1px rgba(0, 0, 0, 0.1)' : 'none'
              }}
            >
              {distance}km
            </button>
          ))}
        </div>
      </div>
      
      <div 
        ref={mapRef} 
        className="w-full h-80 rounded-lg border"
        style={{ minHeight: '320px' }}
      />
      <div className="mt-3 flex items-center justify-center gap-6 text-sm">
        <div className="flex items-center gap-2">
          <div className="w-4 h-4 bg-blue-600 rounded-full"></div>
          <span>Property Location</span>
        </div>
        <div className="flex items-center gap-2">
          <div className="w-4 h-4 bg-green-600 rounded-full"></div>
          <span>Schools ({filteredSchools.length} within {selectedDistance}km)</span>
        </div>
      </div>
    </div>
  );
};