// src/components/SchoolsMapFirst.tsx
import React, { useEffect, useRef, useState } from "react";
import { fetchNearbySchools } from "../services/schoolsService"; // adjust path
import { MapPin } from "lucide-react";

interface School {
  id: string;
  name: string;
  lat: number;
  lon: number;
  distance: number;
  driveTime: string;
  address?: string;
  schoolType?: string;
}

type Props = {
  propertyId: string | number;
  lat: number;
  lon: number;
  radiusKm?: number;
  googleMapsApiKey?: string; // optional: if you want to dynamically load the script
};

export const SchoolsMapFirst: React.FC<Props> = ({
  propertyId,
  lat,
  lon,
  radiusKm = 5,
  googleMapsApiKey, // supply via env or props
}) => {
  const mapRef = useRef<HTMLDivElement | null>(null);
  const googleMap = useRef<google.maps.Map | null>(null);
  const markersRef = useRef<google.maps.Marker[]>([]);
  const infoWindowsRef = useRef<google.maps.InfoWindow[]>([]);
  const [schools, setSchools] = useState<School[]>([]);
  const [loading, setLoading] = useState(false);
  const [error, setError] = useState<string | null>(null);
  const [methodUsed, setMethodUsed] = useState<string | null>(null);

  // Helper: load Google Maps script if not present
  const loadGoogleMaps = (apiKey?: string) =>
    new Promise<void>((resolve, reject) => {
      if ((window as any).google && (window as any).google.maps) return resolve();
      if (!apiKey) return reject(new Error("Google Maps API key not provided"));
      const id = "geo-google-maps-script";
      if (document.getElementById(id)) {
        // script already added but google not ready yet
        const check = setInterval(() => {
          if ((window as any).google && (window as any).google.maps) {
            clearInterval(check);
            resolve();
          }
        }, 200);
        setTimeout(() => {
          clearInterval(check);
          reject(new Error("Google Maps load timeout"));
        }, 15000);
        return;
      }
      const script = document.createElement("script");
      script.id = id;
      script.src = `https://maps.googleapis.com/maps/api/js?key=${apiKey}&libraries=places`;
      script.async = true;
      script.defer = true;
      script.onload = () => resolve();
      script.onerror = () => reject(new Error("Failed to load Google Maps"));
      document.head.appendChild(script);
    });

  // Initialize map
  const initMap = async () => {
    if (!mapRef.current) return;
    // If you prefer not to require a Google key, you can still render the map container and skip markers.
    if (!googleMapsApiKey) {
      // create a blank map with no Google functionality
      mapRef.current.innerHTML = `<div class="flex items-center justify-center h-full text-gray-500">Map disabled (provide Google Maps API key)</div>`;
      return;
    }
    try {
      await loadGoogleMaps(googleMapsApiKey);
      googleMap.current = new (window as any).google.maps.Map(mapRef.current, {
        center: { lat, lng: lon }, // later replaced with bounds
        zoom: 13,
        mapTypeControl: false,
        streetViewControl: false,
      });
    } catch (err) {
      console.error("Google Maps init error:", err);
      // fallback: show message inside map div
      if (mapRef.current) {
        mapRef.current.innerHTML = `<div class="flex items-center justify-center h-full text-red-500">Failed to load Google Maps</div>`;
      }
    }
  };

  // Clear previous markers
  const clearMarkers = () => {
    markersRef.current.forEach((m) => m.setMap(null));
    markersRef.current = [];
    infoWindowsRef.current.forEach((iw) => iw.close());
    infoWindowsRef.current = [];
  };

  // Add markers for property + schools and auto-fit bounds
  const renderMarkers = (schoolsList: School[]) => {
    if (!googleMap.current) return;
    clearMarkers();
    const bounds = new (window as any).google.maps.LatLngBounds();

    // Add property marker (blue house)
    const propertyPos = new (window as any).google.maps.LatLng(lat, lon);
    const propertyMarker = new (window as any).google.maps.Marker({
      position: propertyPos,
      map: googleMap.current,
      title: "Property",
      icon: {
        url:
          "data:image/svg+xml;charset=UTF-8," +
          encodeURIComponent(`<svg width="32" height="32" viewBox="0 0 32 32" xmlns="http://www.w3.org/2000/svg">
            <circle cx="16" cy="16" r="12" fill="#2563eb" stroke="white" stroke-width="2"/>
            <text x="16" y="21" text-anchor="middle" fill="white" font-size="14">🏠</text>
          </svg>`),
        scaledSize: new (window as any).google.maps.Size(32, 32),
      },
      zIndex: 9999,
    });
    markersRef.current.push(propertyMarker);
    bounds.extend(propertyPos);

    // Add school markers
    schoolsList.forEach((s, idx) => {
      const pos = new (window as any).google.maps.LatLng(s.lat, s.lon);
      const marker = new (window as any).google.maps.Marker({
        position: pos,
        map: googleMap.current,
        title: s.name,
        icon: {
          url:
            "data:image/svg+xml;charset=UTF-8," +
            encodeURIComponent(`<svg width="28" height="28" viewBox="0 0 28 28" xmlns="http://www.w3.org/2000/svg">
              <circle cx="14" cy="14" r="12" fill="#16a34a" stroke="white" stroke-width="1.5"/>
              <text x="14" y="18" text-anchor="middle" fill="white" font-size="14">🏫</text>
            </svg>`),
          scaledSize: new (window as any).google.maps.Size(28, 28),
        },
      });

      const infoHtml = `
        <div style="min-width:200px">
          <div style="font-weight:600;margin-bottom:6px">${s.name}</div>
          <div style="font-size:13px;color:#4b5563">${s.address || ""}</div>
          <div style="font-size:13px;color:#4b5563;margin-top:6px">Distance: ${s.distance} km • ${s.driveTime}</div>
        </div>
      `;
      const iw = new (window as any).google.maps.InfoWindow({ content: infoHtml });
      marker.addListener("click", () => {
        infoWindowsRef.current.forEach((w) => w.close());
        iw.open({ map: googleMap.current, anchor: marker });
      });
      markersRef.current.push(marker);
      infoWindowsRef.current.push(iw);
      bounds.extend(pos);
    });

    // Fit bounds to show all markers with padding
    googleMap.current.fitBounds(bounds, 80);
  };

  // Load schools from Geoapify and render map & list
  useEffect(() => {
    let cancelled = false;
    const load = async () => {
      setLoading(true);
      setError(null);
      try {
        const result = await fetchNearbySchools(lat, lon, radiusKm);
        if (cancelled) return;
        setSchools(result.schools);
        setMethodUsed(result.methodUsed || "geoapify");
        // initialize map if not yet
        if (!googleMap.current) {
          await initMap();
          // small timeout so map container is ready
          await new Promise((r) => setTimeout(r, 200));
        }
        // render markers
        if (result.schools.length > 0 && googleMap.current) {
          renderMarkers(result.schools);
        } else if (googleMap.current) {
          // only property marker if no schools
          clearMarkers();
          const pos = new (window as any).google.maps.LatLng(lat, lon);
          const propertyMarker = new (window as any).google.maps.Marker({
            position: pos,
            map: googleMap.current,
            title: "Property",
            icon: {
              url:
                "data:image/svg+xml;charset=UTF-8," +
                encodeURIComponent(`<svg width="32" height="32" viewBox="0 0 32 32" xmlns="http://www.w3.org/2000/svg">
                  <circle cx="16" cy="16" r="12" fill="#2563eb" stroke="white" stroke-width="2"/>
                  <text x="16" y="21" text-anchor="middle" fill="white" font-size="14">🏠</text>
                </svg>`),
              scaledSize: new (window as any).google.maps.Size(32, 32),
            },
            zIndex: 9999,
          });
          markersRef.current.push(propertyMarker);
          googleMap.current.setCenter(pos);
          googleMap.current.setZoom(13);
        }
      } catch (err: any) {
        console.error("SchoolsMapFirst error:", err);
        setError(err?.message || "Failed to load schools");
      } finally {
        setLoading(false);
      }
    };
    load();

    return () => {
      cancelled = true;
    };
    // eslint-disable-next-line react-hooks/exhaustive-deps
  }, [lat, lon, radiusKm, googleMapsApiKey]);

  return (
    <div className="space-y-6">
      {/* Map Card */}
      <div className="bg-white rounded-2xl shadow-xl border p-4">
        <div className="flex items-center justify-between mb-3">
          <div>
            <h3 className="text-xl font-bold">Nearby Schools</h3>
            <p className="text-sm text-gray-500">Map view (shows property + schools)</p>
          </div>

          <div className="flex items-center gap-3">
            {methodUsed === "geoapify" && (
              <span className="px-3 py-1 rounded-full bg-green-100 text-green-700 text-sm font-semibold">
                Real Data (Geoapify)
              </span>
            )}
            <div className="text-sm text-gray-500">{loading ? "Loading..." : `${schools.length} found`}</div>
          </div>
        </div>

        <div ref={mapRef} className="w-full h-[420px] rounded-xl border overflow-hidden" />
      </div>

      {/* List Card */}
      <div className="bg-white rounded-2xl shadow-xl border p-4">
        <div className="flex items-center justify-between mb-4">
          <div>
            <h4 className="text-lg font-semibold">Schools list</h4>
            <p className="text-sm text-gray-500">Sorted by proximity (closest first)</p>
          </div>

          <div className="flex items-center gap-2">
            <label className="text-sm text-gray-600">Radius</label>
            <select
              value={radiusKm}
              onChange={(e) => {
                // Note: to change radius you'll need to re-render component with new prop value from parent
                // or lift state up. Here we just show UI clarity.
                alert("Change radius from property panel (component prop) - or lift radius state up if you want inline control.");
              }}
              className="border rounded px-3 py-1 text-sm"
            >
              <option value={1}>1 km</option>
              <option value={2}>2 km</option>
              <option value={3}>3 km</option>
              <option value={5}>5 km</option>
              <option value={10}>10 km</option>
            </select>
          </div>
        </div>

        {loading && (
          <div className="p-6 flex items-center justify-center text-gray-500">Loading schools…</div>
        )}

        {error && (
          <div className="p-4 bg-red-50 text-red-700 rounded-md">{error}</div>
        )}

        {!loading && !error && schools.length === 0 && (
          <div className="p-6 text-center text-gray-500">No schools found in this area.</div>
        )}

        {!loading && schools.length > 0 && (
          <div className="space-y-3">
            {schools
              .slice()
              .sort((a, b) => a.distance - b.distance)
              .map((s, idx) => (
                <div key={s.id || idx} className="p-4 bg-gray-50 rounded-lg flex items-start gap-4">
                  <div className="flex-none">
                    <div className="w-12 h-12 rounded-lg bg-white border flex items-center justify-center shadow">
                      <MapPin className="w-6 h-6 text-green-600" />
                    </div>
                  </div>

                  <div className="flex-1">
                    <div className="flex items-start justify-between">
                      <div>
                        <div className="font-semibold text-gray-800">{s.name}</div>
                        <div className="text-sm text-gray-500 mt-1">{s.address}</div>
                      </div>
                      <div className="text-right">
                        <div className="text-sm font-medium text-gray-800">{s.distance} km</div>
                        <div className="text-xs text-gray-500">{s.driveTime}</div>
                      </div>
                    </div>

                    <div className="mt-2 text-xs text-gray-500">
                      {s.schoolType}
                    </div>
                  </div>
                </div>
              ))}
          </div>
        )}
      </div>
    </div>
  );
};