import { useNavigate } from "react-router-dom";
import { useRef, useEffect, useState } from "react";
import { ChevronLeft, ChevronRight } from "lucide-react";
import { ScrollFadeIn } from './ui/ScrollAnimation';
import mumbaiImg from "../assets/cities/mumbai.jpg";
import delhiImg from "../assets/cities/delhi-ncr.jpg";
import bengaluruImg from "../assets/cities/bengaluru.jpg";
import hyderabadImg from "../assets/cities/hyderabad.jpg";
import puneImg from "../assets/cities/pune.jpg";
import chennaiImg from "../assets/cities/chennai.jpg";
import ahmedabadImg from "../assets/cities/ahmedabad.jpg";
import kochiImg from "../assets/cities/kochi.jpg";
import lucknowImg from "../assets/cities/lucknow.jpg";

const cities = [
  {
    name: "Mumbai",
    image: mumbaiImg,
    properties: "25,000+ Properties"
  },
  {
    name: "Delhi NCR",
    image: delhiImg,
    properties: "18,000+ Properties"
  },
  {
    name: "Bangalore",
    image: bengaluruImg,
    properties: "15,000+ Properties"
  },
  {
    name: "Hyderabad",
    image: hyderabadImg,
    properties: "10,000+ Properties"
  },
  {
    name: "Pune",
    image: puneImg,
    properties: "12,000+ Properties"
  },
  {
    name: "Chennai",
    image: chennaiImg,
    properties: "8,500+ Properties"
  },
  {
    name: "Ahmedabad",
    image: ahmedabadImg,
    properties: "7,200+ Properties"
  },
  {
    name: "Kochi",
    image: kochiImg,
    properties: "5,500+ Properties"
  },
  {
    name: "Lucknow",
    image: lucknowImg,
    properties: "6,300+ Properties"
  }
];

export function SearchByCities() {
  const navigate = useNavigate();
  const scrollRef = useRef<HTMLDivElement>(null);
  const [isPaused, setIsPaused] = useState(false);

  useEffect(() => {
    const scrollContainer = scrollRef.current;
    if (!scrollContainer || isPaused) return;

    const autoScroll = () => {
      const maxScrollLeft = scrollContainer.scrollWidth - scrollContainer.clientWidth;
      if (scrollContainer.scrollLeft >= maxScrollLeft) {
        scrollContainer.scrollLeft = 0;
      } else {
        scrollContainer.scrollLeft += 1;
      }
    };

    const interval = setInterval(autoScroll, 30);
    return () => clearInterval(interval);
  }, [isPaused]);

  const handleCityClick = (cityName: string) => {
    navigate(`/properties?city=${cityName.toLowerCase()}`);
  };

  const scroll = (scrollOffset: number) => {
    if (scrollRef.current) {
      scrollRef.current.scrollBy({ left: scrollOffset, behavior: "smooth" });
    }
  };

  return (
    <section className="py-12 bg-white">
      <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
        <ScrollFadeIn>
          <div className="text-center mb-8">
            <h2 className="text-2xl md:text-3xl font-bold text-gray-900 mb-2">
              Search by Popular Indian Cities
            </h2>
          </div>
        </ScrollFadeIn>

        <div className="relative">
          <button
            onClick={() => scroll(-320)}
            className="absolute left-0 top-1/2 transform -translate-y-1/2 z-10 p-1 bg-white rounded-full shadow-md hover:bg-gray-100"
            aria-label="Scroll left"
          >
            <ChevronLeft className="w-4 h-4 text-gray-700" />
          </button>

          <div
            ref={scrollRef}
            className="flex space-x-6 overflow-x-auto scroll-smooth px-4"
            style={{ scrollbarWidth: 'none', msOverflowStyle: 'none' }}
          >
            {cities.map((city, index) => (
              <div
                key={index}
                className="relative group cursor-pointer overflow-hidden rounded-lg shadow-lg hover:shadow-xl transition-all duration-300 snap-center"
                style={{ width: '320px', height: '192px', minWidth: '320px', maxWidth: '320px', minHeight: '192px', maxHeight: '192px', flexShrink: 0 }}
                onClick={() => handleCityClick(city.name)}
                onMouseEnter={() => setIsPaused(true)}
                onMouseLeave={() => setIsPaused(false)}
              >
                <div style={{ width: '100%', height: '100%', overflow: 'hidden' }}>
                  <img
                    src={city.image}
                    alt={city.name}
                    className="group-hover:scale-110 transition-transform duration-500"
                    style={{ width: '100%', height: '100%', objectFit: 'cover' }}
                  />
                </div>
                <div className="absolute inset-0 pointer-events-none bg-gradient-to-t from-black/80 via-black/30 to-transparent">
                  <div className="absolute bottom-0 left-0 p-4">
                    <h3 className="text-2xl font-bold text-white">{city.name}</h3>
                  </div>
                </div>
              </div>
            ))}
          </div>

          <button
            onClick={() => scroll(320)}
            className="absolute right-0 top-1/2 transform -translate-y-1/2 z-10 p-1 bg-white rounded-full shadow-md hover:bg-gray-100"
            aria-label="Scroll right"
          >
            <ChevronRight className="w-4 h-4 text-gray-700" />
          </button>
        </div>
      </div>
    </section>
  );
}
