// src/components/SearchSuggestions.tsx
import React, { useState, useRef } from 'react';
import { Search, MapPin, Home, Clock, TrendingUp, Loader2, DollarSign, Bed, Zap, Star, Building, Filter } from 'lucide-react';
import { useSearchSuggestions, useRecentSearches } from '../hooks/useSearchSuggestions';

interface SearchSuggestionsProps {
  query: string;
  onSuggestionSelect: (suggestion: string) => void;
  onQueryChange: (query: string) => void;
  isVisible: boolean;
  onClose: () => void;
  popularSearches?: string[];
}

interface Suggestion {
  id: string;
  text: string;
  type: 'property' | 'location' | 'recent' | 'popular' | 'combined' | 'smart' | 'price_range' | 'bhk_location';
  icon: React.ReactNode;
  count?: number;
  priority?: number;
  category?: string;
  description?: string;
}

export function SearchSuggestions({ 
  query, 
  onSuggestionSelect, 
  onQueryChange, 
  isVisible, 
  onClose,
  popularSearches = []
}: SearchSuggestionsProps) {
  const [selectedIndex, setSelectedIndex] = useState(-1);
  const suggestionsRef = useRef<HTMLDivElement>(null);
  const { suggestions: rawSuggestions, loading } = useSearchSuggestions(query, isVisible);
  const { recentSearches, addRecentSearch } = useRecentSearches();

  // Convert raw suggestions to UI suggestions with enhanced icons
  const suggestions: Suggestion[] = rawSuggestions.map(suggestion => {
    let icon: React.ReactNode;
    let iconColor = 'text-gray-500';
    
    switch (suggestion.type) {
      case 'property':
        icon = <Home className={`w-4 h-4 ${iconColor = 'text-blue-500'}`} />;
        break;
      case 'location':
        icon = <MapPin className={`w-4 h-4 ${iconColor = 'text-green-500'}`} />;
        break;
      case 'recent':
        icon = <Clock className={`w-4 h-4 ${iconColor = 'text-gray-500'}`} />;
        break;
      case 'popular':
        icon = <TrendingUp className={`w-4 h-4 ${iconColor = 'text-orange-500'}`} />;
        break;
      case 'combined':
        icon = <Search className={`w-4 h-4 ${iconColor = 'text-purple-500'}`} />;
        break;
      case 'smart':
        icon = <Zap className={`w-4 h-4 ${iconColor = 'text-yellow-500'}`} />;
        break;
      case 'price_range':
        icon = <span className="w-4 h-4 text-emerald-500 font-bold text-sm flex items-center justify-center">₹</span>;
        break;
      case 'bhk_location':
        icon = <Bed className={`w-4 h-4 ${iconColor = 'text-indigo-500'}`} />;
        break;
      default:
        icon = <Search className={`w-4 h-4 ${iconColor = 'text-gray-500'}`} />;
    }
    
    return {
      ...suggestion,
      icon
    };
  });

  // Add recent searches if no query
  if (!query.trim() && recentSearches.length > 0) {
    const recentSuggestions: Suggestion[] = recentSearches.slice(0, 3).map((search, index) => ({
      id: `recent-${index}`,
      text: search,
      type: 'recent',
      icon: <Clock className="w-4 h-4 text-gray-500" />,
      category: 'Recent Searches'
    }));
    
    suggestions.unshift(...recentSuggestions);
  }
  
  // Group suggestions by category for better organization
  const groupedSuggestions = suggestions.reduce((groups: {[key: string]: Suggestion[]}, suggestion) => {
    const category = suggestion.category || 'Other';
    if (!groups[category]) {
      groups[category] = [];
    }
    groups[category].push(suggestion);
    return groups;
  }, {});

  // Reset selected index when suggestions change
  React.useEffect(() => {
    setSelectedIndex(-1);
  }, [suggestions]);

  const handleKeyDown = (e: React.KeyboardEvent) => {
    if (!isVisible || suggestions.length === 0) return;

    switch (e.key) {
      case 'ArrowDown':
        e.preventDefault();
        setSelectedIndex(prev => 
          prev < suggestions.length - 1 ? prev + 1 : 0
        );
        break;
      case 'ArrowUp':
        e.preventDefault();
        setSelectedIndex(prev => 
          prev > 0 ? prev - 1 : suggestions.length - 1
        );
        break;
      case 'Enter':
        e.preventDefault();
        if (selectedIndex >= 0) {
          const selectedSuggestion = suggestions[selectedIndex];
          addRecentSearch(selectedSuggestion.text);
          onSuggestionSelect(selectedSuggestion.text);
        }
        break;
      case 'Escape':
        e.preventDefault();
        onClose();
        break;
    }
  };

  const handleSuggestionClick = (suggestion: Suggestion) => {
    addRecentSearch(suggestion.text);
    onSuggestionSelect(suggestion.text);
    onClose();
  };

  if (!isVisible || (suggestions.length === 0 && !loading)) {
    return null;
  }

  return (
    <div 
      ref={suggestionsRef}
      className="absolute top-full left-0 right-0 bg-white border border-gray-200 rounded-lg shadow-lg z-50 mt-1 max-h-80 overflow-y-auto"
      onKeyDown={handleKeyDown}
    >
      <div className="py-2">
        {!query.trim() && (
          <div className="px-4 py-2 text-xs font-medium text-gray-500 uppercase tracking-wide border-b flex items-center gap-2">
            <Star className="w-3 h-3" />
            AI Search Suggestions
          </div>
        )}
        
        {query.trim().length >= 2 && (
          <div className="px-4 py-2 text-xs font-medium text-blue-600 uppercase tracking-wide border-b flex items-center gap-2">
            <Zap className="w-3 h-3" />
            Smart Suggestions for "{query.trim()}"
          </div>
        )}
        
        {loading && (
          <div className="px-4 py-3 flex items-center gap-3 text-gray-500">
            <Loader2 className="w-4 h-4 animate-spin" />
            <span>Finding smart suggestions...</span>
          </div>
        )}
        
        {Object.keys(groupedSuggestions).length > 0 ? (
          Object.entries(groupedSuggestions).map(([category, categorySuggestions]) => (
            <div key={category}>
              {Object.keys(groupedSuggestions).length > 1 && (
                <div className="px-4 py-1 text-xs font-medium text-gray-400 bg-gray-50 border-b">
                  {category}
                </div>
              )}
              {categorySuggestions.map((suggestion, index) => {
                const globalIndex = suggestions.findIndex(s => s.id === suggestion.id);
                return (
                  <div
                    key={suggestion.id}
                    className={`px-4 py-3 cursor-pointer flex items-start gap-3 hover:bg-gray-50 transition-colors ${
                      globalIndex === selectedIndex ? 'bg-blue-50 border-l-2 border-blue-500' : ''
                    }`}
                    onMouseDown={(e) => {
                      e.preventDefault();
                      handleSuggestionClick(suggestion);
                    }}
                  >
                    <div className="flex-shrink-0 mt-0.5">
                      {suggestion.icon}
                    </div>
                    <div className="flex-1 min-w-0">
                      <div className="text-gray-900 font-medium truncate">
                        {suggestion.text}
                      </div>
                      {suggestion.description && (
                        <div className="text-xs text-gray-500 mt-1">
                          {suggestion.description}
                        </div>
                      )}
                    </div>
                    <div className="flex items-center gap-2 flex-shrink-0">
                      {suggestion.count && (
                        <span className="text-xs text-gray-400 bg-gray-100 px-2 py-1 rounded-full">
                          {suggestion.count}
                        </span>
                      )}
                      {suggestion.type === 'recent' && (
                        <span className="text-xs text-gray-400">Recent</span>
                      )}
                      {suggestion.type === 'popular' && (
                        <span className="text-xs text-orange-500 font-medium">Popular</span>
                      )}
                      {suggestion.type === 'smart' && (
                        <span className="text-xs text-yellow-600 font-medium">Smart</span>
                      )}
                    </div>
                  </div>
                );
              })}
            </div>
          ))
        ) : (
          !loading && suggestions.map((suggestion, index) => (
            <div
              key={suggestion.id}
              className={`px-4 py-3 cursor-pointer flex items-start gap-3 hover:bg-gray-50 transition-colors ${
                index === selectedIndex ? 'bg-blue-50 border-l-2 border-blue-500' : ''
              }`}
              onMouseDown={(e) => {
              e.preventDefault();
              handleSuggestionClick(suggestion);
            }}
            >
              <div className="flex-shrink-0 mt-0.5">
                {suggestion.icon}
              </div>
              <div className="flex-1 min-w-0">
                <div className="text-gray-900 font-medium truncate">
                  {suggestion.text}
                </div>
                {suggestion.description && (
                  <div className="text-xs text-gray-500 mt-1">
                    {suggestion.description}
                  </div>
                )}
              </div>
              <div className="flex items-center gap-2 flex-shrink-0">
                {suggestion.count && (
                  <span className="text-xs text-gray-400 bg-gray-100 px-2 py-1 rounded-full">
                    {suggestion.count}
                  </span>
                )}
                {suggestion.type === 'recent' && (
                  <span className="text-xs text-gray-400">Recent</span>
                )}
                {suggestion.type === 'popular' && (
                  <span className="text-xs text-orange-500 font-medium">Popular</span>
                )}
                {suggestion.type === 'smart' && (
                  <span className="text-xs text-yellow-600 font-medium">Smart</span>
                )}
              </div>
            </div>
          ))
        )}
        
        {query.trim() && (
          <div className="px-4 py-2 border-t bg-gray-50">
            <div 
              className="flex items-center gap-2 text-blue-600 hover:text-blue-700 cursor-pointer text-sm font-medium p-2 rounded-lg hover:bg-blue-50 transition-colors"
              onClick={() => {
                addRecentSearch(query);
                onSuggestionSelect(query);
              }}
            >
              <Search className="w-4 h-4" />
              Search for "{query}"
              <div className="ml-auto text-xs text-gray-500">Press Enter</div>
            </div>
          </div>
        )}
      </div>
    </div>
  );
}