import { useState, useEffect } from "react";
import { useNavigate, useLocation } from "react-router-dom";
import { toast } from "sonner";

import "../styles/animated-auth.css";
import { GoogleLoginButton } from "./GoogleLoginButton";
import { Captcha } from "./SimpleCaptcha";
import { OTPInput } from "./ui/OTPInput";

export function SellerAuth() {
  const location = useLocation();
  const [isActive, setIsActive] = useState(
    location.pathname === "/seller/register",
  );
  const navigate = useNavigate();

  // Get return URL from query parameters
  const searchParams = new URLSearchParams(location.search);
  const returnUrl = searchParams.get("returnUrl");

  useEffect(() => {
    setIsActive(location.pathname === "/seller/register");
  }, [location.pathname]);

  // Login states
  const [loginEmail, setLoginEmail] = useState("");
  const [loginPassword, setLoginPassword] = useState("");
  const [loginLoading, setLoginLoading] = useState(false);
  const [loginSuccess, setLoginSuccess] = useState(false);
  const [loginError, setLoginError] = useState(false);

  // Register states
  const [step, setStep] = useState<"email" | "otp">("email");
  const [registerEmail, setRegisterEmail] = useState("");
  const [otp, setOtp] = useState("");
  const [password, setPassword] = useState("");
  const [confirmPassword, setConfirmPassword] = useState("");
  const [registerLoading, setRegisterLoading] = useState(false);
  const [registerSuccess, setRegisterSuccess] = useState(false);
  const [registerError, setRegisterError] = useState(false);
  const [toastMessage, setToastMessage] = useState<{
    type: "success" | "error";
    message: string;
  } | null>(null);
  const [passwordValidation, setPasswordValidation] = useState<{
    isValid: boolean;
    errors: string[];
    strength: number;
  }>({ isValid: false, errors: [], strength: 0 });
  const [showPasswordHelp, setShowPasswordHelp] = useState(false);
  const [showPassword, setShowPassword] = useState(false);
  const [showConfirmPassword, setShowConfirmPassword] = useState(false);
  const [captchaValid, setCaptchaValid] = useState(false);

  const validateEmail = (email: string): boolean => {
    const emailRegex = /^[a-zA-Z0-9._%+-]+@[a-zA-Z.-]+\.[a-zA-Z]{2,}$/;
    if (!emailRegex.test(email)) return false;

    const allowedDomains = [
      "gmail.com",
      "yahoo.com",
      "outlook.com",
      "hotmail.com",
      "live.com",
      "icloud.com",
      "proton.me",
      "protonmail.com",
      "zoho.com",
      "gmx.com",
      "mail.com",
      "yandex.com",
      "rediffmail.com",
      "aol.com",
      "msn.com",
      "mit.edu",
      "stanford.edu",
      "harvard.edu",
      "iitd.ac.in",
      "iith.ac.in",
      "vit.ac.in",
      "ox.ac.uk",
      "utoronto.ca",
      "alstonair.com",
      "sitams.in",
    ];
    const domain = email.split("@")[1]?.toLowerCase();
    return allowedDomains.includes(domain);
  };

  const validatePassword = (
    password: string,
    email?: string,
  ): { isValid: boolean; errors: string[]; strength: number } => {
    const errors: string[] = [];
    let strength = 0;

    // Length checks
    if (password.length < 8) errors.push("At least 8 characters required");
    if (password.length > 128) errors.push("Maximum 128 characters allowed");
    else if (password.length >= 8) strength += 1;

    // 🚫 NO SPACES ALLOWED
    if (/\s/.test(password)) {
      errors.push("Spaces are not allowed in the password");
    } else {
      strength += 1;
    }

    // Complexity checks
    if (!/[a-z]/.test(password))
      errors.push("At least one lowercase letter required");
    else strength += 1;

    if (!/[A-Z]/.test(password))
      errors.push("At least one uppercase letter required");
    else strength += 1;

    if (!/\d/.test(password)) errors.push("At least one number required");
    else strength += 1;

    if (!/[!@#$%^&*(),.?":{}|<>]/.test(password))
      errors.push("At least one special character required");
    else strength += 1;

    // Common passwords check
    const commonPasswords = [
      "password",
      "123456",
      "123456789",
      "qwerty",
      "abc123",
      "password123",
      "12345678",
      "admin",
      "letmein",
      "welcome",
    ];
    if (commonPasswords.includes(password.toLowerCase())) {
      errors.push("Password is too common");
      strength = 0;
    }

    // Email similarity check
    if (
      email &&
      password.toLowerCase().includes(email.split("@")[0].toLowerCase())
    ) {
      errors.push("Password cannot contain your email");
      strength = Math.max(0, strength - 1);
    }

    return { isValid: errors.length === 0, errors, strength };
  };

  const getPasswordStrengthText = (
    strength: number,
  ): { text: string; color: string } => {
    if (strength <= 1) return { text: "Very Weak", color: "#ef4444" };
    if (strength === 2) return { text: "Weak", color: "#f97316" };
    if (strength === 3) return { text: "Fair", color: "#eab308" };
    if (strength === 4) return { text: "Good", color: "#22c55e" };
    return { text: "Strong", color: "#16a34a" };
  };

  useEffect(() => {
    if (toastMessage) {
      if (toastMessage.type === "success") {
        toast.success(toastMessage.message);
      } else {
        toast.error(toastMessage.message);
      }
      setToastMessage(null);
    }
  }, [toastMessage]);

  // Login handler
  const handleLogin = async (e: React.FormEvent) => {
    e.preventDefault();
    setLoginLoading(true);
    setLoginError(false);
    setLoginSuccess(false);

    if (!validateEmail(loginEmail)) {
      setLoginError(true);
      setTimeout(() => {
        toast.error("Please enter a valid email address");
        setLoginError(false);
      }, 500);
      setLoginLoading(false);
      return;
    }

    if (!captchaValid) {
      setLoginError(true);
      setTimeout(() => {
        toast.error("Please complete the CAPTCHA verification");
        setLoginError(false);
      }, 500);
      setLoginLoading(false);
      return;
    }

    try {
      const response = await fetch(
        `${import.meta.env.VITE_API_BASE || "http://localhost:8090/api"}/seller-agent-auth/login`,
        {
          method: "POST",
          headers: { "Content-Type": "application/json" },
          body: JSON.stringify({
            email: loginEmail,
            password: loginPassword,
            user_type: "seller",
          }),
        },
      );

      const data = await response.json();

      if (response.ok && data.success) {
        if (data.user_type !== "seller") {
          setLoginError(true);
          setTimeout(() => {
            toast.error("This account is not registered as a seller");
            setLoginError(false);
          }, 500);
          return;
        }
        localStorage.removeItem("agentToken");
        localStorage.removeItem("agentEmail");
        localStorage.removeItem("agentId");
        localStorage.setItem("sellerToken", data.access_token);
        localStorage.setItem("sellerEmail", data.email);
        localStorage.setItem("sellerId", data.user_id);
        // Mark profile as completed for existing users
        localStorage.setItem("sellerProfileCompleted", "true");
        setLoginSuccess(true);
        setTimeout(() => {
          toast.success("Login successful!");
          // For existing users logging in, go directly to dashboard or return URL
          if (returnUrl) {
            navigate(decodeURIComponent(returnUrl));
          } else {
            navigate("/seller/dashboard");
          }
        }, 600);
      } else {
        setLoginError(true);
        setTimeout(() => {
          toast.error("Invalid email or password");
          setLoginError(false);
        }, 500);
      }
    } catch (error) {
      setLoginError(true);
      setTimeout(() => {
        toast.error("Invalid email or password");
        setLoginError(false);
      }, 500);
    } finally {
      setLoginLoading(false);
    }
  };

  // Send OTP handler
  const handleSendOTP = async (e: React.FormEvent) => {
    e.preventDefault();
    setRegisterLoading(true);
    setRegisterError(false);
    setRegisterSuccess(false);

    if (!validateEmail(registerEmail)) {
      setRegisterError(true);
      setTimeout(() => {
        toast.error("Please enter a valid email address");
        setRegisterError(false);
      }, 500);
      setRegisterLoading(false);
      return;
    }

    if (!captchaValid) {
      setRegisterError(true);
      setTimeout(() => {
        toast.error("Please complete the CAPTCHA verification");
        setRegisterError(false);
      }, 500);
      setRegisterLoading(false);
      return;
    }

    try {
      const response = await fetch(
        `${import.meta.env.VITE_API_BASE || "http://localhost:8090/api"}/seller-agent-auth/send-otp`,
        {
          method: "POST",
          headers: { "Content-Type": "application/json" },
          body: JSON.stringify({ email: registerEmail, user_type: "seller" }),
        },
      );

      const data = await response.json();

      if (response.ok && data.success) {
        setStep("otp");
        setToastMessage({
          type: "success",
          message: "OTP sent to your email!",
        });
      } else {
        setRegisterError(true);
        setTimeout(() => {
          const errorMsg =
            typeof data.detail === "string"
              ? data.detail
              : Array.isArray(data.detail)
                ? data.detail[0]?.msg || "Failed to send OTP"
                : "Failed to send OTP";
          toast.error(errorMsg);
          setRegisterError(false);
        }, 500);
      }
    } catch (error) {
      setRegisterError(true);
      setTimeout(() => {
        toast.error("Failed to send OTP. Please try again.");
        setRegisterError(false);
      }, 500);
    } finally {
      setRegisterLoading(false);
    }
  };

  // Verify OTP handler
  const handleVerifyOTP = async (e: React.FormEvent) => {
    e.preventDefault();

    if (password !== confirmPassword) {
      setRegisterError(true);
      setTimeout(() => {
        toast.error("Passwords do not match");
        setRegisterError(false);
      }, 500);
      return;
    }

    const passwordCheck = validatePassword(password, registerEmail);
    if (!passwordCheck.isValid) {
      setRegisterError(true);
      setTimeout(() => {
        toast.error(passwordCheck.errors[0] || "Invalid password");
        setRegisterError(false);
      }, 500);
      return;
    }

    setRegisterLoading(true);
    setRegisterError(false);
    setRegisterSuccess(false);

    try {
      const response = await fetch(
        `${import.meta.env.VITE_API_BASE || "http://localhost:8090/api"}/seller-agent-auth/verify-otp`,
        {
          method: "POST",
          headers: { "Content-Type": "application/json" },
          body: JSON.stringify({
            email: registerEmail,
            otp_code: otp,
            password,
            user_type: "seller",
          }),
        },
      );

      const data = await response.json();

      if (response.ok && data.success) {
        localStorage.removeItem("agentToken");
        localStorage.removeItem("agentEmail");
        localStorage.removeItem("agentId");
        localStorage.setItem("sellerToken", data.access_token);
        localStorage.setItem("sellerEmail", data.email);
        localStorage.setItem("sellerId", data.user_id);
        setRegisterSuccess(true);
        setTimeout(() => {
          toast.success("Registration successful!");
          // Reset to login page initial state
          setIsActive(false);
          setStep("email");
          setOtp('');
          setPassword('');
          setConfirmPassword('');
          // New users always go to profile completion
          navigate("/seller/profile-completion");
        }, 600);
      } else {
        setRegisterError(true);
        setOtp(''); // Clear OTP input on error
        setTimeout(() => {
          const errorMsg =
            typeof data.detail === "string"
              ? data.detail
              : Array.isArray(data.detail)
                ? data.detail[0]?.msg || "Registration failed"
                : "Registration failed";
          toast.error(errorMsg);
          setRegisterError(false);
        }, 500);
      }
    } catch (error) {
      setRegisterError(true);
      setOtp(''); // Clear OTP input on error
      setTimeout(() => {
        toast.error("Registration failed. Please try again.");
        setRegisterError(false);
      }, 500);
    } finally {
      setRegisterLoading(false);
    }
  };

  return (
    <div className={`seller-auth-container ${isActive ? "active" : ""}`}>
      <div className="seller-auth-wrapper">
        {/* Left side illustration */}
        <div className="illustration-side">
          <div className="character-illustration"></div>
        </div>

        {/* Right side form */}
        <div className="form-side">
          <button
            onClick={() => navigate('/')}
            style={{
              position: 'absolute',
              top: '20px',
              right: '20px',
              background: 'none',
              border: 'none',
              cursor: 'pointer',
              color: '#6b7280',
              fontSize: '24px',
              lineHeight: '1',
              padding: '8px',
              zIndex: 10,
              transition: 'color 0.2s'
            }}
            onMouseEnter={(e) => e.currentTarget.style.color = '#1f2937'}
            onMouseLeave={(e) => e.currentTarget.style.color = '#6b7280'}
            title="Close"
          >
            ×
          </button>
        <div className={`container ${isActive ? 'active' : ''}`}>
          {/* Login Form */}
          <div className="form-box Login">
            <h2>Welcome Back</h2>
            <form onSubmit={handleLogin}>
              <div className="input-box">
                <label>Email Address</label>
                <input 
                  type="email" 
                  value={loginEmail}
                  onChange={(e) => setLoginEmail(e.target.value)}
                  className={loginError ? 'error' : loginSuccess ? 'success' : ''}
                  placeholder="Enter your email"
                  required 
                />
              </div>

              <div className="input-box">
                <label>Password</label>
                <div style={{ position: 'relative' }}>
                  <input 
                    type={showPassword ? 'text' : 'password'}
                    value={loginPassword}
                    onChange={(e) => setLoginPassword(e.target.value)}
                    className={loginError ? 'error' : loginSuccess ? 'success' : ''}
                    placeholder="Enter your password"
                    required 
                  />
                  <button
                    type="button"
                    onClick={() => setShowPassword(!showPassword)}
                    style={{
                      position: 'absolute',
                      right: '12px',
                      top: '50%',
                      transform: 'translateY(-50%)',
                      background: 'none',
                      border: 'none',
                      cursor: 'pointer',
                      color: '#6b7280'
                    }}
                  >
                    {showPassword ? (
                      <svg width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" strokeWidth="2">
                        <path d="M17.94 17.94A10.07 10.07 0 0 1 12 20c-7 0-11-8-11-8a18.45 18.45 0 0 1 5.06-5.94M9.9 4.24A9.12 9.12 0 0 1 12 4c7 0 11 8 11 8a18.5 18.5 0 0 1-2.16 3.19m-6.72-1.07a3 3 0 1 1-4.24-4.24"/>
                        <line x1="1" y1="1" x2="23" y2="23"/>
                      </svg>
                    ) : (
                      <svg width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" strokeWidth="2">
                        <path d="M1 12s4-8 11-8 11 8 11 8-4 8-11 8-11-8-11-8z"/>
                        <circle cx="12" cy="12" r="3"/>
                      </svg>
                    )}
                  </button>
                </div>
              </div>

              <div className="input-box">
                <Captcha key="login" onVerify={setCaptchaValid} />
              </div>

              <div style={{ display: 'flex', gap: '12px', marginTop: '16px', alignItems: 'stretch' }}>
                <button
                  type="button"
                  onClick={() => navigate('/')}
                  className="btn"
                  style={{
                    flex: 1,
                    padding: '12px',
                    border: '2px solid transparent',
                    boxSizing: 'border-box'
                  }}
                >
                  Cancel
                </button>
                <button 
                  className={`btn`} 
                  type="submit" 
                  disabled={loginLoading}
                  style={{ 
                    flex: 1, 
                    boxSizing: 'border-box',
                    padding: '12px',
                    border: '2px solid transparent'
                  }}
                >
                  {loginLoading ? "Signing in..." : loginSuccess ? "Success!" : "Sign In"}
                </button>
              </div>

              <div style={{ margin: '16px 0', textAlign: 'center', color: '#6b7280', fontSize: '14px' }}>or</div>
              
              <GoogleLoginButton 
                userType="seller"
                onSuccess={(userData) => {
                  localStorage.removeItem("agentToken");
                  localStorage.removeItem("agentEmail");
                  localStorage.removeItem("agentId");
                  localStorage.setItem("sellerToken", userData.access_token);
                  localStorage.setItem("sellerEmail", userData.email);
                  localStorage.setItem("sellerId", userData.user_id);
                  toast.success("Login successful!");
                  // For Google login, check if this is a new user
                  if (userData.is_new_user) {
                    navigate("/seller/profile-completion");
                  } else {
                    // Mark profile as completed for existing users
                    localStorage.setItem('sellerProfileCompleted', 'true');
                    navigate("/seller/dashboard");
                  }
                }}
                disabled={loginLoading}
              />

              <div className="regi-link">
                <p>Don't have an account? 
                  <a href="#" onClick={(e) => {e.preventDefault(); setIsActive(true);}}>
                    Create Account
                  </a>
                </p>
                <p style={{ marginTop: '8px' }}>
                  <a href="#" onClick={(e) => {e.preventDefault(); navigate('/seller/forgot-password');}}>
                    Forgot Password?
                  </a>
                </p>
              </div>
            </form>
          </div>

          {/* Register Form */}
          <div className="form-box Register">
            <h2>Create Account</h2>
            
            {step === "email" ? (
              <form onSubmit={handleSendOTP}>
                <div className="input-box">
                  <label>Email Address</label>
                  <input
                    type="email"
                    value={registerEmail}
                    onChange={(e) => setRegisterEmail(e.target.value)}
                    className={
                      registerError ? "error" : registerSuccess ? "success" : ""
                    }
                    placeholder="Enter your email"
                    required 
                  />
                </div>

                <div className="input-box">
                  <Captcha key="register" onVerify={setCaptchaValid} />
                </div>

                <div style={{ display: 'flex', gap: '12px', marginTop: '16px', alignItems: 'stretch' }}>
                  <button
                    type="button"
                    onClick={() => navigate('/')}
                    className="btn"
                    style={{
                      flex: 1,
                      padding: '12px',
                      border: '2px solid transparent',
                      boxSizing: 'border-box'
                    }}
                  >
                    Cancel
                  </button>
                  <button 
                    className={`btn`} 
                    type="submit" 
                    disabled={registerLoading}
                    style={{ 
                      flex: 1, 
                      boxSizing: 'border-box',
                      padding: '12px',
                      border: '2px solid transparent'
                    }}
                  >
                    {registerLoading ? "Sending Code..." : registerSuccess ? "Success!" : "Send Verification Code"}
                  </button>
                </div>

                <div className="regi-link">
                  <p>Already have an account? 
                    <a href="#" onClick={(e) => {e.preventDefault(); setIsActive(false);}}>
                      Sign In
                    </a>
                  </p>
                </div>
              </form>
            ) : (
              <form onSubmit={handleVerifyOTP}>
                <div className="input-box">
                  <label>Enter 6-Digit OTP</label>
                  <div className="otp-container">
                    <OTPInput
                      length={6}
                      value={otp}
                      onChange={setOtp}
                      disabled={registerLoading}
                    />
                  </div>
                </div>
                <div style={{ textAlign: 'center', marginTop: '8px', marginBottom: '16px' }}>
                  <p style={{ fontSize: '14px', color: '#6b7280', marginBottom: '8px' }}>Didn't receive the code?</p>
                  {registerLoading ? null : (
                    <>
                      <button
                        type="button"
                        onClick={handleSendOTP}
                        disabled={registerLoading}
                        style={{
                          background: 'none',
                          border: 'none',
                          color: '#3b82f6',
                          fontSize: '14px',
                          fontWeight: '600',
                          cursor: 'pointer',
                          textDecoration: 'underline'
                        }}
                      >
                        Resend OTP
                      </button>
                      <span style={{ margin: '0 8px', color: '#6b7280' }}>|</span>
                      <button
                        type="button"
                        onClick={() => {
                          setStep('email');
                          setOtp('');
                          setPassword('');
                          setConfirmPassword('');
                        }}
                        disabled={registerLoading}
                        style={{
                          background: 'none',
                          border: 'none',
                          color: '#6b7280',
                          fontSize: '14px',
                          fontWeight: '600',
                          cursor: 'pointer',
                          textDecoration: 'underline'
                        }}
                      >
                        Cancel
                      </button>
                    </>
                  )}
                </div>

                <div className="input-box">
                  <label>Password</label>
                  <div style={{ position: "relative" }}>
                    <input
                      type={showPassword ? "text" : "password"}
                      value={password}
                      onChange={(e) => {
                        setPassword(e.target.value);
                        const validation = validatePassword(
                          e.target.value,
                          registerEmail,
                        );
                        setPasswordValidation(validation);
                      }}
                      onFocus={() => setShowPasswordHelp(true)}
                      onBlur={() => setShowPasswordHelp(false)}
                      className={
                        registerError ? "error" : registerSuccess ? "success" : ""
                      }
                      placeholder="Create password (min 8 chars)"
                      required
                    />
                    <button
                      type="button"
                      onClick={() => setShowPassword(!showPassword)}
                      style={{
                        position: "absolute",
                        right: "12px",
                        top: "50%",
                        transform: "translateY(-50%)",
                        background: "none",
                        border: "none",
                        cursor: "pointer",
                        color: "#6b7280",
                      }}
                    >
                      {showPassword ? (
                        <svg
                          width="20"
                          height="20"
                          viewBox="0 0 24 24"
                          fill="none"
                          stroke="currentColor"
                          strokeWidth="2"
                        >
                          <path d="M17.94 17.94A10.07 10.07 0 0 1 12 20c-7 0-11-8-11-8a18.45 18.45 0 0 1 5.06-5.94M9.9 4.24A9.12 9.12 0 0 1 12 4c7 0 11 8 11 8a18.5 18.5 0 0 1-2.16 3.19m-6.72-1.07a3 3 0 1 1-4.24-4.24" />
                          <line x1="1" y1="1" x2="23" y2="23" />
                        </svg>
                      ) : (
                        <svg
                          width="20"
                          height="20"
                          viewBox="0 0 24 24"
                          fill="none"
                          stroke="currentColor"
                          strokeWidth="2"
                        >
                          <path d="M1 12s4-8 11-8 11 8 11 8-4 8-11 8-11-8-11-8z" />
                          <circle cx="12" cy="12" r="3" />
                        </svg>
                      )}
                    </button>
                  </div>
                  {password && (
                    <div
                      className="password-strength"
                      style={{ marginTop: "8px" }}
                    >
                      <div
                        style={{
                          display: "flex",
                          alignItems: "center",
                          gap: "8px",
                        }}
                      >
                        <div
                          style={{
                            flex: 1,
                            height: "4px",
                            backgroundColor: "#e5e7eb",
                            borderRadius: "2px",
                          }}
                        >
                          <div
                            style={{
                              width: `${(passwordValidation.strength / 5) * 100}%`,
                              height: "100%",
                              backgroundColor: getPasswordStrengthText(
                                passwordValidation.strength,
                              ).color,
                              borderRadius: "2px",
                              transition: "all 0.3s ease",
                            }}
                          />
                        </div>
                        <span
                          style={{
                            fontSize: "12px",
                            color: getPasswordStrengthText(
                              passwordValidation.strength,
                            ).color,
                            fontWeight: "500",
                          }}
                        >
                          {
                            getPasswordStrengthText(
                              passwordValidation.strength,
                            ).text
                          }
                        </span>
                      </div>
                      {(showPasswordHelp ||
                        passwordValidation.errors.length > 0) && (
                        <div style={{ marginTop: "4px" }}>
                          {passwordValidation.errors.map((error, index) => (
                            <div
                              key={index}
                              style={{
                                fontSize: "11px",
                                color: "#ef4444",
                                marginTop: "2px",
                              }}
                            >
                              • {error}
                            </div>
                          ))}
                        </div>
                      )}
                    </div>
                  )}
                </div>

                <div className="input-box">
                  <label>Confirm Password</label>
                  <div style={{ position: "relative" }}>
                    <input
                      type={showConfirmPassword ? "text" : "password"}
                      value={confirmPassword}
                      onChange={(e) => setConfirmPassword(e.target.value)}
                      className={
                        registerError ? "error" : registerSuccess ? "success" : ""
                      }
                      placeholder="Confirm your password"
                      required
                    />
                    <button
                      type="button"
                      onClick={() =>
                        setShowConfirmPassword(!showConfirmPassword)
                      }
                      style={{
                        position: "absolute",
                        right: "12px",
                        top: "50%",
                        transform: "translateY(-50%)",
                        background: "none",
                        border: "none",
                        cursor: "pointer",
                        color: "#6b7280",
                      }}
                    >
                      {showConfirmPassword ? (
                        <svg
                          width="20"
                          height="20"
                          viewBox="0 0 24 24"
                          fill="none"
                          stroke="currentColor"
                          strokeWidth="2"
                        >
                          <path d="M17.94 17.94A10.07 10.07 0 0 1 12 20c-7 0-11-8-11-8a18.45 18.45 0 0 1 5.06-5.94M9.9 4.24A9.12 9.12 0 0 1 12 4c7 0 11 8 11 8a18.5 18.5 0 0 1-2.16 3.19m-6.72-1.07a3 3 0 1 1-4.24-4.24" />
                          <line x1="1" y1="1" x2="23" y2="23" />
                        </svg>
                      ) : (
                        <svg
                          width="20"
                          height="20"
                          viewBox="0 0 24 24"
                          fill="none"
                          stroke="currentColor"
                          strokeWidth="2"
                        >
                          <path d="M1 12s4-8 11-8 11 8 11 8-4 8-11 8-11-8-11-8z" />
                          <circle cx="12" cy="12" r="3" />
                        </svg>
                      )}
                    </button>
                  </div>
                </div>

                <button
                  className={`btn ${registerLoading ? "loading" : ""} ${registerSuccess ? "success" : ""}`}
                  type="submit"
                  disabled={registerLoading || !otp.trim()}
                  style={{
                    backgroundColor: registerLoading || !otp.trim() ? '#9ca3af' : '#3b82f6',
                    cursor: registerLoading || !otp.trim() ? 'not-allowed' : 'pointer'
                  }}
                >
                  {registerLoading
                    ? "Creating Account..."
                    : registerSuccess
                      ? "Success!"
                      : "Create Account"}
                </button>

                <div className="regi-link">
                  <p>
                    <a
                      href="#"
                      onClick={(e) => {
                        e.preventDefault();
                        setStep("email");
                      }}
                    >
                      Back
                    </a>{" "}
                    |
                    <a
                      href="#"
                      onClick={(e) => {
                        e.preventDefault();
                        setIsActive(false);
                      }}
                    >
                      Sign In
                    </a>
                  </p>
                </div>
              </form>
              )}
            </div>
          </div>
        </div>
      </div>
    </div>
  );
}
