import { useState, useEffect } from "react";
import { useNavigate } from "react-router-dom";
import { toast } from "sonner";
import '../styles/animated-auth.css';

export function SellerAuthAnimated() {
  const [isActive, setIsActive] = useState(false);
  const [step, setStep] = useState<"email" | "otp">("email");
  
  // Login states
  const [loginEmail, setLoginEmail] = useState("");
  const [loginPassword, setLoginPassword] = useState("");
  const [loginLoading, setLoginLoading] = useState(false);
  
  // Register states
  const [registerEmail, setRegisterEmail] = useState("");
  const [otp, setOtp] = useState("");
  const [password, setPassword] = useState("");
  const [confirmPassword, setConfirmPassword] = useState("");
  const [registerLoading, setRegisterLoading] = useState(false);
  
  const navigate = useNavigate();

  const handleLogin = async (e: React.FormEvent) => {
    e.preventDefault();
    setLoginLoading(true);
    
    try {
      const response = await fetch(`${import.meta.env.VITE_API_BASE || 'http://localhost:8090/api'}/seller-agent-auth/login`, {
        method: 'POST',
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify({ email: loginEmail, password: loginPassword, user_type: 'seller' })
      });
      
      const data = await response.json();
      
      if (response.ok && data.success) {
        localStorage.setItem("sellerToken", data.access_token);
        localStorage.setItem("sellerEmail", data.email);
        localStorage.setItem("sellerId", data.user_id);
        setTimeout(() => toast.success("Login successful!"), 0);
        navigate("/seller/dashboard");
      } else {
        setTimeout(() => toast.error(data.detail || "Invalid credentials"), 0);
      }
    } catch (error) {
      setTimeout(() => toast.error("Login failed. Please try again."), 0);
    } finally {
      setLoginLoading(false);
    }
  };

  const handleSendOTP = async (e: React.FormEvent) => {
    e.preventDefault();
    setRegisterLoading(true);
    
    try {
      const response = await fetch(`${import.meta.env.VITE_API_BASE || 'http://localhost:8090/api'}/seller-agent-auth/send-otp`, {
        method: 'POST',
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify({ email: registerEmail, user_type: 'seller' })
      });
      
      const data = await response.json();
      
      if (response.ok && data.success) {
        setStep("otp");
        toast.success("OTP sent to your email!");
      } else {
        toast.error(data.detail || "Failed to send OTP");
      }
    } catch (error) {
      toast.error("Failed to send OTP. Please try again.");
    } finally {
      setRegisterLoading(false);
    }
  };

  const handleVerifyOTP = async (e: React.FormEvent) => {
    e.preventDefault();
    
    if (password !== confirmPassword) {
      toast.error("Passwords do not match");
      return;
    }
    
    if (password.length < 6) {
      toast.error("Password must be at least 6 characters");
      return;
    }
    
    setRegisterLoading(true);
    
    try {
      const response = await fetch(`${import.meta.env.VITE_API_BASE || 'http://localhost:8090/api'}/seller-agent-auth/verify-otp`, {
        method: 'POST',
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify({ email: registerEmail, otp_code: otp, password })
      });
      
      const data = await response.json();
      
      if (response.ok && data.success) {
        localStorage.setItem("sellerToken", data.access_token);
        localStorage.setItem("sellerEmail", data.email);
        localStorage.setItem("sellerId", data.user_id);
        toast.success("Registration successful!");
        navigate("/seller/dashboard");
      } else {
        toast.error(data.detail || "Registration failed");
      }
    } catch (error) {
      toast.error("Registration failed. Please try again.");
    } finally {
      setRegisterLoading(false);
    }
  };

  return (
    <div className="seller-auth-container">
      <div className="curved-shape"></div>
      <div className="curved-shape2"></div>
      
      <div className={`container ${isActive ? 'active' : ''}`}>
        {/* Login Form */}
        <div className="form-box Login">
          <h2 className="animation" style={{"--D": 0, "--S": 21} as React.CSSProperties}>Login</h2>
          <form onSubmit={handleLogin}>
            <div className="input-box animation" style={{"--D": 1, "--S": 22} as React.CSSProperties}>
              <input 
                type="email" 
                value={loginEmail}
                onChange={(e) => setLoginEmail(e.target.value)}
                required 
              />
              <label>Email</label>
              <i className="input-icon">📧</i>
            </div>

            <div className="input-box animation" style={{"--D": 2, "--S": 23} as React.CSSProperties}>
              <input 
                type="password" 
                value={loginPassword}
                onChange={(e) => setLoginPassword(e.target.value)}
                required 
              />
              <label>Password</label>
              <i className="input-icon">🔒</i>
            </div>

            <div className="input-box animation" style={{"--D": 3, "--S": 24} as React.CSSProperties}>
              <button className="btn" type="submit" disabled={loginLoading}>
                {loginLoading ? "Signing in..." : "Login"}
              </button>
            </div>

            <div className="regi-link animation" style={{"--D": 4, "--S": 25} as React.CSSProperties}>
              <p>Don't have an account? <br /> 
                <a href="#" onClick={(e) => {e.preventDefault(); setIsActive(true);}}>
                  Sign Up
                </a>
              </p>
            </div>
          </form>
        </div>

        <div className="info-content Login">
          <h2 className="animation" style={{"--D": 0, "--S": 20} as React.CSSProperties}>WELCOME BACK!</h2>
          <p className="animation" style={{"--D": 1, "--S": 21} as React.CSSProperties}>
            We are happy to have you with us again. If you need anything, we are here to help.
          </p>
        </div>

        {/* Register Form */}
        <div className="form-box Register">
          <h2 className="animation" style={{"--li": 17, "--S": 0} as React.CSSProperties}>Register</h2>
          
          {step === "email" ? (
            <form onSubmit={handleSendOTP}>
              <div className="input-box animation" style={{"--li": 18, "--S": 1} as React.CSSProperties}>
                <input 
                  type="email" 
                  value={registerEmail}
                  onChange={(e) => setRegisterEmail(e.target.value)}
                  required 
                />
                <label>Email</label>
                <i className="input-icon">📧</i>
              </div>

              <div className="input-box animation" style={{"--li": 20, "--S": 4} as React.CSSProperties}>
                <button className="btn" type="submit" disabled={registerLoading}>
                  {registerLoading ? "Sending OTP..." : "Send OTP"}
                </button>
              </div>

              <div className="regi-link animation" style={{"--li": 21, "--S": 5} as React.CSSProperties}>
                <p>Already have an account? <br /> 
                  <a href="#" onClick={(e) => {e.preventDefault(); setIsActive(false);}}>
                    Sign In
                  </a>
                </p>
              </div>
            </form>
          ) : (
            <form onSubmit={handleVerifyOTP}>
              <div className="input-box animation" style={{"--li": 18, "--S": 1} as React.CSSProperties}>
                <input 
                  type="text" 
                  value={otp}
                  onChange={(e) => setOtp(e.target.value)}
                  maxLength={6}
                  required 
                />
                <label>Enter 6-digit OTP</label>
                <i className="input-icon">🔢</i>
              </div>

              <div className="input-box animation" style={{"--li": 19, "--S": 2} as React.CSSProperties}>
                <input 
                  type="password" 
                  value={password}
                  onChange={(e) => setPassword(e.target.value)}
                  required 
                />
                <label>Password (min 6 chars)</label>
                <i className="input-icon">🔒</i>
              </div>

              <div className="input-box animation" style={{"--li": 19, "--S": 3} as React.CSSProperties}>
                <input 
                  type="password" 
                  value={confirmPassword}
                  onChange={(e) => setConfirmPassword(e.target.value)}
                  required 
                />
                <label>Confirm Password</label>
                <i className="input-icon">🔒</i>
              </div>

              <div className="input-box animation" style={{"--li": 20, "--S": 4} as React.CSSProperties}>
                <button className="btn" type="submit" disabled={registerLoading}>
                  {registerLoading ? "Verifying..." : "Complete Registration"}
                </button>
              </div>

              <div className="regi-link animation" style={{"--li": 21, "--S": 5} as React.CSSProperties}>
                <p>
                  <a href="#" onClick={(e) => {e.preventDefault(); setStep("email");}}>
                    Back to Email
                  </a> | 
                  <a href="#" onClick={(e) => {e.preventDefault(); setIsActive(false);}}>
                    Sign In
                  </a>
                </p>
              </div>
            </form>
          )}
        </div>

        <div className="info-content Register">
          <h2 className="animation" style={{"--li": 17, "--S": 0} as React.CSSProperties}>WELCOME!</h2>
          <p className="animation" style={{"--li": 18, "--S": 1} as React.CSSProperties}>
            We're delighted to have you here. If you need any assistance, feel free to reach out.
          </p>
        </div>
      </div>
    </div>
  );
}