import React, { useState, useEffect } from 'react';
import axios from 'axios';

interface Lead {
  id: string;
  name: string;
  email: string;
  phone: string;
  message?: string;
  budget?: string;
  preferred_contact: string;
  score?: number;
  category?: string;
  property_title?: string;
  status: string;
  created_at: string;
}

export const SellerLeads: React.FC = () => {
  const [leads, setLeads] = useState<Lead[]>([]);
  const [loading, setLoading] = useState(true);
  const [sortBy, setSortBy] = useState('score');
  const [filterScore, setFilterScore] = useState('all');

  useEffect(() => {
    fetchLeads();
  }, []);

  const fetchLeads = async () => {
    try {
      const user = JSON.parse(localStorage.getItem('user') || '{}');
      const response = await axios.get(`/api/leads?seller_id=${user.user_id}`);
      setLeads(response.data.leads);
    } catch (error) {
      console.error('Error fetching leads:', error);
      // Hardcoded data for demo
      setLeads([
        {
          id: '1',
          name: 'John Smith',
          email: 'john@example.com',
          phone: '+91 9876543210',
          message: 'Interested in 3BHK apartment',
          budget: '80 Lakhs',
          preferred_contact: 'call',
          score: 85,
          category: 'High',
          property_title: '3BHK Luxury Apartment',
          status: 'new',
          created_at: new Date().toISOString()
        },
        {
          id: '2',
          name: 'Sarah Johnson',
          email: 'sarah@example.com',
          phone: '+91 9876543211',
          message: 'Looking for investment property',
          budget: '50 Lakhs',
          preferred_contact: 'whatsapp',
          score: 65,
          category: 'Medium',
          property_title: '2BHK Modern Flat',
          status: 'contacted',
          created_at: new Date().toISOString()
        }
      ]);
    } finally {
      setLoading(false);
    }
  };

  const handleContact = async (lead: Lead, method: string) => {
    // Guard: Don't allow actions on archived leads
    if (lead.status === 'archived') {
      alert('Cannot contact archived leads');
      return;
    }

    // Open contact method
    switch (method) {
      case 'call':
        window.open(`tel:${lead.phone}`);
        break;
      case 'sms':
        window.open(`sms:${lead.phone}`);
        break;
      case 'whatsapp':
        window.open(`https://wa.me/${lead.phone.replace(/[^0-9]/g, '')}`);
        break;
      case 'email':
        window.open(`mailto:${lead.email}`);
        break;
    }

    // Update lead status to contacted
    try {
      const user = JSON.parse(localStorage.getItem('user') || '{}');
      const token = localStorage.getItem('access_token');
      const API_BASE = import.meta.env.VITE_API_BASE || 'http://localhost:8090/api';

      const response = await axios.put(
        `${API_BASE}/leads-management/leads/${lead.id}/contact?user_id=${user.user_id}&contact_method=${method}`,
        {},
        {
          headers: {
            'Content-Type': 'application/json',
            ...(token && { Authorization: `Bearer ${token}` }),
          },
        }
      );

      if (response.status === 200) {
        // Update local state
        setLeads((prevLeads) =>
          prevLeads.map((l) =>
            l.id === lead.id ? { ...l, status: 'contacted' } : l
          )
        );
      }
    } catch (error) {
      console.error('Error updating lead status:', error);
    }
  };

  const getScoreColor = (score?: number) => {
    if (!score) return 'bg-gray-100 text-gray-800';
    if (score >= 80) return 'bg-red-100 text-red-800';
    if (score >= 50) return 'bg-yellow-100 text-yellow-800';
    return 'bg-gray-100 text-gray-800';
  };

  const sortedLeads = leads
    .filter(lead => filterScore === 'all' ||
      (filterScore === 'high' && (lead.score || 0) >= 80) ||
      (filterScore === 'medium' && (lead.score || 0) >= 50 && (lead.score || 0) < 80) ||
      (filterScore === 'low' && (lead.score || 0) < 50))
    .sort((a, b) => {
      if (sortBy === 'score') return (b.score || 0) - (a.score || 0);
      if (sortBy === 'name') return a.name.localeCompare(b.name);
      return new Date(b.created_at).getTime() - new Date(a.created_at).getTime();
    });

  if (loading) {
    return <div className="p-6">Loading leads...</div>;
  }

  return (
    <div className="p-6">
      <div className="flex justify-between items-center mb-6">
        <h1 className="text-2xl font-bold">My Leads</h1>
        <div className="flex gap-4">
          <select
            value={sortBy}
            onChange={(e) => setSortBy(e.target.value)}
            className="px-3 py-2 border rounded-lg"
          >
            <option value="score">Sort by Score</option>
            <option value="name">Sort by Name</option>
            <option value="date">Sort by Date</option>
          </select>

          <select
            value={filterScore}
            onChange={(e) => setFilterScore(e.target.value)}
            className="px-3 py-2 border rounded-lg"
          >
            <option value="all">All Scores</option>
            <option value="high">High Score (80+)</option>
            <option value="medium">Medium Score (50-79)</option>
            <option value="low">Low Score (&lt;50)</option>
          </select>
        </div>
      </div>

      <div className="grid gap-4">
        {sortedLeads.map(lead => (
          <div key={lead.id} className="bg-white border rounded-lg p-4 shadow-sm">
            <div className="flex justify-between items-start mb-3">
              <div>
                <h3 className="font-semibold text-lg">{lead.name}</h3>
                <p className="text-gray-600">{lead.property_title}</p>
                <p className="text-sm text-gray-500">{lead.email} • {lead.phone}</p>
              </div>

              <div className="flex items-center gap-2">
                <span className={`px-3 py-1 rounded-full text-sm font-medium ${getScoreColor(lead.score)}`}>
                  Score: {lead.score || 0}
                </span>
                <span className={`px-2 py-1 rounded text-xs ${lead.status === 'new' ? 'bg-blue-100 text-blue-800' :
                    lead.status === 'contacted' ? 'bg-yellow-100 text-yellow-800' :
                      'bg-green-100 text-green-800'
                  }`}>
                  {lead.status}
                </span>
              </div>
            </div>

            {lead.message && (
              <p className="text-gray-700 mb-3">{lead.message}</p>
            )}

            {lead.budget && (
              <p className="text-sm text-gray-600 mb-3">Budget: {lead.budget}</p>
            )}

            <div className="flex justify-between items-center">
              <div className="flex gap-2">
                <button
                  onClick={() => handleContact(lead, 'call')}
                  className="p-2 bg-green-100 text-green-600 rounded-lg hover:bg-green-200"
                  title="Call"
                >
                  📞
                </button>
                <button
                  onClick={() => handleContact(lead, 'sms')}
                  className="p-2 bg-blue-100 text-blue-600 rounded-lg hover:bg-blue-200"
                  title="SMS"
                >
                  💬
                </button>
                <button
                  onClick={() => handleContact(lead, 'whatsapp')}
                  className="p-2 bg-green-100 text-green-600 rounded-lg hover:bg-green-200"
                  title="WhatsApp"
                >
                  🟢
                </button>
                <button
                  onClick={() => handleContact(lead, 'email')}
                  className="p-2 bg-gray-100 text-gray-600 rounded-lg hover:bg-gray-200"
                  title="Email"
                >
                  📨
                </button>
              </div>

              <span className="text-xs text-gray-500">
                {new Date(lead.created_at).toLocaleDateString()}
              </span>
            </div>
          </div>
        ))}
      </div>

      {sortedLeads.length === 0 && (
        <div className="text-center py-12">
          <p className="text-gray-500">No leads found</p>
        </div>
      )}
    </div>
  );
};