import React, { useState, useEffect } from 'react';
import { User, Phone, Mail, Calendar, CheckCircle, Building, IndianRupee, Loader2 } from 'lucide-react';
import { getSellerProfile, type SellerProfile } from '../api/sellerProfile';

const SellerProfileComponent: React.FC = () => {
  const [profile, setProfile] = useState<SellerProfile | null>(null);
  const [loading, setLoading] = useState(true);
  const [error, setError] = useState<string | null>(null);

  useEffect(() => {
    const fetchSellerData = async () => {
      try {
        setLoading(true);
        const profileData = await getSellerProfile();
        setProfile(profileData);
      } catch (err: any) {
        console.error('Error fetching seller data:', err);
        setError(err.message || 'Failed to load seller data');
      } finally {
        setLoading(false);
      }
    };

    fetchSellerData();
  }, []);

  if (loading) {
    return (
      <div className="flex items-center justify-center min-h-[400px]">
        <Loader2 className="w-8 h-8 animate-spin text-[#2B256D]" />
        <span className="ml-2 text-gray-600">Loading profile...</span>
      </div>
    );
  }

  if (error || !profile) {
    return (
      <div className="bg-red-50 border border-red-200 rounded-lg p-6">
        <h3 className="text-red-800 font-semibold mb-2">Error Loading Profile</h3>
        <p className="text-red-600">{error || 'Profile not found'}</p>
      </div>
    );
  }

  return (
    <div className="space-y-6">
      <div className="bg-white rounded-xl shadow-sm p-6">
        <h2 className="text-2xl font-bold text-gray-900 mb-6">Seller Profile</h2>
        
        <div className="flex flex-col lg:flex-row gap-8">
          {/* Profile Picture and Basic Info */}
          <div className="flex flex-col items-center lg:items-start space-y-4">
            <div className="w-32 h-32 rounded-full border-4 border-[#2B256D]/20 bg-[#2B256D] text-white flex items-center justify-center text-5xl font-bold">
              {profile.profile_image_url ? (
                <img 
                  src={profile.profile_image_url} 
                  alt={profile.full_name}
                  className="w-full h-full rounded-full object-cover"
                />
              ) : (
                profile.full_name?.charAt(0) || 'S'
              )}
            </div>
            <div className="text-center lg:text-left">
              <h3 className="text-xl font-semibold text-gray-900">{profile.full_name}</h3>
              <div className="flex items-center space-x-2 mt-2">
                <CheckCircle className={`w-5 h-5 ${profile.status === 'active' ? 'text-green-500' : 'text-yellow-500'}`} />
                <span className={`text-sm font-medium ${profile.status === 'active' ? 'text-green-600' : 'text-yellow-600'}`}>
                  {profile.status === 'active' ? 'Active' : profile.status}
                </span>
              </div>
            </div>
          </div>
          
          {/* Contact Information */}
          <div className="flex-1 space-y-6">
            <div>
              <h4 className="text-lg font-semibold text-gray-900 mb-4">Contact Information</h4>
              <div className="grid md:grid-cols-2 gap-4">
                <div className="flex items-center space-x-3 p-4 bg-gray-50 rounded-lg">
                  <Mail className="w-5 h-5 text-[#2B256D]" />
                  <div>
                    <p className="text-sm text-gray-500">Email</p>
                    <p className="font-medium text-gray-900">{profile.email}</p>
                  </div>
                </div>
                <div className="flex items-center space-x-3 p-4 bg-gray-50 rounded-lg">
                  <Phone className="w-5 h-5 text-green-600" />
                  <div>
                    <p className="text-sm text-gray-500">Phone</p>
                    <p className="font-medium text-gray-900">{profile.phone_number}</p>
                  </div>
                </div>
                <div className="flex items-center space-x-3 p-4 bg-gray-50 rounded-lg">
                  <Calendar className="w-5 h-5 text-purple-600" />
                  <div>
                    <p className="text-sm text-gray-500">Join Date</p>
                    <p className="font-medium text-gray-900">{new Date(profile.created_at).toLocaleDateString('en-US', { year: 'numeric', month: 'long', day: 'numeric' })}</p>
                  </div>
                </div>
                <div className="flex items-center space-x-3 p-4 bg-gray-50 rounded-lg">
                  <User className="w-5 h-5 text-orange-600" />
                  <div>
                    <p className="text-sm text-gray-500">Seller ID</p>
                    <p className="font-medium text-gray-900">#{profile.seller_id}</p>
                  </div>
                </div>
              </div>
            </div>
            
            {/* Property Information */}
            <div>
              <h4 className="text-lg font-semibold text-gray-900 mb-4">Property Portfolio</h4>
              <div className="grid md:grid-cols-2 gap-4">
                <div className="flex items-center space-x-3 p-4 bg-gray-50 rounded-lg">
                  <Building className="w-5 h-5 text-blue-600" />
                  <div>
                    <p className="text-sm text-gray-500">Properties Owned</p>
                    <p className="font-medium text-gray-900">{profile.property_count}</p>
                  </div>
                </div>
                {profile.years_of_experience > 0 && (
                  <div className="flex items-center space-x-3 p-4 bg-gray-50 rounded-lg">
                    <Calendar className="w-5 h-5 text-purple-600" />
                    <div>
                      <p className="text-sm text-gray-500">Experience</p>
                      <p className="font-medium text-gray-900">{profile.years_of_experience} years</p>
                    </div>
                  </div>
                )}
                <div className="flex items-center space-x-3 p-4 bg-gray-50 rounded-lg">
                  <Building className="w-5 h-5 text-green-600" />
                  <div>
                    <p className="text-sm text-gray-500">Properties Listed</p>
                    <p className="font-medium text-gray-900">{profile.total_properties_listed}</p>
                  </div>
                </div>
                <div className="flex items-center space-x-3 p-4 bg-gray-50 rounded-lg">
                  <IndianRupee className="w-5 h-5 text-green-600" />
                  <div>
                    <p className="text-sm text-gray-500">Properties Sold</p>
                    <p className="font-medium text-gray-900">{profile.total_properties_sold}</p>
                  </div>
                </div>
              </div>
            </div>
          </div>
        </div>
      </div>
      
      {/* Property Types & Preferences */}
      {(profile.property_types?.length > 0 || profile.ownership_type || profile.selling_method) && (
        <div className="bg-white rounded-xl shadow-sm p-6">
          <h3 className="text-xl font-semibold text-gray-900 mb-4">Property Details</h3>
          <div className="space-y-4">
            {profile.property_types?.length > 0 && (
              <div className="p-4 bg-gray-50 rounded-lg">
                <p className="text-sm text-gray-500 mb-2">Property Types</p>
                <div className="flex flex-wrap gap-2">
                  {profile.property_types.map((type, index) => (
                    <span key={index} className="px-3 py-1 bg-[#2B256D] text-white text-sm rounded-full">
                      {type.charAt(0).toUpperCase() + type.slice(1)}
                    </span>
                  ))}
                </div>
              </div>
            )}
            {profile.ownership_type && (
              <div className="p-4 bg-gray-50 rounded-lg">
                <p className="text-sm text-gray-500 mb-1">Ownership Type</p>
                <p className="font-medium text-gray-900">{profile.ownership_type.charAt(0).toUpperCase() + profile.ownership_type.slice(1)}</p>
              </div>
            )}
            {profile.selling_method && (
              <div className="p-4 bg-gray-50 rounded-lg">
                <p className="text-sm text-gray-500 mb-1">Preferred Selling Method</p>
                <p className="font-medium text-gray-900">{profile.selling_method.charAt(0).toUpperCase() + profile.selling_method.slice(1)}</p>
              </div>
            )}
          </div>
        </div>
      )}
      
      {/* Bio Section */}
      {profile.bio && (
        <div className="bg-white rounded-xl shadow-sm p-6">
          <h3 className="text-xl font-semibold text-gray-900 mb-4">About</h3>
          <p className="text-gray-700 leading-relaxed">{profile.bio}</p>
        </div>
      )}
      
      {/* Communication Preferences */}
      {(profile.preferred_communication?.length > 0 || profile.availability_hours) && (
        <div className="bg-white rounded-xl shadow-sm p-6">
          <h3 className="text-xl font-semibold text-gray-900 mb-4">Communication Preferences</h3>
          <div className="space-y-4">
            {profile.preferred_communication?.length > 0 && (
              <div className="p-4 bg-gray-50 rounded-lg">
                <p className="text-sm text-gray-500 mb-2">Preferred Communication Methods</p>
                <div className="flex flex-wrap gap-2">
                  {profile.preferred_communication.map((method, index) => (
                    <span key={index} className="px-3 py-1 bg-blue-100 text-blue-800 text-sm rounded-full">
                      {method.replace('_', ' ')}
                    </span>
                  ))}
                </div>
              </div>
            )}
            {profile.availability_hours && (
              <div className="p-4 bg-gray-50 rounded-lg">
                <p className="text-sm text-gray-500 mb-1">Availability Hours</p>
                <p className="font-medium text-gray-900">{profile.availability_hours}</p>
              </div>
            )}
          </div>
        </div>
      )}
    </div>
  );
};

export default SellerProfileComponent;