import React, { useState, useEffect } from 'react';
import { useNavigate } from 'react-router-dom';
import { User, Building, Target, Shield, Check } from 'lucide-react';
import { completeSellerProfile, getSellerProfile, getSellerProfileCompletionStatus, type SellerProfileCreate } from '../api/sellerProfile';
 
interface SellerProfileData extends SellerProfileCreate {
  property_types: string[];
  preferred_communication: string[];
}
 
const SellerProfileCompletion: React.FC = () => {
  const navigate = useNavigate();
  const [currentStep, setCurrentStep] = useState(1);
  const [loading, setLoading] = useState(false);
  const [error, setError] = useState('');
  const [success, setSuccess] = useState('');
 
  const [profileData, setProfileData] = useState<SellerProfileData>({
    full_name: '',
    email: '',
    phone_number: '+91',
    date_of_birth: '',
    gender: '',
    ownership_type: '',
    property_count: 0,
    property_types: [],
    years_of_experience: 0,
    total_portfolio_value_range: '',
    selling_method: '',
    selling_timeline: '',
    price_flexibility: true,
    preferred_communication: [],
    availability_hours: '',
    bio: '',
    profile_image_url: '',
    portfolio_summary: '',
    marketing_consent: false
  });
 
  const ownershipOptions = ['individual', 'joint', 'company', 'trust'];
  const propertyTypeOptions = ['residential', 'commercial', 'land', 'industrial'];
  const portfolioValueOptions = ['under_50L', '50L_1Cr', '1Cr_5Cr', '5Cr_10Cr', 'above_10Cr'];
  const sellingMethodOptions = ['direct', 'agent', 'auction', 'flexible'];
  const timelineOptions = ['immediate', '3_months', '6_months', '1_year', 'flexible'];
  const communicationOptions = ['phone', 'email', 'whatsapp', 'video_call'];
  const genderOptions = ['male', 'female', 'other'];
 
  useEffect(() => {
    loadExistingProfile();
    // Auto-fetch seller email if available
    const sellerEmail = localStorage.getItem('sellerEmail') || localStorage.getItem('login_email');
    if (sellerEmail && !profileData.email) {
      setProfileData(prev => ({ ...prev, email: sellerEmail }));
    }
    
    // Check if profile was just completed to prevent redirect loop
    const profileJustCompleted = sessionStorage.getItem('profileJustCompleted');
    if (profileJustCompleted) {
      sessionStorage.removeItem('profileJustCompleted');
      return;
    }
    
    // Check if this is an existing user with completed profile
    const sellerProfileCompleted = localStorage.getItem('sellerProfileCompleted');
    if (sellerProfileCompleted === 'true') {
      navigate('/seller/dashboard');
      return;
    }
  }, []);
 
  const loadExistingProfile = async () => {
    try {
      const [profile, status] = await Promise.all([
        getSellerProfile().catch(() => null),
        getSellerProfileCompletionStatus().catch(() => null)
      ]);
 
      if (profile) {
        const sellerEmail = localStorage.getItem('sellerEmail') || localStorage.getItem('login_email');
        setProfileData({
          full_name: profile.full_name || '',
          email: profile.email || sellerEmail || '',
          phone_number: profile.phone_number || '+91',
          date_of_birth: profile.date_of_birth || '',
          gender: profile.gender || '',
          ownership_type: profile.ownership_type || '',
          property_count: profile.property_count || 0,
          property_types: profile.property_types || [],
          years_of_experience: profile.years_of_experience || 0,
          total_portfolio_value_range: profile.total_portfolio_value_range || '',
          selling_method: profile.selling_method || '',
          selling_timeline: profile.selling_timeline || '',
          price_flexibility: profile.price_flexibility ?? true,
          preferred_communication: profile.preferred_communication || [],
          availability_hours: profile.availability_hours || '',
          bio: profile.bio || '',
          profile_image_url: profile.profile_image_url || '',
          portfolio_summary: profile.portfolio_summary || '',
          marketing_consent: profile.marketing_consent || false
        });
      } else {
        // If no existing profile, still set the email from localStorage
        const sellerEmail = localStorage.getItem('sellerEmail') || localStorage.getItem('login_email');
        if (sellerEmail) {
          setProfileData(prev => ({ ...prev, email: sellerEmail }));
        }
      }
 
      // Only redirect if profile is truly complete (100%) and not just completed
      if (status && status.completion_percentage >= 100 && !sessionStorage.getItem('profileJustCompleted')) {
        localStorage.setItem('sellerProfileCompleted', 'true');
        navigate('/seller/dashboard');
      }
    } catch (err) {
      console.error('Error loading profile:', err);
    }
  };
 
  const handleInputChange = (field: string, value: any) => {
    setProfileData(prev => ({ ...prev, [field]: value }));
    setError('');
  };
 
  const handleArrayToggle = (field: 'property_types' | 'preferred_communication', value: string) => {
    setProfileData(prev => ({
      ...prev,
      [field]: prev[field].includes(value)
        ? prev[field].filter(item => item !== value)
        : [...prev[field], value]
    }));
  };
 
  const validateStep = (step: number): boolean => {
    switch (step) {
      case 1:
        return !!(profileData.full_name && profileData.email && profileData.phone_number && profileData.gender);
      case 2:
        return !!(profileData.ownership_type && profileData.property_count !== undefined && profileData.property_types.length > 0);
      case 3:
        return !!(profileData.selling_method && profileData.selling_timeline);
      default:
        return true;
    }
  };
 
  const handleNext = () => {
    if (validateStep(currentStep)) {
      setCurrentStep(prev => prev + 1);
    } else {
      setError('Please fill all required fields');
    }
  };
 
  const handleSubmit = async () => {
    setLoading(true);
    setError('');
 
    try {
      const payload = { profile_data: profileData };
      await completeSellerProfile(payload);
      setSuccess('Profile completed successfully!');
      localStorage.setItem('sellerProfileCompleted', 'true');
      // Set a flag to prevent redirect loop
      sessionStorage.setItem('profileJustCompleted', 'true');
      // Clear any existing profile completion checks
      localStorage.removeItem('profileCompletionRequired');
      setTimeout(() => navigate('/seller/dashboard'), 1500);
    } catch (err: any) {
      setError(err.message || 'Failed to complete profile');
    } finally {
      setLoading(false);
    }
  };
 
  const steps = [
    { number: 1, title: 'Personal Info', icon: User },
    { number: 2, title: 'Property Details', icon: Building },
    { number: 3, title: 'Selling Preferences', icon: Target },
    { number: 4, title: 'Profile & Verification', icon: Shield }
  ];
 
  return (
    <div className="min-h-screen bg-gray-50 py-8">
      <div className="max-w-4xl mx-auto px-4">
        <div className="bg-white rounded-2xl shadow-xl overflow-hidden">
          {/* Header */}
          <div className="px-8 py-6 border-b border-gray-200">
            <h1 className="text-2xl font-bold text-gray-900">Complete Your Seller Profile</h1>
            <p className="text-gray-600 mt-1">Build your professional presence</p>
           
            {/* Progress */}
            <div className="flex items-center justify-between mt-6">
              {steps.map((step, index) => (
                <React.Fragment key={step.number}>
                  <div className="flex items-center">
                    <div className={`w-10 h-10 rounded-full flex items-center justify-center ${
                      currentStep >= step.number ? 'bg-blue-600 text-white' : 'bg-gray-200 text-gray-500'
                    }`}>
                      <step.icon className="w-5 h-5" />
                    </div>
                    <span className={`ml-2 text-sm font-medium ${
                      currentStep >= step.number ? 'text-blue-600' : 'text-gray-500'
                    }`}>
                      {step.title}
                    </span>
                  </div>
                  {index < steps.length - 1 && (
                    <div className={`flex-1 h-1 mx-4 ${
                      currentStep > step.number ? 'bg-blue-600' : 'bg-gray-200'
                    }`} />
                  )}
                </React.Fragment>
              ))}
            </div>
          </div>
 
          {/* Content */}
          <div className="p-8">
            {error && (
              <div className="mb-6 bg-red-50 border border-red-200 text-red-700 px-4 py-3 rounded-lg">
                {error}
              </div>
            )}
 
            {success && (
              <div className="mb-6 bg-green-50 border border-green-200 text-green-700 px-4 py-3 rounded-lg flex items-center">
                <Check className="w-4 h-4 mr-2" />
                {success}
              </div>
            )}
 
            {/* Step 1: Personal Info */}
            {currentStep === 1 && (
              <div className="space-y-6">
                <div className="grid grid-cols-2 gap-6">
                  <div>
                    <label className="block text-sm font-medium text-gray-700 mb-2">Full Name *</label>
                    <input
                      type="text"
                      value={profileData.full_name}
                      onChange={(e) => handleInputChange('full_name', e.target.value)}
                      className="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500"
                      required
                    />
                  </div>
                  <div>
                    <label className="block text-sm font-medium text-gray-700 mb-2">Email *</label>
                    <input
                      type="email"
                      value={profileData.email}
                      onChange={(e) => handleInputChange('email', e.target.value)}
                      className="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500"
                      required
                    />
                  </div>
                </div>
 
                <div className="grid grid-cols-2 gap-6">
                  <div>
                    <label className="block text-sm font-medium text-gray-700 mb-2">Phone Number *</label>
                    <input
                      type="tel"
                      value={profileData.phone_number}
                      onChange={(e) => handleInputChange('phone_number', e.target.value)}
                      className="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500"
                      required
                    />
                  </div>
                  <div>
                    <label className="block text-sm font-medium text-gray-700 mb-2">Date of Birth</label>
                    <input
                      type="date"
                      value={profileData.date_of_birth}
                      onChange={(e) => handleInputChange('date_of_birth', e.target.value)}
                      className="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500"
                    />
                  </div>
                </div>
 
                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-2">Gender *</label>
                  <select
                    value={profileData.gender}
                    onChange={(e) => handleInputChange('gender', e.target.value)}
                    className="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500"
                  >
                    <option value="">Select gender</option>
                    {genderOptions.map(option => (
                      <option key={option} value={option}>{option.charAt(0).toUpperCase() + option.slice(1)}</option>
                    ))}
                  </select>
                </div>
              </div>
            )}
 
            {/* Step 2: Property Details */}
            {currentStep === 2 && (
              <div className="space-y-6">
                <div className="grid grid-cols-2 gap-6">
                  <div>
                    <label className="block text-sm font-medium text-gray-700 mb-2">Ownership Type *</label>
                    <select
                      value={profileData.ownership_type}
                      onChange={(e) => handleInputChange('ownership_type', e.target.value)}
                      className="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500"
                      required
                    >
                      <option value="">Select ownership type</option>
                      {ownershipOptions.map(option => (
                        <option key={option} value={option}>{option.charAt(0).toUpperCase() + option.slice(1)}</option>
                      ))}
                    </select>
                  </div>
                  <div>
                    <label className="block text-sm font-medium text-gray-700 mb-2">Number of Properties *</label>
                    <input
                      type="number"
                      min="0"
                      value={profileData.property_count}
                      onChange={(e) => handleInputChange('property_count', parseInt(e.target.value) || 0)}
                      className="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500"
                      required
                    />
                  </div>
                </div>
 
                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-2">Property Types *</label>
                  <div className="grid grid-cols-2 gap-3">
                    {propertyTypeOptions.map(type => (
                      <label key={type} className="flex items-center space-x-2 cursor-pointer">
                        <input
                          type="checkbox"
                          checked={profileData.property_types.includes(type)}
                          onChange={() => handleArrayToggle('property_types', type)}
                          className="rounded border-gray-300 text-blue-600 focus:ring-blue-500"
                        />
                        <span className="text-sm text-gray-700">{type.charAt(0).toUpperCase() + type.slice(1)}</span>
                      </label>
                    ))}
                  </div>
                </div>
 
                <div className="grid grid-cols-2 gap-6">
                  <div>
                    <label className="block text-sm font-medium text-gray-700 mb-2">Years of Experience</label>
                    <input
                      type="number"
                      min="0"
                      value={profileData.years_of_experience}
                      onChange={(e) => handleInputChange('years_of_experience', parseInt(e.target.value) || 0)}
                      className="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500"
                    />
                  </div>
                  <div>
                    <label className="block text-sm font-medium text-gray-700 mb-2">Portfolio Value Range</label>
                    <select
                      value={profileData.total_portfolio_value_range}
                      onChange={(e) => handleInputChange('total_portfolio_value_range', e.target.value)}
                      className="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500"
                    >
                      <option value="">Select range</option>
                      {portfolioValueOptions.map(option => (
                        <option key={option} value={option}>{option.replace('_', ' ')}</option>
                      ))}
                    </select>
                  </div>
                </div>
              </div>
            )}
 
            {/* Step 3: Selling Preferences */}
            {currentStep === 3 && (
              <div className="space-y-6">
                <div className="grid grid-cols-2 gap-6">
                  <div>
                    <label className="block text-sm font-medium text-gray-700 mb-2">Selling Method *</label>
                    <select
                      value={profileData.selling_method}
                      onChange={(e) => handleInputChange('selling_method', e.target.value)}
                      className="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500"
                      required
                    >
                      <option value="">Select method</option>
                      {sellingMethodOptions.map(method => (
                        <option key={method} value={method}>{method.charAt(0).toUpperCase() + method.slice(1)}</option>
                      ))}
                    </select>
                  </div>
                  <div>
                    <label className="block text-sm font-medium text-gray-700 mb-2">Selling Timeline *</label>
                    <select
                      value={profileData.selling_timeline}
                      onChange={(e) => handleInputChange('selling_timeline', e.target.value)}
                      className="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500"
                      required
                    >
                      <option value="">Select timeline</option>
                      {timelineOptions.map(timeline => (
                        <option key={timeline} value={timeline}>{timeline.replace('_', ' ')}</option>
                      ))}
                    </select>
                  </div>
                </div>
 
                <div>
                  <label className="flex items-center space-x-2 cursor-pointer">
                    <input
                      type="checkbox"
                      checked={profileData.price_flexibility}
                      onChange={(e) => handleInputChange('price_flexibility', e.target.checked)}
                      className="rounded border-gray-300 text-blue-600 focus:ring-blue-500"
                    />
                    <span className="text-sm text-gray-700">Price Flexibility</span>
                  </label>
                </div>
 
                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-2">Preferred Communication</label>
                  <div className="grid grid-cols-2 gap-3">
                    {communicationOptions.map(method => (
                      <label key={method} className="flex items-center space-x-2 cursor-pointer">
                        <input
                          type="checkbox"
                          checked={profileData.preferred_communication.includes(method)}
                          onChange={() => handleArrayToggle('preferred_communication', method)}
                          className="rounded border-gray-300 text-blue-600 focus:ring-blue-500"
                        />
                        <span className="text-sm text-gray-700">{method.replace('_', ' ')}</span>
                      </label>
                    ))}
                  </div>
                </div>
 
                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-2">Availability Hours</label>
                  <input
                    type="text"
                    value={profileData.availability_hours}
                    onChange={(e) => handleInputChange('availability_hours', e.target.value)}
                    className="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500"
                    placeholder="Mon-Fri 9AM-6PM"
                  />
                </div>
              </div>
            )}
 
            {/* Step 4: Profile & Verification */}
            {currentStep === 4 && (
              <div className="space-y-6">
                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-2">Bio</label>
                  <textarea
                    value={profileData.bio}
                    onChange={(e) => handleInputChange('bio', e.target.value)}
                    rows={4}
                    className="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500"
                    placeholder="Tell buyers about yourself and your properties..."
                  />
                </div>
 
                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-2">Profile Image URL</label>
                  <input
                    type="url"
                    value={profileData.profile_image_url}
                    onChange={(e) => handleInputChange('profile_image_url', e.target.value)}
                    className="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500"
                    placeholder="https://example.com/your-photo.jpg"
                  />
                </div>
 
                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-2">Portfolio Summary</label>
                  <textarea
                    value={profileData.portfolio_summary}
                    onChange={(e) => handleInputChange('portfolio_summary', e.target.value)}
                    rows={3}
                    className="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500"
                    placeholder="Brief summary of your property portfolio..."
                  />
                </div>
 
                <div>
                  <label className="flex items-center space-x-2 cursor-pointer">
                    <input
                      type="checkbox"
                      checked={profileData.marketing_consent}
                      onChange={(e) => handleInputChange('marketing_consent', e.target.checked)}
                      className="rounded border-gray-300 text-blue-600 focus:ring-blue-500"
                    />
                    <span className="text-sm text-gray-700">I consent to marketing communications</span>
                  </label>
                </div>
              </div>
            )}
          </div>
 
          {/* Footer */}
          <div className="px-8 py-6 border-t border-gray-200 bg-gray-50">
            <div className="flex justify-between">
              <button
                onClick={() => setCurrentStep(prev => prev - 1)}
                disabled={currentStep === 1}
                className="px-6 py-2 border border-gray-300 text-gray-700 rounded-lg hover:bg-gray-50 disabled:opacity-50 disabled:cursor-not-allowed"
              >
                Back
              </button>
             
              {currentStep < 4 ? (
                <button
                  onClick={handleNext}
                  className="px-6 py-2 bg-blue-600 !bg-blue-600 text-black rounded-lg hover:bg-blue-700"
                >
                  Next
                </button>
              ) : (
                <button
                  onClick={handleSubmit}
                  disabled={loading}
                  className="px-6 py-2 bg-blue-600 !bg-blue-600 text-black rounded-lg hover:bg-blue-700 disabled:opacity-50 flex items-center"
                >
                  {loading && <div className="w-4 h-4 border-2 border-black border-t-transparent rounded-full animate-spin mr-2" />}
                  {loading ? 'Completing...' : 'Complete Profile'}
                </button>
              )}
            </div>
          </div>
        </div>
      </div>
    </div>
  );
};
 
export default SellerProfileCompletion;