import { useState, useEffect } from "react";
import { useNavigate } from "react-router-dom";
import { toast } from "sonner";
import '../styles/animated-auth.css';
import { CombinedCaptcha } from './CombinedCaptcha';

export function SellerRegister() {
  const [step, setStep] = useState<"email" | "otp">("email");
  const [email, setEmail] = useState("");
  const [otp, setOtp] = useState("");
  const [password, setPassword] = useState("");
  const [confirmPassword, setConfirmPassword] = useState("");
  const [captchaValid, setCaptchaValid] = useState(false);
  const [loading, setLoading] = useState(false);
  const [toastMessage, setToastMessage] = useState<{type: 'success' | 'error', message: string} | null>(null);
  const navigate = useNavigate();

  useEffect(() => {
    if (toastMessage) {
      if (toastMessage.type === 'success') {
        toast.success(toastMessage.message);
      } else {
        toast.error(toastMessage.message);
      }
      setToastMessage(null);
    }
  }, [toastMessage]);

  const handleSendOTP = async (e: React.FormEvent) => {
    e.preventDefault();
    setLoading(true);
    
    try {
      const response = await fetch(`${import.meta.env.VITE_API_BASE || 'http://localhost:8090/api'}/seller-agent-auth/send-otp`, {
        method: 'POST',
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify({ email, user_type: 'seller' })
      });
      
      const data = await response.json();
      
      if (response.ok && data.success) {
        setStep("otp");
        setToastMessage({type: 'success', message: "OTP sent to your email!"});
      } else {
        setToastMessage({type: 'error', message: data.detail || "Failed to send OTP"});
      }
    } catch (error) {
      setToastMessage({type: 'error', message: "Failed to send OTP. Please try again."});
    } finally {
      setLoading(false);
    }
  };

  const handleVerifyOTP = async (e: React.FormEvent) => {
    e.preventDefault();
    
    if (!captchaValid) {
      setToastMessage({type: 'error', message: "Please complete the captcha verification"});
      return;
    }
    
    if (password !== confirmPassword) {
      setToastMessage({type: 'error', message: "Passwords do not match"});
      return;
    }
    
    if (password.length < 6) {
      setToastMessage({type: 'error', message: "Password must be at least 6 characters"});
      return;
    }
    
    setLoading(true);
    
    try {
      const response = await fetch(`${import.meta.env.VITE_API_BASE || 'http://localhost:8090/api'}/seller-agent-auth/verify-otp`, {
        method: 'POST',
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify({ email, otp_code: otp, password })
      });
      
      const data = await response.json();
      
      if (response.ok && data.success) {
        localStorage.setItem("sellerToken", data.access_token);
        localStorage.setItem("sellerEmail", data.email);
        localStorage.setItem("sellerId", data.user_id);
        setToastMessage({type: 'success', message: "Registration successful!"});
        navigate("/seller/dashboard");
      } else {
        setToastMessage({type: 'error', message: data.detail || "Registration failed"});
      }
    } catch (error) {
      setToastMessage({type: 'error', message: "Registration failed. Please try again."});
    } finally {
      setLoading(false);
    }
  };

  return (
    <div className="seller-auth-container">
      <div className="curved-shape"></div>
      <div className="curved-shape2"></div>
      
      <div className="container">
        <div className="form-box Register">
          <h2 className="animation" style={{"--li": 17, "--S": 0} as React.CSSProperties}>Register</h2>
          
          {step === "email" ? (
            <form onSubmit={handleSendOTP}>
              <div className="input-box animation" style={{"--li": 18, "--S": 1} as React.CSSProperties}>
                <input 
                  type="email" 
                  value={email}
                  onChange={(e) => setEmail(e.target.value)}
                  required 
                />
                <label>Email</label>
                <i className="input-icon">📧</i>
              </div>

              <div className="input-box animation" style={{"--li": 20, "--S": 4} as React.CSSProperties}>
                <button className="btn" type="submit" disabled={loading}>
                  {loading ? "Sending OTP..." : "Send OTP"}
                </button>
              </div>

              <div className="regi-link animation" style={{"--li": 21, "--S": 5} as React.CSSProperties}>
                <p>Already have an account? <br /> 
                  <a href="/seller">Sign In</a>
                </p>
              </div>
            </form>
          ) : (
            <form onSubmit={handleVerifyOTP}>
              <div className="input-box animation" style={{"--li": 18, "--S": 1} as React.CSSProperties}>
                <input 
                  type="text" 
                  value={otp}
                  onChange={(e) => setOtp(e.target.value)}
                  maxLength={6}
                  required 
                />
                <label>Enter 6-digit OTP</label>
                <i className="input-icon">🔢</i>
              </div>

              <div className="input-box animation" style={{"--li": 19, "--S": 2} as React.CSSProperties}>
                <input 
                  type="password" 
                  value={password}
                  onChange={(e) => setPassword(e.target.value)}
                  required 
                />
                <label>Password (min 6 chars)</label>
                <i className="input-icon">🔒</i>
              </div>

              <div className="input-box animation" style={{"--li": 19, "--S": 3} as React.CSSProperties}>
                <input 
                  type="password" 
                  value={confirmPassword}
                  onChange={(e) => setConfirmPassword(e.target.value)}
                  required 
                />
                <label>Confirm Password</label>
                <i className="input-icon">🔒</i>
              </div>

              <CombinedCaptcha 
                onVerify={setCaptchaValid}
              />

              <div className="input-box animation" style={{"--li": 20, "--S": 4} as React.CSSProperties}>
                <button className="btn" type="submit" disabled={loading}>
                  {loading ? "Verifying..." : "Complete Registration"}
                </button>
              </div>

              <div className="regi-link animation" style={{"--li": 21, "--S": 5} as React.CSSProperties}>
                <p>
                  <a href="#" onClick={(e) => {e.preventDefault(); setStep("email");}}>Back</a> | 
                  <a href="/seller">Sign In</a>
                </p>
              </div>
            </form>
          )}
        </div>

        <div className="info-content Register">
          <h2 className="animation" style={{"--li": 17, "--S": 0} as React.CSSProperties}>WELCOME!</h2>
          <p className="animation" style={{"--li": 18, "--S": 1} as React.CSSProperties}>
            We're delighted to have you here. If you need any assistance, feel free to reach out.
          </p>
        </div>
      </div>
    </div>
  );
}