import React, { useState, useEffect } from 'react';
import { MapPin, Layers, Filter, Download, Eye, Info, TrendingUp, Home, Building } from 'lucide-react';
import { Card, CardContent, CardHeader, CardTitle } from './ui/card';
import { Button } from './ui/button';
import { Badge } from './ui/badge';
 
interface HeatmapPoint {
  id: number;
  locality: string;
  city: string;
  latitude: number;
  longitude: number;
  price_per_sqft?: number;
  demand_score?: number;
  rental_yield?: number;
  urgency_score?: number;
  total_listings: number;
  price_trend?: string;
}
 
interface ZoneData {
  id: number;
  zone_name: string;
  city: string;
  center_lat: number;
  center_lng: number;
  total_properties: number;
  avg_price_per_sqft?: number;
  demand_index?: number;
  rental_yield_avg?: number;
  metro_distance?: number;
}
 
export function SmartLandMaps() {
  const [selectedOverlay, setSelectedOverlay] = useState<string>('price');
  const [selectedCity, setSelectedCity] = useState<string>('Bangalore');
  const [heatmapData, setHeatmapData] = useState<HeatmapPoint[]>([]);
  const [zoneData, setZoneData] = useState<ZoneData[]>([]);
  const [selectedZone, setSelectedZone] = useState<ZoneData | null>(null);
  const [loading, setLoading] = useState(false);
 
  const overlayTypes = [
    { id: 'price', name: 'Price per Sq.Ft', color: 'bg-red-500', icon: '₹' },
    { id: 'demand', name: 'Demand Heatmap', color: 'bg-orange-500', icon: '🔥' },
    { id: 'rental', name: 'Rental Yield', color: 'bg-green-500', icon: '📈' },
    { id: 'urgency', name: 'Urgent Sales', color: 'bg-purple-500', icon: '⚡' }
  ];
 
  const cities = ['Bangalore', 'Mumbai', 'Delhi', 'Chennai', 'Hyderabad', 'Pune'];
 
  useEffect(() => {
    loadHeatmapData();
    loadZoneData();
  }, [selectedCity, selectedOverlay]);
 
  const loadHeatmapData = async () => {
    setLoading(true);
    try {
      const response = await fetch(`/api/smart-maps/heatmap?city=${selectedCity}&overlay_type=${selectedOverlay}`);
      if (response.ok) {
        const data = await response.json();
        setHeatmapData(data);
      }
    } catch (error) {
      console.error('Error loading heatmap data:', error);
      // Fallback to mock data
      setHeatmapData(getMockHeatmapData());
    } finally {
      setLoading(false);
    }
  };
 
  const loadZoneData = async () => {
    try {
      const response = await fetch(`/api/smart-maps/zones?city=${selectedCity}&zone_type=locality`);
      if (response.ok) {
        const data = await response.json();
        setZoneData(data);
      }
    } catch (error) {
      console.error('Error loading zone data:', error);
      setZoneData(getMockZoneData());
    }
  };
 
  const getMockHeatmapData = (): HeatmapPoint[] => {
    return [
      {
        id: 1, locality: 'Whitefield', city: 'Bangalore', latitude: 12.9698, longitude: 77.7500,
        price_per_sqft: 8500, demand_score: 85, rental_yield: 3.2, urgency_score: 25,
        total_listings: 450, price_trend: 'up'
      },
      {
        id: 2, locality: 'Koramangala', city: 'Bangalore', latitude: 12.9279, longitude: 77.6271,
        price_per_sqft: 12000, demand_score: 92, rental_yield: 2.8, urgency_score: 15,
        total_listings: 320, price_trend: 'up'
      },
      {
        id: 3, locality: 'Electronic City', city: 'Bangalore', latitude: 12.8456, longitude: 77.6603,
        price_per_sqft: 6500, demand_score: 78, rental_yield: 3.8, urgency_score: 35,
        total_listings: 280, price_trend: 'stable'
      }
    ];
  };
 
  const getMockZoneData = (): ZoneData[] => {
    return [
      {
        id: 1, zone_name: 'Whitefield', city: 'Bangalore', center_lat: 12.9698, center_lng: 77.7500,
        total_properties: 450, avg_price_per_sqft: 8500, demand_index: 85, rental_yield_avg: 3.2, metro_distance: 2.5
      },
      {
        id: 2, zone_name: 'Koramangala', city: 'Bangalore', center_lat: 12.9279, center_lng: 77.6271,
        total_properties: 320, avg_price_per_sqft: 12000, demand_index: 92, rental_yield_avg: 2.8, metro_distance: 1.2
      }
    ];
  };
 
  const getOverlayValue = (point: HeatmapPoint) => {
    switch (selectedOverlay) {
      case 'price': return point.price_per_sqft ? `₹${point.price_per_sqft.toLocaleString()}` : 'N/A';
      case 'demand': return point.demand_score ? `${point.demand_score}/100` : 'N/A';
      case 'rental': return point.rental_yield ? `${point.rental_yield}%` : 'N/A';
      case 'urgency': return point.urgency_score ? `${point.urgency_score}%` : 'N/A';
      default: return 'N/A';
    }
  };
 
  const getIntensityColor = (point: HeatmapPoint) => {
    let value = 0;
    switch (selectedOverlay) {
      case 'price': value = (point.price_per_sqft || 0) / 15000; break;
      case 'demand': value = (point.demand_score || 0) / 100; break;
      case 'rental': value = (point.rental_yield || 0) / 5; break;
      case 'urgency': value = (point.urgency_score || 0) / 100; break;
    }
   
    const intensity = Math.min(value, 1);
    if (intensity > 0.8) return 'bg-red-600';
    if (intensity > 0.6) return 'bg-orange-500';
    if (intensity > 0.4) return 'bg-yellow-500';
    if (intensity > 0.2) return 'bg-green-500';
    return 'bg-blue-500';
  };
 
  return (
    <div className="min-h-screen bg-gray-50">
      {/* Header */}
      <div className="bg-white shadow-sm border-b">
        <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-4">
          <div className="flex items-center justify-between">
            <div>
              <h1 className="text-2xl font-bold text-gray-900">Smart Land Maps</h1>
              <p className="text-gray-600">Interactive heatmaps with GIS intelligence</p>
            </div>
            <div className="flex items-center gap-4">
              <select
                value={selectedCity}
                onChange={(e) => setSelectedCity(e.target.value)}
                className="px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500"
              >
                {cities.map(city => (
                  <option key={city} value={city}>{city}</option>
                ))}
              </select>
              <Button variant="outline" className="flex items-center gap-2">
                <Download className="w-4 h-4" />
                Export Map
              </Button>
            </div>
          </div>
        </div>
      </div>
 
      <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-6">
        <div className="grid grid-cols-1 lg:grid-cols-4 gap-6">
         
          {/* Controls Panel */}
          <div className="lg:col-span-1 space-y-6">
           
            {/* Overlay Selection */}
            <Card>
              <CardHeader>
                <CardTitle className="flex items-center gap-2">
                  <Layers className="w-5 h-5" />
                  Map Overlays
                </CardTitle>
              </CardHeader>
              <CardContent className="space-y-3">
                {overlayTypes.map(overlay => (
                  <div
                    key={overlay.id}
                    onClick={() => setSelectedOverlay(overlay.id)}
                    className={`p-3 rounded-lg cursor-pointer transition-all ${
                      selectedOverlay === overlay.id
                        ? 'bg-blue-50 border-2 border-blue-500'
                        : 'bg-gray-50 border border-gray-200 hover:bg-gray-100'
                    }`}
                  >
                    <div className="flex items-center gap-3">
                      <div className={`w-4 h-4 rounded ${overlay.color}`}></div>
                      <div>
                        <div className="font-medium text-sm">{overlay.name}</div>
                        <div className="text-xs text-gray-500">{overlay.icon}</div>
                      </div>
                    </div>
                  </div>
                ))}
              </CardContent>
            </Card>
 
            {/* Zone List */}
            <Card>
              <CardHeader>
                <CardTitle className="flex items-center gap-2">
                  <MapPin className="w-5 h-5" />
                  Top Zones
                </CardTitle>
              </CardHeader>
              <CardContent className="space-y-3">
                {zoneData.slice(0, 5).map(zone => (
                  <div
                    key={zone.id}
                    onClick={() => setSelectedZone(zone)}
                    className={`p-3 rounded-lg cursor-pointer transition-all ${
                      selectedZone?.id === zone.id
                        ? 'bg-blue-50 border border-blue-500'
                        : 'bg-gray-50 hover:bg-gray-100'
                    }`}
                  >
                    <div className="font-medium text-sm">{zone.zone_name}</div>
                    <div className="text-xs text-gray-500 mt-1">
                      {zone.total_properties} properties
                    </div>
                    {zone.avg_price_per_sqft && (
                      <div className="text-xs text-blue-600 font-medium">
                        ₹{zone.avg_price_per_sqft.toLocaleString()}/sq.ft
                      </div>
                    )}
                  </div>
                ))}
              </CardContent>
            </Card>
 
          </div>
 
          {/* Map Area */}
          <div className="lg:col-span-2">
            <Card className="h-[600px]">
              <CardHeader>
                <CardTitle className="flex items-center justify-between">
                  <span>Interactive Heatmap - {selectedCity}</span>
                  <Badge variant="outline">
                    {overlayTypes.find(o => o.id === selectedOverlay)?.name}
                  </Badge>
                </CardTitle>
              </CardHeader>
              <CardContent className="h-full p-0">
                <div className="relative h-full bg-gradient-to-br from-blue-50 to-green-50 rounded-lg overflow-hidden">
                 
                  {/* Mock Map Background */}
                  <div className="absolute inset-0 bg-gray-100">
                    <div className="w-full h-full relative">
                     
                      {/* Heatmap Points */}
                      {heatmapData.map(point => (
                        <div
                          key={point.id}
                          className="absolute transform -translate-x-1/2 -translate-y-1/2 cursor-pointer group"
                          style={{
                            left: `${((point.longitude - 77.4) / 0.6) * 100}%`,
                            top: `${((13.2 - point.latitude) / 0.6) * 100}%`
                          }}
                        >
                          <div className={`w-8 h-8 rounded-full ${getIntensityColor(point)} opacity-70 group-hover:opacity-100 transition-all group-hover:scale-125`}>
                          </div>
                         
                          {/* Tooltip */}
                          <div className="absolute bottom-full left-1/2 transform -translate-x-1/2 mb-2 opacity-0 group-hover:opacity-100 transition-opacity">
                            <div className="bg-black text-white text-xs rounded px-2 py-1 whitespace-nowrap">
                              <div className="font-medium">{point.locality}</div>
                              <div>{getOverlayValue(point)}</div>
                              <div>{point.total_listings} listings</div>
                            </div>
                          </div>
                        </div>
                      ))}
 
                      {/* Map Labels */}
                      <div className="absolute top-4 left-4 bg-white/90 rounded-lg p-3">
                        <div className="text-sm font-medium text-gray-900">{selectedCity} Heatmap</div>
                        <div className="text-xs text-gray-600 mt-1">
                          {overlayTypes.find(o => o.id === selectedOverlay)?.name}
                        </div>
                      </div>
 
                      {/* Legend */}
                      <div className="absolute bottom-4 left-4 bg-white/90 rounded-lg p-3">
                        <div className="text-xs font-medium text-gray-900 mb-2">Intensity</div>
                        <div className="flex items-center gap-2">
                          <div className="w-3 h-3 bg-blue-500 rounded"></div>
                          <span className="text-xs">Low</span>
                          <div className="w-3 h-3 bg-yellow-500 rounded"></div>
                          <span className="text-xs">Medium</span>
                          <div className="w-3 h-3 bg-red-600 rounded"></div>
                          <span className="text-xs">High</span>
                        </div>
                      </div>
 
                    </div>
                  </div>
                </div>
              </CardContent>
            </Card>
          </div>
 
          {/* Zone Details Panel */}
          <div className="lg:col-span-1">
            {selectedZone ? (
              <Card>
                <CardHeader>
                  <CardTitle className="flex items-center gap-2">
                    <Info className="w-5 h-5" />
                    Zone Details
                  </CardTitle>
                </CardHeader>
                <CardContent className="space-y-4">
                  <div>
                    <h3 className="font-semibold text-lg">{selectedZone.zone_name}</h3>
                    <p className="text-gray-600 text-sm">{selectedZone.city}</p>
                  </div>
 
                  <div className="grid grid-cols-2 gap-4">
                    <div className="text-center p-3 bg-blue-50 rounded-lg">
                      <div className="text-2xl font-bold text-blue-600">{selectedZone.total_properties}</div>
                      <div className="text-xs text-gray-600">Properties</div>
                    </div>
                    <div className="text-center p-3 bg-green-50 rounded-lg">
                      <div className="text-2xl font-bold text-green-600">
                        {selectedZone.demand_index || 0}
                      </div>
                      <div className="text-xs text-gray-600">Demand Index</div>
                    </div>
                  </div>
 
                  <div className="space-y-3">
                    <div className="flex justify-between">
                      <span className="text-sm text-gray-600">Avg Price/Sq.Ft</span>
                      <span className="text-sm font-medium">
                        ₹{selectedZone.avg_price_per_sqft?.toLocaleString() || 'N/A'}
                      </span>
                    </div>
                    <div className="flex justify-between">
                      <span className="text-sm text-gray-600">Rental Yield</span>
                      <span className="text-sm font-medium">
                        {selectedZone.rental_yield_avg || 'N/A'}%
                      </span>
                    </div>
                    <div className="flex justify-between">
                      <span className="text-sm text-gray-600">Metro Distance</span>
                      <span className="text-sm font-medium">
                        {selectedZone.metro_distance || 'N/A'} km
                      </span>
                    </div>
                  </div>
 
                  <Button className="w-full" variant="outline">
                    <Eye className="w-4 h-4 mr-2" />
                    View Properties
                  </Button>
                </CardContent>
              </Card>
            ) : (
              <Card>
                <CardContent className="p-6 text-center">
                  <MapPin className="w-12 h-12 text-gray-400 mx-auto mb-4" />
                  <p className="text-gray-600">Select a zone to view details</p>
                </CardContent>
              </Card>
            )}
 
            {/* Quick Stats */}
            <Card className="mt-6">
              <CardHeader>
                <CardTitle className="flex items-center gap-2">
                  <TrendingUp className="w-5 h-5" />
                  Market Insights
                </CardTitle>
              </CardHeader>
              <CardContent className="space-y-3">
                <div className="flex items-center justify-between">
                  <span className="text-sm text-gray-600">Total Zones</span>
                  <span className="font-medium">{zoneData.length}</span>
                </div>
                <div className="flex items-center justify-between">
                  <span className="text-sm text-gray-600">Avg Demand</span>
                  <span className="font-medium">
                    {Math.round(zoneData.reduce((acc, zone) => acc + (zone.demand_index || 0), 0) / zoneData.length) || 0}
                  </span>
                </div>
                <div className="flex items-center justify-between">
                  <span className="text-sm text-gray-600">Price Range</span>
                  <span className="font-medium text-xs">
                    ₹{Math.min(...zoneData.map(z => z.avg_price_per_sqft || 0)).toLocaleString()} -
                    ₹{Math.max(...zoneData.map(z => z.avg_price_per_sqft || 0)).toLocaleString()}
                  </span>
                </div>
              </CardContent>
            </Card>
          </div>
 
        </div>
      </div>
    </div>
  );
}
 