import React, { useState, useEffect } from "react";
import { useNavigate } from "react-router-dom";
import { motion } from "framer-motion";
import { TrendingUp, MapPin, Loader2 } from "lucide-react";
import { PropertyCard } from "./PropertyCard";
import { api } from "../utils/api";

interface Property {
  id: number;
  title: string;
  locality: string;
  city: string;
  bedrooms?: number;
  bathrooms?: number;
  built_up_area?: number;
  sale_price?: number;
  monthly_rent?: number;
  listing_type: string;
  primary_image?: string;
}

export function SmartSuggestions() {
  const navigate = useNavigate();
  const [trendingProperties, setTrendingProperties] = useState<Property[]>([]);
  const [nearbyProperties, setNearbyProperties] = useState<Property[]>([]);
  const [loading, setLoading] = useState(false);
  const [userCity, setUserCity] = useState<string>("");
  const [locationDenied, setLocationDenied] = useState(false);
  const [clickedHearts, setClickedHearts] = useState<Set<number>>(new Set());

  useEffect(() => {
    const accessToken = localStorage.getItem('access_token');
    if (accessToken) {
      requestLocationAndFetchSuggestions();
    }
  }, []);

  const requestLocationAndFetchSuggestions = async () => {
    if (!navigator.geolocation) {
      console.log('Geolocation not supported');
      return;
    }

    setLoading(true);

    navigator.geolocation.getCurrentPosition(
      async (position) => {
        const { latitude, longitude } = position.coords;
        
        try {
          // Store coordinates for SmartSuggestions
          localStorage.setItem('user_lat', latitude.toString());
          localStorage.setItem('user_lng', longitude.toString());

          // Get city from coordinates using existing geocoding API
          const geocodeResult = await api.reverseGeocode(latitude, longitude);
          
          if (geocodeResult.success) {
            const city = geocodeResult.address?.city || geocodeResult.address?.town || 'Unknown';
            setUserCity(city);
            localStorage.setItem('user_city', city);

            // Fetch both trending and nearby properties
            const [trending, nearby] = await Promise.all([
              api.fetchTrendingProperties(city),
              api.fetchNearbyProperties(latitude, longitude)
            ]);

            setTrendingProperties(trending.slice(0, 4));
            setNearbyProperties(nearby.slice(0, 4));
          }
        } catch (error) {
          console.error('Error fetching suggestions:', error);
        } finally {
          setLoading(false);
        }
      },
      (error) => {
        console.log('Location permission denied:', error);
        setLocationDenied(true);
        setLoading(false);
      }
    );
  };

  const formatPrice = (price: number) => {
    if (price >= 10000000) return `₹${(price / 10000000).toFixed(1)} Cr`;
    if (price >= 100000) return `₹${(price / 100000).toFixed(0)} L`;
    return `₹${price.toLocaleString()}`;
  };

  const handleFavorite = async (propertyId: number) => {
    setClickedHearts(prev => new Set(prev).add(propertyId));
    
    const token = localStorage.getItem('access_token');
    if (token) {
      try {
        await fetch('http://localhost:8000/api/add-to-wishlist/', {
          method: 'POST',
          headers: {
            'Content-Type': 'application/json',
            'Authorization': `Bearer ${token}`
          },
          body: JSON.stringify({ property_id: propertyId })
        });
      } catch (error) {
        console.error('Error adding to wishlist:', error);
      }
    } else {
      localStorage.setItem('pending_wishlist', propertyId.toString());
      setTimeout(() => navigate('/login'), 300);
    }
  };

  const transformProperty = (property: Property) => ({
    id: property.id,
    title: property.title,
    location: `${property.locality}, ${property.city}`,
    beds: property.bedrooms || 'N/A',
    baths: property.bathrooms || 'N/A',
    area: property.built_up_area ? `${property.built_up_area} sq ft` : 'N/A',
    price: formatPrice(property.sale_price || property.monthly_rent || 0),
    imageUrl: property.primary_image || null
  });

  // Don't render if user is not logged in or location was denied
  const accessToken = localStorage.getItem('access_token');
  if (!accessToken || locationDenied) {
    return null;
  }

  // Don't render if no properties found
  if (!loading && trendingProperties.length === 0 && nearbyProperties.length === 0) {
    return null;
  }

  return (
    <section className="py-12 bg-white">
      <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
        {loading ? (
          <div className="text-center py-12">
            <Loader2 className="w-8 h-8 animate-spin mx-auto mb-4 text-blue-600" />
            <p className="text-gray-600">Finding properties for you...</p>
          </div>
        ) : (
          <>


            {/* Nearby Properties Section */}
            {nearbyProperties.length > 0 && (
              <motion.div
                initial={{ opacity: 0, y: 20 }}
                animate={{ opacity: 1, y: 0 }}
                transition={{ duration: 0.6, delay: 0.2 }}
              >
                <div className="flex items-center gap-3 mb-6">
                  <MapPin className="w-6 h-6 text-blue-500" />
                  <h2 className="text-2xl font-bold text-gray-900">
                    📍 Properties Near You
                  </h2>
                </div>
                <div className="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-4 gap-6">
                  {nearbyProperties.map((property) => (
                    <PropertyCard
                      key={property.id}
                      {...transformProperty(property)}
                      onFavorite={handleFavorite}
                      isFavorited={clickedHearts.has(property.id)}
                    />
                  ))}
                </div>
              </motion.div>
            )}
          </>
        )}
      </div>
    </section>
  );
}