import React, { useEffect, useRef } from 'react';

interface TransitMapProps {
  center: { lat: number; lng: number };
  metroDistance?: number;
  busStops?: number;
  height?: string;
}

export const TransitMap: React.FC<TransitMapProps> = ({
  center,
  metroDistance = 1.2,
  busStops = 8,
  height = '400px'
}) => {
  const mapRef = useRef<HTMLDivElement>(null);
  const mapInstanceRef = useRef<any>(null);

  useEffect(() => {
    if (!window.google || !mapRef.current) return;

    // Initialize map
    mapInstanceRef.current = new window.google.maps.Map(mapRef.current, {
      zoom: 14,
      center: center,
      mapTypeId: 'roadmap',
      zoomControl: true,
      mapTypeControl: false,
      streetViewControl: false,
      fullscreenControl: true,
    });

    // Add center marker (locality)
    new window.google.maps.Marker({
      position: center,
      map: mapInstanceRef.current,
      title: 'Selected Locality',
      icon: {
        path: window.google.maps.SymbolPath.CIRCLE,
        scale: 12,
        fillColor: '#3b82f6',
        fillOpacity: 1,
        strokeColor: '#ffffff',
        strokeWeight: 3
      },
      animation: window.google.maps.Animation.DROP
    });

    // Add circle around locality
    new window.google.maps.Circle({
      strokeColor: '#3b82f6',
      strokeOpacity: 0.4,
      strokeWeight: 2,
      fillColor: '#3b82f6',
      fillOpacity: 0.1,
      map: mapInstanceRef.current,
      center: center,
      radius: 2000 // 2km radius
    });

    // Generate mock metro stations
    const metroAngles = [0, 90, 180, 270];
    metroAngles.forEach((angle, idx) => {
      const rad = (angle * Math.PI) / 180;
      const distance = (metroDistance * 1000) / 111320; // Convert km to degrees
      const metroPos = {
        lat: center.lat + distance * Math.cos(rad),
        lng: center.lng + distance * Math.sin(rad) / Math.cos(center.lat * Math.PI / 180)
      };

      new window.google.maps.Marker({
        position: metroPos,
        map: mapInstanceRef.current,
        title: `Metro Station ${idx + 1}`,
        icon: {
          path: window.google.maps.SymbolPath.CIRCLE,
          scale: 8,
          fillColor: '#9333ea',
          fillOpacity: 1,
          strokeColor: '#ffffff',
          strokeWeight: 2
        },
        label: {
          text: 'M',
          color: '#ffffff',
          fontSize: '10px',
          fontWeight: 'bold'
        }
      });
    });

    // Generate mock bus stops
    for (let i = 0; i < Math.min(busStops, 8); i++) {
      const angle = (i * 360) / busStops;
      const rad = (angle * Math.PI) / 180;
      const distance = (0.5 + Math.random() * 1) / 111.32; // 0.5-1.5 km
      const busPos = {
        lat: center.lat + distance * Math.cos(rad),
        lng: center.lng + distance * Math.sin(rad) / Math.cos(center.lat * Math.PI / 180)
      };

      new window.google.maps.Marker({
        position: busPos,
        map: mapInstanceRef.current,
        title: `Bus Stop ${i + 1}`,
        icon: {
          path: window.google.maps.SymbolPath.CIRCLE,
          scale: 6,
          fillColor: '#3b82f6',
          fillOpacity: 1,
          strokeColor: '#ffffff',
          strokeWeight: 2
        },
        label: {
          text: 'B',
          color: '#ffffff',
          fontSize: '8px',
          fontWeight: 'bold'
        }
      });
    }

  }, [center, metroDistance, busStops]);

  return (
    <div className="relative w-full" style={{ height }}>
      <div ref={mapRef} className="w-full h-full rounded-lg" />
      {!window.google && (
        <div className="absolute inset-0 bg-gradient-to-br from-purple-50 to-blue-50 flex items-center justify-center rounded-lg">
          <div className="text-center">
            <div className="animate-spin rounded-full h-12 w-12 border-b-2 border-purple-600 mx-auto mb-4" />
            <p className="text-gray-600 font-medium">Loading Transit Map...</p>
          </div>
        </div>
      )}
    </div>
  );
};
