import React, { useEffect, useState } from "react";
import { useNavigate } from "react-router-dom";
import { PropertyCard } from "./PropertyCard";
import { api } from "../utils/api";
import { ChevronDown, MapPin } from "lucide-react";
import { getPropertyMedia, generateDownloadURL } from '../api/mediaUpload';

interface Property {
  id: number;
  title: string;
  location: string;
  beds: string | number;
  baths: string | number;
  area: string;
  price: string;
  imageUrl: string;
  listingType: string;
  urgent_pricing?: number;
  sale_price?: number;
}

export function TrendingProperties() {
  const navigate = useNavigate();
  const [properties, setProperties] = useState<Property[]>([]);
  const [loading, setLoading] = useState(true);
  const [clickedHearts, setClickedHearts] = useState<Set<number>>(new Set());
  const [selectedCity, setSelectedCity] = useState('Bangalore');
  const [selectedLocality, setSelectedLocality] = useState('All');
  const [showCityDropdown, setShowCityDropdown] = useState(false);
  const [showLocalityDropdown, setShowLocalityDropdown] = useState(false);
  
  const cities = ['Bangalore', 'Mumbai', 'Delhi', 'Chennai', 'Hyderabad', 'Pune', 'Kolkata', 'Ahmedabad'];
  
  const localitiesByCity: { [key: string]: string[] } = {
    'Bangalore': ['All', 'Electronic City', 'Koramangala', 'Indiranagar', 'Whitefield', 'Marathahalli', 'Hebbal', 'HBR Layout'],
    'Chennai': ['All', 'Anna Nagar', 'T Nagar', 'OMR', 'Tambaram', 'Porur', 'Velachery'],
    'Hyderabad': ['All', 'Gachibowli', 'Madhapur', 'Banjara Hills', 'Hitech City', 'Jubilee Hills', 'Kukatpally'],
    'Mumbai': ['All', 'Andheri', 'Bandra', 'Powai', 'Thane', 'Navi Mumbai'],
    'Delhi': ['All', 'Connaught Place', 'Karol Bagh', 'Lajpat Nagar', 'Dwarka'],
    'Pune': ['All', 'Koregaon Park', 'Hinjewadi', 'Wakad', 'Baner'],
    'Kolkata': ['All', 'Salt Lake', 'Park Street', 'Ballygunge'],
    'Ahmedabad': ['All', 'Satellite', 'Vastrapur', 'Bopal']
  };

  useEffect(() => {
    const loadTrendingProperties = async () => {
      try {
        setLoading(true);
        
        let url = `/api/properties/?city=${selectedCity}&ordering=trending&limit=10`;
        if (selectedLocality !== 'All') {
          url += `&locality=${selectedLocality}`;
        }
        
        const response = await fetch(url, {
          method: 'GET',
          headers: { 'Accept': 'application/json' }
        });
        
        if (response.ok) {
          const result = await response.json();
          const trendingProperties = result.properties || [];
          
          if (trendingProperties.length > 0) {
            // Filter trending properties by locality if selected
            let filteredTrending = trendingProperties;
            if (selectedLocality !== 'All') {
              filteredTrending = trendingProperties.filter((property: any) => 
                property.locality.toLowerCase() === selectedLocality.toLowerCase()
              );
            }
            
            if (filteredTrending.length > 0) {
              // Filter out unwanted properties from trending API response too
              const unwantedTitles = ['keerthana 1', 'keerthana 2', 'saad'];
              const cleanTrending = filteredTrending.filter((property: any) => 
                !unwantedTitles.includes(property.title.toLowerCase())
              );
              
              if (cleanTrending.length > 0) {
                const formattedProperties = await Promise.all(
                  cleanTrending
                    .slice(0, 4)
                    .map(async (property: any) => {
                      let imageUrl = property.primary_image || property.photo;
                      
                      // Try to get real image from Azure Blob storage
                      if (!imageUrl) {
                        try {
                          const mediaData = await getPropertyMedia(property.id.toString());
                          if (mediaData?.images && mediaData.images.length > 0) {
                            const primaryImage = mediaData.images.find((img: any) => img.is_primary) || mediaData.images[0];
                            const downloadResponse = await generateDownloadURL(primaryImage.id);
                            imageUrl = downloadResponse?.download_url || downloadResponse?.url || downloadResponse;
                          }
                        } catch (error) {
                          console.log('Could not fetch media for trending property', property.id);
                        }
                      }
                      
                      return {
                        id: property.id,
                        title: property.title,
                        location: `${property.locality}, ${property.city}`,
                        beds: property.bedrooms || 'N/A',
                        baths: property.bathrooms || 'N/A', 
                        area: property.built_up_area ? `${property.built_up_area} sq ft` : 'N/A',
                        price: formatPrice(parseFloat(
                          property.listing_type === 'urgent-sale' && property.urgent_pricing 
                            ? property.urgent_pricing 
                            : property.sale_price
                        )),
                        imageUrl,
                        listingType: property.listing_type
                      };
                    })
                );
                
                setProperties(formattedProperties);
                return;
              }
            }
          }
          
          // Fallback to regular properties if no trending or no matching locality
          // Fallback to regular properties filtered by city
            const sellProperties = await api.fetchSellProperties();
            const cityFilteredProperties = sellProperties.filter((property: any) => {
              // Filter out specific unwanted properties
              const unwantedTitles = ['keerthana 1', 'keerthana 2', 'saad'];
              if (unwantedTitles.includes(property.title.toLowerCase())) {
                return false;
              }
              
              const propertyCity = property.city.toLowerCase();
              const searchCity = selectedCity.toLowerCase();
              
              let cityMatch = false;
              // Handle Bangalore/Bengaluru variations
              if (searchCity === 'bangalore') {
                cityMatch = propertyCity === 'bangalore' || propertyCity === 'bengaluru';
              } else if (searchCity === 'bengaluru') {
                cityMatch = propertyCity === 'bangalore' || propertyCity === 'bengaluru';
              } else {
                cityMatch = propertyCity.includes(searchCity);
              }
              
              if (!cityMatch) return false;
              
              // Filter by locality if selected
              if (selectedLocality !== 'All') {
                return property.locality.toLowerCase() === selectedLocality.toLowerCase();
              }
              
              return true;
            });
            const formattedProperties = await Promise.all(
              cityFilteredProperties
                .slice(0, 4)
                .map(async (property: any) => {
                  let imageUrl = property.primary_image || property.photo;
                  
                  // Try to get real image from Azure Blob storage
                  if (!imageUrl) {
                    try {
                      const mediaData = await getPropertyMedia(property.id.toString());
                      if (mediaData?.images && mediaData.images.length > 0) {
                        const primaryImage = mediaData.images.find((img: any) => img.is_primary) || mediaData.images[0];
                        const downloadResponse = await generateDownloadURL(primaryImage.id);
                        imageUrl = downloadResponse?.download_url || downloadResponse?.url || downloadResponse;
                      }
                    } catch (error) {
                      console.log('Could not fetch media for trending property', property.id);
                    }
                  }
                  
                  return {
                    id: property.id,
                    title: property.title,
                    location: `${property.locality}, ${property.city}`,
                    beds: property.bedrooms || 'N/A',
                    baths: property.bathrooms || 'N/A', 
                    area: property.built_up_area ? `${property.built_up_area} sq ft` : 'N/A',
                    price: formatPrice(parseFloat(
                      property.listing_type === 'urgent-sale' && property.urgent_pricing 
                        ? property.urgent_pricing 
                        : property.sale_price
                    )),
                    imageUrl,
                    listingType: property.listing_type
                  };
                })
            );
            
            setProperties(formattedProperties);
          }
        }
      } catch (error) {
        console.log('Error loading trending properties:', error);
        setProperties([]);
      } finally {
        setLoading(false);
      }
    };
    
    loadTrendingProperties();
  }, [selectedCity, selectedLocality]);
  
  // Reset locality when city changes
  useEffect(() => {
    setSelectedLocality('All');
  }, [selectedCity]);

  const formatPrice = (price: number) => {
    if (price >= 10000000) return `₹${(price / 10000000).toFixed(2)} Cr`;
    if (price >= 100000) return `₹${(price / 100000).toFixed(2)} L`;
    return `₹${price.toLocaleString()}`;
  };

  const handleFavorite = async (propertyId: number) => {
    setClickedHearts(prev => new Set(prev).add(propertyId));
    
    const token = localStorage.getItem('access_token');
    if (token) {
      try {
        await fetch('http://localhost:8000/api/add-to-wishlist/', {
          method: 'POST',
          headers: {
            'Content-Type': 'application/json',
            'Authorization': `Bearer ${token}`
          },
          body: JSON.stringify({ property_id: propertyId })
        });
      } catch (error) {
        console.error('Error adding to wishlist:', error);
      }
    } else {
      localStorage.setItem('pending_wishlist', propertyId.toString());
      setTimeout(() => navigate('/login'), 300);
    }
  };

  return (
    <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-10 bg-white">
      <div className="text-center mb-12">
        <h2 className="text-4xl font-bold text-gray-900 mb-2">Trending Properties</h2>
        <div className="w-24 h-1 bg-orange-500 mx-auto mb-4"></div>
        <p className="text-lg text-gray-600 max-w-3xl mx-auto mb-6">Discover the most popular properties based on current market trends.</p>
        
        <div className="flex gap-4 justify-center">
          {/* City Dropdown */}
          <div className="relative">
            <button
              onClick={() => setShowCityDropdown(!showCityDropdown)}
              className="flex items-center gap-2 px-4 py-2 bg-white border border-gray-300 rounded-lg shadow-sm hover:bg-gray-50"
            >
              <span className="text-gray-700">City: {selectedCity}</span>
              <ChevronDown className="w-4 h-4 text-gray-500" />
            </button>
            
            {showCityDropdown && (
              <div className="absolute top-full left-0 mt-1 w-48 bg-white border border-gray-200 rounded-lg shadow-lg z-50">
                {cities.map((city) => (
                  <button
                    key={city}
                    onClick={() => {
                      setSelectedCity(city);
                      setShowCityDropdown(false);
                    }}
                    className="w-full text-left px-4 py-2 hover:bg-gray-50 first:rounded-t-lg last:rounded-b-lg"
                  >
                    {city}
                  </button>
                ))}
              </div>
            )}
          </div>
          
          {/* Locality Dropdown */}
          <div className="relative">
            <button
              onClick={() => setShowLocalityDropdown(!showLocalityDropdown)}
              className="flex items-center gap-2 px-4 py-2 bg-white border border-gray-300 rounded-lg shadow-sm hover:bg-gray-50"
            >
              <span className="text-gray-700">Area: {selectedLocality}</span>
              <ChevronDown className="w-4 h-4 text-gray-500" />
            </button>
            
            {showLocalityDropdown && (
              <div className="absolute top-full left-0 mt-1 w-48 bg-white border border-gray-200 rounded-lg shadow-lg z-50">
                {localitiesByCity[selectedCity]?.map((locality) => (
                  <button
                    key={locality}
                    onClick={() => {
                      setSelectedLocality(locality);
                      setShowLocalityDropdown(false);
                    }}
                    className="w-full text-left px-4 py-2 hover:bg-gray-50 first:rounded-t-lg last:rounded-b-lg"
                  >
                    {locality}
                  </button>
                ))}
              </div>
            )}
          </div>
        </div>
      </div>
      
      {/* Search Results Indicator */}
      {selectedCity && (
        <div className="bg-blue-50 border border-blue-200 rounded-lg p-4 mb-6">
          <div className="flex items-center justify-between">
            <div className="flex items-center gap-2">
              <MapPin className="w-5 h-5 text-blue-600" />
              <span className="text-blue-800 font-medium">
                Showing trending properties for: <span className="font-bold">{selectedCity}</span>
              </span>
            </div>
            <div className="flex items-center gap-3">
              <div className="text-sm text-blue-600">
                {properties.length} properties found
              </div>
              <button
                onClick={() => {
                  setSelectedCity('Bangalore');
                  setSelectedLocality('All');
                }}
                className="text-sm text-blue-600 hover:text-blue-800 font-medium underline"
              >
                Clear Search
              </button>
            </div>
          </div>
        </div>
      )}
      
      <div className="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-4 xl:grid-cols-5 gap-4 justify-items-center">
        {loading ? (
          <div className="col-span-full text-center py-8">
            <p className="text-gray-500">Loading trending properties...</p>
          </div>
        ) : properties.length > 0 ? properties.map((property: Property) => (
          <PropertyCard
            key={property.id}
            id={property.id}
            title={property.title}
            location={property.location}
            beds={property.beds}
            baths={property.baths}
            area={property.area}
            price={property.price}
            imageUrl={property.imageUrl}
            onFavorite={handleFavorite}
            isFavorited={clickedHearts.has(property.id)}
          />
        )) : (
          <div className="col-span-full text-center py-8">
            <p className="text-gray-500">No trending properties available at the moment.</p>
          </div>
        )}
      </div>
    </div>
  );
}