import { useState } from 'react';
import { Button } from './ui/button';
import { Input } from './ui/input';
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from './ui/select';
import { X, Building, MapPin, AlertCircle, CheckCircle, TrendingUp, Calculator, FileText, ArrowRight, ChevronDown, ChevronUp, Download } from 'lucide-react';
import { validatePropertyData, valuationEngine } from '../services/valuationEngine';
import { valuationApi } from '../services/valuationApi';

interface ValuationFlowProps {
  isOpen: boolean;
  onClose: () => void;
  onComplete?: (result: ValuationResult) => void;
}

type FlowStep = 'form' | 'processing' | 'results' | 'error';

interface FormErrors {
  propertyType?: string;
  location?: string;
  area?: string;
  bedrooms?: string;
}

export function ValuationFlow({ isOpen, onClose, onComplete }: ValuationFlowProps) {
  const [step, setStep] = useState<FlowStep>('form');
  const [showAdvanced, setShowAdvanced] = useState(false);
  const [formData, setFormData] = useState<PropertyData>({
    propertyType: 'apartment' as const,
    location: '',
    area: 0,
    bedrooms: '',
    bathrooms: undefined,
    floor: undefined,
    totalFloors: undefined,
    age: undefined,
    furnishing: 'unfurnished',
    facing: undefined,
    amenities: []
  });
  const [errors, setErrors] = useState<FormErrors>({});
  const [result, setResult] = useState<ValuationResult | null>(null);
  const [errorMessage, setErrorMessage] = useState('');

  const availableAmenities = [
    "Parking", "Gym", "Swimming Pool", "Security", "Power Backup",
    "Lift", "Garden", "Club House", "Play Area", "Intercom",
    "Gas Pipeline", "Water Supply", "Maintenance Staff"
  ];

  const toggleAmenity = (amenity: string) => {
    setFormData(prev => ({
      ...prev,
      amenities: prev.amenities?.includes(amenity)
        ? prev.amenities.filter(a => a !== amenity)
        : [...(prev.amenities || []), amenity]
    }));
  };

  if (!isOpen) return null;

  const validateForm = (): FormErrors => {
    const validationErrors = validatePropertyData(formData);
    const newErrors: FormErrors = {};
    
    validationErrors.forEach(error => {
      if (error.includes('Property type')) newErrors.propertyType = error;
      if (error.includes('Location')) newErrors.location = error;
      if (error.includes('area') || error.includes('Area')) newErrors.area = error;
      if (error.includes('Bedrooms')) newErrors.bedrooms = error;
    });
    
    return newErrors;
  };

  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault();
    setErrors({});
    
    const formErrors = validateForm();
    if (Object.keys(formErrors).length > 0) {
      setErrors(formErrors);
      return;
    }
    
    setStep('processing');
    
    try {
      // Try API first
      const requestData = {
        property_type: formData.propertyType,
        location: formData.location,
        area: formData.area,
        bedrooms: parseInt(formData.bedrooms),
        floor_number: formData.floor,
        total_floors: formData.totalFloors,
        property_age: formData.age,
        furnishing: formData.furnishing,
        facing: formData.facing,
        amenities: formData.amenities || []
      };
      
      const response = await valuationApi.calculateValuation(requestData);
      
      if (response.success && response.data) {
        const apiResult = response.data;
        const valuationResult = {
          estimatedValue: apiResult.estimated_value,
          pricePerSqft: apiResult.price_per_sqft,
          confidence: apiResult.confidence_score,
          range: {
            min: apiResult.valuation_range_min,
            max: apiResult.valuation_range_max
          },
          methodology: apiResult.calculation_methodology || apiResult.methodology,
          mlPrediction: apiResult.ml_prediction,
          mlConfidence: apiResult.ml_confidence,
          coordinates: apiResult.coordinates,
          comparables: (apiResult.comparables || []).map((comp: any) => ({
            address: comp.address,
            price: comp.price,
            pricePerSqft: comp.price_per_sqft,
            distance: comp.distance_km
          })),
          marketTrends: {
            appreciation: apiResult.market_trends?.appreciation || 0,
            rentalYield: apiResult.market_trends?.rental_yield || 0,
            demandSupply: apiResult.market_trends?.demand_supply || 'medium'
          }
        };
        
        setResult(valuationResult);
        setStep('results');
        onComplete?.(valuationResult);
      } else {
        // Fallback to local engine if API fails
        console.log('API failed, using local valuation engine');
        const localResult = await valuationEngine.calculateValuation({
          propertyType: formData.propertyType,
          location: formData.location,
          area: formData.area,
          bedrooms: formData.bedrooms
        });
        
        setResult(localResult);
        setStep('results');
        onComplete?.(localResult);
      }
    } catch (error) {
      // Fallback to local engine on error
      console.log('Error occurred, using local valuation engine:', error);
      try {
        const localResult = await valuationEngine.calculateValuation({
          propertyType: formData.propertyType,
          location: formData.location,
          area: formData.area,
          bedrooms: formData.bedrooms
        });
        
        setResult(localResult);
        setStep('results');
        onComplete?.(localResult);
      } catch (engineError) {
        setErrorMessage(engineError instanceof Error ? engineError.message : 'Valuation calculation failed');
        setStep('error');
      }
    }
  };

  const handleClose = () => {
    setStep('form');
    setFormData({ propertyType: 'apartment', location: '', area: 0, bedrooms: '' });
    setErrors({});
    setResult(null);
    setErrorMessage('');
    onClose();
  };

  const formatCurrency = (amount: number) => {
    return new Intl.NumberFormat('en-IN', {
      style: 'currency',
      currency: 'INR',
      maximumFractionDigits: 0
    }).format(amount);
  };

  const generatePDF = async () => {
    if (!result) return;

    // Convert favicon to base64
    const getBase64Image = async (url: string): Promise<string> => {
      try {
        const response = await fetch(url);
        const blob = await response.blob();
        return new Promise((resolve) => {
          const reader = new FileReader();
          reader.onload = () => resolve(reader.result as string);
          reader.readAsDataURL(blob);
        });
      } catch {
        return '';
      }
    };

    const logoBase64 = await getBase64Image('/favicon.png');
    const analysisMode = showAdvanced && (formData.bathrooms || formData.floor || formData.furnishing !== 'unfurnished' || formData.facing || (formData.amenities && formData.amenities.length > 0)) ? 'Enhanced' : 'Basic';
    const factorsUsed = analysisMode === 'Enhanced' ? 10 : 6;

    const reportData = {
      generatedAt: new Date().toISOString(),
      propertyDetails: {
        propertyType: formData.propertyType,
        location: formData.location,
        area: formData.area,
        bedrooms: formData.bedrooms,
        bathrooms: formData.bathrooms || 'Not specified',
        floor: formData.floor || 'Not specified',
        totalFloors: formData.totalFloors || 'Not specified',
        age: formData.age || 'Not specified',
        furnishing: formData.furnishing || 'Not specified',
        facing: formData.facing || 'Not specified',
        amenities: formData.amenities || []
      },
      valuationResults: result,
      analysisMode,
      factorsUsed
    };

    const getPropertyTypeDisplay = (type: string) => {
      const typeMap: Record<string, string> = {
        'apartment': '🏢 Apartment',
        'villa': '🏡 Villa',
        'plot': '🏞️ Plot',
        'commercial': '🏢 Commercial'
      };
      return typeMap[type] || type;
    };

    const getFurnishingDisplay = (furnishing: string) => {
      const furnishingMap: Record<string, string> = {
        'unfurnished': 'Unfurnished',
        'semi_furnished': 'Semi Furnished',
        'fully_furnished': 'Fully Furnished'
      };
      return furnishingMap[furnishing] || furnishing;
    };

    const getFacingDisplay = (facing: string) => {
      const facingMap: Record<string, string> = {
        'north': '🧭 North',
        'south': '🧭 South',
        'east': '🧭 East',
        'west': '🧭 West',
        'northeast': '🧭 North-East',
        'northwest': '🧭 North-West',
        'southeast': '🧭 South-East',
        'southwest': '🧭 South-West'
      };
      return facingMap[facing] || facing;
    };

    const getValuationRecommendation = () => {
      const confidence = result.confidence;
      const appreciation = result.marketTrends.appreciation;
      const rentalYield = result.marketTrends.rentalYield;
      
      if (confidence >= 80 && appreciation >= 8) {
        return `Excellent investment opportunity with high confidence valuation (${confidence}%). The property shows strong market fundamentals with ${appreciation}% annual appreciation and ${rentalYield}% rental yield. This ${analysisMode.toLowerCase()} analysis using ${factorsUsed} factors provides reliable valuation for confident investment decisions. Current market conditions favor property acquisition at this valuation level.`;
      } else if (confidence >= 70 && appreciation >= 5) {
        return `Good investment potential with reliable valuation confidence (${confidence}%). The property demonstrates solid market performance with ${appreciation}% annual growth and ${rentalYield}% rental returns. The ${analysisMode.toLowerCase()} analysis provides comprehensive insights for strategic investment planning. Market conditions suggest favorable timing for property transactions.`;
      } else if (confidence >= 60) {
        return `Moderate confidence valuation (${confidence}%) indicates reasonable market understanding with scope for additional analysis. The property shows ${appreciation}% appreciation potential and ${rentalYield}% rental yield. Consider providing additional property details for enhanced accuracy. Current market trends suggest cautious optimism for property investments.`;
      } else {
        return `Initial valuation assessment (${confidence}% confidence) provides baseline market insights. The property indicates ${appreciation}% growth potential with ${rentalYield}% rental yield expectations. For more accurate valuation, consider comprehensive property analysis including detailed specifications and market comparisons. Professional consultation recommended for investment decisions.`;
      }
    };

    // Create HTML content for PDF
    const htmlContent = `<!DOCTYPE html>
<html>
<head>
  <meta charset="utf-8">
  <title>NAL Property Valuation Report</title>
  <style>
    @media print {
      body { margin: 0; }
      .no-print { display: none; }
    }
    body { font-family: Arial, sans-serif; margin: 20px; color: #333; line-height: 1.4; }
    .header { position: relative; text-align: center; margin-bottom: 30px; border-bottom: 2px solid #2B256D; padding-bottom: 20px; }
    .header-logo { position: absolute; top: 0; right: 0; width: 60px; height: 60px; }
    .property-summary { background: linear-gradient(135deg, #f0f9ff 0%, #e0f2fe 100%); padding: 20px; border-radius: 12px; margin: 20px 0; border: 1px solid #0ea5e9; }
    .valuation-highlight { text-align: center; background: linear-gradient(135deg, #dcfce7 0%, #bbf7d0 100%); padding: 30px; border-radius: 12px; margin: 20px 0; border: 2px solid #10b981; }
    .valuation-amount { font-size: 48px; font-weight: bold; color: #059669; margin: 15px 0; }
    .market-metrics { display: flex; justify-content: space-around; margin: 20px 0; gap: 15px; }
    .metric-card { text-align: center; padding: 20px; border-radius: 12px; flex: 1; }
    .appreciation-card { background-color: #dbeafe; color: #1e40af; border: 2px solid #93c5fd; }
    .rental-card { background-color: #f3e8ff; color: #7c3aed; border: 2px solid #c4b5fd; }
    .demand-card { background-color: #fed7aa; color: #ea580c; border: 2px solid #fdba74; }
    .comparables-section { background: #f8fafc; padding: 20px; border-radius: 12px; margin: 20px 0; border: 1px solid #e2e8f0; }
    .comparable-item { background: white; padding: 15px; border-radius: 8px; margin: 10px 0; border: 1px solid #e5e7eb; display: flex; justify-content: space-between; align-items: center; }
    .recommendation { background: linear-gradient(135deg, #eff6ff 0%, #dbeafe 100%); padding: 25px; border-radius: 12px; margin-top: 25px; border: 2px solid #3b82f6; }
    .property-details { background: #f8fafc; padding: 20px; border-radius: 12px; margin: 20px 0; border: 1px solid #e2e8f0; }
    .details-grid { display: grid; grid-template-columns: repeat(2, 1fr); gap: 15px; margin-top: 15px; }
    .detail-item { background: white; padding: 12px; border-radius: 8px; border: 1px solid #e5e7eb; }
    .amenities-list { display: flex; flex-wrap: wrap; gap: 8px; margin-top: 10px; }
    .amenity-tag { background: #3b82f6; color: white; padding: 4px 12px; border-radius: 20px; font-size: 12px; }
    h1 { color: #2B256D; margin: 0 0 10px 0; font-size: 28px; }
    h2 { color: #374151; margin: 25px 0 15px 0; font-size: 20px; }
    h3 { color: #1f2937; margin: 15px 0 10px 0; font-size: 16px; }
    .confidence-badge { padding: 8px 16px; border-radius: 25px; font-size: 14px; font-weight: bold; text-transform: uppercase; display: inline-block; }
    .confidence-high { background-color: #dcfce7; color: #166534; border: 2px solid #bbf7d0; }
    .confidence-medium { background-color: #fef3c7; color: #92400e; border: 2px solid #fde68a; }
    .confidence-low { background-color: #fee2e2; color: #991b1b; border: 2px solid #fecaca; }
    p { margin: 10px 0; }
    .print-btn { background: #2B256D; color: white; padding: 12px 24px; border: none; border-radius: 8px; cursor: pointer; margin: 20px; font-size: 14px; }
    .footer { text-align: center; margin-top: 40px; padding-top: 20px; border-top: 1px solid #e5e7eb; color: #6b7280; font-size: 12px; }
  </style>
</head>
<body>
  <button class="print-btn no-print" onclick="window.print()">Print/Save as PDF</button>
  
  <div class="header">
    ${logoBase64 ? `<img src="${logoBase64}" alt="NAL Logo" class="header-logo" />` : ''}
    <h1>NAL Property Valuation Report</h1>
    <p><strong>Professional Property Assessment & Market Analysis</strong></p>
    <p>Generated on: ${new Date().toLocaleDateString('en-US', { year: 'numeric', month: 'long', day: 'numeric', hour: '2-digit', minute: '2-digit' })}</p>
  </div>

  <div class="property-summary">
    <h2>🏠 Property Overview</h2>
    <div class="details-grid">
      <div class="detail-item"><strong>Property Type:</strong> ${getPropertyTypeDisplay(reportData.propertyDetails.propertyType)}</div>
      <div class="detail-item"><strong>Location:</strong> ${reportData.propertyDetails.location}</div>
      <div class="detail-item"><strong>Area:</strong> ${reportData.propertyDetails.area} sq ft</div>
      <div class="detail-item"><strong>Bedrooms:</strong> ${reportData.propertyDetails.bedrooms} BHK</div>
    </div>
    ${analysisMode === 'Enhanced' ? `
    <h3>🔍 Enhanced Analysis Details</h3>
    <div class="details-grid">
      <div class="detail-item"><strong>Bathrooms:</strong> ${reportData.propertyDetails.bathrooms}</div>
      <div class="detail-item"><strong>Floor:</strong> ${reportData.propertyDetails.floor} of ${reportData.propertyDetails.totalFloors}</div>
      <div class="detail-item"><strong>Age:</strong> ${reportData.propertyDetails.age} years</div>
      <div class="detail-item"><strong>Furnishing:</strong> ${getFurnishingDisplay(reportData.propertyDetails.furnishing)}</div>
      <div class="detail-item"><strong>Facing:</strong> ${getFacingDisplay(reportData.propertyDetails.facing)}</div>
    </div>
    ${reportData.propertyDetails.amenities.length > 0 ? `
    <div style="margin-top: 15px;">
      <strong>Amenities (${reportData.propertyDetails.amenities.length} selected):</strong>
      <div class="amenities-list">
        ${reportData.propertyDetails.amenities.map(amenity => `<span class="amenity-tag">${amenity}</span>`).join('')}
      </div>
    </div>` : ''}
    ` : ''}
    <div style="margin-top: 20px; text-align: center;">
      <span class="confidence-badge confidence-${reportData.valuationResults.confidence >= 80 ? 'high' : reportData.valuationResults.confidence >= 70 ? 'medium' : 'low'}">
        ${reportData.analysisMode} Analysis Mode • ${reportData.factorsUsed} Factors Used
      </span>
    </div>
  </div>

  <div class="valuation-highlight">
    <h2>💰 Property Valuation</h2>
    <div class="valuation-amount">${formatCurrency(reportData.valuationResults.estimatedValue)}</div>
    <span class="confidence-badge confidence-${reportData.valuationResults.confidence >= 80 ? 'high' : reportData.valuationResults.confidence >= 70 ? 'medium' : 'low'}">
      ${reportData.valuationResults.confidence}% Confidence
    </span>
    <p style="margin-top: 15px; font-size: 14px;">Price per sq ft: ${formatCurrency(reportData.valuationResults.pricePerSqft)}</p>
    <p style="font-size: 14px;">Valuation Range: ${formatCurrency(reportData.valuationResults.range.min)} - ${formatCurrency(reportData.valuationResults.range.max)}</p>
  </div>

  <h2>📊 Market Intelligence</h2>
  <div class="market-metrics">
    <div class="metric-card appreciation-card">
      <h3>+${reportData.valuationResults.marketTrends.appreciation}%</h3>
      <p><strong>Annual Appreciation</strong></p>
      <p>Year-over-year growth rate</p>
    </div>
    <div class="metric-card rental-card">
      <h3>${reportData.valuationResults.marketTrends.rentalYield}%</h3>
      <p><strong>Rental Yield</strong></p>
      <p>Expected annual returns</p>
    </div>
    <div class="metric-card demand-card">
      <h3 style="text-transform: capitalize;">${reportData.valuationResults.marketTrends.demandSupply}</h3>
      <p><strong>Market Demand</strong></p>
      <p>Current supply-demand balance</p>
    </div>
  </div>

  <div class="comparables-section">
    <h2>🏘️ Comparable Properties Analysis</h2>
    <p style="margin-bottom: 15px;"><strong>Methodology:</strong> ${reportData.valuationResults.methodology}</p>
    ${reportData.valuationResults.comparables.map((comp, index) => `
      <div class="comparable-item">
        <div>
          <div style="font-weight: bold; margin-bottom: 5px;">${comp.address}</div>
          <div style="font-size: 14px; color: #6b7280;">${comp.distance} km away</div>
        </div>
        <div style="text-align: right;">
          <div style="font-weight: bold; font-size: 16px;">${formatCurrency(comp.price)}</div>
          <div style="font-size: 14px; color: #6b7280;">${comp.pricePerSqft ? formatCurrency(comp.pricePerSqft) + '/sq ft' : 'Price per sq ft not available'}</div>
        </div>
      </div>
    `).join('')}
  </div>

  <div class="recommendation">
    <h2>🎓 Professional Valuation Analysis</h2>
    <p><strong>Valuation Confidence:</strong> 
      <span class="confidence-badge confidence-${reportData.valuationResults.confidence >= 80 ? 'high' : reportData.valuationResults.confidence >= 70 ? 'medium' : 'low'}">
        ${reportData.valuationResults.confidence >= 80 ? 'HIGH' : reportData.valuationResults.confidence >= 70 ? 'MEDIUM' : 'LOW'}
      </span>
    </p>
    <p style="margin-top: 20px; font-size: 16px; line-height: 1.6;">${getValuationRecommendation()}</p>
  </div>

  <div class="footer">
    <p><strong>Disclaimer:</strong> This valuation report is generated based on available market data and property information provided. 
    Actual market conditions may vary. For investment decisions, please consult with real estate professionals.</p>
    <p>© 2024 New Age Land. All rights reserved. | Generated by NAL Property Valuation System</p>
  </div>
</body>
</html>`;

    // Create blob and download
    const blob = new Blob([htmlContent], { type: 'text/html' });
    const url = URL.createObjectURL(blob);
    
    // Navigate in same tab instead of opening new window
    window.location.href = url;
  };

  const renderForm = () => (
    <div className="bg-white rounded-xl shadow-2xl w-full max-w-md flex flex-col" style={{ maxHeight: '90vh' }}>
      <div className="flex items-center justify-between p-4 border-b flex-shrink-0">
        <h2 className="text-lg font-semibold flex items-center">
          <Building className="w-5 h-5 mr-2" />
          Property Valuation
        </h2>
        <Button variant="ghost" size="sm" onClick={handleClose}>
          <X className="w-4 h-4" />
        </Button>
      </div>
      
      <div className="overflow-y-auto flex-1" style={{ minHeight: 0 }}>
        <div className="p-4 space-y-3">
        <div className="space-y-1.5">
          <label className="text-sm font-medium">Property Type</label>
          <Select 
            value={formData.propertyType} 
            onValueChange={(value: PropertyData['propertyType']) => 
              setFormData(prev => ({ ...prev, propertyType: value }))
            }
          >
            <SelectTrigger className={errors.propertyType ? "border-red-300" : ""}>
              <SelectValue placeholder="Select property type" />
            </SelectTrigger>
            <SelectContent>
              <SelectItem value="apartment">Apartment</SelectItem>
              <SelectItem value="villa">Villa</SelectItem>
              <SelectItem value="plot">Plot</SelectItem>
              <SelectItem value="commercial">Commercial</SelectItem>
            </SelectContent>
          </Select>
          {errors.propertyType && <p className="text-sm text-red-600">{errors.propertyType}</p>}
        </div>
        
        <div className="space-y-1.5">
          <label className="text-sm font-medium">Location</label>
          <div className="relative">
            <MapPin className="absolute left-3 top-2.5 w-4 h-4 text-gray-400" />
            <Input
              placeholder="e.g., Koramangala, Bangalore"
              value={formData.location}
              onChange={(e) => setFormData(prev => ({ ...prev, location: e.target.value }))}
              className={`pl-10 ${errors.location ? "border-red-300" : ""}`}
            />
          </div>
          {errors.location && <p className="text-sm text-red-600">{errors.location}</p>}
        </div>
        
        <div className="space-y-1.5">
          <label className="text-sm font-medium">Area (sq ft)</label>
          <Input
            type="number"
            placeholder="Enter area in sq ft"
            value={formData.area || ''}
            onChange={(e) => setFormData(prev => ({ ...prev, area: parseInt(e.target.value) || 0 }))}
            min="100"
            max="50000"
            className={`${errors.area ? "border-red-300" : ""} [&::-webkit-outer-spin-button]:appearance-none [&::-webkit-inner-spin-button]:appearance-none [-moz-appearance:textfield]`}
          />
          {errors.area && <p className="text-sm text-red-600">{errors.area}</p>}
        </div>
        
        <div className="space-y-1.5">
          <label className="text-sm font-medium">Bedrooms</label>
          <Select 
            value={formData.bedrooms} 
            onValueChange={(value) => setFormData(prev => ({ ...prev, bedrooms: value }))}
          >
            <SelectTrigger className={errors.bedrooms ? "border-red-300" : ""}>
              <SelectValue placeholder="Select bedrooms" />
            </SelectTrigger>
            <SelectContent>
              <SelectItem value="1">1 BHK</SelectItem>
              <SelectItem value="2">2 BHK</SelectItem>
              <SelectItem value="3">3 BHK</SelectItem>
              <SelectItem value="4">4+ BHK</SelectItem>
            </SelectContent>
          </Select>
          {errors.bedrooms && <p className="text-sm text-red-600">{errors.bedrooms}</p>}
        </div>

        {/* Advanced Options Toggle */}
        <div className="pt-1 border-t mt-2">
          <button
            type="button"
            className="w-full flex items-center justify-between p-2 hover:bg-gray-50 rounded transition-colors"
            onClick={() => setShowAdvanced(!showAdvanced)}
          >
            <span className="text-sm font-medium text-gray-700">Advanced Details (Optional)</span>
            {showAdvanced ? <ChevronUp className="w-4 h-4 text-gray-500" /> : <ChevronDown className="w-4 h-4 text-gray-500" />}
          </button>
          {!showAdvanced && <p className="text-xs text-gray-500 text-center mt-0.5">For more accurate valuation</p>}
        </div>

        {/* Advanced Fields */}
        {showAdvanced && (
          <div className="space-y-2.5 pb-2">
            <div className="grid grid-cols-2 gap-2">
              <div>
                <label className="text-xs font-medium text-gray-700">Bathrooms</label>
                <Select
                  value={formData.bathrooms?.toString()}
                  onValueChange={(value) => setFormData(prev => ({ ...prev, bathrooms: parseInt(value) }))}
                >
                  <SelectTrigger className="h-8 text-sm mt-1">
                    <SelectValue placeholder="Select bathrooms" />
                  </SelectTrigger>
                  <SelectContent>
                    <SelectItem value="1">1</SelectItem>
                    <SelectItem value="2">2</SelectItem>
                    <SelectItem value="3">3</SelectItem>
                    <SelectItem value="4">4</SelectItem>
                    <SelectItem value="5">5+</SelectItem>
                  </SelectContent>
                </Select>
              </div>
              <div>
                <label className="text-xs font-medium text-gray-700">Age (years)</label>
                <Select
                  value={formData.age?.toString()}
                  onValueChange={(value) => setFormData(prev => ({ ...prev, age: parseInt(value) }))}
                >
                  <SelectTrigger className="h-8 text-sm mt-1">
                    <SelectValue placeholder="Select" />
                  </SelectTrigger>
                  <SelectContent>
                    <SelectItem value="0">New (0-1)</SelectItem>
                    <SelectItem value="3">3-5 years</SelectItem>
                    <SelectItem value="7">5-10 years</SelectItem>
                    <SelectItem value="12">10-15 years</SelectItem>
                    <SelectItem value="20">15+ years</SelectItem>
                  </SelectContent>
                </Select>
              </div>
            </div>

            <div className="grid grid-cols-2 gap-2">
              <div>
                <label className="text-xs font-medium text-gray-700">Floor No.</label>
                <Select
                  value={formData.floor?.toString()}
                  onValueChange={(value) => setFormData(prev => ({ ...prev, floor: parseInt(value) }))}
                >
                  <SelectTrigger className="h-8 text-sm mt-1">
                    <SelectValue placeholder="Select" />
                  </SelectTrigger>
                  <SelectContent>
                    <SelectItem value="0">Ground</SelectItem>
                    <SelectItem value="1">1st</SelectItem>
                    <SelectItem value="2">2nd</SelectItem>
                    <SelectItem value="3">3rd</SelectItem>
                    <SelectItem value="4">4th</SelectItem>
                    <SelectItem value="5">5th</SelectItem>
                    <SelectItem value="6">6th</SelectItem>
                    <SelectItem value="7">7th</SelectItem>
                    <SelectItem value="8">8th</SelectItem>
                    <SelectItem value="9">9th</SelectItem>
                    <SelectItem value="10">10th+</SelectItem>
                  </SelectContent>
                </Select>
              </div>
              <div>
                <label className="text-xs font-medium text-gray-700">Total Floors</label>
                <Select
                  value={formData.totalFloors?.toString()}
                  onValueChange={(value) => setFormData(prev => ({ ...prev, totalFloors: parseInt(value) }))}
                >
                  <SelectTrigger className="h-8 text-sm mt-1">
                    <SelectValue placeholder="Select" />
                  </SelectTrigger>
                  <SelectContent>
                    <SelectItem value="1">1</SelectItem>
                    <SelectItem value="2">2</SelectItem>
                    <SelectItem value="3">3</SelectItem>
                    <SelectItem value="4">4</SelectItem>
                    <SelectItem value="5">5</SelectItem>
                    <SelectItem value="7">7</SelectItem>
                    <SelectItem value="10">10</SelectItem>
                    <SelectItem value="15">15</SelectItem>
                    <SelectItem value="20">20+</SelectItem>
                  </SelectContent>
                </Select>
              </div>
            </div>

            <div>
              <label className="text-xs font-medium text-gray-700">Furnishing</label>
              <Select
                value={formData.furnishing}
                onValueChange={(value) => setFormData(prev => ({ ...prev, furnishing: value }))}
              >
                <SelectTrigger className="h-8 text-sm mt-1">
                  <SelectValue />
                </SelectTrigger>
                <SelectContent>
                  <SelectItem value="unfurnished">Unfurnished</SelectItem>
                  <SelectItem value="semi_furnished">Semi Furnished</SelectItem>
                  <SelectItem value="fully_furnished">Fully Furnished</SelectItem>
                </SelectContent>
              </Select>
            </div>

            <div>
              <label className="text-xs font-medium text-gray-700">Facing</label>
              <Select
                value={formData.facing}
                onValueChange={(value) => setFormData(prev => ({ ...prev, facing: value }))}
              >
                <SelectTrigger className="h-8 text-sm mt-1">
                  <SelectValue placeholder="Select" />
                </SelectTrigger>
                <SelectContent>
                  <SelectItem value="north">North</SelectItem>
                  <SelectItem value="south">South</SelectItem>
                  <SelectItem value="east">East</SelectItem>
                  <SelectItem value="west">West</SelectItem>
                  <SelectItem value="northeast">North-East</SelectItem>
                  <SelectItem value="northwest">North-West</SelectItem>
                  <SelectItem value="southeast">South-East</SelectItem>
                  <SelectItem value="southwest">South-West</SelectItem>
                </SelectContent>
              </Select>
            </div>

            <div>
              <label className="text-xs font-medium text-gray-700 mb-1 block">Amenities ({formData.amenities?.length || 0})</label>
              <div className="grid grid-cols-2 gap-1 max-h-28 overflow-y-auto p-2 border rounded bg-gray-50">
                {availableAmenities.map((amenity) => (
                  <label key={amenity} className="flex items-center gap-1.5 cursor-pointer hover:bg-white p-1 rounded">
                    <input
                      type="checkbox"
                      checked={formData.amenities?.includes(amenity)}
                      onChange={() => toggleAmenity(amenity)}
                      className="rounded border-gray-300 w-3 h-3 flex-shrink-0"
                    />
                    <span className="text-xs leading-tight">{amenity}</span>
                  </label>
                ))}
              </div>
            </div>
          </div>
        )}
        
        </div>
      </div>
      
      <div className="border-t p-4 flex-shrink-0 bg-white">
        <form onSubmit={handleSubmit}>
          <Button 
            type="submit"
            disabled={!formData.propertyType || !formData.location.trim() || !formData.area || !formData.bedrooms}
            className="w-full bg-white hover:bg-gray-100 text-black border-2 border-emerald-600 disabled:opacity-50 disabled:cursor-not-allowed"
          >
            Calculate Valuation
          </Button>
        </form>
        <p className="text-xs text-gray-500 text-center mt-2">
          Estimates based on market analysis
        </p>
      </div>
    </div>
  );

  const renderProcessing = () => (
    <div className="bg-white rounded-xl shadow-2xl w-full max-w-md p-8 text-center">
      <div className="animate-spin w-12 h-12 border-4 border-emerald-200 border-t-emerald-600 rounded-full mx-auto mb-4"></div>
      <h3 className="text-lg font-semibold mb-2">Analyzing Property</h3>
      <p className="text-gray-600">Processing market data and comparable properties...</p>
    </div>
  );

  const renderResults = () => {
    if (!result) return null;
    
    return (
      <div className="bg-white rounded-xl shadow-2xl w-full max-w-2xl max-h-[85vh] overflow-y-auto">
        <div className="flex items-center justify-between p-4 border-b">
          <h2 className="text-lg font-semibold">Valuation Report</h2>
          <Button variant="ghost" size="sm" onClick={handleClose}>
            <X className="w-4 h-4" />
          </Button>
        </div>
        
        <div className="p-4 space-y-3">
          {/* Main Valuation */}
          <div className="bg-emerald-50 p-4 rounded-lg border border-emerald-200">
            <div className="text-center">
              <h3 className="text-xl font-bold text-emerald-800 mb-1">
                {formatCurrency(result.estimatedValue)}
              </h3>
              <p className="text-sm text-emerald-600 font-medium">Estimated Market Value</p>
              <p className="text-xs text-emerald-600 mt-0.5">
                {formatCurrency(result.pricePerSqft)}/sq ft
              </p>
            </div>
            
            <div className="mt-2 flex items-center justify-center gap-3">
              <div className="flex items-center gap-1">
                <div className={`w-2 h-2 rounded-full ${
                  result.confidence >= 80 ? 'bg-green-500' : 
                  result.confidence >= 70 ? 'bg-yellow-500' : 'bg-red-500'
                }`}></div>
                <span className="text-xs font-medium">{result.confidence}% Confidence</span>
              </div>
              <span className="text-xs text-emerald-700">
                Range: {formatCurrency(result.range.min)} - {formatCurrency(result.range.max)}
              </span>
            </div>
          </div>

          {/* ML Prediction Info */}
          {result.mlPrediction && (
            <div className="bg-gradient-to-r from-purple-50 to-blue-50 p-3 rounded-lg border border-purple-200">
              <div className="flex items-center justify-between">
                <div>
                  <div className="text-sm font-medium text-purple-800">AI-Enhanced Analysis</div>
                  <div className="text-xs text-purple-600">ML Prediction: {formatCurrency(result.mlPrediction)}</div>
                </div>
                <div className="text-right">
                  <div className="text-xs text-purple-600">ML Confidence: {Math.round((result.mlConfidence || 0) * 100)}%</div>
                  <div className="text-xs text-purple-500">{result.methodology}</div>
                </div>
              </div>
            </div>
          )}

          {/* Market Trends */}
          <div className="grid grid-cols-3 gap-2">
            <div className="text-center p-3 bg-blue-50 rounded-lg">
              <TrendingUp className="w-4 h-4 text-blue-600 mx-auto mb-1" />
              <div className="text-base font-bold text-blue-800">+{result.marketTrends.appreciation}%</div>
              <div className="text-xs text-blue-600">Annual Growth</div>
            </div>
            <div className="text-center p-3 bg-purple-50 rounded-lg">
              <Calculator className="w-4 h-4 text-purple-600 mx-auto mb-1" />
              <div className="text-base font-bold text-purple-800">{result.marketTrends.rentalYield}%</div>
              <div className="text-xs text-purple-600">Rental Yield</div>
            </div>
            <div className="text-center p-3 bg-orange-50 rounded-lg">
              <FileText className="w-4 h-4 text-orange-600 mx-auto mb-1" />
              <div className="text-base font-bold text-orange-800 capitalize">{result.marketTrends.demandSupply}</div>
              <div className="text-xs text-orange-600">Demand</div>
            </div>
          </div>

          {/* Comparables */}
          <div>
            <h4 className="text-sm font-semibold mb-2">Comparable Properties</h4>
            <div className="space-y-1.5">
              {result.comparables.map((comp, index) => (
                <div key={index} className="flex justify-between items-center p-2 bg-gray-50 rounded">
                  <div>
                    <div className="font-medium text-xs">{comp.address}</div>
                    <div className="text-xs text-gray-600">{comp.distance} km away</div>
                  </div>
                  <div className="text-right">
                    <div className="text-sm font-semibold">₹{comp.pricePerSqft?.toLocaleString('en-IN') || 'N/A'}/sq ft</div>
                  </div>
                </div>
              ))}
            </div>
          </div>

          {/* Actions */}
          <div className="flex gap-2 pt-2">
            <Button 
              variant="outline" 
              className="flex-1 text-sm py-2"
              onClick={generatePDF}
            >
              <Download className="w-4 h-4 mr-1" />
              Download Report
            </Button>
            <Button 
              variant="outline" 
              className="flex-1 text-sm py-2"
              onClick={() => window.location.href = '/valuation-reports'}
            >
              Get Detailed Report
            </Button>
            <Button 
              className="flex-1 bg-white hover:bg-gray-100 text-black border-2 border-emerald-600 text-sm py-2"
              onClick={handleClose}
            >
              <ArrowRight className="w-4 h-4 mr-1" />
              Continue
            </Button>
          </div>
        </div>
      </div>
    );
  };

  const renderError = () => (
    <div className="bg-white rounded-xl shadow-2xl w-full max-w-md p-6 text-center">
      <AlertCircle className="w-16 h-16 text-red-500 mx-auto mb-4" />
      <h3 className="text-lg font-semibold mb-2 text-red-800">Valuation Failed</h3>
      <p className="text-gray-600 mb-4">{errorMessage}</p>
      
      {errorMessage.includes('timeout') && (
        <div className="bg-yellow-50 border border-yellow-200 rounded-lg p-3 mb-4">
          <p className="text-sm text-yellow-800">
            The service is experiencing high demand. Please try again in a few moments.
          </p>
        </div>
      )}
      
      {errorMessage.includes('location') && (
        <div className="bg-blue-50 border border-blue-200 rounded-lg p-3 mb-4">
          <p className="text-sm text-blue-800">
            Try entering a more specific location (e.g., "Koramangala, Bangalore")
          </p>
        </div>
      )}
      
      <div className="flex gap-3">
        <Button variant="outline" onClick={() => setStep('form')} className="flex-1">
          Try Again
        </Button>
        <Button onClick={handleClose} className="flex-1">
          Close
        </Button>
      </div>
    </div>
  );

  return (
    <div className="fixed inset-0 bg-black/50 backdrop-blur-sm z-50 flex items-center justify-center p-4">
      {step === 'form' && renderForm()}
      {step === 'processing' && renderProcessing()}
      {step === 'results' && renderResults()}
      {step === 'error' && renderError()}
    </div>
  );
}